/**********************************************************
 *
 * Snare for Solaris version 2.1
 *
 * Author: InterSect Alliance Pty Ltd
 *
 * Copyright 2001-2003 InterSect Alliance Pty Ltd
 *
 * Last Modified: 29/8/2003
 *
 * Available under the terms of the GNU Public Licence.
 * - See www.gnu.org
 *
 **********************************************************
 *
 * Snare for Solaris is a user-space program
 * that interacts with the audit facility within the
 * solaris kernel using the auditsvc() system call.
 *
 * The program takes the output of the auditsvc call,
 * pipes it through the praudit command, and sends the
 * resulting text data over the network to a remote server
 * using UDP.
 *
 * The process also monitors the praudit process, and
 * respawns praudit if a segmentation fault occurs.
 *
 * Backlog also only responds to a very limited number of
 * signals, and so is unlikely to be killed by minor service
 * problems.
 *
 **********************************************************
 *
 * History:
 *      23/02/2001  Initial working version
 *       2/03/2001  Cleaned up error information.
 *       9/03/2001  Added signal handler to check for
 *                  praudit failure, and restart the service.
 *      22/03/2001  Set up each process to ignore most signals.
 *      23/03/2001  Oedipus update: Child kills parent if a
 *                  terminal error occurs in child.
 *      21/12/2001  Now sends a stop signal to the audit subsystem on exit.
 *                  This is to deal with the buggy Sol 8 +cnt flag.
 *       2/ 8/2002  Can now dynamically turn on audit events.
 *       8/ 8/2002  Started integrating SNARE functionality in order to
 *                  facilitate remote control.
 *       9/ 8/2002  Event start/stop in response to audit configuration file.
 *       1/11/2002  Web server functionality bought over from Windows.
 *      20/12/2002  Finalising code modifications for alpha version.
 *      24/ 1/2003  Added comments relating to the requirement for
 *                  /etc/security/audit_control and /etc/security/audit_class
 *                  to be available, otherwise new processes will NOT
 *                  inherit audit class settings.
 *      15/ 8/2003  Added the User / Groups / Group Members capabilities
 *                  to the micro-embedded web server.
 *                  Fixed header returns to work with older netscapes.
 *      24/ 8/2003  Reduced CPU usage of snarecore by changing the way
 *                  information is read from the praudit pipe.
 *
 **********************************************************
 *
 * Compilation Instructions:
 *    gcc -o snarecore snaresolaris.c webserver.c WebPages.c -lbsm -lsocket -lnsl -lintl
 *
 **********************************************************/


#include <sys/types.h> 
#include <sys/wait.h> 
#include <sys/errno.h>
#include <sys/fcntl.h>
#include <sys/termios.h>
#include <sys/socket.h>
#include <bsm/audit.h>
#include <bsm/audit_kevents.h>
#include <bsm/audit_uevents.h>
#include <bsm/audit_record.h>
#include <bsm/libbsm.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <limits.h>
#include <dirent.h>

#include <regex.h>

#include "snaresolaris.h"
#include "webserver.h"

////// DEBUG
#include <stdarg.h>
void DebugMsg(const char* pszFormat, ...);
////// END DEBUG

// Linked List Functions
static Node *head, *currentnode;
static int AuditDestination=AUDIT_TO_STDOUT;            // Send to STDOUT unless otherwise stated.

char hostid[MAX_HOSTID]="";
int remote_allow=0;
int remote_webport=6161;
char remote_restrictip[16]="";
char remote_password[256]="";

FILE * AuditFile=(FILE *) NULL;
int AuditSocket=0;
struct sockaddr_in      AuditSocketName;

/* Define an error variable so we can communicate with auditsvc */
extern int errno;
int caught_pipe;                /* variable that lets us know when a SIGPIPE has been received */
int caught_kill;                /* variable that lets us know when a SIGKILL has been received */

void pipe_signal(sig)
{
	caught_pipe++;
	return;
}

void kill_signal(sig)
{
	caught_kill++;
	return;
}

void audit_off()
{
	static int auditingoff = AUC_NOAUDIT;
	auditon(A_SETCOND, (caddr_t) & auditingoff, sizeof (int));
}

int audit_on()
{
	static int auditingon = AUC_AUDITING;
	if(auditon(A_SETCOND, (caddr_t) & auditingon, sizeof (int)) < 0) {
		return(0);
	}
	return(1);
}

int main(int argc, char *argv[])
{
	/* General program related variables */
	char cache[CACHESIZE] = "";
	int bytesread;			/* number of bytes read from the pipe */
	int daemon2praudit[2];		/* Pipe from the daemon to praudit */
	int praudit2daemon[2];		/* Pipe from praudit back to the daemon */
	char logbuffer[8192];	/* Holds the data from praudit */
	int logbuffercount=0;		/* current position in the log buffer */
	int startlogbuffer=0;		/* the starting point in the log buffer for general audit log information */
	int error;
	int continueloop=1;			/* Make sure we restart praudit if the pipe fails */

	/* Signal related variables */
	sigset_t signalset;

	/* Variables that relate to the network socket */
	struct hostent *hp;		/* Host Pointer */
	struct sockaddr_in socketname;
	struct in_addr ia;

	/* Audit related variables */
	auditinfo_t auditinfo;
	au_id_t euid;
	int fd;				/* file descriptor */
	int auditstatus;
	long auditpolicy=0;
	int pid,pid2,ppid=0;			/* process id */
	int webpid=0;
	
	// NODE that matches the objective search term.
	Node * nodematch;

	/* The only person who can run auditsvc is root, so lets
	   make sure that we discourage other users */
	if(getuid() != 0)
	{
		fprintf(stderr,"This program can only be run by root.\n");
		exit(1);
	}

	/* Make sure that we do not get into a endless audit loop
	   - turn off auditing for this process completely. */

	getaudit(&auditinfo);
	auditinfo.ai_mask.as_success = 0;
	auditinfo.ai_mask.as_failure = 0;
	setaudit(&auditinfo);

	/* However, if the administrator has set the system up as
	   per sun recommendations (ie: creating an audit user with
	   a UID of 0), then set our effective uid to 'audit'.
	   That way, if the administrator modifies
	   /etc/security/audit_user and tries to force root to always
	   audit some events, then the audit user will still be excluded.
	   It would be much better to do a setauid(AU_NOAUDITID) as
           indicated in the solaris header information, but 
	   AU_NOAUDITID is not type uid_t so we cannot. */
	
	euid=geteuid();
	setauid(&euid);

	/* This does not work. Leaving here for reference purposes */
	/* setauid(AU_NOAUDITID); */

	/* Release the controlling terminal from this program */
	setpgrp();
	if((fd=open("/dev/tty",0,O_RDWR)) > 0) {
		ioctl(fd,TIOCNOTTY,0);
		close(fd);
	}

	if(auditon(A_GETCOND,(caddr_t)&auditstatus,sizeof(auditstatus)) == -1)
	{
		printf("Could not get auditing condition from the kernel.\nUnable to determine whether auditing has been enabled or not.\n");
		exit(1);
	} else {

		/* Lets make sure that a exec arguments are included in the
		   audit information, as well as sequence numbers. */
		if(auditon(A_GETPOLICY,(caddr_t)&auditpolicy,sizeof(auditpolicy)) == -1)
		{
			fprintf(stderr,"Could not get auditing condition from the kernel.\nUnable to query the current audit policy.\n");
			exit(1);
		} else {
			/* Turn on some specific audit conditions. */
			auditpolicy = auditpolicy | AUDIT_ARGV | AUDIT_SEQ | AUDIT_CNT;
			if(auditon(A_SETPOLICY,(caddr_t)&auditpolicy,sizeof(auditpolicy)) == -1)
			{
				fprintf(stderr,"Could not set the audit policy to include exec, execv and sequence data in audit information.\n");
				exit(1);
			}
		}
	}




	/* If the kernel is letting us read the audit condition */
	if(auditstatus == AUC_DISABLED)
	{
		fprintf(stderr,"Auditing has not been enabled on this machine!\n");
		exit(1);
	}

	/* Set the class of all active processes to our custom class. */
	setclass(0x10000000);


	/* Make sure that we divorce ourselves from the parent process
	   as we may do not want to inherit signals
	   Note that we will definately be divorced from the parent tty
	   after this, so error messages to stdout will not be particularly
	   useful. */

	if(fork())	{
		/* This is the parent process - exit, and ditch the parent.
		   We will be inherited by process 1 (init) */
		exit(0);
	}

	// Read our configuration file, and set up audit.
	read_config_file(0);
		
	if(!hostid) {
		getfqdn(hostid);
	} else {
		if(strlen(hostid)==0) {
			getfqdn(hostid);
		}
	}


	/* Work out the current process ID */
	ppid=getpid();
	if(!ppid)
	{
		fprintf(stderr,"Cannot find current process ID. Exiting.\n");
		audit_off();
		exit(1);
	} else {
		/* Write out process ID */
		FILE *pidfile;
		if(pidfile = fopen("/var/run/snarecore.pid","w")) {
			fprintf(pidfile,"%d",ppid);
			fclose(pidfile);
		}
	}

	
	
	 
	/* ceci n'est pas une pipe */
	if (pipe(daemon2praudit) == -1 || pipe(praudit2daemon) == -1)
	{
		fprintf(stderr,"Cannot open pipe for audit information\n");
		kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
		audit_off();
		exit(1);
	}

	/* This is effectively a while(true) */
	while(continueloop)
	{
		/* Ok. We should be just about ready to audit now. */
		/* Lets tell the kernel. */
	
		if (!audit_on()) {
			fprintf(stderr,"Cannot start auditing!\n");
			kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
        		exit(1);
		}

		/* Configure / Reinstate signals */
		/* Trap signals relating to PIPE failures,
		    and Child process termination */
		/* Reset this each time through the loop, just in case a signal handler resets our values */
		
		if(!setsignals(&signalset)) {
			fprintf(stderr,"Cannot set important signals - exiting\n");
			kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
			audit_off();
			continueloop=0;
			break;
		}
	
		
		if(remote_allow) {
			webpid=fork();
			if(webpid==0) {
				int ret=0;
				// Child process - Run our web server
				while(InitWebServer(remote_webport,remote_restrictip,remote_password) == -1) {
					// Problem binding. Sleep for a moment.
					sleep(5);
				}
				while((ret = StartThread()) == 1) {
					// Problem binding. Keep trying.
					sleep(5);
				}
				exit(0);
			}
		}
		
		
		/* Fork again here */
		/* The child process leads to the praudit/auditsvc processes,
		   and the other continues down to reading/writing from the
		   pipe that joins praudit to the udp sending module */
		/* Note that the auditsvc process needs to be the parent process
		   of any subsidary forks - that way, if praudit dies due to a bug
		   (as is common in the sun BSM implementation - I have discovered
		    three bugs so far of various criticality in auditd and praudit),
		   then at least we will be able to signal the auditsvc fork to
		   terminate also, as we have the process ID */

		pid=fork();
		if(pid==-1)
		{
			fprintf(stderr,"Could not fork a process to run praudit\n");
			kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
			audit_off();
			break;
		} else if(pid == 0) {
			FILE *pidfile;
			int prauditpid;

			/* Fork child */
			switch (pid2=fork())
			{
			  case -1:
				fprintf(stderr,"fork failure. Cannot start the praudit and auditsvc fork.\n");
				kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
				audit_off();
				exit(1);
		
			  case 0:
				/* Child process section */
				/* Duplicate child's reading end of daemon2praudit into stdin,
				 and child's writing end of praudit2daemon into stdout */
				if (dup2(daemon2praudit[0], STDIN_FILENO) ==-1 ||
				    dup2(praudit2daemon[1], STDOUT_FILENO)==-1
				    )
				{
					fprintf(stderr,"Could not reroute stdin and stdout for praudit\n");
					audit_off();
					kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
					exit(1);
				}

				// Close unused fildes.
				close(daemon2praudit[0]);
				close(daemon2praudit[1]);
				close(praudit2daemon[0]);
				close(praudit2daemon[1]);
						
				execlp("/usr/sbin/praudit", "praudit", "-l", "-d	", (char *)0);

				// We should never actually get here
				fprintf(stderr,"Could not exec praudit\n");
				audit_off();
				kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
				exit(1);
				
			  default:
				/* Parent section */
				/* Close unused fildes */
				close(daemon2praudit[0]);
				close(praudit2daemon[1]);
			
				/* Loop for as long as we don't receive a signal */
				while(1)
				{
					/* Receive the data from auditsvc */
					/* args: file, minimum free blocks */
					/* try a min free block count of zero */
					/* Note that the audit-to-pipe patch had
					   better be installed before we make this
					   system call, or the machine will halt! */


					auditsvc(daemon2praudit[1],0);
					error=errno;

					/* ignore non fatal errors at this time. */
					if(error==EINVAL)
					{
						/* Auditing dissapeared from under us? */
						fprintf(stderr,"SNARE reports that auditing has been turned off. Exiting.\n");
						// Just in case.
						audit_off();
						unlink("/var/run/snarecore.pid");
						kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
						exit(1);
					} else if(error == EBUSY) {
						/* Auditing is already being performed by another process */
						fprintf(stderr,"Another process has control of audit.\n");
						audit_off();
						if(webpid) {
							kill(webpid,SIGTERM);
						}
						kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
						exit(1);
					} else if(error == EINTR) {
						fprintf(stderr,"Interrupt received.\n");
						/* Signal handler has been tripped.
						   This is likely to mean that the
						   praudit program has died.
						   Kill this thread and try to restart */
						// Try to take out the praudit process, just in case.
						// kill(pid2,SIGTERM); waitpid(pid2,NULL,NULL);
						exit(0);
					} else if(error == EBADF) {
						// Pipe has been closed.
						// Try to take out the praudit process, just in case.


						kill(pid2,SIGTERM); waitpid(pid2,NULL,NULL);
						exit(1);
	
					} else {
						sleep(1);	// Limit respawns for no reason
					}
				}
			} 
		} else {
			/* Setup a kill signal handler so that when the primary
			   terminates, it closes nicely and takes the other forks
			    down with it */
			if(signal(SIGTERM,kill_signal) == SIG_ERR)
			{
				fprintf(stderr,"Cannot set signal SIGTERM\n");
				audit_off();
				kill(ppid,SIGTERM); waitpid(pid,NULL,NULL);
				exit(1);
			}

			logbuffercount=0;
			char *newline=logbuffer;
			char *endbuffer = logbuffer+sizeof(logbuffer)-1;
			char *bufferptr = logbuffer;
			long spaceavail=sizeof(logbuffer)-1;
			const long buffersize=sizeof(logbuffer)-1;

			char username[MAX_USERNAME];
			char searchterm[MAX_PATH];
			char eventname[MAX_EVENTNAME];
			char options[MAX_OPTIONS]="";
			char *startrecord=logbuffer;
			int returncode;

			char *tbuffer;

			// while ((bytesread=read(praudit2daemon[0],bufferptr,spaceavail) >0) && continueloop)
			while (continueloop)
			{
				if(spaceavail==0) {
					// Oh dear.. filled our buffer without seeing a newline. Must be filled with corrupt data from praudit.
					// Clear it.
					bufferptr=logbuffer;
					spaceavail=buffersize;
				}

				bytesread=read(praudit2daemon[0],bufferptr,spaceavail);
				if(!bytesread) {
					break;
				}

				if(bytesread < 1) continue;

				tbuffer=bufferptr+bytesread-1;
				tbuffer++;*tbuffer=0;

				// While there are more newlines in the buffer
				startrecord=logbuffer;
				do {
					newline=strstr(startrecord,"\n");
					if(!newline) { break; }

					*newline='\0';
					newline++;

					/* received a newline from praudit. */
					/* Time to send the udp packet out. */
					/* make sure the newline is included */
					/* This is where we implement the SNARE event checker process */
					returncode = GetSolDetails(startrecord,username,searchterm,eventname,options);
					while((nodematch=CheckObjective(username,searchterm,eventname,options,returncode)) != (Node *) NULL) {

						FormatDelimiters(startrecord);

						if(sendevent(startrecord,nodematch->criticality) != 0)
						{
							/* Report an error, but continue */
							fprintf(stderr,"error sending log entry to destination\n");
						}
					}

					startrecord=newline;
				} while(newline);

				if(startrecord != logbuffer) {
					int copysize=bytesread-(startrecord-bufferptr);
					memcpy(logbuffer,startrecord,copysize);
					bufferptr=logbuffer+copysize;
					*bufferptr=0;
					spaceavail=endbuffer-bufferptr;
				} else {
					bufferptr+=bytesread;
					spaceavail-=bytesread;
				}
			}

			if(caught_kill) {
				caught_kill=0;
				/* Someone has terminated the parent process. */
				/* Take our other processes down with it. */
				/* Kill the other child process (auditsvc),
				   and restart praudit and auditsvc */
				audit_off();

				kill(pid,SIGTERM);
				/* Wait for the process to die */
				waitpid(pid,NULL,NULL);
				close(daemon2praudit[1]);		
		
				continueloop=0;
				if(webpid) {
					kill(webpid,SIGTERM);
				}
				
				if(AuditSocket) {
					close(AuditSocket);
				}

				DestroyList();
				unlink("/var/run/snarecore.pid");
			
				/* All done */
				return 0;
			} else if(caught_pipe) {


				/* Oh dear, our child process has been terminated. */
				caught_pipe=0;
				/* Kill the other child process (auditsvc),
				   and restart praudit and auditsvc */
				kill(pid,SIGTERM);
				/* Wait for the process to die */
				waitpid(pid,NULL,NULL);
				/* Also send out notification to our central
				   collection system */
				/* our logbuffer variable will be available for our use. */
				if(sendevent("Solaris praudit has died unexpectedly. Restarting the praudit process.",0) != 0)
				{
					/* Report an error, but continue */
					fprintf(stderr,"error sending datagram\n");
				}
				/* Sleep for a moment to limit gratuitous respawns */
				sleep(1);
			}
		}
	}

	/* We will never actually get here. The only way we can die is with signals. */
	/* Files will be automatically cleaned up by the operating system in this case. */
	close(daemon2praudit[1]);
	// close(udpsocket);
	close(AuditSocket);
	
	DestroyList();
	unlink("/var/run/snarecore.pid");
			
	/* All done */
	audit_off();
	return 0;
}


int setsignals(sigset_t *signalset)
{
	sigfillset(signalset);
	sigdelset(signalset,SIGTERM);
	sigdelset(signalset,SIGALRM);
	sigdelset(signalset,SIGINT);
	sigdelset(signalset,SIGPIPE);
	sigdelset(signalset,SIGCHLD);
	sigprocmask(SIG_BLOCK,signalset, (void *) NULL);

	if(signal(SIGPIPE,pipe_signal) == SIG_ERR)
	{
		fprintf(stderr,"Cannot set signal SIGPIPE\n");
		return(0);
	}
		
	if(signal(SIGCHLD,pipe_signal) == SIG_ERR)
	{
		fprintf(stderr,"Cannot set signal SIGCHLD\n");
		return(0);
	}

	return(1);
}

int turn_event_on(int eventnumber)
{
	//struct au_evclass_map map;
	au_evclass_map_t map;
	int returncode;

	// Cannot turn on events above 512. Must rely on audit_class.
	if(eventnumber > 512) {
		return(0);
	}

	map.ec_number=eventnumber;
	map.ec_class=0x10000000;	// Custom class (ie: 'snare')
	
	returncode=auditon(A_SETCLASS,(caddr_t)&map,sizeof(au_evclass_map_t));

	if(returncode == -1) {
		printf("AUDITON FAILURE for event number %d:",eventnumber);
		perror("");
	}
}

int turn_event_off(int eventnumber)
{
	au_evclass_map_t map;
	map.ec_number=eventnumber;
	map.ec_class=0x00000000;	// No class
	
	auditon(A_SETCLASS,(caddr_t)&map,sizeof(au_evclass_map_t));
}


// NOTE: DO NOT turn on auditing for NSCD - the solaris system WILL DIE.
//
// NOTE2: In order for this to work for NEWLY created processes, the
// snare class must be listed in /etc/security/audit_control (and
// therefore, in /etc/security/audit_class).
// It looks as though the sun engineers used a severe HACK to get
// audit working. Rather than inheriting the parent processes audit
// class, every newly created process seems to check
// /etc/security/audit_control and /etc/security/audit_class, and
// applies the appropriate audit_class.
// Yuk.. At least it lets them divorce auditing from a standard kernel
// I guess.

int setclass(unsigned int class)
{
	au_mask_t mask;
	auditinfo_t ai;
	int c;
	int auditstatus;
	int pid;
	int pspid;

	DIR *procdir;
	struct dirent *dp;            /* for readdir() return values */

	struct auditpinfo pinfo;

	mask.am_success = class;
	mask.am_failure = class;
	
	// auditon(A_SETKMASK,(caddr_t)&mask,sizeof(au_mask_t));

	if ((procdir = opendir ("/proc")) == NULL)
	{
		perror("Cannot set audit on process list\n");
		exit(2);
	}

	pid=getpid();
	while ((dp = readdir (procdir)) != NULL) /* now loop over processes */
	{
		// Ignore . and ..
		if (dp->d_name[0] == '.') continue;
		pspid=atoi(dp->d_name);

		pinfo.ap_pid = pspid;

/* THIS IS ANOTHER WAY TO DO A SIMILAR THING to the code below.
   - HOWEVER, we need to be sure not to apply audit to
     NSCD, as it seems to kill solaris.
// DEBUG ONLY 
#include <sys/proc.h>
#include <procfs.h>
{
	int psdata;
	struct psinfo psbuf;
	char fname[1024];

	snprintf(fname,sizeof(fname),"/proc/%s/psinfo",dp->d_name);
 	if( (psdata = open( fname, O_RDONLY )) != -1 ){ 
		read(psdata, (void *) &psbuf, sizeof(struct psinfo) );
		close(psdata);
		if(strcmp(psbuf.pr_fname,"nscd") != 0) {

			if(pspid != pid) {
				pinfo.ap_pid = pspid;
				pinfo.ap_mask.am_success=class;
				pinfo.ap_mask.am_failure=class;
				auditon(A_SETPMASK,(caddr_t)&pinfo,sizeof(pinfo));
			}
		}
	}
}*/


		auditon(A_GETPINFO,(caddr_t)&pinfo,sizeof(pinfo));

		// Only turn on auditing for those processes that Solaris has already set an audit mask for.
		// NOTE: This means that /etc/security/audit_startup needs to be run before snare starts, or we won't see any audit.
		if(pspid != pid && (pinfo.ap_mask.am_success || pinfo.ap_mask.am_failure)) {
			pinfo.ap_pid = pspid;
			pinfo.ap_mask.am_success=class;
			pinfo.ap_mask.am_failure=class;
			auditon(A_SETPMASK,(caddr_t)&pinfo,sizeof(pinfo));
		} else if(pspid == pid) {
			// Turn off audit for our own process though!
			pinfo.ap_pid = pspid;
			pinfo.ap_mask.am_success=0x00000000;
			pinfo.ap_mask.am_failure=0x00000000;
			auditon(A_SETPMASK,(caddr_t)&pinfo,sizeof(pinfo));
		}
	}
}

// Turn off all audit events in preparation for updating our audit policy.
int flush_audit_events()
{
	int count;

	for(count=0;count<=MAXAUDIT;count++) {
		turn_event_off(count);
	}
}


// Shortread: don't re-read all the objectives - just the general configuration items.
int read_config_file(int shortread)
{
	FILE * configfile = (FILE *)NULL;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];	// Should be enough for most config lines.
							// Would love to use gchar here instead, but need to keep this simple.
	// Config file header.
	int headertype=0;
	
	// Clear some default variables.
	remote_allow=0;
	remote_webport=6161;
	strncpy(remote_restrictip,"",sizeof(remote_restrictip));
	strncpy(remote_password,"",sizeof(remote_password));

	// Clear the audit destination
	AuditDestination=0;
	
	configfile=fopen(CONFIG_FILENAME,"r");

	if(configfile==(FILE *)NULL) {
		printf("Cannot open audit configuration file.");
		return(0);
	}

	while(fgets(inputbuffer,MAX_AUDIT_CONFIG_LINE,configfile)) {
		// Kill whitespace from start and end of line.
		trim(inputbuffer);
		
		if(!iscomment(inputbuffer))	{
			// Is this line a header?
			if(isheader(inputbuffer)) {
				headertype=getheader(inputbuffer);
			} else {
				if(headertype == CONFIG_OBJECTIVES) {
					// Save off enough space to store the data we need.
					char event[MAX_AUDIT_CONFIG_LINE], user[MAX_AUDIT_CONFIG_LINE], path[MAX_AUDIT_CONFIG_LINE];
					char event2[MAX_AUDIT_CONFIG_LINE], options[MAX_OPTIONS]="";
					char *eventpointer=NULL;
					char *eventpointer2=NULL;
					int criticality;
					int eventnumber;
					int returncode;
					int excludeflag=0;

					// Do we need to skip over the objectives?
					if(shortread==1) {
						continue;
					}
					
					if((criticality = splitobjective(inputbuffer,event,user,path,&returncode,&excludeflag)) > -1) {
						// add the objective to the linked list.
						trimallwhitespace(event);

						// HERE: Split event into comma separated list of events				
						eventpointer=event;
						eventpointer2=event;
						
						// While there are no more commas
						while(eventpointer2) {
							eventpointer2=strstr(eventpointer,",");
							if(eventpointer2 == (char *)NULL) {
								// No commas left. Just copy to the end of the line.
								strncpy(event2,eventpointer,MAX_AUDIT_CONFIG_LINE);
							} else {
								strncpy(event2,eventpointer,(eventpointer2-eventpointer));
								// Make sure we have a null on the end of the line.
								event2[eventpointer2-eventpointer]='\0';
							}
							if(eventpointer2) {
								// Skip the comma
								eventpointer=eventpointer2+1;
							}
							// Clear out any options strings
							strcpy(options,"");
							if(regmatch(event2,"\\(.*\\)$")) {
								char *position;

								// Bonus, we have something. Grab it, add to options.
								// First, trim off the last bracket.
								event2[strlen(event2)-1]='\0';
								position=strstr(event2,"(");
								
								strncpy(options,position+1,MAX_OPTIONS);
								// truncate the event string at the bracket.
								*position='\0';
							}

							if((eventnumber = eventname2number(event2)) > -1) {
								AddToList(event2,user,path,criticality,options,excludeflag,returncode);

								// Also, ensure this event is turned on
								turn_event_on(eventnumber);

							} else {
								if(!strcmp(event,"*")) {
									AddToList("AUDIT_ALL",user,path,criticality,"",excludeflag,returncode);
								}
							}
						}
					} else {
						fprintf(stderr,"WARNING: Cannot process objective - please ensure configuration line\n\tcontains valid criticality, event, criticality, return, user and match elements\n\t%s\n",inputbuffer);
					}
				} else if(headertype == CONFIG_OUTPUT) {
					if(isfilename(inputbuffer)) {
						if(! open_audit_output(inputbuffer)) {
							fprintf(stderr,"WARNING: Could not open the file specified in the audit configuration: %s - Auditing to STDOUT\n",inputbuffer);
						}
					} else if(isnetwork(inputbuffer)) {
						if(! open_audit_network(inputbuffer)) {
							fprintf(stderr,"WARNING: Could not open the network specified in the audit configuration: %s - Auditing to STDOUT\n",inputbuffer);
						}
					}
				} else if(headertype == CONFIG_HOSTID) {
					gethostident(inputbuffer,hostid,MAX_HOSTID);
				} else if(headertype == CONFIG_REMOTE) {
					// Grab the remote control stuff here
					if(regmatchi(inputbuffer,"^allow=")) {
						if(regmatchi(inputbuffer,"=1")) {
							remote_allow=1;
						}
					} else if(regmatchi(inputbuffer,"^listen_port=")) {
						remote_webport=getport(inputbuffer);
					} else if(regmatchi(inputbuffer,"^restrict_ip=")) {
						if(!getconfstring(inputbuffer,remote_restrictip,sizeof(remote_restrictip))) {
							strncpy(remote_restrictip,"",sizeof(remote_restrictip));
						}
					} else if(regmatchi(inputbuffer,"^accesskey=")) {
						if(!getconfstring(inputbuffer,remote_password,sizeof(remote_password))) {
							strncpy(remote_password,"",sizeof(remote_password));
						}
					}
				} else {
					fprintf(stderr,"WARNING: Configuration file line does not fit in any recognised header.\n\t%s\n",inputbuffer);
				}
			}
		}
	}

	fclose((FILE *)configfile);
	return(1);
}

int iscomment(char * line)
{
	// Verify that there is something to check.
	if(line == (char *) NULL) return(1);
	
	// And that there is some data within.
	if(strlen(line) == 0 || strlen(line) > MAX_AUDIT_CONFIG_LINE) return(1);
	
	// If the first non-whitespace character is a hash, this is a comment line.
	while(*line) {
		// Space or tab or newline / formfeed
		if(*line == ' ' || *line == 9 || *line == 10 || *line == 12) {
			line++;
		} else if(*line == '#') {
				return(1);
		} else {
			// Ahh. A non-whitespace, non hash character.
			return(0);
		}
	}
	
	// If we are here, then the whole line must have been whitespace
	return(1);
}

int isheader(char *string)
{
	if(string[0] == '[' && string[strlen(string)-1] == ']') {
		return(1);
	}
	return(0);
}

int isfilename(char *string)
{
	if(regmatch(string,"^file=")) {
		return(1);
	} else {
		return(0);
	}	
}

// Network output
int isnetwork(char *string)
{
	if(regmatch(string,"^network=")) {
		return(1);
	} else {
		return(0);
	}	
}

int open_audit_output(char *string)
{
	string+=strlen("file=");
	if(strlen(string)) {
		if(regmatchi(string,"stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return(1);
		}
		AuditFile=fopen(string,"a");
		if(AuditFile != (FILE *) NULL) {
			AuditDestination |= AUDIT_TO_FILE;
			return(1);
		} else {
			AuditDestination |= AUDIT_TO_STDOUT;
			return(0);
		}
	}

	return(0);
}


// Open the network - destination host and port
int open_audit_network(char *string)
{
	char desthost[MAX_HOSTID];
	int port=6161;	// Default port
	char *pos;
	struct hostent *hp;

	// Initialise desthost.
	strncpy(desthost,"localhost",MAX_HOSTID);
	
	string+=strlen("network=");
	if(strlen(string)) {
		if(regmatchi(string,"stdout")) {
			AuditDestination |= AUDIT_TO_STDOUT;
			return(1);
		}

		AuditSocket = socket(AF_INET,SOCK_DGRAM,0);
		if(AuditSocket < 0)
		{
			fprintf(stderr,"Cannot open network socket - auditing to STDOUT\n");
			AuditDestination |=AUDIT_TO_STDOUT;
			return(0);
		}

		pos=strstr(string,":");
		if(pos) {
			if((pos-string) < MAX_HOSTID) {
				strncpy(desthost,string,(pos-string));
				desthost[pos-string]='\0';
			} else {
				strncpy(desthost,string,MAX_HOSTID);
				desthost[MAX_HOSTID-1]='\0';
			}
			pos++;
			// As long as the user has a number there.
			if(strlen(pos)) {
				port=atoi(pos);
			}
		} else {
			// No colon. Assume the default port to be 6161
			strncpy(desthost,string,MAX_HOSTID);
		}
		
		// Just in case the user has entered a blank field
		if(strlen(desthost) == 0) {
			strncpy(desthost,"localhost",MAX_HOSTID);
		}
		
		hp = gethostbyname(desthost);
		if(hp == 0) {
			fprintf(stderr,"Cannot resolve host %s - auditing to STDOUT\n",hostid);
			close(AuditSocket);
			AuditDestination |= AUDIT_TO_STDOUT;
			return(0);
		}
		bcopy(hp->h_addr, &AuditSocketName.sin_addr, hp->h_length);
		AuditSocketName.sin_family = AF_INET;
		AuditSocketName.sin_port = htons(port);
		
		AuditDestination |= AUDIT_TO_NETWORK;
		return(1);
	}

	AuditDestination |= AUDIT_TO_STDOUT;
	return(0);
}

int getheader(char *string)
{
	char temp[256];
	char *stringp;
	strncpy(temp,string,256);
	
	stringp=temp;
	
	// Remove the first and last bracket.
	stringp++;
	stringp[strlen(stringp)-1]='\0';
	
	if(regmatchi(stringp,"^objectives$")) {
		return(CONFIG_OBJECTIVES);
	} else if(regmatchi(stringp,"^events$")) {
		return(CONFIG_EVENTS);
	} else if(regmatchi(stringp,"^file$") || regmatchi(stringp,"^output$")) {
		return(CONFIG_OUTPUT);
	} else if(regmatchi(stringp,"^hostid$")) {
		return(CONFIG_HOSTID);
	} else if(regmatchi(stringp,"^remote$")) {
		return(CONFIG_REMOTE);
	}

	fprintf(stderr,"Unknown header in configuration file: %s\n",stringp);
	return(0);
}

// Return the host identifier
char * getconfstring(char *string, char *file,int length)
{
	char *stringp=string;

	stringp=strstr(string,"=");
	if(stringp != (char *) NULL) {
		stringp++;
		if(strlen(stringp)) {
			strncpy(file,stringp,length-1);
		} else {
			return((char *)NULL);
		}
	} else {
		return((char *)NULL);
	}
	return((char *)file);
}

int getport(char *string)
{
	char *stringp=string;
	char *pos;
	char strPort[10];

	stringp=strstr(string,"=");
	
	if(stringp != (char *) NULL) {
		stringp++;
		if(strlen(stringp)) {
			strncpy(strPort,stringp,10);
			return(atoi((char *)strPort));
		} else {
			return(0);
		}
	} else {
		return(0);
	}
}

int geteventswitch(char *string)
{
	char *stringp=string;
	char *endname=string;

	stringp=strstr(string,"=");
	if(stringp != (char *) NULL) {
		endname=stringp;
		stringp++;
		if(!strcmp(stringp,"1")) {
			// Set this event.
			// Grab the event name. Insert a null into the string in place of the = character.
			*endname='\0';
			return(eventname2number(string));
		}
	} else {
		return(-1);
	}
	return(-1);
}

// Return the host identifier
char * gethostident(char *string, char *host,int length)
{
	char *stringp=string;

	stringp=strstr(string,"=");
	if(stringp != (char *) NULL) {
		stringp++;
		if(strlen(stringp)) {
			strncpy(host,stringp,length-1);
		} else {
			return((char *)NULL);
		}
	} else {
		return((char *)NULL);
	}
	return((char *)host);
}

// Return a string that contains the criticality
// Note that string is assumed to be MAX_AUDITREC in size.
char * criticalitystring(int criticality, char * string)
{
	switch(criticality)
	{
		case CRITICALITY_CRITICAL:	return(strncpy(string,"critical",MAX_AUDITREC));
		case CRITICALITY_PRIORITY:	return(strncpy(string,"priority",MAX_AUDITREC));
		case CRITICALITY_WARNING:	return(strncpy(string,"warning",MAX_AUDITREC));
		case CRITICALITY_INFO:		return(strncpy(string,"information",MAX_AUDITREC));
		default:					return(strncpy(string,"clear",MAX_AUDITREC));
	}
}


// Remove start / end whitespace, including the newline
void trim(char *string)
{
	char * pointer;
	char * pointer2;
	
	// Verify that there is something to check.
	if(string == (char *) NULL) return;
	
	// And that there is some data within.
	if(strlen(string) == 0 ) return;

	// Start from the end, work backwards.
	pointer=&string[strlen(string)-1];

	while((*pointer == ' ' || *pointer == 9 || *pointer == 10 || *pointer == 12) && pointer >= string) {
		*pointer='\0';
		pointer--;
	}
	
	// Are we back at the start of the string? If so, this line must be null.
	if(pointer==string) return;
	// Pointer is now at the last non-whitespace character of the string.
	
	pointer2=string;
	while((*pointer2 == ' ' || *pointer2 == 9 || *pointer2 == 10 || *pointer2 == 12) && pointer2 < pointer) {
		pointer2++;
	}
	// pointer2 will now point to the start of the first non-null character
	// Copy the truncated string back to the original.
	strcpy(string,pointer2);
}

void trimallwhitespace(char *string)
{
	char * readpointer;
	char * writepointer;
	
	// Verify that there is something to check.
	if(string == (char *) NULL) return;
	
	// And that there is some data within.
	if(strlen(string) == 0 ) return;

	// Start from the end, work backwards.
	readpointer=string;
	writepointer=string;
	while(readpointer < (string + strlen(string))) {
		if(*readpointer == ' ' || *readpointer == 9 || *readpointer == 10 || *readpointer == 12) {
			readpointer++;
		} else {
			if(writepointer != readpointer) {
				*writepointer=*readpointer;
			}
			readpointer++;
			writepointer++;
		}
			
	}
}

// Strip tabs and newlines out of a string.
char * strip (char * string)
{
	char *stringp=string;
		
	while(*stringp != '\0') {
		if(*stringp == '\t' || *stringp == '\n') {
			*stringp = ' ';
		}
		stringp++;
	}
	return(string);
}

// Take an identified objective line
// NOTE: string, event, user and match must all be the same buffer size!
int splitobjective(char *string, char *event, char *user, char *match, int *returncode, int *excludeflag)
{
	char *startevent, *startuser, *startmatch, *startcrit, *startreturn;
	char *stringpointer, *endstring;
	char *eventp, *userp, *matchp, *critp;
	char criticality[MAX_AUDIT_CONFIG_LINE];
	int crit=0;

	*returncode=RETURNCODE_ANY;

	// Do some basic sanity checks.
	if(string == (char *) NULL || event == (char *) NULL || user == (char *) NULL || match == (char *) NULL) {
		return(-1);
	}
	if(!strlen(string)) {
		return(-1);
	}

	startcrit=strstr(string,"criticality=");
	startevent=strstr(string,"event=");
	startreturn=strstr(string,"return=");
	startuser=strstr(string,"user=");
	if(startuser==(char *)NULL) {
		startuser=strstr(string,"user!=");
		if(startuser!=(char *)NULL) {
			// EXCLUDE USERS rather than include.
			*excludeflag=1;
		}
	}
	startmatch=strstr(string,"match=");
	
	// string pointers for iteration.
	stringpointer=string;
	eventp=event; userp=user; matchp=match; critp=criticality;
	
	// Pointer to the last character in the string.
	endstring=&string[strlen(string)];
	
	if(startevent == (char *)NULL || startuser == (char *)NULL ||
	   startmatch == (char *)NULL || startcrit == (char *)NULL || startreturn == (char *)NULL) {
		// Problem, this line is malformed. We really cannot proceed with this line.
		fprintf(stderr,"The following line does not contain the criticality, event, return, user and match elements: %s",string);
		return(-1);
	}
	
	// Start with the event
	stringpointer=startcrit+strlen("criticality=");
	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startuser && stringpointer!=startmatch && stringpointer != endstring)) {
		*critp=*stringpointer;
		critp++;
		stringpointer++;
	}
	*critp='\0';
	trim(criticality);
	crit=atoi(criticality);
	if(crit < CRITICALITY_CLEAR) crit=CRITICALITY_CLEAR;
	if(crit > CRITICALITY_CRITICAL) crit=CRITICALITY_CRITICAL;
	
	// Start with the event
	stringpointer=startevent+strlen("event=");
	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startuser && stringpointer!=startmatch && stringpointer != endstring)) {
		*eventp=*stringpointer;
		eventp++;
		stringpointer++;
	}
	*eventp='\0';
	// Remove extra whitespace at start and end.
	trim(event);
	
	stringpointer=startreturn+strlen("return=");
	if(regmatchi(stringpointer,"^success")) {
		*returncode=RETURNCODE_SUCCESS;
	} else if (regmatchi(stringpointer,"^failure")) {
		*returncode=RETURNCODE_FAILURE;
	} else {
		// Any thing else is either success or failure.
		*returncode=RETURNCODE_ANY;
	}

	if(*excludeflag) {
		stringpointer=startuser+strlen("user!=");
	} else {
		stringpointer=startuser+strlen("user=");
	}

	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startmatch && stringpointer!=startevent && stringpointer != endstring)) {
		*userp=*stringpointer;
		stringpointer++;
		userp++;
	}
	*userp='\0';
	// Remove extra whitespace at start and end.
	trim(user);
	
	stringpointer=startmatch+strlen("match=");
	while(*stringpointer && (stringpointer!=startcrit && stringpointer != startreturn && stringpointer!=startevent && stringpointer!=startuser && stringpointer != endstring)) {
		*matchp=*stringpointer;
		stringpointer++;
		matchp++;
	}
	*matchp='\0';
	// Remove extra whitespace at start and end.
	trim(match);

	// Return the criticality value.
	return(crit);
}


// Match string against an extended regular expression.
// Return 1 for match, 0 for no-match or error.
int regmatch(const char *string, const char *pattern)
{
	int status;
	regex_t re;
	
	if(regcomp(&re,pattern,REG_EXTENDED|REG_NOSUB) != 0) {
		return(0);
	}
	status=regexec(&re,string,(size_t) 0, NULL, 0);
	regfree(&re);
	if(status != 0) {
		return(0);
	}
	return(1);
}	

// Match string against an extended regular expression. Case insensitive.
// Return 1 for match, 0 for no-match or error.
int regmatchi(const char *string, const char *pattern)
{
	int status;
	regex_t re;
	
	if(regcomp(&re,pattern,REG_EXTENDED|REG_NOSUB|REG_ICASE) != 0) {
		return(0);
	}

	status=regexec(&re,string,(size_t) 0, NULL, 0);
	regfree(&re);
	if(status != 0) {
		return(0);
	}
	return(1);
}	

// Linked List Functions

void CreateLinkedList(void)
{
    head = currentnode = NULL;
}

int IsListEmpty(void)
{
    if (NULL == head)
        return 1;
    else
        return 0;
}

// Node * AddToList(int eventnumber, char *username, char *path, int criticality, char *options, int excludeflag, int returncode)
Node * AddToList(char *eventname, char *username, char *path, int criticality, char *options, int excludeflag, int returncode)
{
    Node *newNode=NULL;
    char comment[128];

    newNode = (Node *) malloc(sizeof(Node));

    if(strcmp(eventname,"AUDIT_ALL")) {
        // Find real eventname..
        eventname2comment(eventname,comment,sizeof(comment));
    } else {
	strncpy(comment,"AUDIT_ALL",sizeof(comment));
    }

    if (newNode == NULL)
    {
        fprintf(stderr, "AddToList(): error in dynamic memory allocation\nCould not add a new objective into our linked list. You may be low on memory.\n");
        return((Node *)NULL);
    }

	//newNode->event_number=eventnumber;
	// strncpy(newNode->event_name,eventname,MAX_EVENTNAME);
	strncpy(newNode->event_name,comment,MAX_EVENTNAME);
	newNode->criticality=criticality;
	newNode->returncode=returncode;
	newNode->excludeflag=excludeflag;

	strncpy(newNode->username,username,MAX_USERREG);
	strncpy(newNode->path,path,PATH_MAX);
	strncpy(newNode->options,options,MAX_OPTIONS);

    if (head == NULL)
    {
        head = newNode;
        newNode->next = NULL;
    }
    else
    {
        newNode->next = head;
        head = newNode;
    }

    return newNode;
}

void RemoveFromListHead(void)
{
    Node *tempPtr;

    if (NULL == head)
        return;

    tempPtr = head;
    head = head->next;

    free(tempPtr);
}

void RemoveFromList(Node* node)
{
    Node *tempPtr, *previousPtr;

    if (NULL == node)
        return;

    if (head == node)
    {
        RemoveFromListHead();
        return;
    }

    tempPtr = head;

    while (NULL != tempPtr)
    {
        previousPtr = tempPtr;
        tempPtr = tempPtr ->next;

        if (tempPtr == node)
        {
            previousPtr->next = tempPtr->next;
            free(tempPtr);
            break;
        }
    }
}

void ResetCurrentNode(void)
{
    currentnode = head;
}

int IsValidItem(void)
{
    return (NULL == currentnode) ? 0 : 1;
}

Node * GetCurrentItem()
{
    return (currentnode);
}

void NextItemInList(void)
{
    if (NULL == currentnode)
        return;

    currentnode = currentnode->next;
}

void DestroyList(void)
{
    if (NULL == head)
    {
        return;
    }

    while (NULL != head)
    {
        Node *tempPtr = head;
        head = head->next;

        free(tempPtr);
    }
}


// Pull out the fully qualified domain name, if possible.
char * getfqdn(char * FQDN)
{
	char hname[MAX_HOSTID];
	struct hostent *hp;

	if (gethostname(hname, MAX_HOSTID)) {
		strncpy(FQDN,"localhost.unknown",MAX_HOSTID);
		return(FQDN);
	}
	
	strncpy(FQDN,hname,MAX_HOSTID);
 
	hp = gethostbyname(hname);
	if (hp) {
		while (hp->h_aliases && *hp->h_aliases) {
			if (strlen(*(hp->h_aliases)) > strlen(hname) && !strncmp(hname, *(hp->h_aliases), strlen(hname))) {
				strncpy(FQDN, *(hp->h_aliases),MAX_HOSTID);
			}
			hp->h_aliases++;
		}
		if (strlen(hp->h_name) > strlen(hname) && !strncmp(hname, hp->h_name, strlen(hname))) {
				strncpy(FQDN, hp->h_name,MAX_HOSTID);
		}
	}

     return(FQDN);
}

int SendToSocket(char * message)
{
	if(AuditSocket) {
		if(sendto(AuditSocket, message,
			strlen(message), 0, 
			(const struct sockaddr *)&AuditSocketName,
			sizeof(struct sockaddr))
					< 0) {
			puts(message);
			return(1);
		}
	} else {
		puts(message);
		return(1);
	}
	return(0);
}

// If we want to use a config file that uses text-type event names to enable/disable events
int	eventname2number(char * eventname)
{
	int eventnum=-1;

	FILE * audit_event;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];
	char teventname[256];
	char *position;
	char *position2;
	int teventnumber;

	audit_event=fopen("/etc/security/audit_event","r");

	if(audit_event==(FILE *)NULL) {
		printf("Cannot open /etc/security/audit_event. Exiting\n");
		return(0);
	}

	while(fgets(inputbuffer,MAX_AUDIT_CONFIG_LINE,audit_event)) {
		if(!iscomment(inputbuffer))	{
			if(position = strstr(inputbuffer,":")) {
				// Grab the number
				// Convert colon to null.
				*position='\0';
				teventnumber=atoi(inputbuffer);
				// Jump over the AUE_
				position+=5;
				if(position2 = strstr(position,":")) {
					char tseventname[256];
					int size;

					size=position2-position;
					if(size>255) size=255;
					strncpy(teventname,position,size);
					teventname[size]='\0';
					snprintf(tseventname,255,"^%s$",eventname);
					if(regmatchi(teventname,tseventname)) {
						eventnum=teventnumber;
						break;
					}
				}
			}
		}
	}

	fclose(audit_event);

	return(eventnum);
}

// We need to find out the comment associated with event names - and getauevnam SUCKS badly (case sensitive).
int	eventname2comment(char * eventname,char *comment,int length)
{
	FILE * audit_event;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];
	char teventname[256];
	char *position;
	char *position2;
	int teventnumber;

	if(!comment || !eventname) {
		return(0);
	}

	strncpy(comment,"",length);

	audit_event=fopen("/etc/security/audit_event","r");

	if(audit_event==(FILE *)NULL) {
		printf("Cannot open /etc/security/audit_event. Exiting\n");
		return(0);
	}

	while(fgets(inputbuffer,MAX_AUDIT_CONFIG_LINE,audit_event)) {
		if(!iscomment(inputbuffer))	{
			if(position = strstr(inputbuffer,":")) {
				// Grab the number
				// Convert colon to null.
				*position='\0';
				// Jump over the AUE_
				position+=5;
				if(position2 = strstr(position,":")) {
					int size;
					char tseventname[256];

					size=position2-position;
					if(size>255) size=255;
					strncpy(teventname,position,size);
					teventname[size]='\0';
					snprintf(tseventname,255,"^%s$",eventname);
					if(regmatchi(teventname,tseventname)) {
						position=position2+1;
						// Find the real name.
						if(position2 = strstr(position,":")) {
							size=position2-position;
							if(size>length) {
								size=length-1;
							}
							strncpy(comment,position,size);
							comment[size]='\0';
						}
						break;
					}
				}
			}
		}
	}

	fclose(audit_event);
	return(1);
}

// If we want to use a config file that uses text-type event names to enable/disable events
char * eventnumber2name(int number, char * eventname, int buffer)
{
	FILE * audit_event;
	char inputbuffer[MAX_AUDIT_CONFIG_LINE];
	char *position;
	int eventnumber;

	strncpy(eventname,"Unknown",buffer);

	audit_event=fopen("/etc/security/audit_event","r");

	if(audit_event==(FILE *)NULL) {
		printf("Cannot open /etc/security/audit_event. Exiting\n");
		return(0);
	}

	while(fgets(inputbuffer,MAX_AUDIT_CONFIG_LINE,audit_event)) {
		if(!iscomment(inputbuffer))	{
			if(position = strstr(inputbuffer,":")) {
				// Grab the number
				eventnumber=0;
				sscanf(inputbuffer,"%d",eventnumber);
				if(eventnumber == number) {
					char *position2;
					// Jump over the AUE_
					position+=5;
					if(position2 = strstr(position,":")) {
						int size;
						position2--;
						size=position2-position;
						if(size>=buffer) size=buffer-1;
						strncpy(eventname,position,size);
						break;
					}
				}
			}
		}
	}

	fclose(audit_event);

	return(eventname);
}

////////////////////////////////////////////////////////////////////////////////
// This routine will send an event to the user-selected output device.
////////////////////////////////////////////////////////////////////////////////
int sendevent(char *string,int criticality)
{
	char stringout[MAX_AUDITREC]="";
	
	// NEW audit record format, including host and SolarisBSM identifier string:
	snprintf(stringout,MAX_AUDITREC,"%s\tSolarisBSM\t%d\t%s",hostid,criticality,string);
	
	if(AuditDestination & AUDIT_TO_STDOUT) {
		// Send to STDOUT.
		printf("%s\n",stringout);
	}
	
	if(AuditDestination & AUDIT_TO_FILE) {
		// Write to file
		if(AuditFile != (FILE *) NULL) {
			fprintf(AuditFile,"%s\n",stringout);
			fflush(AuditFile);
		}
	}
	
	if(AuditDestination & AUDIT_TO_NETWORK) {
		if(AuditSocket) {		
			SendToSocket(stringout);
		}
	}
	
	return(0);
}

Node * CheckObjective(char *username,char *searchterm,char *eventname,char *options,int returncode)
{
	static int firstcall=0;
	Node *currentnode;

	if(firstcall==0) {
		ResetCurrentNode();
		firstcall=1;
	}
	
	while(IsValidItem()) {
		currentnode=GetCurrentItem();

		if((!strcmp(eventname,currentnode->event_name) || !strcmp(currentnode->event_name,"AUDIT_ALL")) && (returncode == currentnode->returncode || currentnode->returncode == RETURNCODE_ANY)) {
			// Are we including users, or excluding.
			if(currentnode->excludeflag) {
				if(!(regmatch(username,currentnode->username)) && regmatch(searchterm,currentnode->path)) {
					if(regmatchi(options,currentnode->options)) {
						NextItemInList();
						return(currentnode);
					}
				}
			} else {
				if(regmatch(username,currentnode->username) && regmatch(searchterm,currentnode->path)) {
					if(regmatchi(options,currentnode->options)) {
						NextItemInList();
						return(currentnode);
					}
				}
			}
		}
		NextItemInList();
	}

	ResetCurrentNode();
	return((Node *) NULL);
}

// Pick out the core details from an audit event, and set the username/searchterm/eventname
// Note: Expects event to be in tabbed praudit format, not snare modified format.
int GetSolDetails(char *logbuffer,char *username,char *searchterm,char *eventname,char *options)
{
	char *position=logbuffer;
	char tempreturn[20];

	searchterm[0]='\0';
	eventname[0]='\0';
	username[0]='\0';
	options[0]='\0';

	position=FindHeaderToken(logbuffer);
	if(position) GetElement(position,3,eventname,MAX_EVENTNAME);

	if(!eventname) {
		strncpy(eventname,"Unknown",MAX_EVENTNAME);
	}

	position=FindSubjectToken(logbuffer);
	if(position) GetElement(position,2,username,MAX_USERNAME);

	// login - we want the target user name as the search term
	// execve - path + exec_args
	if(!strncmp(eventname,"execve",6)) {
		char cargcount[5];
		char temparg[MAX_PATH];
		int argcount;

		// path token for some events. open, execve...
		position=FindPathToken(logbuffer);
		if(position) GetElement(position,1,searchterm,MAX_PATH);
		
		position=FindExecArgsToken(logbuffer);
		if(position) GetElement(position,1,cargcount,5);
		argcount=atoi(cargcount);
		if(argcount >1) {
			int count;
			for(count=1;count<argcount;count++) {
				GetElement(position,count+2,temparg,MAX_PATH);
				if(strlen(temparg)) {
					snprintf(searchterm,MAX_PATH,"%s %s",searchterm,temparg);
				}
			}
		}
	} else if(!strncmp(eventname,"chmod",5) || !strncmp(eventname,"ioctl",5)) {
		// path token for some events. open, execve...
		position=FindPathToken(logbuffer);
		if(position) GetElement(position,1,searchterm,MAX_PATH);
		position=FindAttributeToken(logbuffer);
		if(position) GetElement(position,1,options,MAX_OPTIONS);
	} else if(!strncmp(eventname,"old setgid",10) || !strncmp(eventname,"setregid",8) || !strncmp(eventname,"setegid",7)) {
		position=FindSubjectToken(logbuffer);
		if(position) GetElement(position,5,searchterm,MAX_PATH);
	} else if(!strncmp(eventname,"old setuid",10) || !strncmp(eventname,"setreuid",8) || !strncmp(eventname,"seteuid",7)) {
		position=FindSubjectToken(logbuffer);
		if(position) GetElement(position,4,searchterm,MAX_PATH);
	} else if(!strncmp(eventname,"setauid",7)) {
		position=FindSubjectToken(logbuffer);
		if(position) GetElement(position,1,searchterm,MAX_PATH);
	} else {
		// path token for some events. open, execve...
		position=FindPathToken(logbuffer);
		if(position) {
			 GetElement(position,1,searchterm,MAX_PATH);
		} else {
			// No Path token.. Ok.. copy in the entire string
			strncpy(searchterm,logbuffer,MAX_PATH);
		}

		position=FindAttributeToken(logbuffer);
		if(position) {
			GetElement(position,1,options,MAX_OPTIONS);
		} else {
			strncpy(options,"",MAX_OPTIONS);
		}
	}


	position=FindReturnToken(logbuffer);
	GetElement(position,1,tempreturn,20);

	if(!strcmp(tempreturn,"success")) {
		return(RETURNCODE_SUCCESS);
	} else {
		return(RETURNCODE_FAILURE);
	}
}

char * FindHeaderToken(char *buffer)
{
	return(strstr(buffer,"header	"));
}

char * FindSubjectToken(char *buffer)
{
	return(strstr(buffer,"subject	"));
}

char * FindReturnToken(char *buffer)
{
	return(strstr(buffer,"return	"));
}

char * FindPathToken(char *buffer)
{
	return(strstr(buffer,"path	"));
}

char * FindExecArgsToken(char *buffer)
{
	return(strstr(buffer,"exec_args	"));
}

char * FindAttributeToken(char *buffer)
{
	return(strstr(buffer,"attribute	"));
}

int GetElement(char *source,int count,char *dest,int buffersize)
{
	char * currentpos=source;
	char *position;
	int currentelement=0;
	int copysize=0;

	if(!source) return(0);
	if(!buffersize) return(0);

	dest[0]='\0';

	while(currentelement <= count) {
		position=strstr(currentpos,"	");
		if(!position) {
			if(currentelement==count) {
				// No problem, we're at the last element.
				strncpy(dest,currentpos,buffersize);
				dest[buffersize]='\0';
				return(1);
			} else {
				// Not enough TABS in our input
				return(0);
			}
		} else {
			if(currentelement==count) {
				copysize=position-currentpos;
				if(copysize > (buffersize-1)) {
					copysize=buffersize-1;
				}
				strncpy(dest,currentpos,copysize);
				dest[copysize]='\0';
				return(1);
			}
			currentpos=position+1;
		}

		currentelement++;
	}
}

char * FormatDelimiters(char *eventbuffer)
{
	char *eventpointer=eventbuffer;
	int tokenheader=0;

	while(*eventpointer) {
		// If we have a TAB
		if(*eventpointer=='	') {
			tokenheader=0;
			if(!strncmp(eventpointer+1,"ip	",3)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"arg	",4)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"acl	",4)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"ipc	",4)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"file	",5)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"path	",5)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"exit	",5)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"text	",5)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"iport	",6)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"header	",7)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"opaque	",7)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"return	",7)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"groups	",7)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"socket	",7)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"subject	",8)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"in_addr	",8)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"process	",8)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"trailer	",8)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"expanded	",9)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"exec_env	",9)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"ipc_perm	",9)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"sequence	",9)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"arbitrary	",10)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"attribute	",10)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"newgroups	",10)) { tokenheader=1; }
			else if(!strncmp(eventpointer+1,"exec_args	",10)) { tokenheader=1; }

			if(!tokenheader) {
				*eventpointer=',';
			}
		} else if(*eventpointer == ',') {
			// Cull commas
			*eventpointer=' ';
		}

		eventpointer++;
	}
	return(eventbuffer);
}

void DebugMsg(const char* pszFormat, ...)
{
	char buf[8192]="";
	FILE *fp;

	va_list arglist;
	va_start(arglist, pszFormat);
	vsnprintf(&buf[strlen(buf)],8192-strlen(buf)-1,pszFormat,arglist);
	va_end(arglist);

	fp=fopen("/tmp/SNARE-OUT","a");
	fputs(buf,fp);
	fclose(fp);
}

