#ifndef _SNARECORE_H
#define _SNARECORE_H
/**********************************************************
 *
 * Snare for Solaris version 2.5.7
 *
 * Author: InterSect Alliance Pty Ltd
 *
 * Copyright 2001-2002 InterSect Alliance Pty Ltd
 *
 * Last Modified: 21/12/2001
 *
 * Available under the terms of the GNU Public Licence.
 * - See www.gnu.org
 *
 **********************************************************
 *
 * Snare for Solaris is a user-space program
 * that interacts with the audit facility within the
 * solaris kernel using the auditsvc() system call.
 *
 * The program takes the output of the auditsvc call,
 * pipes it through the praudit command, and sends the
 * resulting text data over the network to a remote server
 * using UDP.
 *
 * The process also monitors the praudit process, and
 * respawns praudit if a segmentation fault occurs.
 *
 * Backlog also only responds to a very limited number of
 * signals, and so is unlikely to be killed by minor service
 * problems.
 *
 **********************************************************
 *
 * History:
 *      23/02/2001  Initial working version
 *       2/03/2001  Cleaned up error information.
 *       9/03/2001  Added signal handler to check for
 *                  praudit failure, and restart the service.
 *      22/03/2001  Set up each process to ignore most signals.
 *      23/03/2001  Oedipus update: Child kills parent if a
 *                  terminal error occurs in child.
 *      21/12/2001  Now sends a stop signal to the audit subsystem on exit.
 *                  This is to deal with the buggy Sol 8 +cnt flag.
 *       2/ 8/2002  Can now dynamically turn on audit events.
 *       8/ 8/2002  Started integrating SNARE functionality in order to
 *                  facilitate remote control.
 *       9/ 8/2002  Event start/stop in response to audit configuration file.
 *       
 *
 **********************************************************
 *
 * Compilation Instructions:
 *    gcc -o snarecore snarecore.c webserver.c WebPages.c -lbsm -lsocket -lnsl -lintl
 *
 **********************************************************/

// TODO: Cut up audit log data into TAB and COMMA delim stuff (like SNARE)

#include <sys/types.h> 
#include <sys/wait.h> 
#include <sys/errno.h>
#include <sys/fcntl.h>
#include <sys/termios.h>
#include <sys/socket.h>
#include <bsm/audit.h>
#include <bsm/audit_kevents.h>
#include <bsm/audit_uevents.h>
#include <bsm/audit_record.h>
#include <bsm/libbsm.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <pthread.h>
#include <limits.h>
#include <dirent.h>

#include <regex.h>

#include "webserver.h"

// Should migrate these to a header.

#define CACHESIZE 2		/* MAX cache size */
#define LOGBUFFERSIZE 4096	/* MAX UDP string buffer size */
#define HOSTSIZE 1024		/* MAX hostname */
#define MAXAUDIT 65535		/* Maximum number of audit events */
#define MAX_AUDIT_CONFIG_LINE	8192
#define CONFIG_FILENAME "/etc/security/snare.conf"
// #define CONFIG_FILENAME "snare.conf"

#define CONFIG_AUDITTYPE	1
#define	CONFIG_OBJECTIVES	2
#define CONFIG_EVENTS		3
#define CONFIG_OUTPUT		4
#define CONFIG_DELIVERY		5
#define CONFIG_HOSTID		6
#define CONFIG_REMOTE		7

#define MAX_HOSTID		256	// Host identifier - usually the fully qualified hostname.
#define MAX_AUDITREC PATH_MAX+4096		// How much buffer to reserve for the textual representation of an audit event.
#define MAX_AUDIT_CONFIG_LINE	8192
#define MAX_USERNAME		256
#define MAX_OPTIONS		256
#define MAX_EVENTNAME		32
#define MAX_USERREG		PATH_MAX
#define MAX_HOSTID		256		// Host identifier - usually the fully qualified hostname.
#define MAX_PATH			1024		// PATH_MAX in linux is 4096, but we cannot afford to
#define MAXCOMMAND 		16
#define WAITTIME		30			// Time, in seconds, to wait until attempting to send to a dead host

#define CRITICALITY_CLEAR	0
#define CRITICALITY_INFO	1
#define CRITICALITY_WARNING 2
#define CRITICALITY_PRIORITY 3
#define CRITICALITY_CRITICAL 4

#define	RETURNCODE_FAILURE	0
#define RETURNCODE_SUCCESS	1
#define RETURNCODE_ANY		999

#define AUDIT_TO_STDOUT		1
#define AUDIT_TO_FILE           2
#define AUDIT_TO_NETWORK        4
#define AUDIT_TO_SYSLOG         8

#define WEB_CACHE_SIZE 50
#define DEFAULT_CACHE 320000
#define SOCKETTYPE_UDP 0
#define SOCKETTYPE_TCP 1

#define AUDIT_ALL	-1

// Linked List
struct _node
{
	// int event_number;	// The event number this is supposed to match
	char event_name[MAX_EVENTNAME];	// The event number this is supposed to match
	int criticality;	// How critical is this particular node
	int returncode;		// return code required.
	int excludeflag;	// Include or exclude users?
	int excludematchflag;	// Include or exclude the match?
	char username[MAX_USERREG];	// Remember, this will be a regular expression.
	regex_t usernameRE;
	char path[PATH_MAX];			// NOTE: this could match either path or destpath
	regex_t pathRE;
	char options[MAX_OPTIONS];	// Options associated with a audit type (eg: O_RDONLY|O_CREAT)
	regex_t optionsRE;
	struct _node *next;
};

typedef struct _node Node;

// Linked List
struct _msgcache {
	char * msg;
	int msglen;
	struct _msgcache *next;
};

typedef struct _msgcache MsgCache;

struct _webcache {
	char hostid[MAX_HOSTID];
	//char string[MAX_AUDITREC];
	char CurrentDate[16];
	char username[MAX_USERNAME];
	char searchterm[MAX_PATH];
	char eventname[MAX_EVENTNAME];
	char options[MAX_OPTIONS];
	int returncode;
	int criticality;
	int snareseq;
	int seenflag;
	struct _webcache *next;
	struct _webcache *prev;
};

typedef struct _webcache WebCache;

struct _hostnode
{
	int AuditSocket;
	struct sockaddr_in      AuditSocketName;
	socklen_t dest_addr_size;
	time_t last_error;
	char desthost[MAX_HOSTID];
	int port;
	int protocol;
	int cacheflag;
	struct _hostnode *next;
};

typedef struct _hostnode HostNode;

void	pipe_signal();		/* Routine to handle SIGPIPEs */
void	kill_signal();		/* Routine to handle SIGTERMs */
void	usr1_signal();

int		turn_event_on(int);
int		turn_event_off(int);
int		setclass(unsigned int);
int		read_config_file(int);
char *	gethostident(char *string, char *host,int length);
char *	getconfstring(char *string, char *file,int length);
int		getport(char *string);

int		setsignals(sigset_t *signalset);

void	trim(char *string);
void	trimallwhitespace(char *string);
int		IsListEmpty(void);

int sendevent(char *string,int criticality);
int iscomment(char * line);
int isheader(char *string);
int getheader(char *string);
int splitobjective(char *string, char *event, char *user, char *match, int *excludeflag, int *excludematchflag, int *returncode);
int regmatch(const char *string, const char *pattern);
int regmatchi(const char *string, const char *pattern);
int eventname2number(char * eventname);
int isfilename(char *string);
int open_audit_output(char *string);
int open_audit_network(char *string);
int isnetwork(char *string);
int eventname2comment(char * eventname,char *comment,int length);
void syslogdate(char *sdate, struct tm *cdate);



Node *	AddToList(char *eventname, char *username, char *path, int criticality, char *options, int excludeflag, int excludematchflag, int returncode);
char *	getfqdn(char * FQDN);

Node *	CheckObjective(char *username,char *searchterm,char *eventname,char *options,int returncode);
char *	FormatDelimiters(char *eventbuffer);
int		GetSolDetails(char *logbuffer,char *username,char *searchterm,char *eventname, char *options);
int		GetElement(char *source,int count,char *dest,int buffersize);

char * FindHeaderToken(char *buffer);
char * FindSubjectToken(char *buffer);
char * FindReturnToken(char *buffer);
char * FindPathToken(char *buffer);
char * FindExecArgsToken(char *buffer);
char * FindAttributeToken(char *buffer);

void DestroyList(void);
int IsSocketValid(int socket);
int cachemsg(HostNode *hn, char * msg);
void purge_cache();
int fix_last_error(HostNode *hn);
void start_web ();

// definitions for the ring buffer communication between the auditsvc reader
//   and the daemon writing to praudit

#define AUDIT_DATA_BLOCK_SIZE 256	// data size of one block within a record
#define MAX_AUDIT_DATA_BLOCKS 10000	// number of blocks available for records

struct _auditblock
{
	unsigned long recordnumber;	// The record number this block belongs to
	unsigned short numberblocks;	// The amount of blocks belonging to the record
	unsigned short sequencenumber;	// The sequence number of this block within the record (1-based)
	unsigned short datalength;	// The data length of this block within the record
	char data[AUDIT_DATA_BLOCK_SIZE]; // The actual data
};

typedef struct _auditblock Auditblock;

void RecordToRingbuffer(const char *newrecord, int recordlength,
					    unsigned long *recordnumber);
int RecordFromRingbuffer(char *readrecord, unsigned long *recordnumber);

#endif //_SNARECORE_H
