#!/bin/sh
########################################################################
#    
#    (c) 2001-2003 Intersect Alliance Pty Ltd
# 
#   This script will automate the installation of the snare audit daemon.
#   
#   Version 1.5
#    
#    Usage: ./install.sh
#
########################################################################

########################################################################
#   Check that the necessary files are contained within the local
#   directory. 
########################################################################

file_check()
{
     EVENTFILEWS=snare.conf.workstation
     EVENTFILESV=snare.conf.server
     START_SCRIPT=audit_start.sh
     AUDITCONTROL=snare_audit_control
     AUDITCLASS=snare_audit_class
     AUDITEVENT=snare_audit_event
     INSTALL_DIR=/etc/security 
     INSTALL_SCRIPT_DIR=/etc/init.d

     if [ ! -f snarecore ]
     then
        echo "The program snarecore binary file is missing. Terminating script."
        exit 1
     fi

     if [ ! -f $EVENTFILEWS ]
     then
        echo "The new workstation configuration file is missing. Terminating script."
        exit 2
     fi

     if [ ! -f $EVENTFILESV ]
     then
        echo "The new server configuration file is missing. Terminating script."
        exit 3
     fi

     if [ ! -f $START_SCRIPT ]
     then
        echo "The audit startup script is missing. Terminating script."
        exit 6 
     fi

     if [ ! -f $AUDITCONTROL ]
     then
        echo "The audit control script is missing. Terminating script."
        exit 6 
     fi

     if [ ! -f $AUDITCLASS ]
     then
        echo "The audit class script is missing. Terminating script."
        exit 6 
     fi

     if [ ! -f $AUDITEVENT ]
     then
        echo "The audit event script is missing. Terminating script."
        exit 6 
     fi

     if [ ! -d doco ]
     then
        echo "Documentation directory is missing. Terminating script."
        exit 6
     fi

}

########################################################################
#   Check that the user has an effective userid of 0 (root) 
########################################################################

root_check()
{
     id | grep -c uid=0 > /dev/null
     RETVAL=$?

	if [ $RETVAL -gt 0 ]
	then
           echo "You need to be root to run this script. Terminating script."
           exit 7
        else
           echo "Confirmed your are logged in as root"
	fi

}

########################################################################
#   Check that the software patch level is OK. The two patches (one 
#   for Solaris 6 and one for Solaris 7. These two scripts MUST
#   be installed.
########################################################################

check_os_and_patch_level()
{
   uname -r | grep 5.[6-8] > /dev/null
   RETVAL=$?

   if [ $RETVAL -gt 0 ]
   then
     echo "This updated Solaris daemon is designed for Solaris 6, 7 and 8 only. Terminating script"
     exit 8
   fi 
	
   uname -r | grep 5.6 > /dev/null 
   RETVAL=$?

   if [ $RETVAL -lt 1 ]
   then		
        showrev -p | egrep "^Patch: +105621" > /dev/null
        RETVAL=$?
		
	if [ $RETVAL -gt 0 ]
	then
           echo "You need the Solaris 6 patch 105621 installed. Terminating script"
           exit 9
        else 
           echo "Confirmed you have the necessary Solaris 6 patches" 
	fi
   fi 


   uname -r | grep 5.7 > /dev/null
   RETVAL=$?

   if [ $RETVAL -lt 1 ]
   then
        showrev -p | egrep "^Patch: +106541" > /dev/null
        RETVAL=$?
       
        if [ $RETVAL -gt 0 ]
        then
           echo "You need the Solaris 7 patch 106541 installed. Terminating script"
           exit 10 
        else
           echo "Confirmed you have the necessary Solaris 7 patches"
        fi
   fi

   uname -r | grep 5.8 > /dev/null
   RETVAL=$?

   if [ $RETVAL -lt 1 ]
   then
	echo "No additional patches are required for Solaris 8. Continuing."
   fi
}

########################################################################
#   Check that the Solaris audit sub-system has been executed. The
#   audit sub-system is installed by executing "bsmconv". This will then
#   add an entry in the /etc/system file, which can be used to check
#   if audit has been installed. 
########################################################################

check_audit_is_on()
{
    cat /etc/system | egrep "c2audit:audit_load *= *1" > /dev/null
    RETVAL=$?
      
    if [ $RETVAL -gt 0 ] 
    then
       echo "The audit sub-system has not been installed."
       echo "Please convert this system to be audit-ready by running"
       echo "bsmconv from /etc/security."
       echo "Terminating script"
       exit 11
    else 
       echo "Confirmed that you have the audit sub-system installed"
    fi              
}


########################################################################
#   This routine will copy the key audit_[control|class|event]  
#   and the /etc/init.d/audit files to the /etc/security directory
#   and rename them to ".snare-backup". If they exist, it will not back them up.
#   The point of this approach is to backup the old BSM files, and not
#   the snarecore files. If a recovery is required, the user can then 
#   recover to an old BSM state. 
########################################################################

backup_key_files()
{

   echo "Backing up key files..........."

   if [ ! -f $INSTALL_DIR/audit.snare-backup ]
   then
      cp $INSTALL_SCRIPT_DIR/audit $INSTALL_DIR/audit.snare-backup
      RETVAL=$?
      if [ $RETVAL -gt 0 ]
      then
        echo "Unable to backup the /etc/init.d/audit file to /etc/security. Terminating script"
        exit 15
      fi
   else
      echo "Backup of audit.snare-backup file already exists. Not overwriting"
   fi

   cp $INSTALL_DIR/audit_control $INSTALL_DIR/audit_control.snare-backup
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
      echo "Unable to backup the /etc/init.d/audit_control file to /etc/security. Terminating script"
      exit 15
   fi

   cp $INSTALL_DIR/audit_class $INSTALL_DIR/audit_class.snare-backup
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
      echo "Unable to backup the /etc/init.d/audit_class file to /etc/security. Terminating script"
      exit 15
   fi

   cp $INSTALL_DIR/audit_event $INSTALL_DIR/audit_event.snare-backup
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
      echo "Unable to backup the /etc/init.d/audit_event file to /etc/security. Terminating script"
      exit 15
   fi
}


########################################################################
#  This is the main loop  
########################################################################

echo
echo "Note: This script is for Solaris 6, 7 and 8 installations only"
echo "      Snare has not been tested on other systems."
echo

check_audit_is_on
file_check 
root_check
check_os_and_patch_level
backup_key_files

INSTAL=0
SNAREC=0

if [ -f /etc/security/snare.conf ]; then
   SNAREC=1
fi
 
while [ "$INSTAL" -ne "1" ] && [ "$INSTAL" -ne "2" ] && [ "$INSTAL" -ne "3" ]
do
  echo 
  echo "Enter [1] for a workstation install"
  echo "Enter [2] for a server install"
  if [ "$SNAREC" -eq "1" ]; then
     echo "Enter [3] to preserve the EXISTING snare configuration file"
  fi
  echo "Enter anything else to terminate the script"
  echo "Enter your choice..."
    read INSTAL
    case $INSTAL in
       1) echo "Installing for workstation"; INSTALL_TYPE=1 ;;
       2) echo "Installing for a server"; INSTALL_TYPE=2 ;;
       3) if [ "$SNAREC" -eq "1" ]; then
		echo "Preserving existing SNARE configuration"
		INSTALL_TYPE=3
	  else
		echo "Script terminating at user request";
		exit 11;
	  fi;;
       *) echo "Script terminating at user request"; exit 11 ;;
    esac 
done
 

########################################################################
#  This provides the final warning before the software is loaded onto
#  the system. In case of mistakes, it allows the user to escape before 
#  the software is loaded.   
########################################################################

echo 
echo "*************************************************"
echo 
echo "You are about to install Snare on this host."
echo "Your selections are as follows:"
if [ $INSTALL_TYPE -eq 1 ]; then
  echo "Workstation Installation"
elif [ $INSTALL_TYPE -eq 2 ]; then
  echo "Server Installation"
else
  echo "Preserving existing configuration file."
fi



########################################################################
#  This will copy the common workstation and server files to
#  to /etc directory.  
########################################################################

if [ $INSTALL_TYPE -eq 1 ] || [ $INSTALL_TYPE -eq 2 ] || [ $INSTALL_TYPE -eq 3 ]; then
   echo
   echo "Installing common files..........."

   cp ./snarecore $INSTALL_DIR/snarecore
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the snarecore program to /etc/security. Terminating script"
     exit 19
   fi

   chown root:root $INSTALL_DIR/snarecore
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to change owner for file to /etc/security/snarecore. Terminating script"
     exit 20
   fi
   chmod  550 $INSTALL_DIR/snarecore
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to change the permissions for file to /etc/security/snarecore. Terminating script"
     exit 21
   fi

   cp ./audit_start.sh  $INSTALL_SCRIPT_DIR/audit
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the startup script: /etc/init.d/audit. Terminating script"
     exit 22
   fi

   cp ./snare_start.sh  $INSTALL_SCRIPT_DIR/snare
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the startup script: /etc/init.d/snare. Terminating script"
     exit 23
   fi

   cp ./snare_audit_control $INSTALL_DIR/audit_control
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the snare audit_control replacement to /etc/security. Terminating script"
     exit 19
   fi

   cp ./snare_audit_class $INSTALL_DIR/audit_class
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the snare audit_class replacement to /etc/security. Terminating script"
     exit 19
   fi

   cp ./snare_audit_event $INSTALL_DIR/audit_event
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the snare audit_event replacement to /etc/security. Terminating script"
     exit 19
   fi

   cp ./snare /usr/bin/
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
      echo "Unable to copy the SNARE GUI into /usr/bin. Continuing."
   fi

   ln /etc/init.d/snare /etc/rc0.d/K33snare
   ln /etc/init.d/snare /etc/rc1.d/K33snare
   ln /etc/init.d/snare /etc/rc2.d/S99zsnare
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to create etc/rc*.d files. Terminating script"
     exit 24
   fi

   mkdir -p /usr/share/doc/snare-core-0.9/doc/
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to create snare document directory. Terminating script"
     exit 24
   fi

   cp doco/* /usr/share/doc/snare-core-0.9/doc/
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy files to the document directory. Terminating script"
     exit 24
   fi

   echo
   echo "Successfully installed the common files"
   echo
fi

	
########################################################################
#  This will copy the workstation files to /etc/security, for a
#  workstation setup. It will also change the owner and permissions
#  for these files.
########################################################################

if [ $INSTALL_TYPE -eq 1 ]
then
   echo
   echo "Installing for a workstation..........."

   cp ./$EVENTFILEWS $INSTALL_DIR/snare.conf
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the audit_event file to /etc/security. Terminating script"
     exit 23 
   fi

fi


########################################################################
#  This will copy the server files to /etc/security, for a
#  workstation setup. It will also change the owner and permissions
#  for these files.
########################################################################


if [ $INSTALL_TYPE -eq 2 ]
then
   echo
   echo "Installing for a server..........."


   cp ./$EVENTFILESV $INSTALL_DIR/snare.conf
   RETVAL=$?
   if [ $RETVAL -gt 0 ]
   then
     echo "Unable to copy the audit_event file to /etc/security. Terminating script"
     exit 24 
   fi
fi

########################################################################
#  This will kill the old audit daemon, stop the new daemon (if it exists) 
#  and start the new daemon with the updated parameters. 
########################################################################


# Kill the audit daemon, if it is alive
audit -t > /dev/null 2>&1
sleep 2

/etc/init.d/audit stop

echo
echo "Successfully installed the necessary files. Activating auditing..."
echo 


/etc/init.d/audit start
/etc/init.d/snare start
sleep 2

PID=`/usr/bin/ps -e -u 0|/usr/bin/fgrep snarecor|/usr/bin/egrep -v fgrep|/usr/bin/awk '{print $1}'`
if [ -z "$PID" ] ;  then
        echo "Snare not running. Please check installation parameters."
        exit 25;
fi

echo "Done."
echo
echo "Please connect to the local machine using your web browser"
echo "using the following URL: http://"`hostname`":6161/"
echo "and a userid and password of 'snare'"
echo "Once connected, please CHANGE THE DEFAULT PASSWORD, configure"
echo "any additonal audit objectives you require, and restart"
echo "the audit server".

exit 0		
