/***************************************************************************
 *
 * Copyright (c) 1998 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "afunix.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/time.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#include <string.h>

#include "syslog-ng.h"
#include "cfgdefs.h"
#include "log.h"
#include "utils.h"

#ifndef lint
static const char rcsid[]="$Id: afunix.c,v 1.23 1999/01/12 16:53:07 bazsi Exp $";
#endif

int 
afunix_output(lg, fte)
	logmsg_t *lg;
	filetable_entry_t *fte;
{
	char line[MAXLINE + 1];
	unixspec_t *xs = fte->fte_spec;
	int nw = 0;
	
	snprintf(line, sizeof(line), "%s %s %s\n", lg->lg_time, lg->lg_host, lg->lg_line);
	if (opts & OPT_DEBUG)
		log_debug("%smsg to unix %s\n",
			(xs->xs_mode & VAR_RAW) ? "raw " : "",
			xs->xs_un.sun_path);
	if (xs->xs_fd >= 0) {
		nw = write(xs->xs_fd, line, strlen(line));
		if (nw < 0) {
			log_printf(LOG_SYSLOG | LOG_ERR, "error sending unix socket %s: %s, reopening", xs->xs_un.sun_path, strerror(errno));
			close(xs->xs_fd);
			ftable_del_fd(xs->xs_fd);
			xs->xs_fd = -1;
			return -1;	/* indicate unsuccessful completition */
		}
	}
	else {
		return -1;
	}
	return 0;
}

void
afunix_open_socket(xs)
	unixspec_t *xs;
{
	int fd = -1;
	int type = xs->xs_type & VAR_DSF;

	if (xs->xs_proto == NET_UNIX_DGRAM)
		fd = socket(AF_UNIX, SOCK_DGRAM, 0);
	else if (xs->xs_proto == NET_UNIX_STREAM)
		fd = socket(AF_UNIX, SOCK_STREAM, 0);
	else 
		log_printf(LOG_SYSLOG | LOG_ERR, "unknown unix protocol");

	if (fd == -1)
		log_printf(LOG_SYSLOG | LOG_ERR, "socket: %s", strerror(errno));
	else {
		(void) fcntl(fd, F_SETFD, fcntl(fd, F_GETFD, 0) | FD_CLOEXEC);
		if (type == SRC_UNIX) {
			(void) unlink(xs->xs_un.sun_path);
			if (bind(fd, (struct sockaddr *) &xs->xs_un, sizeof(xs->xs_un))) {
				log_printf(LOG_SYSLOG | LOG_ERR, "bind failed: %s", strerror(errno));
				close(fd);
				fd = -1;
			}
			(void) chmod(xs->xs_un.sun_path, 0666);
			if (xs->xs_proto == NET_UNIX_STREAM) 
				listen(fd, 5);
		} else if (type == DEST_UNIX) {
			if (connect(fd, (struct sockaddr *) &xs->xs_un, sizeof(xs->xs_un))) {
				close(fd);
				fd = -1;
			}
		}
	}
	xs->xs_fd = fd;
}

int
afunix_reconnect(ft)
	filetype_t *ft;
{
	unixspec_t *xs = ft->ft_ptr;
	
	afunix_open_socket(xs);
	ftable_add_fd(xs->xs_fd, ft->ft_owner, ft->ft_ndx, ft->ft_type, ft->ft_ptr);
	return 0;
}

int
afunix_check_fd(fd, fte)
	int fd;
	filetable_entry_t *fte;
{
	unixspec_t *xs = (unixspec_t *) fte->fte_spec;
	struct sockaddr_un un;
	int slen, len, nfd;
	filetype_t *ftype;
	char buf[MAXLINE + 1];

	switch (VAR_TYPE(fte->fte_type)) {
	case SRC_UNIX:
		switch (xs->xs_proto) {
		case NET_UNIX_DGRAM:
			slen = sizeof(un);
			len = recvfrom(fd, buf, MAXLINE, 0,
					(struct sockaddr *)&un, &slen);
			log_printall(buf, len, fte);
			break;
		case NET_UNIX_STREAM:
			/* new unix connection */
			slen = sizeof(un);
			nfd = accept(fd, (struct sockaddr *) &un, &slen);
			if (nfd == -1) {
				log_printf(LOG_SYSLOG | LOG_ERR, "accept(%d,%s) failed: %s", fd,
					un.sun_path, strerror(errno));
				return -1;
			}
			else {
				ftable_add_fd(nfd, fte->fte_owner, fte->fte_ndx, SRC_UNIX_ACCEPTED, fte->fte_spec);
				ftable_add_read_fd(nfd);
			}
			break;
		}
		break;
	case SRC_UNIX_ACCEPTED:
		/* handle opened unix connection */
		len = read(fd, buf, MAXLINE);
		if (len < 0) {
			log_printf(LOG_SYSLOG | LOG_ERR, "read(%d,%s) failed: %s", fd,
				un.sun_path, strerror(errno));
			return -1;
		}
		else if (len == 0) {
 			close(fd);
 			ftable_del_fd(fd);
 			return -1;
		}
		log_printall(buf, len, fte);
		break;
	case DEST_UNIX:
		ftype = &fte->fte_owner->v_list[fte->fte_ndx];
		if (log_queue_run(ftype, (log_output_func_t) afunix_output, fte)) {
			ftable_del_write_fd(((unixspec_t *) fte->fte_spec)->xs_fd);
		}
		break;
	}
	return 0;
}
