/***************************************************************************
 *
 * Copyright (c) 1998 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "ftable.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdlib.h>

#include "var.h"
#include "afunix.h"
#include "afinet.h"
#include "affile.h"
#include "afuser.h"
#include "afpipe.h"
#include "afstreams.h"

#ifndef lint
static const char rcsid[]="$Id: ftable.c,v 1.21 1998/12/10 20:16:42 bazsi Exp $";
#endif

static	fd_set rd_fds;		/* master read fds */
static	int n_rd_fds;
static	fd_set wr_fds;		/* master write fds */
static	int n_wr_fds;

static	filetable_entry_t	*fttable;
static	u_int			fttsize;

void
ftable_init()
{
	n_rd_fds = 0;
	memset(&rd_fds, 0, sizeof(rd_fds));
	n_wr_fds = 0;
	memset(&wr_fds, 0, sizeof(wr_fds));

	fttsize = getdtablesize();
	fttable = (filetable_entry_t *)malloc(fttsize * sizeof(*fttable));
}

void
ftable_reinit()
{
	n_rd_fds = 0;
	memset(&rd_fds, 0, sizeof(rd_fds));
	n_wr_fds = 0;
	memset(&wr_fds, 0, sizeof(wr_fds));

	memset((char *)fttable, 0, fttsize * sizeof(*fttable));
}

void 
ftable_done()
{
	free(fttable);
}

void
ftable_add_read_fd(fd)
	int fd;
{
	if (fd >= 0) {
		FD_SET(fd, &rd_fds);
		n_rd_fds = MAX(n_rd_fds, fd + 1);
	}
}

void
ftable_del_read_fd(fd)
	int fd;
{
	if (fd >= 0) {
		FD_CLR(fd, &rd_fds);
		n_rd_fds = MAX(n_rd_fds, fd + 1);
		while (n_rd_fds && !FD_ISSET(n_rd_fds-1, &rd_fds)) n_rd_fds--;
	}
}

void 
ftable_add_write_fd(fd)
	int fd;
{
	if (fd >= 0) {
		FD_SET(fd, &wr_fds);
		n_wr_fds = MAX(n_wr_fds, fd + 1);
	}
}

void 
ftable_del_write_fd(fd)
	int fd;
{
	if (fd >= 0) {
		FD_CLR(fd, &wr_fds);
		n_wr_fds = MAX(n_wr_fds, fd + 1);
		while (n_wr_fds && !FD_ISSET(n_wr_fds-1, &wr_fds)) n_wr_fds--;
	}
}

void
ftable_add_fd(fd, owner, ndx, type, spec)
	int fd;
	var_t *owner;
	int ndx;
	void *spec;
{
	if (fd >= 0) {
		fttable[fd].fte_owner = owner;
		fttable[fd].fte_ndx = ndx;
		fttable[fd].fte_type = type;
		fttable[fd].fte_spec = spec;
	}
}

void 
ftable_del_fd(fd)
{
	if (fd >= 0) {
		memset(&fttable[fd], 0, sizeof(filetable_entry_t));
		ftable_del_read_fd(fd);
		ftable_del_write_fd(fd);
	}
}

filetable_entry_t *
ftable_get_fte(fd)
	int fd;
{
	return (fd >= 0) ? &fttable[fd] : NULL;
}

int
ftable_check_fd(fd)
	int fd;
{
	filetable_entry_t *fte = fttable + fd;

	switch (VAR_TYPE(fte->fte_type))
	{
	case SRC_UNIX:
	case SRC_UNIX_ACCEPTED:
		afunix_check_fd(fd, fte);
		break;
	case SRC_INET:
	case SRC_INET_ACCEPTED:
	case DEST_INET:
		afinet_check_fd(fd, fte);
		break;
	case SRC_FILE:
	case DEST_FILE:
		affile_check_fd(fd, fte);
		break;
	case DEST_CMD:
		afpipe_check_fd(fd, fte);
		break;
#ifdef USE_SUN_STREAMS
	case SRC_SUN_STREAM:
		afstreams_check_fd(fd, fte);
		break;
#endif
	}
	return 0;
}

int
ftable_check_fds(delay)
	time_t delay;
{
	struct timeval tv;
	fd_set rfd, wfd;
	int rc, i, cnt;

	tv.tv_sec = 0;
	tv.tv_usec = 0;
	rc = 1;
	afuser_check_users();
	while (rc > 0) {	/* loop while we have any output */
		FD_ZERO(&wfd);
		memcpy(&wfd, &wr_fds, sizeof(wr_fds));
		rc = select(n_wr_fds, NULL, &wfd, NULL, &tv);
		cnt = rc;
		for (i = 0; cnt > 0; i++) {
			if (FD_ISSET(i, &wfd)) {
				ftable_check_fd(i);
				cnt--;
			}
		}
	}
	if (rc == -1) return -1;

	tv.tv_sec = delay;
	tv.tv_usec = 0;
	/* now process input */
	memcpy(&rfd, &rd_fds, sizeof(rd_fds));
	rc = select(n_rd_fds, &rfd, NULL, NULL, &tv);
	if (rc == -1) return -1;
	if (rc == 0) return -2;
	cnt = rc;
	for (i = 0; cnt > 0; i++) {
		if (FD_ISSET(i, &rfd)) {
			ftable_check_fd(i);
			cnt--;
		}
	}
	return 0;
}
