/***************************************************************************
 *
 * Copyright (c) 1998 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "hash.h"
#include "affile.h"
#include "utils.h"

void usage(void)
{
	printf( "chechhash - checks the digital signature of a given logfile\n"
		"syntax:\n"
		"checkhash [options] files...\n\n"
		"options:\n"
		"  -p <password>       use supplied password instead of asking one\n"
		"  -h <hashtype>       use this hashtype instead of default SHA1\n\n");
}

int checkfile(char *password, hashalgo_t *hash, char *path)
{
	char buf[256], salt[MAX_FILE_SALT], line[MAXLINE + 1], phash[MAX_DIGEST_LEN], nhash[MAX_DIGEST_LEN];
	FILE *fd, *logfile, *hashfile;
	hashrec_t hrec, holdrec;
	int hlen;

	snprintf(buf, sizeof(buf), "%s.salt", path);
	fd = fopen(buf, "rb");
	if (!fd)
		return -1;
	fread(salt, 1, sizeof(salt), fd);
	fclose(fd);
	
	logfile = fopen(path, "rb");
	if (!logfile)
		return -1;
		
	snprintf(buf, sizeof(buf), "%s%s", path, hash->ha_type->ht_ext);
	hashfile = fopen(buf, "rb");
	if (!hashfile) {
		fprintf(stderr, "hash file not available: %s\n", buf);
		fclose(logfile);
		return -1;
	}
	
	hash_init(hash);
	hash_update(hash, salt, sizeof(salt));
	hash_update(hash, password, strlen(password));
	hash_finalize(hash, phash);		/* now we have all data for H0 */
	hlen = hash_getdigestlen(hash);
	
	hrec.hr_lcnt = 0;
	while (fread(&holdrec, sizeof(holdrec), 1, hashfile) == 1) {
		fgets(line, sizeof(line), logfile);
		memset(&hrec.hr_mac, 0, sizeof(hrec.hr_mac));
		hash_init(hash);
		hash_update(hash, phash, hlen);
		hash_update(hash, line, strlen(line));
		hash_update(hash, (char *) &hrec, sizeof(hrec));
		hash_finalize(hash, nhash);
		if (holdrec.hr_lcnt != hrec.hr_lcnt || memcmp(nhash, holdrec.hr_mac, hlen) != 0) {
			fprintf(stderr, "file %s hash mismatch at line %ld, quitting\n", path, hrec.hr_lcnt);
			return -2;
		}
		hash_init(hash);	/* generate next hash */
		hash_update(hash, phash, hlen);
		hash_update(hash, nhash, hlen);
		hash_finalize(hash, phash);
		hrec.hr_lcnt++;
		
	}
	
	fclose(logfile);
	fclose(hashfile);
	return 0;
}

int main(int argc, char *argv[])
{
	char password[50], hashtype[12] = "SHA1";
	hashalgo_t *hash;
	int askpass = 0, i;
	int rc = 0, opt;
	
	while ((opt = getopt(argc, argv, "p:h:")) != -1) {
		switch (opt) {
		case 'p':
			strncpy(password, optarg, sizeof(password));
			askpass = 0;
			break;
		case 'h':
			strncpy(hashtype, optarg, sizeof(hashtype));
			break;
		default:
			fprintf(stderr, "Invalid arguments\n");
			usage();
			exit(1);
		}
	}
	if ((hash = hash_create(hashtype)) == NULL) {
		fprintf(stderr, "Unknown hash type: %s\n", hashtype);
		usage();
		exit(1);
	}
	if (optind >= argc) {
		fprintf(stderr, "No files found on the commandline\n");
		usage();
		exit(1);
	}
	if (askpass) {
		printf("Enter initial password: ");
		fgets(password, sizeof(password), stdin);
		chomp(password);
	}
	for (i = optind; i < argc; i++) {
		if (checkfile(password, hash, argv[i]) == -2) 
			rc++;
	}
	return rc;
}
