/***************************************************************************
 *
 * Copyright (c) 1998 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "hash.h"

#include <stdio.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>

#include "syslog-ng.h"
#include "utils.h"
#include "var.h"
#include "sha1.h"

#ifndef lint
static const char rcsid[]="$Id: hash.c,v 1.18 1999/01/18 10:30:37 bazsi Exp $";
#endif

static hashtype_t hashes[] = {
	{ "SHA1", ".sha1", HASH_SHA_1, 20, SHA1Create, SHA1Destroy, SHA1Init, SHA1Update, SHA1Final },
	{ "", ".non", HASH_NONE, 0, NULL, NULL, NULL, NULL },
};


/*
 * find a hash algo by its type
 */
hashtype_t *
hash_find(name)
	char *name;
{
	int i;

	for (i = 0; hashes[i].ht_type != HASH_NONE; i++)
		if (strcmp(hashes[i].ht_name, name) == 0)
			return hashes + i;
	return NULL;
}

hashalgo_t *
hash_create(name)
	char *name;
{
	hashalgo_t *ha;
	hashtype_t *ht;
	
	ht = hash_find(name);
	if (!ht || !(NEW(ha)))
		return NULL;
	
	ha->ha_type = ht;
	ha->ha_ctx = (*ht->ht_create)();
	return ha;
}

void
hash_destroy(ha)
	hashalgo_t *ha;
{
	if (ha && ha->ha_type) {
		(*ha->ha_type->ht_destroy)(ha->ha_ctx);
		free(ha);
	}
}

void 
hash_init(ha)
	hashalgo_t *ha;
{
	if (ha && ha->ha_type) {
		(*ha->ha_type->ht_init)(ha->ha_ctx);
	}
}

void
hash_update(ha, data, dlen)
	hashalgo_t *ha;
	char *data;
	size_t dlen;
{
	if (ha && ha->ha_type) {
		(*ha->ha_type->ht_update)(ha->ha_ctx, data, dlen);
	}
}

void
hash_finalize(ha, digest)
	hashalgo_t *ha;
	char *digest;
{
	if (ha && ha->ha_type) {
		(*ha->ha_type->ht_final)(ha->ha_ctx, digest);
	}
}

void
hash_getsalt(buf, bufsz)
	char *buf;
	size_t bufsz;
{
	struct timeval tv;
	int fd, hv, n;
	char *s;

	fd = open(PATH_RANDOM, O_RDONLY);
	if (fd >= 0) {
		/*
		 * Urgh, some /dev/random's don't supply a very continuous
		 * stream of bits...
		 */
		for (s = buf; s - buf < (signed) bufsz; ) {
			n = read(fd, s, 1);
			if (n == -1)
				break;
			s += n;
		}

		close(fd);
		if (s - buf == (signed) bufsz)
			return;
	}
	/*
	 * /dev/random is not available, revert to a much weaker method
	 *
	 * XXX - This is not exactly strong, but it'll have to do for now
	 * as I don't want to write a real pseudo-random generator.
	 */
	(void) gettimeofday(&tv, NULL);
	hv = tv.tv_sec * tv.tv_usec;
	(void) gettimeofday(&tv, NULL);
	hv *= tv.tv_usec ^ hv;
	srandom(hv);
	for (s = buf; s - buf < (signed) bufsz; s++) {
		hv = random();
		*s = hv & 0x0f;
		hv = random();
		*s |= hv & 0xf0;
	}
}

