/***************************************************************************
 *
 * Copyright (c) 1998 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "htcp.h"

#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include "syslog-ng.h"
#include <syslog.h>
#include "var.h"
#include "ftable.h"
#include "log.h"
#include "utils.h"

#ifndef lint
static const char rcsid[]="$Id: htcp.c,v 1.34 1999/03/08 15:05:46 bazsi Exp $";
#endif

static int		htcp_checkin_reply	__P((int, htcp_state_t *));
static int		htcp_auth_reply		__P((int, htcp_state_t *));

static htcp_command_t   *htcp_find_command	__P((char *));

static htcp_header_t	*htcp_findheader	__P((htcp_header_t *, char *));
static void		htcp_addheader		__P((htcp_header_t **, htcp_header_t *));
static void		htcp_freeheaders	__P((htcp_header_t *));


static htcp_command_t htcp_commands[] = {
	{ "CHECKIN",	htcp_checkin_reply },
	{ "AUTH",	htcp_auth_reply },
	{ "",		NULL }
};

static char hexchars[] = "0123456789ABCDEF";

char *
htcp_bintohex(bin, hex, len)
	char *bin;
	char *hex;
	int len;
{
	int i;
	char *p = hex;
	
	for (i = 0; i < len; i++) {
		*(p++) = hexchars[(bin[i] & 0xf0) >> 4];
		*(p++) = hexchars[(bin[i] & 0x0f)];
	}
	*p = 0;
	return hex;
}

int
htcp_hexvalue(char c)
{
	if (c >= '0' && c <= '9') {
		return c - '0';
	}
	else if (c >= 'A' && c <= 'F') {
		return c - 'A' + 10;
	}
	return -1;
}

char *
htcp_hextobin(hex, bin)
	char *hex, *bin;
{
	char *p, *p2;
	int i;
	
	if (hex && bin) {
		p = hex;
		p2 = bin;
		i = strlen(p);
		while (i >= 2) {
			*p2 = (htcp_hexvalue(*p) << 4) + htcp_hexvalue(*(p+1));
			p += 2;
			p2++;
			i -= 2;
		}
	}
	return bin;
}

int
htcp_readline(fd, buf, buflen)
	int fd;
	char *buf;
	int buflen;
{
	char c, *p = buf;
	int cnt = 0, rc;
	
	cnt = 0;
	buflen--;
	alarm(1);
	rc = read(fd, &c, 1);
	alarm(0);
	while (buflen && rc == 1) {
		cnt++;
		*p = c;
		if (c == '\n') break;
		alarm(1);
		rc = read(fd, &c, 1);
		alarm(0);
		p++; buflen--;
	} 
	*p = 0;
	if (rc == -1 && errno == EWOULDBLOCK)
		return -1;
	return cnt;
}


int
htcp_init_client(st, secret, secretlen)
	htcp_state_t *st;
	char *secret;
	int secretlen;
{
	strncpy(st->hs_secret, secret, MIN(secretlen, MAX_SECRET_SIZE) - 1);
	return 0;
}

int  
htcp_init_server(st, secret, secretlen, sa, slen, input)
	htcp_state_t *st;
	char *secret;
	int secretlen;
	struct sockaddr *sa;
	int slen;
	htcp_input_proc input;
{
	strncpy(st->hs_secret, secret, MIN(secretlen, MAX_SECRET_SIZE) - 1);
	st->hs_clientaddr = malloc(slen);
	if (st->hs_clientaddr) {
		st->hs_input_proc = input;
		memcpy(st->hs_clientaddr, sa, slen);
		return 0;
	}
	return -1;
}

int
htcp_sendlog(fd, st, logline)
	int fd;
	htcp_state_t *st;
	char *logline;
{
	char line[MAXLINE + MAX_DIGEST_STR + 1];
	char digest[MAX_DIGEST_LEN];

	if (st) {
		if (st->hs_conn_flags & HTCP_F_LOGGING_IN) {
			return -2;	/* don't delete message, don't reopen connection */
		}
		strcpy(line, logline);
		chomp(line);
		hash_init(st->hs_hashing);
		hash_update(st->hs_hashing, st->hs_my_salt, st->hs_my_salt_size);
		hash_update(st->hs_hashing, line, strlen(line));
		hash_update(st->hs_hashing, st->hs_secret, strlen(st->hs_secret));
		hash_finalize(st->hs_hashing, digest);
		
		strncat(line, "\r\n", sizeof(line));
		htcp_bintohex(digest, line + strlen(line), hash_getdigestlen(st->hs_hashing));
		strncat(line, "\r\n", sizeof(line));
		if (write(fd, line, strlen(line)) < 0)
			return -1;
		else
			return 0;
	}
	return -1;
}

/*
 * htcp_client()
 *
 *    connect to a receiving TCP connection
 *
 *    returns 0 on success
 */
int
htcp_client(fd, st)
	int fd;
	htcp_state_t *st;
{
	htcp_header_t *hd;
	char buf[MAXLINE + MAX_DIGEST_STR + 1], digest[MAX_DIGEST_LEN], digest2[MAX_DIGEST_LEN], *p;
	
	switch (st->hs_req_state) {
	case HTCP_PREOPEN:
		st->hs_req_state = HTCP_CLIENT_CHECKIN;
		ftable_add_write_fd(fd);
		break;
	case HTCP_CLIENT_COMMAND:
		snprintf(buf, sizeof(buf), "%s %s\r\n", st->hs_req_cmd, st->hs_req_param ? st->hs_req_param : "");
		strcpy(buf, st->hs_req_cmd);
		if (st->hs_req_param) {
			strcat(buf, " ");
			strcat(buf, st->hs_req_param);
		}
		strcat(buf, "\r\n");
		if (write(fd, buf, strlen(buf)) < 0) {
			return -1;
		}
		free(st->hs_req_param);
		st->hs_req_param = NULL;
		st->hs_req_state = HTCP_CLIENT_HEADERS;
		break;
	case HTCP_CLIENT_HEADERS:
		hd = st->hs_req_headers;
		if (hd) {
			st->hs_req_headers = hd->hh_next;
			snprintf(buf, sizeof(buf), "%s: %s\r\n", hd->hh_name, hd->hh_value);
			if (write(fd, buf, strlen(buf)) < 0) {
				return -1;
			}
			hd->hh_next = NULL;
			htcp_freeheaders(hd);
		}
		else {
			write(fd, "\r\n", 2);
			st->hs_req_state = HTCP_CLIENT_GETSTAT;
			ftable_add_read_fd(fd);
			ftable_del_write_fd(fd);
		}
		break;
	case HTCP_CLIENT_GETSTAT:
		switch (htcp_readline(fd, buf, sizeof(buf))) {
		case 0:
			return -1;
		case -1:
			return 0;
		default:
			break;
		}
		chomp(buf);
		sscanf(buf, "%d", &st->hs_reply_status);
		htcp_freeheaders(st->hs_reply_headers);
		st->hs_reply_headers = NULL;
		st->hs_req_state = HTCP_CLIENT_GETHEADERS;
		break;
	case HTCP_CLIENT_GETHEADERS:
		switch (htcp_readline(fd, buf, sizeof(buf))) {
		case 0:
			return -1;
		case -1:
			return 0;
		default:
			break;
		}
		chomp(buf);
		if (buf[0] != 0) {
			hd = malloc(sizeof(htcp_header_t));
			if (!hd) return -1;
			hd->hh_name = strdup(strtok(buf, ":"));
			p = strtok(NULL, "");
			while (p && (*p == ' ' || *p == '\t')) p++;
			hd->hh_value = p ? strdup(p) : NULL;
			htcp_addheader(&st->hs_reply_headers, hd);
		}
		else {
			st->hs_req_state = st->hs_next_state;

			ftable_del_read_fd(fd);
			ftable_add_write_fd(fd);
		}
		break;
	case HTCP_CLIENT_CHECKIN:
		strcpy(st->hs_req_cmd, "CHECKIN");
		st->hs_req_param = NULL;
		hd = malloc(sizeof(htcp_header_t));
		if (!hd) return -1;
		hd->hh_name = strdup("HashType");
		hd->hh_value = strdup("SHA1");
		htcp_addheader(&st->hs_req_headers, hd);
		st->hs_conn_flags = HTCP_F_LOGGING_IN;
		st->hs_req_state = HTCP_CLIENT_COMMAND;
		st->hs_next_state = HTCP_CLIENT_POST_CHECKIN;
		break;
	case HTCP_CLIENT_POST_CHECKIN:
		if (st->hs_reply_status == 100) {
			htcp_header_t *hd;
			
			hd = htcp_findheader(st->hs_reply_headers, "HashType");
			if (hd) {
				st->hs_hashing = hash_create(hd->hh_value);
				if (st->hs_hashing == NULL) {
					return -1;
				}
				st->hs_conn_flags |= HTCP_F_CHECKEDIN;
				st->hs_req_state = HTCP_CLIENT_AUTH;
			}
			else {
				return -1;
			}
		}
		else {
			return -1;
		}
		break;
	case HTCP_CLIENT_AUTH:
		strcpy(st->hs_req_cmd, "AUTH");
		st->hs_req_param = NULL;
		/* generate salt */
		st->hs_my_salt_size = GEN_SALT_SIZE;
		st->hs_my_salt = malloc(st->hs_my_salt_size);
		if (!st->hs_my_salt) return -1;
		hash_getsalt(st->hs_my_salt, st->hs_my_salt_size);

		hd = malloc(sizeof(htcp_header_t));
		if (!hd) return -1;
		hd->hh_name = strdup("Salt");
		hd->hh_value = malloc(st->hs_my_salt_size * 2 + 1);
		if (!hd->hh_value) return -1;
		htcp_bintohex(st->hs_my_salt, hd->hh_value, st->hs_my_salt_size);
		
		htcp_addheader(&st->hs_req_headers, hd);

		hash_init(st->hs_hashing);
		hash_update(st->hs_hashing, st->hs_my_salt, st->hs_my_salt_size);
		hash_update(st->hs_hashing, st->hs_secret, strlen(st->hs_secret));
		hash_finalize(st->hs_hashing, digest);
		
		hd = malloc(sizeof(htcp_header_t));
		if (!hd) return -1;
		hd->hh_name = strdup("Hash");
		hd->hh_value = malloc(hash_getdigestlen(st->hs_hashing) * 2 + 1);
		if (!hd->hh_value) return -1;
		htcp_bintohex(digest, hd->hh_value, st->hs_hashing->ha_type->ht_diglen);
		htcp_addheader(&st->hs_req_headers, hd);

		st->hs_req_state = HTCP_CLIENT_COMMAND;
		st->hs_next_state = HTCP_CLIENT_POST_AUTH;
		break;
	case HTCP_CLIENT_POST_AUTH:
		if (st->hs_reply_status == 100) {
			htcp_header_t *hash, *salt;
			
			hash = htcp_findheader(st->hs_reply_headers, "Hash");
			salt = htcp_findheader(st->hs_reply_headers, "Salt");
			if (hash && salt) {
				st->hs_his_salt_size = strlen(salt->hh_value) / 2;
				st->hs_his_salt = malloc(st->hs_his_salt_size);
				if (!st->hs_his_salt) return -1;
				htcp_hextobin(salt->hh_value, st->hs_his_salt);
				
				hash_init(st->hs_hashing);
				hash_update(st->hs_hashing, st->hs_his_salt, st->hs_his_salt_size);
				hash_update(st->hs_hashing, st->hs_secret, strlen(st->hs_secret));
				hash_finalize(st->hs_hashing, digest);
				
				htcp_hextobin(hash->hh_value, digest2);
				if ((hash_getdigestlen(st->hs_hashing) != strlen(hash->hh_value) / 2) || memcmp(digest, digest2, hash_getdigestlen(st->hs_hashing)) != 0) {
					return -1;
				}
				st->hs_conn_flags &= ~HTCP_F_LOGGING_IN;
				st->hs_conn_flags |= HTCP_F_AUTHORIZED;
				st->hs_req_state = HTCP_CLIENT_IDLE;
				ftable_del_read_fd(fd);
			}
			else {
				return -1;
			}
		}
		else {
			return -1;
		}
		break;
	case HTCP_CLIENT_IDLE:
		ftable_del_read_fd(fd);
		ftable_del_write_fd(fd);
		break;
	}
	return 0;
}

int
htcp_checkin_reply(fd, st)
	int fd;
	htcp_state_t *st;
{
	htcp_header_t *hd;
	
	/* todo: hash-type negotiation */
	if (st->hs_conn_flags == 0) {
		st->hs_conn_flags = HTCP_F_CHECKEDIN;
		st->hs_reply_status = 100;
		hd = malloc(sizeof(htcp_header_t));
		if (!hd) return -1;
		hd->hh_name = strdup("HashType");
		hd->hh_value = strdup("SHA1");
		htcp_addheader(&st->hs_reply_headers, hd);
		st->hs_hashing = hash_create("SHA1");
	}
	else {
		st->hs_reply_status = 999;
	}
	return 0;
}

int
htcp_auth_reply(fd, st)
	int fd;
	htcp_state_t *st;
{
	htcp_header_t *salt, *hash, *hd;
	char digest[MAX_DIGEST_LEN], hisdigest[MAX_DIGEST_LEN];
	int dlen = hash_getdigestlen(st->hs_hashing);
	
	if (st->hs_conn_flags & HTCP_F_CHECKEDIN) {
		salt = htcp_findheader(st->hs_req_headers, "Salt");
		hash = htcp_findheader(st->hs_req_headers, "Hash");
		if (salt && hash) {
			st->hs_his_salt_size = strlen(salt->hh_value) / 2;
			st->hs_his_salt = malloc(st->hs_his_salt_size);
			if (!st->hs_his_salt) return -1;
			htcp_hextobin(salt->hh_value, st->hs_his_salt);
			st->hs_my_salt_size = GEN_SALT_SIZE;
			st->hs_my_salt = malloc(st->hs_my_salt_size);
			if (!st->hs_my_salt) return -1;
			hash_getsalt(st->hs_my_salt, st->hs_my_salt_size);
			
			htcp_hextobin(hash->hh_value, hisdigest);
			
			hash_init(st->hs_hashing);
			hash_update(st->hs_hashing, st->hs_his_salt, st->hs_his_salt_size);
			hash_update(st->hs_hashing, st->hs_secret, strlen(st->hs_secret));
			hash_finalize(st->hs_hashing, digest);
			
			if ((strlen(hash->hh_value) / 2) == dlen && memcmp(digest, hisdigest, dlen) == 0) {
				st->hs_reply_status = 100;		/* auth ok */
				hd = malloc(sizeof(htcp_header_t));
				if (!hd) return -1;
				hd->hh_name = strdup("Salt");
				hd->hh_value = malloc(st->hs_my_salt_size * 2 + 1);
				if (!hd->hh_value) return -1;
				htcp_bintohex(st->hs_my_salt, hd->hh_value, st->hs_my_salt_size);
				htcp_addheader(&st->hs_reply_headers, hd);
				hd = malloc(sizeof(htcp_header_t));
				if (!hd) return -1;
				
				hash_init(st->hs_hashing);
				hash_update(st->hs_hashing, st->hs_my_salt, st->hs_my_salt_size);
				hash_update(st->hs_hashing, st->hs_secret, strlen(st->hs_secret));
				hash_finalize(st->hs_hashing, digest);
				
				hd->hh_name = strdup("Hash");
				hd->hh_value = malloc(dlen * 2 + 1);
				if (!hd->hh_value) return -1;
				htcp_bintohex(digest, hd->hh_value, dlen);
				htcp_addheader(&st->hs_reply_headers, hd);
				st->hs_conn_flags |= HTCP_F_AUTHORIZED;
				/* afinet specific !!! */
				log_printf(LOG_SYSLOG | LOG_ERR, "htcp authorization ok for %s\n", inet_ntoa(((struct sockaddr_in *)st->hs_clientaddr)->sin_addr));
			}
			else {
				log_printf(LOG_SYSLOG | LOG_ERR, "htcp authorization failed for %s\n", inet_ntoa(((struct sockaddr_in *)st->hs_clientaddr)->sin_addr));
				st->hs_reply_status = 901;
			}
		}
		else {
			st->hs_reply_status = 901;
		}
	}
	else {
		st->hs_reply_status = 999;
	}
	return 0;
}

/*
 * htcp_server()
 *
 */
int
htcp_server(fd, st)
	int fd;
	htcp_state_t *st;
{
	char line[MAXLINE + 1], *p, digest[MAX_DIGEST_LEN], rdigest[MAX_DIGEST_LEN];
	htcp_header_t *hd;
	
	switch (st->hs_req_state) {
	case HTCP_PREOPEN:
		st->hs_req_state = HTCP_SRV_PRECOMMAND;
		ftable_add_read_fd(fd);
		break;
	case HTCP_SRV_PRECOMMAND:
		/* fetch command */
		if ((st->hs_conn_flags & HTCP_F_LOGALLOWED) == HTCP_F_LOGALLOWED) {
			st->hs_req_state = HTCP_SRV_ACCEPT_LOG;
			return 0;
		}
		switch (htcp_readline(fd, line, sizeof(line))) {
		case 0:
			return -1;
		case -1:
			return 0;
		default:
			break;
		}

		chomp(line);
		p = strtok(line, " ");
		strncpy(st->hs_req_cmd, p ? p : line, sizeof(st->hs_req_cmd) - 1);
		st->hs_req_cmd[sizeof(st->hs_req_cmd) - 1] = 0;
		p = strtok(NULL, "");
		st->hs_req_param = p ? strdup(p) : NULL;
		st->hs_req_state = HTCP_SRV_PREHEADER;
		break;
	case HTCP_SRV_PREHEADER:
		/* fetch a header line */
		switch (htcp_readline(fd, line, sizeof(line))) {
		case 0:
			return -1;
		case -1:
			return 0;
		default:
			break;
		}

		chomp(line);
		if (line[0] != 0) {
			hd = malloc(sizeof(htcp_header_t));
			if (!hd) return -1;
			hd->hh_name = strdup(strtok(line, ":"));
			p = strtok(NULL, "");
			while (p && (*p == ' ' || *p == '\t')) p++;
			hd->hh_value = p ? strdup(p) : NULL;
			htcp_addheader(&st->hs_req_headers, hd);
		}
		else {
			htcp_command_t *hc;
			
			st->hs_req_state = HTCP_SRV_REPLY_STATUS;
			hc = htcp_find_command(st->hs_req_cmd);
			if (hc) {
				if (hc->hc_reply(fd, st) == -1) {
					return -1;
				}
			}
			else {
				st->hs_reply_status = 999;
			}
			free(st->hs_req_param);	/* we don't need that anymore */
			st->hs_req_param = NULL;
			htcp_freeheaders(st->hs_req_headers);
			st->hs_req_headers = NULL;
			ftable_del_read_fd(fd);
			ftable_add_write_fd(fd);
			st->hs_req_state = HTCP_SRV_REPLY_STATUS;
		}
		break;
	case HTCP_SRV_REPLY_STATUS:
		snprintf(line, sizeof(line), "%d\r\n", st->hs_reply_status);
		write(fd, line, strlen(line));
		st->hs_req_state = HTCP_SRV_REPLY_HEADERS;
		break;
	case HTCP_SRV_REPLY_HEADERS:
		hd = st->hs_reply_headers;
		if (hd) {
			st->hs_reply_headers = st->hs_reply_headers->hh_next;
			snprintf(line, sizeof(line), "%s: %s\r\n", hd->hh_name, hd->hh_value);
			write(fd, line, strlen(line));
			hd->hh_next = NULL;
			htcp_freeheaders(hd);
		}
		else {
			write(fd, "\r\n", 2);
			st->hs_req_state = HTCP_SRV_PRECOMMAND;
			ftable_del_write_fd(fd);
			ftable_add_read_fd(fd);
		}
		break;
	case HTCP_SRV_ACCEPT_LOG:
		switch (htcp_readline(fd, st->hs_linebuf, sizeof(st->hs_linebuf))) {
		case 0:
			return -1;
		case -1:
			return 0;
		default:
			break;
		}

		chomp(st->hs_linebuf);
		st->hs_req_state = HTCP_SRV_ACCEPT_HASH;
		break;
	case HTCP_SRV_ACCEPT_HASH:
		switch (htcp_readline(fd, st->hs_linehash, sizeof(st->hs_linehash))) {
		case 0:
			return -1;
		case -1:
			return 0;
		default:
			break;
		}

		st->hs_req_state = HTCP_SRV_ACCEPT_LOG;
		chomp(st->hs_linehash);
		
		hash_init(st->hs_hashing);
		hash_update(st->hs_hashing, st->hs_his_salt, st->hs_his_salt_size);
		hash_update(st->hs_hashing, st->hs_linebuf, strlen(st->hs_linebuf));
		hash_update(st->hs_hashing, st->hs_secret, strlen(st->hs_secret));
		hash_finalize(st->hs_hashing, digest);

		htcp_hextobin(st->hs_linehash, rdigest);
		if (hash_getdigestlen(st->hs_hashing) == (strlen(st->hs_linehash) / 2) && memcmp(digest, rdigest, hash_getdigestlen(st->hs_hashing)) == 0) {
			strcpy(line, st->hs_linebuf);
		}
		else {
			snprintf(line, sizeof(line), "%s (hash mismatch)\n", st->hs_linebuf);
		}
		st->hs_input_proc(ftable_get_fte(fd), st->hs_clientaddr, line);
		break;
	}
	return 0;
}

int
htcp_close(fd, st)
	int fd;
	htcp_state_t *st;
{
	if (st && fd != -1) {
		ftable_del_fd(fd);
		close(fd);
		free(st->hs_clientaddr);
		free(st->hs_req_param);
		free(st->hs_his_salt);
		free(st->hs_my_salt);
		htcp_freeheaders(st->hs_req_headers);
		htcp_freeheaders(st->hs_reply_headers);
		hash_destroy(st->hs_hashing);
		memset(st, 0, sizeof(htcp_state_t));
	}
	return 0;
}

htcp_command_t *
htcp_find_command(char *name)
{
	int i;
	
	for (i = 0; htcp_commands[i].hc_cmd[0]; i++) {
		if (strcmp(htcp_commands[i].hc_cmd, name) == 0)
			return htcp_commands + i;
	}
	return NULL;
}

void
htcp_addheader(head, hd)
	htcp_header_t **head;
	htcp_header_t *hd;
{
	hd->hh_next = *head;
	*head = hd;
}

htcp_header_t *
htcp_findheader(head, name)
	htcp_header_t *head;
	char *name;
{
	while (head) {
		if (strcmp(head->hh_name, name) == 0) {
			return head;
		}
		head = head->hh_next;
	}
	return NULL;
}

void
htcp_freeheaders(hd1)
	htcp_header_t *hd1;
{
	htcp_header_t *hd2;
	
	/* free headers */
	hd2 = NULL;
	while (hd1) {
		hd2 = hd1->hh_next;
		free(hd1->hh_name);
		free(hd1->hh_value);
		free(hd1);
		hd1 = hd2;
	}
}
