/***************************************************************************
 *
 * Copyright (c) 1998 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Based on the original nsyslog by
 *
 * Copyright (C) 1997 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 ***************************************************************************/

#include "log.h"

#include <stdio.h>
#include <utmp.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <syslog.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <ctype.h>
#include <signal.h>
#include <string.h>
#include <errno.h>

#ifdef USE_STDARG
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifdef USE_PTHREADS
#include <pthread.h>
#endif

#include "syslog-ng.h"
#include "utils.h"

#include "var.h"
#include "log.h"
#include "afinet.h"
#include "afunix.h"
#include "affile.h"
#include "afuser.h"
#include "afpipe.h"
#include "ftable.h"
#include "hash.h"
#include "htcp.h"
#include "y.tab.h"

#if !defined(lint)
static const char rcsid[] = "$Id: log.c,v 1.34 1999/01/19 14:45:17 bazsi Exp $";
#endif

static	int		log_recursions = 0;

static	int		log_chkfilter	__P((logmsgtype_t *, var_t *));
static	int		log_queue_add	__P((filetype_t *, logmsg_t *));
static	logmsg_t *	log_queue_last	__P((filetype_t *));
static	void		log_write_dups	__P((filetype_t *));
static	int		log_add_msg	__P((filetype_t *, logmsgtype_t *));
static	void		log_add_dest_fd	__P((filetype_t *));
static	int		log_match_pri	__P((int, int, int));
static	void		log_distribute_logmsg	__P((logmsgtype_t *));

#ifdef USE_PTHREADS
pthread_mutex_t loglock = PTHREAD_MUTEX_INITIALIZER;
#endif

/*
 * If a log message doesn't have a reocgnisable time-date stamp at the front,
 * insert one.
 */
static void
log_add_date(lmp)
	logmsgtype_t *lmp;
{
	struct tm *tm;
	char ts[17];

	if (!lmp->lm_time) {
		if (!(tm = localtime(&lmp->lm_recvd))) {
			snprintf(ts, sizeof(ts), "%d", lmp->lm_recvd);
		} else {
			strftime(ts, sizeof(ts), "%h %e %H:%M:%S", tm);
			ts[16] = '\0';
		}
		lmp->lm_time = strdup(ts);
	}
}

/*
 * add the hostname of the source into the message.  This goes between the
 * date-time stamp and the rest of the message so it may need proper
 * insertion.
 */
static void
log_add_host(lmp)
	logmsgtype_t *lmp;
{
	char buf[MAXLINE + 1];
	char *host;

	host = lmp->lm_host ? lmp->lm_host : get_shorthostname();
	if (strchr(host, '/') == NULL && strchr(host, '@') == NULL) {
		if (lmp->lm_fte)
			snprintf(buf, sizeof(buf), "%s@%s", lmp->lm_fte->fte_owner->v_name, host);
		else
			snprintf(buf, sizeof(buf), "internal@%s", host);
		free(lmp->lm_host);
		lmp->lm_host = strdup(buf);
	}
	else {
		if (!lmp->lm_host) lmp->lm_host = strdup(host);
	}
}

logmsg_t *
new_logmsg(msg)
	char *msg;
{
	logmsg_t *lg;
	char *s, *ssave, oldch;
	int pri = 0;

	lg = (logmsg_t *)calloc(1, sizeof(logmsg_t));
	if (lg) {
	
		lg->lg_ref = 1;
		s = msg;
		if (*s == '<' && isdigit(*(s+1))) {
			s++;
			for (; *s && isdigit(*s); s++)
				pri = 10 * pri + (*s - '0');
			if (*s && *s == '>')
				s++;
			if (pri >= (LOG_NFACILITIES << 3))
				pri = LOG_USER|LOG_NOTICE;
		} else
			pri = LOG_USER|LOG_NOTICE;
		lg->lg_pri = pri;

		if (strlen(s) >= 16 && s[3] == ' ' && s[6] == ' ' &&
		    s[9] == ':' && s[12] == ':' && s[15] == ' ') {
		    	/* timestamp found */
			s[15] = 0;
			lg->lg_time = strdup(s);
			s += 16;
			/* check whether there's a host name */
			ssave = s;
			while (*s != ' ' && *s != '[' && *s != ':' && *s) s++;
			if (*s == ' ') {
				/* found hostname */
				*s = 0;
				lg->lg_host = strdup(ssave);
				s++;
			} else {
				s = ssave;
			}
			ssave = s;
			while (*s != ':' && *s != '[' && *s) s++;
			if (*s == ':' || *s == '[') {
				/* found progname */
				oldch = *s;
				*s = 0;
				lg->lg_prog = strdup(ssave);
				*s = oldch;
			}
			s = ssave;
		}
		chomp(s);
		lg->lg_line = strdup(s);
	}
	return lg;
}

void
free_logmsg(lgp)
	logmsg_t **lgp;
{
	logmsg_t *lg = *lgp;

	lg->lg_ref--;
	if (!lg->lg_ref) {
		free(lg->lg_prog);
		free(lg->lg_host);
		free(lg->lg_time);
		free(lg->lg_line);
		free(lg);
	}
	*lgp = NULL;
}

void
#ifdef USE_STDARG
log_debug(char *fmt, ...)
#else
log_debug(fmt, va_alist)
	char *fmt;
	va_dcl
#endif
{
	va_list pvar;
#ifdef USE_STDARG
	va_start(pvar, fmt);
#else
	va_start(pvar);
#endif
	vfprintf(stderr, fmt, pvar);

	va_end(pvar);
}

void
#ifdef USE_STDARG
log_printf(int pri, char *fmt, ...)
#else
log_printf(pri, fmt, va_alist)
	int pri;
	char	*fmt;
	va_dcl
#endif
{
	char buf[MAXLINE + 1];
	va_list pvar;
	logmsgtype_t lm;
	int n;

#ifdef USE_STDARG
	va_start(pvar, fmt);
#else
	va_start(pvar);
#endif
	if (started && (log_recursions == 0)) {
		n = snprintf(buf, sizeof(buf), "<%d>syslog-ng[%d]: ", pri, getpid());
		vsnprintf(buf + n, sizeof(buf) - n, fmt, pvar);
		lm.lm_log = new_logmsg(buf);
		log_fakemsg(&lm);
	} else {
		vfprintf(stderr, fmt, pvar);
		fputc('\n', stderr);
	}
	va_end(pvar);
}

/* fakes a log message */
void
log_fakemsg(lmp)
	logmsgtype_t *lmp;
{
	lmp->lm_recvd = time(NULL);
	lmp->lm_fte = NULL;
	lmp->lm_flags = FLG_INT;
	logmsg(lmp);
}

#define NEXT_LOG_NDX(x)		(((x) >= MAXLOGQUEUE - 1) ? 0 : ((x) + 1))

static int
log_queue_add(ft, lmp)
	filetype_t *ft;
	logmsg_t *lmp;
{
	log_add_dest_fd(ft);
	if (NEXT_LOG_NDX(ft->ft_wrptr) != ft->ft_rdptr) {
		ft->ft_logqueue[ft->ft_wrptr] = lmp;
		ft->ft_wrptr = NEXT_LOG_NDX(ft->ft_wrptr);
		return 0;
	}
	return -1;
}

logmsg_t *
log_queue_get(ft)
	filetype_t *ft;
{
	int rd;
	
	if (ft->ft_rdptr != ft->ft_wrptr) {
		rd = ft->ft_rdptr;
		ft->ft_rdptr = NEXT_LOG_NDX(ft->ft_rdptr);
		return ft->ft_logqueue[rd];
	}
	return NULL;
}

logmsg_t *
log_queue_last(ft)
	filetype_t *ft;
{
	if (ft->ft_rdptr != ft->ft_wrptr) {
		return ft->ft_logqueue[ft->ft_rdptr];
	}
	return NULL;
}

int
log_queue_run(ft, func, param)
	filetype_t *ft;
	log_output_func_t func;
	void *param;
{
	logmsg_t *lg;
	
	lg = log_queue_last(ft);
	if (lg) {
		if (func(lg, param) == 0) {
			log_queue_get(ft);
			free_logmsg(&lg);
		}
		return log_queue_last(ft) == NULL; /* more to check */
	}
	else {
		return 1;	/* fd may be unselected */
	}
}

static void
log_write_dups(ft)
	filetype_t *ft;
{
	char msg[MAXLINE + 1];
	logmsgtype_t lm;

	snprintf(msg, sizeof(msg), "syslog-ng[%d]: last message repeated %d times",
		getpid(), ft->ft_mcount);
	lm.lm_log = new_logmsg(msg);
	lm.lm_recvd = time(NULL);
	lm.lm_fte = NULL;
	lm.lm_flags = FLG_INT;
	lm.lm_pri = LOG_SYSLOG | LOG_INFO;
	log_add_date(&lm);
	log_add_host(&lm);
	log_queue_add(ft, lm.lm_log);
}

static int
log_add_msg(ft, lmp)
	filetype_t *ft;
	logmsgtype_t *lmp;
{

	if (ft->ft_lastentry && ft->ft_line) {

		if ((lmp->lm_flags & FLG_INT) ||
		    (ft->ft_pri != lmp->lm_pri) ||
		    ((0 == ft->ft_prog) != (0 == lmp->lm_prog)) ||
		    (ft->ft_prog && lmp->lm_prog &&
		     strcmp(ft->ft_prog, lmp->lm_prog)) ||
		     strcmp(ft->ft_line, lmp->lm_line) ||
		     strcmp(ft->ft_host, lmp->lm_host)) {

			/* different from the previous one */
			if (ft->ft_mcount > 1) {
				log_write_dups(ft);
			}
			free_logmsg(&ft->ft_lastentry);
			ft->ft_mcount = 1;
		} else {
			/* same as the previous one */
			
			ft->ft_lasttime = time(NULL);
			ft->ft_mcount++;
			return 1;
		}

	}
	if (log_queue_add(ft, lmp->lm_log) == -1) { 
		return 1;
	} 
	lmp->lm_log->lg_ref += 2;
	ft->ft_lastentry = lmp->lm_log;
	ft->ft_lasttime = time(NULL);
	return 0;
}


static void
log_add_dest_fd(ft)
	filetype_t *ft;
{
	int fd = 0;
	
	switch (VAR_TYPE(ft->ft_type)) {
	case DEST_FILE:
		fd = ((filespec_t *) ft->ft_ptr)->fs_fd;
		if (fd == -1)
			affile_reconnect(ft);
		fd = ((filespec_t *) ft->ft_ptr)->fs_fd;
		break;
	case DEST_INET:
		fd = ((netspec_t *) ft->ft_ptr)->ns_fd;
		if (fd == -1)
			afinet_reconnect(ft);
		fd = ((netspec_t *) ft->ft_ptr)->ns_fd;
		break;
	case DEST_UNIX:
		fd = ((unixspec_t *) ft->ft_ptr)->xs_fd;
		if (fd == -1)
			afunix_reconnect(ft);
		fd = ((unixspec_t *) ft->ft_ptr)->xs_fd;
		break;
	case DEST_CMD:
		fd = ((cmdspec_t *) ft->ft_ptr)->cm_fd;
		if (fd == -1)
			afpipe_reconnect(ft);
		fd = ((cmdspec_t *) ft->ft_ptr)->cm_fd;
		break;
	case DEST_USER:
		afuser_output_avail = 1;
		break;
	}
	ftable_add_write_fd(fd);
	return;
}

void
log_timeout_dups()
{
	filetype_t *ft;
	sdlist_t *sd;
	var_t *d;
	int i;

	for (sd = sdlist; sd; sd = sd->sd_next) {
		d = sd->sd_dst;

		for (i = 0; i < d->v_num; i++) {
			ft = d->v_list + i;

			if (ft->ft_lastentry) {
				if (ft->ft_mcount > 1) log_write_dups(ft);
				free_logmsg(&ft->ft_lastentry);
				ft->ft_mcount = 0;
			}
		}
	}
}

void log_timeout_marks()
{
	filetype_t *ft;
	logmsgtype_t lm;
	sdlist_t *sd;
	var_t *d;
	int i;
	char msg[MAXLINE + 1];
	
	for (sd = sdlist; sd; sd = sd->sd_next) {
		d = sd->sd_dst;

		for (i = 0; i < d->v_num; i++) {
			ft = d->v_list + i;

			snprintf(msg, sizeof(msg), "syslog-ng[%d]:  --- MARK ---", getpid());
			lm.lm_log = new_logmsg(msg);
			lm.lm_pri = LOG_SYSLOG | LOG_INFO;
			lm.lm_flags |= FLG_MARK;
			lm.lm_fte = NULL;
			lm.lm_recvd = time(NULL);
			log_add_date(&lm);
			log_add_host(&lm);
			log_queue_add(ft, lm.lm_log);
		}
	}
}

static int
log_match_pri(p1, p2, comp)
	int p1, p2, comp;
{

	p1 &= LOG_PRIMASK;
	p2 &= LOG_PRIMASK;

	switch (comp)
	{
	case IL_EQ :
		return (p1 == p2);
	case IL_NEQ :
		return (p1 != p2);
	case IL_LT :
		return (p1 > p2);
	case IL_LE :
		return (p1 >= p2);
	case IL_GT :
		return (p1 < p2);
	case IL_GE :
		return (p1 <= p2);
	}
	log_printf(LOG_SYSLOG | LOG_ERR, "unknown comparator: %d\n", comp);
	return -1;
}


/*
 * log_chkfilter - check if a message matches a given list of filters.
 *             return -1 if it doesn't match
 *             return 0 if it does match
 */
static int
log_chkfilter(lmp, v)
	logmsgtype_t *lmp;
	var_t *v;
{
	matchspec_t	*ms;
	matchfilespec_t	*mf;
	matchprogspec_t	*mp;
	nmaskspec_t	*nm;
	facspec_t	*fs;
	prispec_t	*ps;
	facprispec_t	*fp;
	filetype_t *ft;
	int i, ok = 0;

	for (i = 0, ft = v->v_list; i < v->v_num; i++, ft++) {
		switch (VAR_TYPE(ft->ft_type))
		{
		case FIL_FAC :
			fs = (facspec_t *)ft->ft_ptr;
			ok = fs->fs_comp;

			if (!OK_FAC(lmp->lm_pri, fs->fs_fac))
				return !ok;
			break;
		case FIL_PRI :
			ps = (prispec_t *)ft->ft_ptr;
			ok = ps->ps_not;

			if (!log_match_pri(lmp->lm_pri, ps->ps_pri, ps->ps_comp))
				return !ok;
			break;
		case FIL_FACPRI :
			fp = (facprispec_t *)ft->ft_ptr;
			ok = fp->fp_not;
			if (!OK_FAC(lmp->lm_pri, fp->fp_facpri))
				return !ok;
			if (!log_match_pri(lmp->lm_pri, fp->fp_facpri, fp->fp_comp))
				return !ok;
			break;
		case FIL_WRAP :
			if (VAR_TYPE(lmp->lm_fte->fte_type) != SRC_INET)
				return !ok;

			nm = (nmaskspec_t *)ft->ft_ptr;
			ok = nm->nm_comp;

#if 0
			if (!OK_INET(lmp->lm_src, nm))
				return !ok;
#endif
			break;

		case FIL_MATCH :
			ms = (matchspec_t *)ft->ft_ptr;
			ok = ms->ms_comp;
#ifdef USE_POSIX_REGEX
			if (regexec(&ms->ms_buf, lmp->lm_line, 0, NULL, 0))
#else
			if (!step(lmp->lm_line, ms->ms_buf))
#endif
				return !ok;
			break;
		case FIL_MATCHFILE :
			mf = (matchfilespec_t *)ft->ft_ptr;
			ok = mf->mf_comp;

			for (i = 0; i < mf->mf_nlines; i++)
#ifdef USE_POSIX_REGEX
				if (regexec(&mf->mf_bufs[i], lmp->lm_line, 0,
					     NULL, 0))
#else
				if (!step(lmp->lm_line, mf->mf_bufs[i]))
#endif
					return !ok;
			break;
		case FIL_MATCHPROG :
			mp = (matchprogspec_t *)ft->ft_ptr;
			ok = mp->mp_comp;

			if (!lmp->lm_prog || strcmp(mp->mp_prog, lmp->lm_prog))
				return !ok;
			break;
		default :
			break;
		}
	}
	return ok;
}


/*
 * scan the list of log rules which join sources, filters and destinations
 * for rules which match this source.
 */
static void
log_distribute_logmsg(lmp)
	logmsgtype_t *lmp;
{
	fillist_t *fa, *fo;
	sdlist_t *s;
	var_t *d;
	int i, j;

	for (s = sdlist; s; s = s->sd_next)
		if ((lmp->lm_flags & FLG_INT) || (lmp->lm_fte && s->sd_src == lmp->lm_fte->fte_owner)) {
			if (s->sd_fil && ((lmp->lm_flags & FLG_MARK) == 0)) {
				/*
				 * Check filters which are to be applied in
				 * sequence and check each chain of these.
				 */
				for (fo = s->sd_fil; fo; fo = fo->fl_or) {
					for (fa = fo; fa; fa = fa->fl_and) {
						j = log_chkfilter(lmp, fa->fl_fil);
						if ((j && !fa->fl_comp) ||
						    (!j && fa->fl_comp))
							break;
					}
					if ((!fa && !fo->fl_comp) ||
					    (fa && fo->fl_comp))
						break;
				}
				if (!fo)
					continue;
			}

			/*
			 * passed the filtering, so write it out to all the
			 * various destinations.
			 */
			d = s->sd_dst;
			for (i = 0; i < d->v_num; i++) {
				log_add_msg(d->v_list + i, lmp);
			}
		}
}

void
logmsg(lmp)
	logmsgtype_t *lmp;
{
	
#ifdef USE_PTHREADS
	pthread_mutex_lock(&loglock);
#endif

	log_recursions++;

	log_add_date(lmp);
	log_add_host(lmp);
	
	log_distribute_logmsg(lmp);

	free_logmsg(&lmp->lm_log);
#ifdef USE_PTHREADS
	pthread_mutex_unlock(&loglock);
#endif
	log_recursions--;
	return;
}

void
log_printall(char *buf, int len, filetable_entry_t *fte)
{
	char buf2[MAXLINE + 1];
	logmsgtype_t lm;
	char *dst, *p, *last;

	buf[len] = 0;	
	if (fte->fte_part) {
		strcpy(buf2, fte->fte_part);
		dst = buf2 + strlen(fte->fte_part);
		free(fte->fte_part);
		fte->fte_part = NULL;
	}
	else {
		dst = buf2;
	}
	p = buf;
	while (p - buf < len) {
		last = p;
		while (((dst - buf2) < MAXLINE) && ((p - buf) < len) && (*p != '\n' && *p != '\0')) {
			*dst = *p;
			p++;
			dst++;
		}
		if (dst - buf2 == MAXLINE) {
			while (((p - buf) < len) && (*p != '\n' && *p != '\0')) p++;
		}
		else {
			if (p - buf == len) {
				fte->fte_part = strdup(last);
				break;
			}
		}
		*dst = 0;
		lm.lm_flags = FLG_SYS;
		lm.lm_fte = fte;
		lm.lm_log = new_logmsg(buf2);
		lm.lm_recvd = time(NULL);
		logmsg(&lm);
		dst = buf2;
		while ((p - buf < len) && (*p == '\n' || *p == '\0')) p++;
	}

}
