/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: main.c,v 1.15 1999/06/08 19:47:31 bazsi Exp $
 *
 ***************************************************************************/

#include "syslog-ng.h"
#include "cfgfile.h"
#include "afinter.h"
#include "werror.h"

#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>

#include "main.c.x"

char cfgfilename[128] = PATH_SYSLOG_NG_CONF;

/* CLASS:
   (class
     (name reread_config)
     (super callback)
     (vars
       (backend object syslog_backend)
       (state simple int)))
*/

static void do_reread_config(struct callback *c)
{
	CAST(reread_config, self, c);

	switch (self->state) {
	case 0:
	case 2:
		CONFIG_DESTROY(self->backend->configuration);
		self->state++;
		self->backend->oldconfig = self->backend->configuration;
		self->backend->configuration = self->backend->newconfig;
		io_callout(&self->backend->super, 0, c);
		break;
	case 1:
		if (CONFIG_INIT(self->backend->configuration)) {
			notice("new configuration initialized\n");
			self->backend->newconfig = NULL;
			self->backend->oldconfig = NULL;
		}
		else {
			/* initializing the new configuration was not 
			   successful, try to revert to the old one */
			self->state++;
			self->backend->newconfig = self->backend->oldconfig;
			self->backend->oldconfig = NULL;
			io_callout(&self->backend->super, 0, c);

		}
		break;
	case 3:
		if (CONFIG_INIT(self->backend->configuration)) {
			notice("Initializing new configuration failed, reverting to old config.\n");
		}
		else {
			/* this is a fatal error, but no stderr & no log output */
			exit(137);
		}
		break;
	}
}

/* CLASS:
   (class
     (name syslog_backend)
     (super io_backend)
     (vars
       (newconfig object syslog_conf)
       (oldconfig object syslog_conf)
       (configuration object syslog_conf)))
*/

int sighuprecvd = 0;

void sig_hup(int signo)
{
	sighuprecvd = 1;
	signal(SIGHUP, sig_hup);
}

int sigtermrecvd = 0;

void sig_term(int signo)
{
	sigtermrecvd = 1;
	signal(SIGTERM, sig_term);
}

int main_loop(struct syslog_backend *backend)
{
	signal(SIGPIPE, SIG_IGN);
	signal(SIGHUP, sig_hup);

	while (io_iter(&backend->super)) {
		if (sighuprecvd) {

			notice("SIGHUP received, restarting syslog-ng\n");
			backend->newconfig = make_syslog_conf(cfgfilename, &backend->super);
			if (backend->newconfig) {
				static struct reread_config r = 
				{ { STATIC_HEADER, do_reread_config }, NULL, 0 };
				r.backend = backend;
				r.state = 0;
				io_callout(&backend->super, 0, &r.super);
			}
			else {
				notice("Syntax error reading config file, using old configuration\n");
			}
			sighuprecvd = 0;
		}
		if (sigtermrecvd) {
			notice("syslog-ng version %z exiting\n", VERSION);
			CONFIG_DESTROY(backend->configuration);
			break;
		}
	}
	return 0;
}

void go_background()
{
	pid_t pid;
	
	pid = fork();
	if (pid == 0) {
		return;
	}
	else if (pid == -1) {
		werror("Cannot fork(), (%z)\n", strerror(errno));
		exit(1);
	}
	signal(SIGTERM, sig_term);
	while (sigtermrecvd == 0)
		pause();
	exit(0);
}

void usage(void)
{
	werror("Usage: syslog-ng [options]\n"
	       "Accept and manage system log messages\n\n"
	       "Options:\n"
	       "  -d, --debug                      Turn on debugging messages\n"
	       "  -v, --verbose                    Be a bit more verbose\n"
	       "  -f <fname>, --cfgfile=<fname>    Set config file name, default=" PATH_SYSLOG_NG_CONF "\n"
#ifdef YYDEBUG
	       "  -y, --yydebug                    Turn on yacc debug messages\n"
#endif
		);

	exit(0);
}

#ifdef YYDEBUG
extern int yydebug;
#endif

int main(int argc, char **argv)
{
	int do_fork = 1;
	int opt;

	NEW(syslog_backend, backend);
	struct option syslog_ng_options[] = {
		{ "cfgfile", required_argument, NULL, 'f' },
		{ "debug", no_argument, NULL, 'd' },
		{ "verbose", no_argument, NULL, 'v' },
		{ "help", no_argument, NULL, 'h' },
#ifdef YYDEBUG
		{ "yydebug", no_argument, NULL, 'y' },
#endif
		{ NULL, 0, NULL, 0 }
	};
#ifdef HAVE_MALLOPT
        mallopt(M_TOP_PAD, 0x10000);
#endif
/*	gc_threshold = 0;*/

	while ((opt = getopt_long(argc, argv, "f:dvhy", syslog_ng_options, NULL)) != -1) {
		switch (opt) {
		case 'f':
			strncpy(cfgfilename, optarg, sizeof(cfgfilename));
			break;
		case 'd':
			debug_flag++;
			break;
		case 'v':
			verbose_flag++;
			break;
#ifdef YYDEBUG
		case 'y':
			yydebug = 1;
			break;
#endif
		case 'h':
		default:
			usage();
			break;
		}
	}

	init_backend(&backend->super);
	
	backend->configuration = make_syslog_conf(cfgfilename, &backend->super);
	if (!backend->configuration) {
		werror("Parse error reading configuration file, exiting.\n");
		return 1;
	}
	if (debug_flag)
		do_fork = 0;
	if (do_fork)
		go_background();
	if (!CONFIG_INIT(backend->configuration)) {
		werror("Error initializing configuration, exiting.\n");
		if (do_fork) kill(getppid(), SIGTERM);
		return 2;
	}

	if (do_fork) {
		set_error_internal();
		close(0);
		close(1);
		close(2);
		setsid();
		kill(getppid(), SIGTERM); 
	}
	notice("syslog-ng version %z starting\n", VERSION);
	return main_loop(backend);
}
