/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afdoor.c,v 1.1 1999/06/06 11:36:17 bazsi Exp $
 *
 ***************************************************************************/

#include "afdoor.h"
#include "sources.h"
#include "format.h"

#include "afdoor.c.x"

/* CLASS:
   (class
     (name afdoor_source)
     (super log_source_driver)
     (vars
       (name string)
       (fd . int)))
*/

#ifdef HAVE_DOOR_H

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <door.h>


void afdoor_server_proc(void *cookie, char *argp, size_t arg_size, door_desc_t *dp, size_t n_desc)
{
	CAST(afdoor_source, handler, cookie);
	struct log_info *msg = make_log_info(c_format(argp, arg_size));
	HANDLE_LOG(handler, msg);
}

static int do_init_afdoor_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afdoor_source, self, c);
	struct stat st;
	int fd;

	if (stat(self->name->data, &st) == -1) {
		/* door file not found, attempt to create one */
		fd = creat(self->name->data, 0666);
		if (fd == -1) {
			werror("Cannot create door file %S (%z)\n", self->name, strerror(errno));
			return ST_FAIL | ST_QUIT;
		}
		close(fd);
	}
	fdetach(self->name->data);
	fd = door_create(afdoor_server_proc, (void *) c, 0);
	if (fd == -1) {
		werror("Cannot initialize door server %S (%z)\n", self->name, strerror(errno));
		return ST_FAIL | ST_QUIT;
	}
	fattach(fd, self->name->data);
	self->fd = fd;
	return ST_OK | ST_GOON;
}

static int do_destroy_afdoor_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afdoor_source, self, c);

	door_revoke(self->fd);
	return ST_OK | ST_GOON;
}

static void do_handle_afdoor_source(struct log_handler *c, struct log_info *log)
{
	CAST(afdoor_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

struct log_source_driver *make_afdoor_source(const char *name)
{
	NEW(afdoor_source, self);

	self->super.super.super.init = do_init_afdoor_source;
	self->super.super.super.destroy = do_destroy_afdoor_source;
	self->super.super.super.handler = do_handle_afdoor_source;
	self->name = c_format_cstring("%z", name);
	self->fd = -1;
	return &self->super;
}

#else

static int do_init_afdoor_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afdoor_source, self, c);
	werror("sun-door support not compiled in (door=%S)\n", self->name);
	return ST_FAIL | ST_QUIT;
}

struct log_source_driver *make_afdoor_source(const char *name)
{
	NEW(afdoor_source, self);

	self->super.super.super.init = do_init_afdoor_source;
	self->name = c_format_cstring("%z", name);
	self->fd = -1;
	return &self->super;
}


#endif
