/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: cfg-grammar.y,v 1.21 1999/06/10 11:38:57 bazsi Exp $
 *
 ***************************************************************************/

%{

#include "cfgfile.h"
#include "filters.h"
#include "syslog-names.h"
#include "afinter.h"
#include "affile.h"
#include "afsocket.h"
#include "afuser.h"
#include "afdoor.h"

#include <netinet/in.h>

void yyerror(char *msg);
int yylex();

struct affile_dest *last_dest_affile;

%}

%union {
	UINT32 num;
	char *cptr;
	void *ptr;
	struct filter_expr_node *node;
	struct in_addr ip;
}

/* statements */
%token	KW_SOURCE KW_DESTINATION KW_LOG KW_OPTIONS KW_FILTER

/* source & destination items */
%token	KW_INTERNAL KW_FILE KW_UNIX_STREAM KW_UNIX_DGRAM KW_UDP KW_TCP KW_USER
%token  KW_SUN_DOOR

/* option items */
%token KW_MARK_FREQ KW_SYNC_FREQ KW_LONG_HOSTNAMES

/* filter items*/
%token KW_FACILITY KW_LEVEL KW_PROGRAM KW_HOST KW_MATCH

/* yes/no switches */
%token	KW_YES KW_NO

%token  DOTDOT
%token	<cptr> IDENTIFIER
%token	<cptr> FILTER_PARAMS
%token  <cptr> REGEXP
%token	<num>  NUMBER
%token	<cptr> STRING
%token  <ip>   IPADDR

%left	KW_OR
%left	KW_AND
%left   KW_NOT

%type	<ptr> source_stmt
%type	<ptr> dest_stmt
%type	<ptr> log_stmt
%type	<ptr> options_stmt

%type	<ptr> source_items
%type	<ptr> source_item
%type   <ptr> source_afinter
%type	<ptr> source_affile
%type	<ptr> source_afsocket
%type	<ptr> source_afdoor

%type	<ptr> dest_items
%type	<ptr> dest_item
%type   <ptr> dest_affile
%type	<ptr> dest_afsocket
%type   <ptr> dest_afuser

%type	<ptr> log_items
%type	<ptr> log_item

%type	<ptr> options_items
%type	<ptr> options_item

%type	<ptr> filter_stmt
%type	<node> filter_expr
%type	<node> filter_simple_expr

%type   <num> filter_fac_list
%type   <num> filter_fac
%type	<num> filter_level_list
%type   <num> filter_level

%type	<num> yesno

%type	<cptr> content

%%

start   
        : stmts
	;

stmts   
        : stmt ';' stmts
	|	
	;

stmt    
        : KW_SOURCE source_stmt			{ add_source_group($2); }
	| KW_DESTINATION dest_stmt		{ add_dest_group($2); }
	| KW_LOG log_stmt			{ add_log_connection($2); }
	| KW_FILTER filter_stmt			{ add_filter_rule($2); }
	| KW_OPTIONS options_stmt		{  }
	;

source_stmt
        : content '{' source_items '}'	{ $$ = make_source_group($1, $3); free($1); }        
	;

dest_stmt
        : content '{' dest_items '}'		{ $$ = make_dest_group($1, $3); free($1); }
	;

log_stmt
        : '{' log_items '}'			{ $$ = make_log_connection($2); }
	;

options_stmt
        : '{' options_items '}'			{ $$ = NULL; }
	;

source_items
        : source_item ';' source_items		{ append_source_driver($1, $3); $$ = $1; }
	|					{ $$ = NULL; }
	;

source_item
  	: source_afinter			{ $$ = $1; }
	| source_affile				{ $$ = $1; }
	| source_afsocket			{ $$ = $1; }
	| source_afdoor 			{ $$ = $1; }
	;

source_afinter
	: KW_INTERNAL				{ $$ = make_afinter_source(); }

source_affile
	: KW_FILE content 			{ $$ = make_affile_source($2); free($2); }
	;

source_afsocket
	: KW_UNIX_DGRAM content			{ $$ = make_afsocket_source(
							make_unix_addr($2), 
							AFUNIX_PROTO_DGRAM); 
							free($2); 
						}
	| KW_UNIX_STREAM content		{ $$ = make_afsocket_source(
							make_unix_addr($2), 
							AFUNIX_PROTO_STREAM); 
							free($2); 
						}
	| KW_UDP IPADDR ',' NUMBER		{ $$ = make_afsocket_source(
							make_inet_addr($2, $4), 
							AFINET_PROTO_DGRAM);
						}
	| KW_TCP IPADDR ',' NUMBER		{ $$ = make_afsocket_source(
							make_inet_addr($2, $4), 
							AFINET_PROTO_STREAM);
						}
	;

source_afdoor
	: KW_SUN_DOOR content			{ $$ = make_afdoor_source($2); free($2); }
	;

dest_items
	: dest_item ';' dest_items		{ append_dest_driver($1, $3); $$ = $1; }
	|					{ $$ = NULL; }
	;

dest_item
	: dest_affile				{ $$ = $1; }
	| dest_afsocket				{ $$ = $1; }
	| dest_afuser				{ $$ = $1; }
	;

dest_affile
	: KW_FILE content { last_dest_affile = make_affile_dest($2, 0); free($2); } dest_affile_options { $$ = last_dest_affile; }
	;

dest_affile_options
	: dest_affile_option dest_affile_options		
        |
	;
	

dest_affile_option
	: KW_SYNC_FREQ '(' NUMBER ')'		{ last_dest_affile->sync_freq = $3; }
	;

dest_afsocket
	: KW_UDP IPADDR ',' NUMBER		{ $$ = make_afsocket_dest(
							make_inet_addr($2, $4), 
							AFINET_PROTO_DGRAM);

						}
	| KW_TCP IPADDR ',' NUMBER		{ $$ = make_afsocket_dest(
							make_inet_addr($2, $4), 
							AFINET_PROTO_STREAM); 
						}
	| KW_UNIX_DGRAM content			{ $$ = make_afsocket_dest(
							make_unix_addr($2), 
							AFUNIX_PROTO_DGRAM);
							free($2);
						}
	| KW_UNIX_STREAM content		{ $$ = make_afsocket_dest(
							make_unix_addr($2), 
							AFUNIX_PROTO_STREAM);
							free($2);
						}
	;

dest_afuser
	: KW_USER content 			{ $$ = make_afuser_dest($2); free($2); }
	;

log_items
	: log_item ';' log_items		{ append_endpoint_info($1, $3); $$ = $1; }
	|					{ $$ = NULL; }
	;

log_item
	: KW_SOURCE content			{ $$ = make_log_endpoint_info(EP_SOURCE, $2); free($2); }
	| KW_FILTER content			{ $$ = make_log_endpoint_info(EP_FILTER, $2); free($2); }
	| KW_DESTINATION content		{ $$ = make_log_endpoint_info(EP_DESTINATION, $2); free($2); }
	;

options_items
	: options_item ';' options_items	{ $$ = $1; }
	|					{ $$ = NULL; }
	;

options_item
	: KW_MARK_FREQ '(' NUMBER ')'		{ configuration->mark_freq = $3; }
	| KW_SYNC_FREQ '(' NUMBER ')'		{ configuration->sync_freq = $3; }
	| KW_LONG_HOSTNAMES '(' yesno ')'	{ configuration->long_hostnames = $3; }
	;

filter_stmt
	: content '{' filter_expr ';' '}'	{ $$ = make_filter_rule($1, $3); free($1); }
	;

filter_expr
	: filter_simple_expr			{ $$ = $1; }
        | KW_NOT filter_expr			{ $2->comp = !($2->comp); $$ = $2; }
	| filter_expr KW_OR filter_expr		{ $$ = make_filter_or($1, $3); }
	| filter_expr KW_AND filter_expr	{ $$ = make_filter_and($1, $3); }
	| '(' filter_expr ')'			{ $$ = $2; }
	;

filter_simple_expr
	: KW_FACILITY '(' filter_fac_list ')'	{ $$ = make_filter_facility($3);  }
	| KW_LEVEL '(' filter_level_list ')' 	{ $$ = make_filter_level($3); }
	| KW_PROGRAM '(' content ')'		{ $$ = make_filter_prog($3); free($3); }
	| KW_HOST '(' content ')'		{ $$ = make_filter_host($3); free($3); }	
	| KW_MATCH '(' content ')'		{ $$ = make_filter_match($3); free($3); }	
	;

filter_fac_list
	: filter_fac ',' filter_fac_list	{ $$ = $1 + $3; }
	| filter_fac				{ $$ = $1; }
	;

filter_fac
	: IDENTIFIER				{ $$ = (1 << syslog_lookup_facility($1)); free($1); }
	;

filter_level_list
	: filter_level ',' filter_level_list	{ $$ = $1 + $3; }
	| filter_level				{ $$ = $1; }
	;

filter_level
	: IDENTIFIER				{ $$ = (1 << syslog_lookup_level($1)); }
	| IDENTIFIER DOTDOT IDENTIFIER		{ $$ = syslog_make_range(syslog_lookup_level($1), syslog_lookup_level($3)); }
	;

yesno
	: KW_YES				{ $$ = 1; }
	| KW_NO					{ $$ = 0; }
	| NUMBER				{ $$ = $1; }
	;

content
	: IDENTIFIER
	| STRING
	;

%%

extern int linenum;

void yyerror(char *msg)
{
	fprintf(stderr, "%s at %d\n", msg, linenum);
}
