/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afdoor.c,v 1.2 1999/06/15 20:32:34 bazsi Exp $
 *
 ***************************************************************************/

#include "afdoor.h"
#include "sources.h"
#include "format.h"
#include "cfgfile.h"

#include "afdoor.c.x"

/* CLASS:
   (class
     (name afdoor_source)
     (super log_source_driver)
     (vars
       (name string)
       (pipe array (simple int) 2)
       (pipe_fd object io_fd)
       (door simple int)))
*/

#ifdef HAVE_DOOR_H

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <door.h>


void afdoor_server_proc(void *cookie, char *argp, size_t arg_size, door_desc_t *dp, size_t n_desc)
{
	CAST(afdoor_source, handler, cookie);
	if (argp) {
		write(handler->pipe[1], argp, arg_size);
	}
	door_return(NULL, 0, NULL, 0);
}

static int do_init_afdoor_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afdoor_source, self, c);
	struct stat st;
	int fd;

	if (stat(self->name->data, &st) == -1) {
		/* door file not found, attempt to create one */
		fd = creat(self->name->data, 0666);
		if (fd == -1) {
			werror("Cannot create door file %S (%z)\n", self->name, strerror(errno));
			return ST_FAIL | ST_QUIT;
		}
		close(fd);
	}
	fdetach(self->name->data);
	if (pipe(self->pipe) != -1) {
		fd = door_create(afdoor_server_proc, (void *) c, 0);
		if (fd == -1) {
			werror("do_init_afdoor_source: Cannot initialize door server %S (%z)\n", self->name, strerror(errno));
			close(self->pipe[0]);
			close(self->pipe[1]);
			return ST_FAIL | ST_QUIT;
		}
		if (fattach(fd, self->name->data) == -1) {
			werror("do_init_afdoor_source: Cannot attach door to %S (%z)\n", self->name, strerror(errno));
			close(self->pipe[0]);
			close(self->pipe[1]);
			return ST_FAIL | ST_QUIT;
		}
		self->door = fd;
		self->pipe_fd = io_read(make_io_fd(cfg->backend, self->pipe[0]), 
					make_log_reader(0, c),
					NULL);
		REMEMBER_RESOURCE(cfg->resources, &self->pipe_fd->super.super);
	}
	else {
		werror("do_init_afdoor_source: Error creating pipe (%z)\n", strerror(errno));
		return ST_FAIL | ST_QUIT;
	}
	return ST_OK | ST_GOON;
}

static int do_destroy_afdoor_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afdoor_source, self, c);

	door_revoke(self->door);
	return ST_OK | ST_GOON;
}

static void do_handle_afdoor_source(struct log_handler *c, struct log_info *log)
{
	CAST(afdoor_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

struct log_source_driver *make_afdoor_source(const char *name)
{
	NEW(afdoor_source, self);

	self->super.super.super.init = do_init_afdoor_source;
	self->super.super.super.destroy = do_destroy_afdoor_source;
	self->super.super.super.handler = do_handle_afdoor_source;
	self->name = c_format_cstring("%z", name);
	self->door = -1;
	return &self->super;
}

#else

static int do_init_afdoor_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afdoor_source, self, c);
	werror("sun-door support not compiled in (door=%S)\n", self->name);
	return ST_FAIL | ST_QUIT;
}

struct log_source_driver *make_afdoor_source(const char *name)
{
	NEW(afdoor_source, self);

	self->super.super.super.init = do_init_afdoor_source;
	self->name = c_format_cstring("%z", name);
	self->door = -1;
	return &self->super;
}


#endif
