/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: affile.c,v 1.15 1999/06/08 19:47:31 bazsi Exp $
 *
 ***************************************************************************/

#include "affile.h"
#include "format.h"
#include "pkt_buffer.h"
#include "cfgfile.h"

#include <fcntl.h>
#include <assert.h>
#include <errno.h>
#include <string.h>

#define CLASS_DEFINE
#include "affile.h.x"
#undef CLASS_DEFINE

#include "affile.c.x"

static int do_open_file(struct ol_string *name, int flags, int mode, int *fd)
{
	*fd = open(name->data, flags, mode);
	if (*fd != -1) {
		return 1;
	}
	return 0;
}

/* CLASS:
     (class 
       (name affile_source)
       (super log_source_driver)
       (vars
         (src object io_fd)
         (name string)))
*/


static int do_init_affile_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(affile_source, self, c);
	int fd;
  
	if (do_open_file(self->name, O_RDONLY | O_NOCTTY | O_NONBLOCK, 0, &fd)) {
		
		self->src = io_read(make_io_fd(cfg->backend, fd), 
			make_log_reader(0, c), 
			NULL);
		REMEMBER_RESOURCE(cfg->resources, &self->src->super.super);
		return ST_OK | ST_GOON;
	}
	else {
		werror("Error opening file %S for reading (%z)\n", self->name, strerror(errno));
	}
	return ST_FAIL | ST_QUIT;
}

static void do_affile_handle_line(struct log_handler *c, struct log_info *log)
{
	CAST(affile_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

struct log_source_driver *make_affile_source(const char *name)
{
	NEW(affile_source, self);
  
	self->super.super.super.init = do_init_affile_source;
	self->super.super.super.handler = do_affile_handle_line;
	self->name = c_format_cstring("%z%c", name, 0);
	return &self->super;
}

/* affile_dest */

static int do_init_affile_dest(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(affile_dest, self, c);
	int fd;
	
	if (self->sync_freq == -1)
		self->sync_freq = cfg->sync_freq;
	/* self->cfg = cfg;*/
	self->dest = NULL;
	if (do_open_file(self->name, O_WRONLY | O_CREAT | O_APPEND | O_NOCTTY | O_NONBLOCK, 0600, &fd)) {
		struct resource_node *res;

		self->dest = io_write(make_io_fd(cfg->backend, fd), 
				      make_pkt_buffer_ext(self->sync_freq, 
							  MAX(LOG_FIFO_SIZE + 1, 
							      self->sync_freq)),
				      NULL);
		res = REMEMBER_RESOURCE(cfg->resources, &self->dest->super.super);
		self->dest->super.close_callback = make_dest_close(c, res, TIME_REOPEN, cfg);
		/* self->attempts = 0; */
	}
	else {
		if (errno == ENXIO) {
			/* opening the write side of the fifo failed */
			/* self->attempts++; */
			io_callout(cfg->backend, TIME_REOPEN, make_dest_reopen(c, cfg));
		}
		else {
			werror("Cannot open file %S for writing (%z)\n", self->name, strerror(errno));
			return ST_FAIL | ST_QUIT;
		}
	}
	return ST_OK | ST_GOON;
}

static void do_handle_affile_log(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest, self, c);

	if (self->dest && self->dest->buffer) {
		A_WRITE_STRING(&self->dest->buffer->super, c_format("%S %S %S\n", msg->date, msg->host, msg->msg));
	}
	log_info_free(msg);
}

struct affile_dest *make_affile_dest(const char *name, int flags)
{
	NEW(affile_dest, self);
	
	self->super.super.init = do_init_affile_dest;
	self->super.super.handler = do_handle_affile_log;
	self->name = c_format_cstring("%z", name);
	self->flags = flags;
	self->sync_freq = -1; /* use global setting */
	return self;
}
