/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afstreams.c,v 1.1 1999/06/15 20:32:34 bazsi Exp $
 *
 ***************************************************************************/

#include "afstreams.h"
#include "sources.h"
#include "format.h"
#include "cfgfile.h"

#include "afstreams.c.x"

/* CLASS:
   (class
     (name afstreams_source)
     (super log_source_driver)
     (vars
       (name string)
       (stream_fd object io_fd)))
*/

#ifdef HAVE_STROPTS_H

#include <sys/types.h>
#include <sys/stat.h>
#include <stropts.h>
#include <sys/strlog.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

static void prepare_stream(struct nonblocking_fd *fd)
{
	if (fd->to_be_closed)
		kill_fd(fd);
}

/* CLASS:
   (class
     (name stream_read)
     (super abstract_read)
     (vars
       (fd simple int)))
*/

static int do_stream_read(struct abstract_read **r, UINT32 length, UINT8 *buf)
{
	CAST(stream_read, self, *r);
	struct strbuf ctl, data;
	struct log_ctl lc;
	int flags;
	char tmp[MAX_LINE + 1];

	ctl.maxlen = ctl.len = sizeof(lc);
	ctl.buf = (char *) &lc;
	data.maxlen = sizeof(tmp);
	data.len = 0;
	data.buf = tmp;
	flags = 0;

	if (getmsg(self->fd, &ctl, &data, &flags) == 0) {
		return c_format_write("<%d>%s", length, buf, lc.pri, data.len, data.buf);
	}
	else {
		werror("do_stream_read: getmsg() failed %z\n", strerror(errno));
		return A_FAIL;
	}
}

static void stream_read_callback(struct nonblocking_fd *fd)
{
	CAST(io_fd, self, fd);
	struct stream_read r = { { STACK_HEADER, do_stream_read, NULL }, fd->fd };
	int res;
	
        /* The handler function may install a new handler */
        res = READ_HANDLER(self->handler, &r.super);

        if (res & ST_DIE)
        {
                closekill_fd(fd,
                             ST_FAILUREP(res) ? CLOSE_PROTOCOL_FAILURE : 0);
        }
        else if (res & ST_CLOSE)
        {
                close_fd(fd, ST_FAILUREP(res) ? CLOSE_PROTOCOL_FAILURE : CLOSE_EOF);
        }

}

struct io_fd *io_stream_get(struct io_fd *f,
			    struct read_handler *handler,
			    struct close_callback *close)
{
	f->super.prepare = prepare_stream;
	f->super.read = stream_read_callback;
	f->super.want_read = !!handler;

	f->handler = handler;
	f->super.close_callback = close;
	return f;
}

static int do_init_afstreams_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afstreams_source, self, c);
	int fd;

	fd = open(self->name->data, O_RDONLY | O_NOCTTY | O_NONBLOCK);
	if (fd != -1) {
		struct strioctl ioc;
		memset(&ioc, 0, sizeof(ioc));
		
		ioc.ic_cmd = I_CONSLOG;
		if (ioctl(fd, I_STR, &ioc) < 0) { 
			werror("do_init_afstreams_source: Cannot enable console logging on sun-stream %S (%z)\n", self->name, strerror(errno));
			close(fd);
			return ST_FAIL | ST_QUIT;
		}
		self->stream_fd = 
			io_stream_get(
				make_io_fd(cfg->backend, fd),
				make_log_reader(0, c),
				NULL);
	}
	else {
		werror("do_init_afstreams_source: Cannot open sun-stream %S (%z)\n", self->name, strerror(errno));
		return ST_FAIL | ST_QUIT;
	}
	return ST_OK | ST_GOON;
}

static void do_handle_afstreams_source(struct log_handler *c, struct log_info *log)
{
	CAST(afstreams_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

struct log_source_driver *make_afstreams_source(const char *name)
{
	NEW(afstreams_source, self);

	self->super.super.super.init = do_init_afstreams_source;
	self->super.super.super.handler = do_handle_afstreams_source;
	self->name = c_format_cstring("%z", name);
	return &self->super;
}

#else

static int do_init_afstreams_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afstreams_source, self, c);
	werror("sun-streams support not compiled in (STREAM=%S)\n", self->name);
	return ST_FAIL | ST_QUIT;
}

struct log_source_driver *make_afstreams_source(const char *name)
{
	NEW(afstreams_source, self);

	self->super.super.super.init = do_init_afstreams_source;
	self->name = c_format_cstring("%z", name);
	return &self->super;
}


#endif
