/***************************************************************************
 *
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afsocket.c,v 1.11 1999/06/08 19:47:31 bazsi Exp $
 *
 ***************************************************************************/

#include "afsocket.h"
#include "format.h"
#include "cfgfile.h"
#include "pkt_buffer.h"
#include "utils.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

static void free_sockaddr(struct sockaddr *sa);

#define CLASS_DEFINE
#include "afsocket.h.x"
#undef CLASS_DEFINE

#include "afsocket.c.x"

void free_sockaddr(struct sockaddr *sa)
{
	ol_space_free(sa);
}

int do_connect_socket(struct socket_addr *self, int fd)
{
	return connect(fd, self->sa, self->salen) != -1;
}

int do_bind_unix_socket(struct socket_addr *self, int fd)
{
	if (fd != -1) {
		struct sockaddr_un *su = (struct sockaddr_un *) self->sa;
		if (su) {
			unlink(su->sun_path);
			errno = 0;
			if (bind(fd, (struct sockaddr *) su, sizeof(struct sockaddr_un)) == -1) {
				werror("cannot bind af_unix socket to %z (%z)\n", su->sun_path, strerror(errno));
				return 0;
			}
			chmod(su->sun_path, 0666);
		}
		return 1;
	}
	/* indicate failure */
	return 0;
}

struct socket_addr *make_unix_addr(const char *name)
{
	NEW(socket_addr, self);
	struct sockaddr_un *s_un;
	
	NEW_SPACE(s_un);
	s_un->sun_family = AF_UNIX;
	strncpy(s_un->sun_path, name, sizeof(s_un->sun_path));
	self->bind = do_bind_unix_socket;
	self->connect = do_connect_socket;
	self->sa = (struct sockaddr *) s_un;
	self->salen = sizeof(struct sockaddr_un);
	return self;
}

int do_bind_inet_socket(struct socket_addr *self, int fd)
{
	if (fd != -1) {
		struct sockaddr_in *sin = (struct sockaddr_in *) self->sa;
		
		if (sin) {
			int tmp = 1;
			setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &tmp, sizeof(tmp));
			if (bind(fd, (struct sockaddr *) sin, sizeof(struct sockaddr_in)) == -1) {
				werror("cannot bind af_inet socket to %I:%i (%z)\n", sin->sin_addr, ntohs(sin->sin_port), strerror(errno));
				return 0;
			}
		}
		return 1;
	}
	/* indicate failure */
	return 0;
}

struct socket_addr *make_inet_addr(struct in_addr host, unsigned short port)
{
	NEW(socket_addr, self);
	struct sockaddr_in *s_in;
	
	NEW_SPACE(s_in);
	s_in->sin_family = AF_INET;
	s_in->sin_addr = host;
	s_in->sin_port = htons(port);
	self->bind = do_bind_inet_socket;
	self->connect = do_connect_socket;
	self->sa = (struct sockaddr *) s_in;
	self->salen = sizeof(struct sockaddr_in);
	return self;
}

/* CLASS:
     (class
       (name afsocket_source_close_callback)
       (super close_callback)
       (vars
          (res simple "struct resource_node *")
	  (cfg object syslog_conf)))
*/

static int do_afsocket_src_close(struct close_callback *c, int close_reason)
{
	CAST(afsocket_source_close_callback, self, c);
	
	KILL_RESOURCE_NODE(self->cfg->resources, self->res);
	return 0;
}

struct afsocket_source_close_callback *make_afsocket_source_close_callback(struct syslog_conf *cfg)
{
	NEW(afsocket_source_close_callback, self);
	self->super.f = do_afsocket_src_close;
	self->cfg = cfg;
	return self;
}

/* CLASS:
     (class
       (name afsocket_source_connection)
       (super log_handler_pipe)
       (vars
         (fd simple int)
         (src object io_fd)))
*/

static int do_init_afsocket_source_connection(
	struct log_handler *c, 
	struct syslog_conf *cfg)
{
	CAST(afsocket_source_connection, self, c);
	struct afsocket_source_close_callback *cb = make_afsocket_source_close_callback(cfg);
	
	self->src = io_read(make_io_fd(cfg->backend, self->fd), 
			    make_log_reader(1, c),
			    &cb->super);
	cb->res = REMEMBER_RESOURCE(cfg->resources, &self->src->super.super);
	return 0;
}


static void do_handle_afinet_line(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_source_connection, self, c);
	HANDLE_LOG(self->super.next, msg);
}

struct afsocket_source_connection *make_afinet_source_connection(
	int fd, 
	struct log_handler *next)
{
	NEW(afsocket_source_connection, self);

	self->super.super.init = do_init_afsocket_source_connection;
	self->super.super.handler = do_handle_afinet_line;
	self->super.next = next;
	self->fd = fd;
	return self;
}

static void do_handle_afunix_line(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_source_connection, self, c);

	if (msg->host) {
		ol_string_free(msg->host);
		msg->host = NULL;
	}
	msg->flags |= LF_LOCAL;

	HANDLE_LOG(self->super.next, msg);
}

struct afsocket_source_connection *make_afunix_source_connection(
	int fd, 
	struct log_handler *next)
{
	NEW(afsocket_source_connection, self);

	self->super.super.init = do_init_afsocket_source_connection;
	self->super.super.handler = do_handle_afunix_line;
	self->super.next = next;
	self->fd = fd;
	return self;
}

static struct afsocket_source_connection *get_source_connection(int fd, int family, struct log_handler *next)
{
	struct afsocket_source_connection *conn;

	switch (family) {
	case AF_UNIX:
		conn = make_afunix_source_connection(fd, next);
		break;
	case AF_INET:
		conn = make_afinet_source_connection(fd, next);
		break;
	default:
		fatal("Unknown afsocket type: %i\n", family);
	}
	return conn;
}

/* CLASS:
     (class
       (name afsocket_accept_callback)
       (super fd_listen_callback)
       (vars
         (cfg object syslog_conf)
         (backend object io_backend)
	 (next object log_handler)))
*/

static int do_accept_socket_connection(struct fd_listen_callback *c, int fd, size_t salen, struct sockaddr *sa)
{
	CAST(afsocket_accept_callback, self, c);
	struct afsocket_source_connection *conn;

	conn = get_source_connection(fd, sa->sa_family, self->next);

	return LOG_HANDLER_INIT(conn, self->cfg);
}

struct fd_listen_callback *make_afsocket_accept_callback(struct io_backend *backend, struct log_handler *next, struct syslog_conf *cfg)
{
	NEW(afsocket_accept_callback, self);
	
	self->super.f = do_accept_socket_connection;
	self->backend = backend;
	self->next = next;
	self->cfg = cfg;
	return &self->super;
}

static int socktypes[] = { SOCK_STREAM, SOCK_DGRAM };
static int protos[] = { IPPROTO_TCP, IPPROTO_UDP, 0, 0 };

/* CLASS:
     (class
       (name afsocket_source)
       (super log_source_driver)
       (vars
         ; used in connection mode
	 (flags simple UINT32)
	 (sa object socket_addr)))
*/

int do_init_afsocket_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afsocket_source, self, c);
	int fd;
	
	fd = socket(self->sa->sa->sa_family, socktypes[self->flags & AFSOCKET_TYPE], protos[self->flags & 3]);
	if (BIND_SOCKET(self->sa, fd)) {
		struct listen_fd *listen;

		switch (self->flags & AFSOCKET_TYPE) {
		case AFSOCKET_STREAM:
			if ((listen = 
			        io_listen(cfg->backend, fd, 
					  make_afsocket_accept_callback(
						  cfg->backend, 
						  self->super.super.next,
						  cfg)))) {
				REMEMBER_RESOURCE(cfg->resources, &listen->super.super);
				return ST_OK | ST_GOON;
			}

			break;
		case AFSOCKET_DGRAM: {
			struct afsocket_source_connection *conn;
			conn = get_source_connection(fd, self->sa->sa->sa_family, self->super.super.next);
			LOG_HANDLER_INIT(conn, cfg);
			return ST_OK | ST_GOON;
		}
		}
	}
	else {
		werror("Cannot bind socket (%z)\n", strerror(errno));
		close(fd);
	}
	return ST_FAIL | ST_QUIT;
}

struct log_source_driver *make_afsocket_source(struct socket_addr *sa, UINT32 flags)
{
        NEW(afsocket_source, self);

        self->super.super.super.init = do_init_afsocket_source;
	self->sa = sa;
        self->flags = flags;
        return &self->super;
}

/* CLASS:
     (class
       (name afsocket_dest)
       (super log_dest_driver)
       (vars
         (dest object io_fd)
	 (dest_res simple "struct resource_node *")
         (sa object socket_addr)
	 (flags simple UINT32)))
*/

static int do_init_afsocket_dest(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afsocket_dest, self, c);
	int fd;
	
	fd = socket(self->sa->sa->sa_family, socktypes[self->flags & AFSOCKET_TYPE], protos[self->flags & 3]);
	if (CONNECT_SOCKET(self->sa, fd)) {
		struct resource_node *n;
		self->dest = io_write(make_io_fd(cfg->backend, fd), 
				      make_pkt_buffer(LOG_FIFO_SIZE),
				      NULL);
		n = REMEMBER_RESOURCE(cfg->resources, 
				      &self->dest->super.super);
		self->dest->super.close_callback = make_dest_close(c, n, TIME_REOPEN, cfg);
		return ST_OK | ST_GOON;
	}
	else {
		werror("Cannot connect socket (%z)\n", strerror(errno));
		close(fd);
	}
	return ST_FAIL | ST_QUIT;
}

static void do_handle_afsocket_dest(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_dest, self, c);

	A_WRITE_STRING(&self->dest->buffer->super, c_format("<%i> %S %S %S\n", msg->pri, msg->date, msg->host, msg->msg));
	log_info_free(msg);
}

struct log_dest_driver *make_afsocket_dest(struct socket_addr *sa, UINT32 flags)
{
	NEW(afsocket_dest, self);
	
	self->super.super.init = do_init_afsocket_dest;
	self->super.super.handler = do_handle_afsocket_dest;
	self->sa = sa;
	self->flags = flags;
	return &self->super;
}
