%{

#include "syslog-ng.h"
#include "utils.h"

#include <string.h>
#include <strings.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "cfg-grammar.h"

struct keyword {
	char	*kw_name;
	int	kw_token;
};

static struct keyword keywords[] = {
	/* statements */
	{ "source", 		KW_SOURCE },
	{ "destination",	KW_DESTINATION },
	{ "filter",             KW_FILTER },
	{ "log",		KW_LOG },
	{ "options",		KW_OPTIONS },

	/* option items */
	{ "mark",		KW_MARK_FREQ },
	{ "sync", 		KW_SYNC_FREQ },
	{ "long_hostnames",	KW_LONG_HOSTNAMES },
       
	/* source or destination items */
	{ "file",		KW_FILE },
        { "internal",           KW_INTERNAL },
	{ "unix-dgram",		KW_UNIX_DGRAM },
	{ "unix-stream",	KW_UNIX_STREAM },
        { "udp",                KW_UDP },
        { "tcp",                KW_TCP },
        { "usertty", 		KW_USER },
        { "door",               KW_DOOR },
        { "sun-stream",		KW_SUN_STREAMS },
        { "sun-streams",	KW_SUN_STREAMS },

	/* filter items */
        { "or",                 KW_OR },
	{ "and",                KW_AND },
        { "not",                KW_NOT },
	{ "level",              KW_LEVEL },
	{ "facility",           KW_FACILITY },
	{ "program",		KW_PROGRAM },
        { "host",               KW_HOST },
        { "match",		KW_MATCH },

	/* on/off switches */
	{ "yes",		KW_YES },
	{ "on",			KW_YES },
	{ "no",			KW_NO },
	{ "off", 		KW_NO }
};

#define YY_NO_UNPUT   1

#define MAX_REGEXP_LEN	1024

int linenum = 1;
int lex_filter_params = 0;
char buf[MAX_REGEXP_LEN];
char *str;

static int check_reserved_words(char *token);
static void append_string(int length, char *str);

%}

%option noyywrap

white	[ \t]
digit	[0-9]
alpha		[a-zA-Z]
alphanum	[a-zA-Z0-9]

%x string
%%

^\#.*$                     ;
\n			   { linenum++; }
{white}+		   ;
\.\.                       { return DOTDOT; }
[0-9]+\.[0-9]+\.[0-9]+.[0-9]+ {
                            if (inet_aton(yytext, &yylval.ip)) 
                                     return IPADDR;
                             else
				     REJECT;
			   }
{digit}+		   { yylval.num = atoi(yytext); return NUMBER; }
[^ \#"\(\)\{\}\\;\n\t,|]+    { return check_reserved_words(yytext); }
\(	      		   { return '('; }
\)			   { return ')'; }
\;			   { return ';'; }
\{			   { return '{'; }
\}			   { return '}'; }
\|			   { return '|'; }
\,			   { return ','; }

\"                         {
				str = buf;
				/* yy_push_state(string);*/
				BEGIN(string);
			   }
<string>\\.		   { 
				append_string(1, yytext + 1); 
			   }
<string>\"		   { 
				BEGIN(INITIAL);
				/* yy_pop_state();*/
				yylval.cptr = strdup(buf);
				return STRING; 
		           }
<string>[^"\\]+		   { append_string(strlen(yytext), yytext); }

%%

int lex_init(FILE *file)
{
	yyrestart(file);
	linenum = 0;
	return 0;
}

int check_reserved_words(char *token)
{
	int i;
	
	for (i = 0; i < (sizeof(keywords) / sizeof(struct keyword)); i++) {
		if (strcmp(keywords[i].kw_name, token) == 0) {
			return keywords[i].kw_token;
		}
	}
	yylval.cptr = strdup(token);
	return IDENTIFIER;
}

static void append_string(int length, char *s)
{
	int to_copy = MIN(MAX_REGEXP_LEN - (str - buf) - 1, length);

	memcpy(str, s, to_copy);
	str += to_copy;
	*str = 0;
}
