/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afsocket.c,v 1.16 1999/10/20 19:30:01 bazsi Exp $
 *
 ***************************************************************************/

#include "afsocket.h"
#include "format.h"
#include "cfgfile.h"
#include "pkt_buffer.h"
#include "utils.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>

#include "afsocket.c.x"

/* CLASS:
   (class
     (name afsocket_params)
     (vars
       (init method int "struct log_handler *c" "struct syslog_conf *cfg")))
*/

#define AFSOCKET_PARAMS_INIT(p, h, c) ((p)->init(p, h, c))

/* CLASS:
     (class
       (name afsocket_source_close_callback)
       (super close_callback)
       (vars
          (res simple "struct resource_node *")
	  (cfg object syslog_conf)))
*/

static int do_afsocket_src_close(struct close_callback *c, int close_reason)
{
	CAST(afsocket_source_close_callback, self, c);
	
	KILL_RESOURCE_NODE(self->cfg->resources, self->res);
	return 0;
}

struct afsocket_source_close_callback *make_afsocket_source_close_callback(struct syslog_conf *cfg)
{
	NEW(afsocket_source_close_callback, self);
	self->super.f = do_afsocket_src_close;
	self->cfg = cfg;
	return self;
}

/* CLASS:
     (class
       (name afsocket_source_connection)
       (super log_handler_pipe)
       (vars
         (fd simple int)
         (src object io_fd)))
*/

static int do_init_afsocket_source_connection(
	struct log_handler *c, 
	struct syslog_conf *cfg)
{
	CAST(afsocket_source_connection, self, c);
	struct afsocket_source_close_callback *cb = make_afsocket_source_close_callback(cfg);
	
	self->src = io_read(make_io_fd(cfg->backend, self->fd), 
			    make_log_reader(1, c),
			    &cb->super);
	cb->res = REMEMBER_RESOURCE(cfg->resources, &self->src->super.super);
	return 0;
}


static void do_handle_afinet_line(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_source_connection, self, c);
	/* FIXME: somewhat ugly BSD workaround */
	if (!msg->saddr) {
		char sockaddr_buf[256];
		struct sockaddr *sa = (struct sockaddr *) &sockaddr_buf;
		int salen = sizeof(sockaddr_buf);

		if (getpeername(self->fd, sa, &salen) != -1) {
			msg->saddr = ol_space_alloc(salen);
			memcpy(msg->saddr, sa, salen);
		}
	}
	HANDLE_LOG(self->super.next, msg);
}

struct afsocket_source_connection *make_afinet_source_connection(
	int fd, 
	struct log_handler *next)
{
	NEW(afsocket_source_connection, self);

	self->super.super.init = do_init_afsocket_source_connection;
	self->super.super.handler = do_handle_afinet_line;
	self->super.next = next;
	self->fd = fd;
	return self;
}

static void do_handle_afunix_line(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_source_connection, self, c);

	if (msg->host) {
		ol_string_free(msg->host);
		msg->host = NULL;
	}
	msg->flags |= LF_LOCAL;

	HANDLE_LOG(self->super.next, msg);
}

struct afsocket_source_connection *make_afunix_source_connection(
	int fd, 
	struct log_handler *next)
{
	NEW(afsocket_source_connection, self);

	self->super.super.init = do_init_afsocket_source_connection;
	self->super.super.handler = do_handle_afunix_line;
	self->super.next = next;
	self->fd = fd;
	return self;
}

static struct afsocket_source_connection *get_source_connection(int fd, int family, struct log_handler *next)
{
	struct afsocket_source_connection *conn;

	switch (family) {
	case AF_UNIX:
		conn = make_afunix_source_connection(fd, next);
		break;
	case AF_INET:
		conn = make_afinet_source_connection(fd, next);
		break;
	default:
		fatal("Unknown afsocket type: %i\n", family);
	}
	return conn;
}

/* CLASS:
     (class
       (name afsocket_accept_callback)
       (super fd_listen_callback)
       (vars
         (cfg object syslog_conf)
         (backend object io_backend)
	 (next object log_handler)))
*/

static int do_accept_socket_connection(struct fd_listen_callback *c, int fd, struct address_info *a)
{
	CAST(afsocket_accept_callback, self, c);
	struct afsocket_source_connection *conn;

	conn = get_source_connection(fd, a->family, self->next);

	return LOG_HANDLER_INIT(conn, self->cfg);
}

struct fd_listen_callback *make_afsocket_accept_callback(struct io_backend *backend, struct log_handler *next, struct syslog_conf *cfg)
{
	NEW(afsocket_accept_callback, self);
	
	self->super.f = do_accept_socket_connection;
	self->backend = backend;
	self->next = next;
	self->cfg = cfg;
	return &self->super;
}

/* CLASS:
   (class
     (name afunix_source_params)
     (super afsocket_params)
     (vars
       (owner string)
       (group string)
       (perm simple int)))
*/

static int do_init_afunix_source(struct afsocket_params *c, struct log_handler *h, struct syslog_conf *cfg)
{
	CAST(afunix_source_params, self, c);
	CAST(afsocket_source, source, h);
	CAST(unix_address_info, addr, source->src_addr);
	struct group *grp = NULL;
	struct passwd *pwd = NULL;

	if (self->owner)
		pwd = getpwnam(self->owner->data);
	if (self->group)
		grp = getgrnam(self->group->data);
	chown(addr->path->data, pwd ? pwd->pw_uid : -1, grp ? grp->gr_gid : -1);
	chmod(addr->path->data, self->perm);
	return ST_OK | ST_GOON;
}

struct afsocket_params *make_afunix_source_params(void)
{
	NEW(afunix_source_params, self);
	self->super.init = do_init_afunix_source;
	self->perm = 0666;
	return &self->super;
}

void afunix_set_owner(struct log_source_driver *c, const char *owner)
{
	CAST(afsocket_source, source, c);
	CAST(afunix_source_params, self, source->params);

	if (!self) {
		source->params = make_afunix_source_params();
		self = (struct afunix_source_params *) CHECK_TYPE(afunix_source_params, source->params);
	}
	if (self->owner)
		ol_string_free(self->owner);
	self->owner = c_format_cstring("%z", owner);
}

void afunix_set_group(struct log_source_driver *c, const char *group)
{
	CAST(afsocket_source, source, c);
	CAST(afunix_source_params, self, source->params);

	if (!self) {
		source->params = make_afunix_source_params();
		self = (struct afunix_source_params *) CHECK_TYPE(afunix_source_params, source->params);
	}
	if (self->group)
		ol_string_free(self->group);
	self->group = c_format_cstring("%z", group);
}

void afunix_set_perm(struct log_source_driver *c, int perm)
{
	CAST(afsocket_source, source, c);
	CAST(afunix_source_params, self, source->params);

	if (!self) {
		source->params = make_afunix_source_params();
		self = (struct afunix_source_params *) CHECK_TYPE(afunix_source_params, source->params);
	}
	self->perm = perm;
}

void inet_address_setip(struct address_info **a, const char *ip)
{
	CAST(inet_address_info, addr, *a);
	if (addr) {
		*a = &make_inet_address_c(ip, NULL)->super;
	}
	else {
		ol_string_free(addr->ip);
		addr->ip = c_format("%z", ip);
	}
}

void inet_address_setport(struct address_info **a, UINT32 port)
{
	CAST(inet_address_info, addr, *a);
	if (!addr) {
		*a = &make_inet_address(NULL, port)->super;
	}
	else {
		addr->port = port;
	}
}

void afinet_src_set_localip(struct log_source_driver *c, const char *ip)
{
	CAST(afsocket_source, source, c);

	inet_address_setip(&source->src_addr, ip);
}

void afinet_src_set_localport(struct log_source_driver *c, UINT32 port)
{
	CAST(afsocket_source, source, c);

	inet_address_setport(&source->src_addr, port);
}

void afinet_src_set_auth(struct log_source_driver *c, UINT32 v)
{
	CAST(afsocket_source, l, c);

	l->flags = set_flags(l->flags, AFSOCKET_AUTH_MASK, AFSOCKET_AUTH_SHIFT, v);
}

void afinet_src_set_mac(struct log_source_driver *c, UINT32 v)
{
	CAST(afsocket_source, l, c);

	l->flags = set_flags(l->flags, AFSOCKET_MAC_MASK, AFSOCKET_MAC_SHIFT, v);
}

void afinet_src_set_encrypt(struct log_source_driver *c, UINT32 v)
{
	CAST(afsocket_source, l, c);

	l->flags = set_flags(l->flags, AFSOCKET_ENCRYPT_MASK, AFSOCKET_ENCRYPT_SHIFT, v);
}

static int socktypes[] = { SOCK_STREAM, SOCK_DGRAM };
static int protos[] = { IPPROTO_TCP, IPPROTO_UDP, 0, 0 };

/* CLASS:
     (class
       (name afsocket_source)
       (super log_source_driver)
       (vars
         (params object afsocket_params)
         ; used in connection mode
	 (flags simple UINT32)
	 (src_addr object address_info)))
*/

int do_init_afsocket_source(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afsocket_source, self, c);
	struct listen_fd *listen;
	int fd;

	fd = io_open_socket(self->src_addr->family, 
			    socktypes[self->flags & AFSOCKET_TYPE],
			    protos[self->flags & 3], 
			    self->src_addr);
	if (fd < 0) {
		return ST_QUIT | ST_FAIL;
	}
	switch (self->flags & AFSOCKET_TYPE) {
	case AFSOCKET_STREAM: {
		if ((listen = 
		     io_listen(cfg->backend, fd,
			       make_afsocket_accept_callback(
				       cfg->backend, 
				       self->super.super.next,
				       cfg)))) {
			REMEMBER_RESOURCE(cfg->resources, &listen->super.super);
		}
		else
			return ST_FAIL | ST_QUIT;
		break;
	}
			
	case AFSOCKET_DGRAM: {
		struct afsocket_source_connection *conn;
		conn = get_source_connection(fd, self->src_addr->family, self->super.super.next);
		LOG_HANDLER_INIT(conn, cfg);

		break;
	}
	default:
		return ST_FAIL | ST_QUIT;
	}
	if (self->params)
		AFSOCKET_PARAMS_INIT(self->params, c, cfg);
	return ST_OK | ST_GOON;		
}

struct log_source_driver *make_afsocket_source(struct address_info *src_addr, UINT32 flags)
{
        NEW(afsocket_source, self);

        self->super.super.super.init = do_init_afsocket_source;
	self->src_addr = src_addr;
        self->flags = flags;
	if (src_addr && src_addr->family == AF_UNIX)
		afunix_set_perm(&self->super, 0666);
        return &self->super;
}

void afinet_dest_set_auth(struct log_dest_driver *c, UINT32 v)
{	
	CAST(afsocket_dest, l, c);
	l->flags = set_flags(l->flags, AFSOCKET_AUTH_MASK, AFSOCKET_AUTH_SHIFT, v);
}

void afinet_dest_set_mac(struct log_dest_driver *c, UINT32 v)
{
	CAST(afsocket_dest, l, c);
	l->flags = set_flags(l->flags, AFSOCKET_MAC_MASK, AFSOCKET_MAC_SHIFT, v);
}

void afinet_dest_set_encrypt(struct log_dest_driver *c, UINT32 v)
{
	CAST(afsocket_dest, l, c);
	l->flags = set_flags(l->flags, AFSOCKET_ENCRYPT_MASK, AFSOCKET_ENCRYPT_SHIFT, v);
}

void afinet_dest_set_localip(struct log_dest_driver *c, const char *ip)
{
	CAST(afsocket_dest, dest, c);

	inet_address_setip(&dest->src_addr, ip);
}

void afinet_dest_set_localport(struct log_dest_driver *c, UINT32 port)
{
	CAST(afsocket_dest, dest, c);

	inet_address_setport(&dest->src_addr, port);
}

void afinet_dest_set_destport(struct log_dest_driver *c, UINT32 port)
{
	CAST(afsocket_dest, dest, c);

	inet_address_setport(&dest->dest_addr, port);
}


/* CLASS:
     (class
       (name afsocket_dest_connected)
       (super fd_callback)
       (vars
         (logger object afsocket_dest)
	 (cfg object syslog_conf)))
*/

static int do_afsocket_dest_connected(struct fd_callback **r, int fd)
{
	CAST(afsocket_dest_connected, self, *r);
	struct resource_node *n;

	if (self->logger->dest)
		self->logger->dest->buffer = NULL;
	if (fd != -1) {
		self->logger->dest = io_write(make_io_fd(self->cfg->backend, fd), 
					    self->logger->dest_buf,
					    NULL);
		n = REMEMBER_RESOURCE(self->cfg->resources, 
				      &self->logger->dest->super.super);
		self->logger->dest->super.close_callback = make_dest_close(&self->logger->super.super, n, self->cfg->time_reopen, self->cfg);
		
	}
	else {
		werror("error connecting remote loghost, reattempting in %i seconds\n", self->cfg->time_reopen);
		self->logger->dest = NULL;
		io_callout(self->cfg->backend, self->cfg->time_reopen, make_dest_reopen(&self->logger->super.super, self->cfg));
	}
	return ST_OK | ST_GOON;
}

struct fd_callback *make_afsocket_dest_connected(struct afsocket_dest *logger, struct syslog_conf *cfg)
{
	NEW(afsocket_dest_connected, self);
	self->super.f = do_afsocket_dest_connected;
	self->logger = logger;
	self->cfg = cfg;
	return &self->super;
}

/* CLASS:
     (class
       (name afsocket_dest)
       (super log_dest_driver)
       (vars
         (dest object io_fd)
	 (dest_buf object abstract_buffer)
	 (dest_res simple "struct resource_node *")
	 (src_addr object address_info)
         (dest_addr object address_info)
	 (flags simple UINT32)))
*/

static int do_init_afsocket_dest(struct log_handler *c, struct syslog_conf *cfg)
{
	CAST(afsocket_dest, self, c);
	int fd;

	if (self->super.log_fifo_size == -1)
		self->super.log_fifo_size = cfg->log_fifo_size;
	
	fd = io_open_socket(self->dest_addr->family, socktypes[self->flags & AFSOCKET_TYPE], protos[self->flags & 3], self->src_addr);
	
	if (fd < 0) {
		return ST_FAIL | ST_QUIT;
	}
	if (!self->dest_buf)
		self->dest_buf = make_pkt_buffer(self->super.log_fifo_size);
	if (io_connect(cfg->backend, fd, self->dest_addr, make_afsocket_dest_connected(self, cfg))) {
		return ST_OK | ST_GOON;
	}
	else {
		werror("do_init_afsocket_dest(): Cannot connect socket (%z)\n", strerror(errno));
	}
	return ST_FAIL | ST_QUIT;
}

static void do_handle_afsocket_dest(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_dest, self, c);

	if (self->dest_buf)
		A_WRITE_STRING(&self->dest_buf->super, c_format("<%i>%S %S %S\n", msg->pri, msg->date, msg->host, msg->msg));
	log_info_free(msg);
}

struct log_dest_driver *make_afsocket_dest(struct address_info *dest_addr, UINT32 flags)
{
	NEW(afsocket_dest, self);
	
	self->super.super.init = do_init_afsocket_dest;
	self->super.super.handler = do_handle_afsocket_dest;
	self->super.log_fifo_size = -1;
	self->dest_addr = dest_addr;
	self->flags = flags;
	return &self->super;
}
