/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: afinet.c,v 1.4 1999/12/18 22:25:38 bazsi Exp $
 *
 ***************************************************************************/

#include "afsocket.h"
#include "format.h"
#include "cfgfile.h"
#include "pkt_buffer.h"

#include <errno.h>
#include <string.h>
#include <assert.h>

#include "afinet.c.x"

void inet_address_setip(struct address_info **a, const char *ip)
{
	CAST(inet_address_info, addr, *a);
	if (addr) {
		*a = &make_inet_address_c(ip, NULL)->super;
	}
	else {
		ol_string_free(addr->ip);
		addr->ip = c_format("%z", ip);
	}
}

void inet_address_setport(struct address_info **a, UINT32 port)
{
	CAST(inet_address_info, addr, *a);
	if (!addr) {
		*a = &make_inet_address(NULL, port)->super;
	}
	else {
		addr->port = port;
	}
}

/* af_inet source connection */
static int 
do_init_afinet_source_connection(struct log_handler *c, 
				 struct syslog_config *cfg,
				 struct persistent_config *persistent)
{
	CAST(afsocket_source_connection, self, c);
	if (self->client_addr) {
		CAST(inet_address_info, inet, self->client_addr);
		
		notice("AF_INET client connected from %S, port %i\n", 
		       inet->ip, inet->port);
	}
	io_read(self->client, 
		make_log_reader(1, c), 
		make_afsocket_source_close_callback(self));
	
	return ST_OK | ST_GOON;
}

static void 
do_handle_afinet_line(struct log_handler *c, struct log_info *msg)
{
	CAST(afsocket_source_connection, self, c);

	if (!msg->saddr) {
		msg->saddr = self->client_addr;
	}
	HANDLE_LOG(self->super.next, msg);
}

static int
do_close_afinet_connection(struct afsocket_source_connection *c,
                           int close_reason)
{
	assert(c->owner);
	if (c->client_addr) {
		CAST(inet_address_info, inet, c->client_addr);

		notice("AF_INET client dropped connection from %S, port %i\n", 
		       inet->ip, inet->port);
	}
	object_queue_remove(c->queue_node);
	c->owner->num_connections--;
	return ST_OK | ST_GOON;
}

static void
do_destroy_afinet_connection(struct log_handler *c,
			     struct syslog_config *cfg,
			     struct persistent_config *persistent)
{
	CAST(afsocket_source_connection, self, c);

	close_fd(&self->client->super, 0);
}

struct afsocket_source_connection *
make_afinet_source_connection(struct io_fd *client,
			      struct address_info *client_addr, 
			      struct afsocket_source *owner,
			      struct log_handler *next)
{
	NEW(afsocket_source_connection, self);

	self->super.super.init = do_init_afinet_source_connection;
	self->super.super.handler = do_handle_afinet_line;
	self->super.super.destroy = do_destroy_afinet_connection;
	self->super.next = next;
	self->owner = owner;
	self->client = client;
	self->client_addr = client_addr;
	self->close = do_close_afinet_connection;
	return self;
}

/* afinet_source */
/* CLASS:
   (class
     (name afinet_source)
     (super afsocket_source)
     (vars
       (cfg object syslog_config)))
*/

void afinet_src_set_localip(struct log_source_driver *c, const char *ip)
{
	CAST_SUBTYPE(afsocket_source, source, c);

	inet_address_setip(&source->bind_addr, ip);
}

void afinet_src_set_localport(struct log_source_driver *c, UINT32 port)
{
	CAST_SUBTYPE(afsocket_source, source, c);

	inet_address_setport(&source->bind_addr, port);
}

void afinet_src_set_auth(struct log_source_driver *c, UINT32 v)
{
	CAST_SUBTYPE(afsocket_source, l, c);

	l->flags = set_flags(l->flags, AFINET_AUTH_MASK, AFINET_AUTH_SHIFT, v);
	werror("Warning: network authentication not implemented yet.\n");
}

void afinet_src_set_mac(struct log_source_driver *c, UINT32 v)
{
	CAST_SUBTYPE(afsocket_source, l, c);

	l->flags = set_flags(l->flags, AFINET_MAC_MASK, AFINET_MAC_SHIFT, v);
	werror("Warning: network mac protection not implemented yet.\n");
}

void afinet_src_set_encrypt(struct log_source_driver *c, UINT32 v)
{
	CAST_SUBTYPE(afsocket_source, l, c);

	l->flags = set_flags(l->flags, AFINET_ENCRYPT_MASK, AFINET_ENCRYPT_SHIFT, v);
	werror("Warning: network encryption not implemented yet.\n");
}

static int 
do_open_afinet_connection(struct afsocket_source *c,
			  struct io_fd *client,
			  struct address_info *client_addr)
{
	CAST(afinet_source, self, c);
	struct afsocket_source_connection *conn;
	UINT32 res;

	if (c->num_connections >= c->max_connections) {
		CAST(inet_address_info, inet_addr, client_addr);

		notice("Error accepting AF_INET connection from: %S:%i, opened connections: %i, max: %i\n", inet_addr->ip, inet_addr->port, c->num_connections, c->max_connections);
		close_fd(&client->super, 0);
		return ST_OK | ST_GOON;
	}
	else {
		conn = make_afinet_source_connection(client, client_addr, c, self->super.super.super.next);
		res = LOG_HANDLER_INIT(conn, self->cfg, NULL);
		if ((res & ST_FAIL) == 0) {
			conn->queue_node = 
				object_queue_add_tail(self->super.connections, &conn->super.super.super);
			c->num_connections++;
		}
		return res;
	}


}

static int 
do_init_afinet_source(struct log_handler *c, 
		      struct syslog_config *cfg,
		      struct persistent_config *persistent)
{
	CAST(afinet_source, self, c);
	struct listen_fd *listen;
	int fd;

	self->cfg = cfg;

	if (persistent && (self->super.flags & AFSOCKET_KEEP_ALIVE)) {
		CAST(inet_address_info, inet_addr, self->super.bind_addr);
		struct ol_string *persistent_name;
		struct persistent_info *p_info;

		persistent_name = c_format("afinet_%S_%i", inet_addr->ip, inet_addr->port);
		p_info = find_persistent_info(persistent, persistent_name->length, persistent_name->data);
		ol_string_free(persistent_name);

		if (p_info) {
			CAST(object_queue, p, p_info->o);
			self->super.num_connections = 0;
			{
				FOR_OBJECT_QUEUE(p, n) {
					CAST_SUBTYPE(log_handler, h, n);
					append_log_handler(h, self->super.super.super.next);
					self->super.num_connections++;
				}
			}
			self->super.connections = p;
			p_info->kill = NULL;

		}
	}

	fd = io_open_socket(self->super.bind_addr->family, 
			    (self->super.flags & AFSOCKET_DGRAM)
			    ? SOCK_DGRAM : SOCK_STREAM,
			    (self->super.flags & AFSOCKET_DGRAM) 
			    ? IPPROTO_UDP : IPPROTO_TCP, 
			    self->super.bind_addr);
	if (fd < 0) {
		return ST_QUIT | ST_FAIL;
	}
	switch (self->super.flags & 0x0003) {
	case AFSOCKET_STREAM: {
		if ((listen = 
		     io_listen(cfg->backend, fd,
			       make_afsocket_accept_callback(
				       cfg->backend, 
				       &self->super)))) {
			REMEMBER_RESOURCE(cfg->resources, &listen->super.super);
		}
		else
			return ST_FAIL | ST_QUIT;
		break;
	}
			
	case AFSOCKET_DGRAM: {
		struct afsocket_source_connection *conn;
		struct io_fd *client = make_io_fd(cfg->backend, fd);
		int res;

		self->super.num_connections = 1;
		conn = make_afinet_source_connection(client, NULL, &self->super, self->super.super.super.next);
		res = LOG_HANDLER_INIT(conn, cfg, NULL);
		if ((res & ST_FAIL) == 0) {
			conn->queue_node = 
				object_queue_add_tail(self->super.connections, &conn->super.super.super);
		}
		break;
	}
	default:
		return ST_FAIL | ST_QUIT;
	}
	return ST_OK | ST_GOON;		
}

static void
do_destroy_afinet_source(struct log_handler *c,
			 struct syslog_config *cfg,
			 struct persistent_config *persistent)
{
	CAST(afinet_source, self, c);

	if ((self->super.flags & AFSOCKET_KEEP_ALIVE) && persistent) {
		CAST(inet_address_info, inet_addr, self->super.bind_addr);
		struct persistent_info *p;

		p = make_persistent_info
			(c_format("afinet_%S_%i", 
				  inet_addr->ip, inet_addr->port),
			 &self->super.connections->super, 
			 NULL);
		add_persistent_info(persistent, p);
	}
	else {
		FOR_OBJECT_QUEUE(self->super.connections, n) {
			CAST_SUBTYPE(log_handler, p, n);

			LOG_HANDLER_DESTROY(p, cfg, NULL);
		}
	}
}

struct log_source_driver *make_afinet_source(struct address_info *bind_addr, UINT32 flags)
{
        NEW(afinet_source, self);

        self->super.super.super.super.init = do_init_afinet_source;        
        self->super.super.super.super.destroy = do_destroy_afinet_source;
	self->super.open_connection = do_open_afinet_connection;
	self->super.bind_addr = bind_addr;
        self->super.flags = flags;
	self->super.connections = make_object_queue();
	self->super.max_connections = 10;
        return &self->super.super;
}

/* CLASS:
   (class
     (name afinet_dest)
     (super afsocket_dest)
     (vars
       (dest object io_fd)
       (dest_buf object abstract_buffer)
       (cfg object syslog_config)))
*/

/* af_inet destination */
void afinet_dest_set_auth(struct log_dest_driver *c, UINT32 v)
{	
	CAST_SUBTYPE(afsocket_dest, l, c);
	l->flags = set_flags(l->flags, AFINET_AUTH_MASK, AFINET_AUTH_SHIFT, v);
}

void afinet_dest_set_mac(struct log_dest_driver *c, UINT32 v)
{
	CAST_SUBTYPE(afsocket_dest, l, c);
	l->flags = set_flags(l->flags, AFINET_MAC_MASK, AFINET_MAC_SHIFT, v);
}

void afinet_dest_set_encrypt(struct log_dest_driver *c, UINT32 v)
{
	CAST_SUBTYPE(afsocket_dest, l, c);
	l->flags = set_flags(l->flags, AFINET_ENCRYPT_MASK, AFINET_ENCRYPT_SHIFT, v);
}

void afinet_dest_set_localip(struct log_dest_driver *c, const char *ip)
{
	CAST_SUBTYPE(afsocket_dest, dest, c);

	inet_address_setip(&dest->dest_addr, ip);
}

void afinet_dest_set_localport(struct log_dest_driver *c, UINT32 port)
{
	CAST_SUBTYPE(afsocket_dest, dest, c);

	inet_address_setport(&dest->dest_addr, port);
}

void afinet_dest_set_destport(struct log_dest_driver *c, UINT32 port)
{
	CAST_SUBTYPE(afsocket_dest, dest, c);

	inet_address_setport(&dest->dest_addr, port);
}

static int do_init_afinet_dest(struct log_handler *c, 
			       struct syslog_config *cfg,
			       struct persistent_config *persistent)
{
	CAST(afinet_dest, self, c);
	int fd;

	self->cfg = cfg;
	if (self->super.super.log_fifo_size == -1)
		self->super.super.log_fifo_size = cfg->log_fifo_size;
	
	fd = io_open_socket(self->super.dest_addr->family, 
			    (self->super.flags & AFSOCKET_DGRAM) 
			    ? SOCK_DGRAM : SOCK_STREAM, 
			    (self->super.flags & AFSOCKET_DGRAM)
			    ? IPPROTO_UDP : IPPROTO_TCP, 
			    self->super.bind_addr);
	
	if (fd < 0) {
		return ST_FAIL | ST_QUIT;
	}
	if (!self->dest_buf)
		self->dest_buf = make_pkt_buffer(self->super.super.log_fifo_size);

	if (io_connect(cfg->backend, 
		       fd, 
		       self->super.dest_addr, 
		       make_afsocket_dest_connected(cfg->backend, &self->super))) {
		return ST_OK | ST_GOON;
	}
	else {
		werror("Error creating AF_INET socket (%z)\n", strerror(errno));
	}
	return ST_FAIL | ST_QUIT;
}

static int
do_afinet_dest_connected(struct afsocket_dest *c,
			 struct io_fd *server)
{
	CAST(afinet_dest, self, c);
	struct resource_node *n;

	if (self->dest)
		self->dest->buffer = NULL;
	if (server) {
		n = REMEMBER_RESOURCE(self->cfg->resources, 
				      &server->super.super);
		self->dest = io_write(server, 
				      self->dest_buf,
				      make_driver_reinit_on_close
				      (&self->super.super.super, n, 
				       self->cfg->time_reopen, 
				       self->cfg));
	}
	else {
		CAST(inet_address_info, inet_addr, self->super.dest_addr);

		werror("Error connecting to remote host (%S), reattempting in %i seconds\n", inet_addr->ip, self->cfg->time_reopen);
		self->dest = NULL;
		io_callout(self->cfg->backend, 
			   self->cfg->time_reopen, 
			   make_driver_reinit(&self->super.super.super, self->cfg));
	}
	return ST_OK | ST_GOON;

}

void do_handle_afinet_dest(struct log_handler *c, struct log_info *msg)
{
	CAST(afinet_dest, self, c);

	if (self->dest_buf) {
		A_WRITE_STRING(&self->dest_buf->super, c_format("<%i>%S %S %S\n", msg->pri, msg->date, msg->host, msg->msg));
	}
	log_info_free(msg);
}

struct log_dest_driver *make_afinet_dest(struct address_info *dest_addr, UINT32 flags)
{
	NEW(afinet_dest, self);
	
	self->super.super.super.init = do_init_afinet_dest;
	self->super.super.super.handler = do_handle_afinet_dest;
	self->super.super.log_fifo_size = -1;
	self->super.dest_addr = dest_addr;
	self->super.flags = flags;
	self->super.connected = do_afinet_dest_connected;
	return &self->super.super;
}
