/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: affile.c,v 1.25 1999/12/18 22:25:38 bazsi Exp $
 *
 ***************************************************************************/

#include "affile.h"
#include "format.h"
#include "pkt_buffer.h"
#include "cfgfile.h"
#include "utils.h"
#include "center.h"

#include <fcntl.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include "syslog-names.h"

#include "affile.c.x"

static int do_open_file(struct ol_string *name, int flags, int mode, int *fd)
{
	*fd = open(name->data, flags, mode);
	if (*fd != -1) {
		return 1;
	}
	return 0;
}

/* CLASS:
   (class 
     (name affile_source)
     (super log_source_driver)
     (vars
       (flags . UINT32)
       (src object io_fd)
       (name string)))
*/


static int do_init_affile_source(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(affile_source, self, c);
	int fd;
  
	if (do_open_file(self->name, O_RDONLY | O_NOCTTY | O_NONBLOCK, 0, &fd)) {
		struct resource_node *res;

		self->src = io_read(make_io_fd(cfg->backend, fd), 
			make_log_reader(0, c), 
			NULL);
		res = REMEMBER_RESOURCE(cfg->resources, &self->src->super.super);
		if (self->flags & AFFILE_PIPE) {
			self->src->super.close_callback = 
				make_driver_reinit_on_close(c, res, 0, cfg);
		}
		return ST_OK | ST_GOON;
	}
	else {
		werror("Error opening file %S for reading (%z)\n", self->name, strerror(errno));
	}
	return ST_FAIL | ST_QUIT;
}

static void do_affile_handle_line(struct log_handler *c, struct log_info *log)
{
	CAST(affile_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

struct log_source_driver *make_affile_source(const char *name, int flags)
{
	NEW(affile_source, self);
  
	self->super.super.super.init = do_init_affile_source;
	self->super.super.super.handler = do_affile_handle_line;
	self->name = c_format_cstring("%z%c", name, 0);
	self->flags = flags;
	return &self->super;
}

/* CLASS:
     (class
       (name affile_dest_reaper)
       (super callback)
       (vars
         (affile_dest object affile_dest_writer)))
*/

static void do_reap_affile_dest(struct callback *c)
{
	CAST(affile_dest_reaper, self, c);

	if (self->affile_dest->prev) {
		self->affile_dest->prev->next = self->affile_dest->next;
	}
	else {
		self->affile_dest->owner->writers = self->affile_dest->next;
	}
	if (self->affile_dest->next) {
		self->affile_dest->next->prev = self->affile_dest->prev;
	}
	if (self->affile_dest && self->affile_dest->dest)
		close_fd(&self->affile_dest->dest->super, 0);
}

static struct callback *
make_affile_dest_reaper(struct affile_dest_writer *affile_dest)
{
	NEW(affile_dest_reaper, self);
	
	self->super.f = do_reap_affile_dest;
	self->affile_dest = affile_dest;
	return &self->super;
}

/* CLASS:
     (class
       (name affile_dest_writer)
       (super log_handler)
       (vars
         (owner object affile_dest)
         (dest object io_fd)
	 (dest_buf object abstract_buffer)
	 (reap simple "struct callout *")
         (expanded_fname string)
	 (prev object affile_dest_writer)
	 (next object affile_dest_writer)))
*/

static int
do_init_dest_writer(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(affile_dest_writer, self, c);
	int fd;
	
	if (self->dest)
		self->dest->buffer = NULL;
	if (!self->dest_buf)
		self->dest_buf = make_pkt_buffer_ext(self->owner->sync_freq, 
				    MAX(self->owner->super.log_fifo_size + 1, 
					self->owner->sync_freq));
	self->dest = NULL;
	if (do_open_file(self->expanded_fname, O_WRONLY | O_CREAT | O_APPEND | O_NOCTTY | O_NONBLOCK, self->owner->perm, &fd)) {
		struct resource_node *res;

		chown(self->expanded_fname->data, self->owner->uid, self->owner->gid);

		self->dest = io_write(make_io_fd(cfg->backend, fd), 
				      self->dest_buf,
				      NULL);
		res = REMEMBER_RESOURCE(cfg->resources, &self->dest->super.super);
		self->dest->super.close_callback = 
			make_driver_reinit_on_close(c, res, cfg->time_reopen, cfg);
		if ((self->owner->flags & AFFILE_DST_NO_EXPAND) == 0)
			self->reap = io_callout(cfg->backend, TIME_REAP_DEST, make_affile_dest_reaper(self));
		/* self->attempts = 0; */
	}
	else {
		if (errno == ENXIO) {
			/* opening the write side of the fifo failed */
			/* self->attempts++; */
			io_callout(cfg->backend, cfg->time_reopen, make_driver_reinit(c, cfg));
			if ((self->owner->flags & AFFILE_DST_NO_EXPAND) == 0) 
				self->reap = io_callout(cfg->backend, TIME_REAP_DEST, make_affile_dest_reaper(self));
		}
		else {
			werror("Cannot open file %S for writing (%z)\n", self->expanded_fname, strerror(errno));
			return ST_FAIL | ST_QUIT;
		}
	}
	return ST_OK | ST_GOON;

}

static void do_handle_dest_writer(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest_writer, self, c);

	if (self->dest_buf) {
		A_WRITE_STRING(&self->dest_buf->super, c_format("%S %S %S\n", msg->date, msg->host, msg->msg));
		if (self->reap)
			io_callout_set_timeout(self->reap, TIME_REAP_DEST);
	}
	log_info_free(msg);
}

struct affile_dest_writer *make_affile_dest_writer(struct affile_dest *owner, struct ol_string *expanded_fname)
{
	NEW(affile_dest_writer, self);

	self->super.init = do_init_dest_writer;
	self->super.handler = do_handle_dest_writer;
	self->owner = owner;
	self->expanded_fname = expanded_fname;
	return self;
}

/* CLASS:
     (class
       (name affile_dest)
       (super log_dest_driver)
       (vars
         (uid . int)
	 (gid . int)
	 (perm simple int)
         (flags simple int)
	 (sync_freq simple UINT32)
	 (template_fname string)
	 (writers object affile_dest_writer)
	 (cfg object syslog_config)))
*/

void affile_set_syncfreq(struct log_dest_driver *c, int syncfreq)
{
	CAST(affile_dest, self, c);

	self->sync_freq = syncfreq;
}

void affile_set_compress(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_COMPRESS, AFFILE_COMPRESS_SHIFT, enable);
	werror("Warning: File compression not implemented yet.\n");
}

void affile_set_encrypt(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_ENCRYPT, AFFILE_ENCRYPT_SHIFT, enable);
	werror("Warning: File encryption not implemented yet.\n");
}

void affile_set_owner(struct log_dest_driver *c, char *owner)
{
	CAST(affile_dest, self, c);
	struct passwd *pwd;

	pwd = getpwnam(owner);
	if (pwd)
		self->uid = pwd->pw_uid;
	else
		self->uid = atoi(owner);
}

void affile_set_group(struct log_dest_driver *c, char *group)
{
	CAST(affile_dest, self, c);
	struct group *grp;

	grp = getgrnam(group);
	if (grp)
		self->gid = grp->gr_gid;
	else
		self->gid = atoi(group);
}

void affile_set_perm(struct log_dest_driver *c, int perm)
{
	CAST(affile_dest, self, c);

	self->perm = perm;
}

#define MAX_MACRO_ARGS        32
#define MAX_EXPANDED_MACRO    1024

struct macro_def {
	char *name;
	int id;
};

void expand_macro(int id, char **fmt, char **args[], struct log_info *msg)
{
#define ADD_MACRO_FMT(f) \
        do { \
                int len; \
                len = strlen(f); \
                memcpy(*fmt, f, len); \
                *fmt += len; \
        } while(0)
	
#define ADD_MACRO_ARG(a) \
        do { \
                (*args)[0] = (char *) (a); \
                (*args)++; \
	} while(0)
	
	switch (id) {
	case 'h': {
		UINT8 *p1 = memchr(msg->host->data, '@', msg->host->length);
		UINT8 *p2;
		int remaining;
		
		if (p1) 
			p1++; 
		else 
			p1 = msg->host->data;
		remaining = msg->host->length - (p1 - msg->host->data);
		p2 = memchr(p1, '/', remaining);
		ADD_MACRO_FMT("%s");
		if (p2) 
			ADD_MACRO_ARG(p2 - p1);
		else
			ADD_MACRO_ARG(remaining);
		ADD_MACRO_ARG(p1);

		break;
	}
	case 'f': {
		char *n = syslog_lookup_value(msg->pri & LOG_FACMASK, sl_facilities);
		if (!n) {
			ADD_MACRO_FMT("%z");
			ADD_MACRO_ARG(n);
		}
		else {
			ADD_MACRO_FMT("0x%xi");
			ADD_MACRO_ARG(msg->pri & LOG_FACMASK);
		}
		break;
	}
	case 'p': {
		ADD_MACRO_FMT("%S");
		ADD_MACRO_ARG(msg->program);
		break;
	}
	case 'l': {
		char *n = syslog_lookup_value(msg->pri & LOG_PRIMASK, sl_levels);
		if (!n) {
			ADD_MACRO_FMT("%z");
			ADD_MACRO_ARG(n);
		}
		else {
			/* should never happen */
			ADD_MACRO_FMT("0x%xi");
			ADD_MACRO_ARG(msg->pri & LOG_PRIMASK);
		}

		break;
	}
	case 'y':
	case 'm':
	case 'd': {
		struct tm *tm = localtime(&msg->stamp);

		ADD_MACRO_FMT("%fS");
		switch (id) {
		case 'y':
			ADD_MACRO_ARG(c_format("%04i", tm->tm_year + 1900));
			break;
		case 'm':
			ADD_MACRO_ARG(c_format("%02i", tm->tm_mon + 1));
			break;
		case 'd':
			ADD_MACRO_ARG(c_format("%02i", tm->tm_mday));
			break;
		}
		break;
	}
	default:
		break;
	}

}

struct ol_string *
expand_macros(struct ol_string *template, struct log_info *msg)
{
	static struct macro_def macros[] = {
		{ "HOST", 'h' },
		{ "FACILITY", 'f' },
		{ "PRIORITY", 'l' },
		{ "LEVEL", 'l' },
		{ "PROGRAM", 'p' },
		{ "YEAR", 'y' },
		{ "MONTH", 'm' },
		{ "DAY", 'd' }
	};
	char format[MAX_EXPANDED_MACRO];
	char *format_ptr = format;
	char *args[MAX_MACRO_ARGS];
	char **arg_ptr = &args[0];
	int i, j;
	int length;
	struct ol_string *result;

	i = 0;
	while ((format_ptr - format < MAX_EXPANDED_MACRO) && 
	       (i < template->length)) {

		if (template->data[i] == '$') {
			/* beginning of a macro */
			for (j = 0; j < (sizeof(macros) / sizeof(struct macro_def)); j++) {
				if (strncmp(macros[j].name, &template->data[i + 1], strlen(macros[j].name)) == 0) {
					break;
				}
			}
			if (j == (sizeof(macros) / sizeof(struct macro_def))) {
				i++;
				while ((template->data[i] >= 'A' && 
					template->data[i] <= 'Z') ||
				       template->data[i] == '_') 
					i++;
			}
			else {
				i += strlen(macros[j].name) + 1;
				expand_macro(macros[j].id, &format_ptr, &arg_ptr, msg);
			}
		}
		else {
			*format_ptr = template->data[i];
			format_ptr++;
			i++;
		}
	}
	*format_ptr = 0;
	length = c_vformat_length(format, (va_list) args);
	result = ol_string_alloc(length + 1);
	c_vformat_write(format, length, result->data, (va_list) args);
	result->data[length] = 0;
	result->length = length;
	return result;
}

/* FIXME: a binary search would be nicer */
static struct affile_dest_writer *
affile_find_writer(struct affile_dest *self, 
		   struct log_info *msg)
{
	struct ol_string *expanded_fname = NULL;
	struct affile_dest_writer *p;
	UINT32 res;	

	if (self->flags & AFFILE_DST_NO_EXPAND) {
		if (self->writers) 
			return self->writers;
		self->writers = make_affile_dest_writer(self, self->template_fname);
		res = LOG_HANDLER_INIT(self->writers, self->cfg, NULL);
		if (res & ST_QUIT)
			return NULL;
	}

	expanded_fname = expand_macros(self->template_fname, msg);
	p = self->writers;
	while (p && 
	       ((p->expanded_fname->length != expanded_fname->length) || 
		(memcmp(p->expanded_fname->data, expanded_fname->data, expanded_fname->length) != 0))) {
		p = p->next;
	}
	if (p) {
		ol_string_free(expanded_fname);
		return p;
	}
	else {
	
		/* allocate new writer */
		p = make_affile_dest_writer(self, expanded_fname);
		res = LOG_HANDLER_INIT(p, self->cfg, NULL);
		if (res & ST_QUIT)
			return NULL;
		p->next = self->writers;
		if (p->next) p->next->prev = p;
		self->writers = p;
		return p;
	}
}

static int 
do_init_affile_dest(struct log_handler *c, 
		    struct syslog_config *cfg, 
		    struct persistent_config *persistent)
{
	CAST(affile_dest, self, c);
	
	if (self->sync_freq == -1)
		self->sync_freq = cfg->sync_freq;
	if (self->super.log_fifo_size == -1)
		self->super.log_fifo_size = cfg->log_fifo_size;
	self->cfg = cfg;
	return ST_OK | ST_GOON;
}

static void do_handle_affile_log(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest, self, c);
	struct affile_dest_writer *w;

	w = affile_find_writer(self, msg);
	if (w) {
		HANDLE_LOG(&w->super, msg);
	}
	else {
		log_info_free(msg);
	}
}

struct log_dest_driver *make_affile_dest(const char *name, int flags)
{
	NEW(affile_dest, self);
	
	self->super.super.init = do_init_affile_dest;
	self->super.super.handler = do_handle_affile_log;
	self->super.log_fifo_size = -1;
	self->template_fname = c_format_cstring("%z", name);
	if (strchr(name, '$') == NULL)
		flags |= AFFILE_DST_NO_EXPAND;
	self->flags = flags;
	self->sync_freq = -1; /* use global setting */
	return &self->super;
}
