/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: destinations.c,v 1.15 1999/11/20 19:13:39 bazsi Exp $
 *
 ***************************************************************************/

#include "destinations.h"
#include "format.h"
#include "utils.h"
#include "cfgfile.h"
#include "resource.h"

#define CLASS_DEFINE
#include "destinations.h.x"
#undef CLASS_DEFINE

#include "destinations.c.x"

static void add_date(struct log_info *msg)
{
        struct tm *tm;
        char ts[20];

        if (!msg->date) {
                if (!(tm = localtime(&msg->recvd))) {
                        msg->date = c_format("%i", msg->recvd);
                } else {
                        strftime(ts, sizeof(ts) - 1, "%h %e %H:%M:%S", tm);
                        ts[16] = '\0';
			msg->date = c_format("%z", ts);
                }
        }
}

static void add_host(struct log_info *msg)
{
	if (!msg->host) {
		char buf[MAX_HOSTNAME];
		
		msg->host = c_format_cstring("%z", getshorthostname(buf, sizeof(buf)));
	}
}

/* log_dest_group */

static int do_init_group(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(log_dest_group, self, c);
	int res;
	
	struct log_dest_driver *drv;
	
	res = 0;
	for (drv = self->drivers; drv; drv = drv->next_driver) {
		res |= LOG_HANDLER_INIT(drv, cfg, persistent);
		if (res & ST_QUIT) 
			break;
	}
	return res;
}

static void 
do_destroy_group(struct log_handler *c, 
		 struct syslog_config *cfg, 
		 struct persistent_config *persistent)
{
	CAST(log_dest_group, self, c);
	
	struct log_dest_driver *drv;
	
	for (drv = self->drivers; drv; drv = drv->next_driver) {
		if (drv->super.destroy)
			LOG_HANDLER_DESTROY(drv, cfg, persistent);
	}
}

static void do_handle_log(struct log_handler *c, struct log_info *msg)
{
	CAST(log_dest_group, self, c);
	struct log_dest_driver *drv;

	add_date(msg);
	add_host(msg);	

	for (drv = self->drivers; drv; drv = drv->next_driver) {
		HANDLE_LOG(drv, log_info_use(msg));
	}
	log_info_free(msg);
}

struct log_dest_group *make_dest_group(const char *name, struct log_dest_driver *drvs)
{
	NEW(log_dest_group, self);

	self->super.handler = do_handle_log;
	self->super.destroy = do_destroy_group;
	self->super.init = do_init_group;
	self->name = c_format("%z", name);
	self->drivers = drvs;
	
	return self;
}

/* CLASS:
     (class
       (name log_dest_reopen)
       (super callback)
       (vars
         (handler object log_handler)
	 (cfg object syslog_config)))
*/

static void do_reopen(struct callback *c)
{
	CAST(log_dest_reopen, self, c);

	LOG_HANDLER_INIT(self->handler, self->cfg, NULL);
}

struct callback *make_dest_reopen(struct log_handler *handler, struct syslog_config *cfg)
{
	NEW(log_dest_reopen, self);
	self->super.f = do_reopen;
	self->handler = handler;
	self->cfg = cfg;
	return &self->super;
}

/* CLASS:
     (class
       (name dest_close)
       (super close_callback)
       (vars
         (handler object log_handler)
	 (res simple "struct resource_node *")
	 (cfg object syslog_config)
         (time_reopen simple int)))
*/

static int do_dest_close(struct close_callback *c, int reason)
{
	CAST(dest_close, self, c);

	if (reason != CLOSE_PROTOCOL_FAILURE) {
		/* schedule a reopen */
		werror("destination connection broken, reopening in %i seconds\n", self->time_reopen);
		KILL_RESOURCE_NODE(self->cfg->resources, self->res);
		io_callout(self->cfg->backend, self->time_reopen, make_dest_reopen(self->handler, self->cfg));
	}
	return ST_OK | ST_GOON;
}

struct close_callback *make_dest_close(struct log_handler *handler, struct resource_node *res, int time_reopen, struct syslog_config *cfg)
{
	NEW(dest_close, self);

	self->super.f = do_dest_close;
	self->handler = handler;
	self->res = res;
	self->time_reopen = time_reopen;
	self->cfg = cfg;
	return &self->super;
}
