/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999-2001 BalaBit IT Ltd.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: affile.c,v 1.31.2.2 2001/02/25 12:30:22 bazsi Exp $
 *
 ***************************************************************************/

#include "affile.h"
#include "format.h"
#include "pkt_buffer.h"
#include "cfgfile.h"
#include "utils.h"
#include "center.h"
#include "resource.h"
#include "cfgfile.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <stdio.h>
#include "syslog-names.h"

#include "affile.c.x"

static int do_open_file(struct ol_string *name, int flags, 
                        int uid, int gid, int mode, 
                        int dir_uid, int dir_gid, int dir_mode,
                        int create_dirs, int *fd)
{
	if (strstr(name->data, "../") ||
	    strstr(name->data, "/..")) {
		werror("Spurious path %S, logfile not created\n", name);
		return 0;
	}

	*fd = open(name->data, flags, mode);
	if (create_dirs && *fd == -1 && errno == ENOENT) {
		/* directory does not exist */
		char *p = name->data + 1;

		p = strchr(p, '/');
		while (p) {
			struct stat st;
			*p = 0;
			if (stat(name->data, &st) == 0) {
				if (!S_ISDIR(st.st_mode))
					return 0;
			}
			else if (errno == ENOENT) {
				if (mkdir(name->data, dir_mode) == -1)
					return 0;
				chown(name->data, dir_uid, dir_gid);
			}
			*p = '/';
			p = strchr(p + 1, '/');
		}
		*fd = open(name->data, flags, mode);
	}
	chown(name->data, uid, gid);
	return *fd != -1;
}

/* CLASS:
   (class 
     (name affile_source)
     (super log_source_driver)
     (vars
       (flags . UINT32)
       (src object io_fd)
       (name string)))
*/


static int do_init_affile_source(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(affile_source, self, c);
	int fd, flags;
  
  	if (self->flags & AFFILE_PIPE)
  		flags = O_RDWR | O_NOCTTY | O_NONBLOCK;
  	else
	  	flags = O_RDONLY | O_NOCTTY | O_NONBLOCK;
	if (do_open_file(self->name, flags, 0, 0, 0, 0, 0, 0, 0, &fd)) {
		struct resource_node *res;

		lseek(fd, 0, SEEK_END);
		self->src = io_read(make_io_fd(cfg->backend, fd), 
			make_log_reader(0, c), 
			NULL);
		res = REMEMBER_RESOURCE(cfg->resources, &self->src->super.super);
		return ST_OK | ST_GOON;
	}
	else {
		werror("Error opening file %S for reading (%z)\n", self->name, strerror(errno));
	}
	return ST_FAIL | ST_QUIT;
}

static void do_affile_handle_line(struct log_handler *c, struct log_info *log)
{
	CAST(affile_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

struct log_source_driver *make_affile_source(const char *name, int flags)
{
	NEW(affile_source, self);
  
	self->super.super.super.init = do_init_affile_source;
	self->super.super.super.handler = do_affile_handle_line;
	self->name = c_format_cstring("%z", name);
	self->flags = flags;
	return &self->super;
}

/* CLASS:
     (class
       (name affile_dest_reaper)
       (super callback)
       (vars
         (cfg object syslog_config)
         (affile_dest object affile_dest_writer)))
*/

static void do_reap_affile_dest(struct callback *c)
{
	CAST(affile_dest_reaper, self, c);

	if (self->affile_dest->prev) {
		self->affile_dest->prev->next = self->affile_dest->next;
	}
	else {
		self->affile_dest->owner->writers = self->affile_dest->next;
	}
	if (self->affile_dest->next) {
		self->affile_dest->next->prev = self->affile_dest->prev;
	}
	if (self->affile_dest && self->affile_dest->dest) {
		KILL_RESOURCE_NODE(self->cfg->resources, self->affile_dest->res);
	}
}

static struct callback *
make_affile_dest_reaper(struct syslog_config *cfg,
                        struct affile_dest_writer *affile_dest)
{
	NEW(affile_dest_reaper, self);
	
	self->super.f = do_reap_affile_dest;
	self->cfg = cfg;
	self->affile_dest = affile_dest;
	return &self->super;
}

/* CLASS:
     (class
       (name affile_dest_writer)
       (super log_handler)
       (vars
         (res pointer "struct resource_node")
         (time_reap simple UINT32)
         (owner object affile_dest)
         (dest object io_fd)
	 (reap simple "struct callout *")
         (expanded_fname string)
	 (prev object affile_dest_writer)
	 (next object affile_dest_writer)))
*/

static int
do_init_dest_writer(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(affile_dest_writer, self, c);
	int fd, flags;
	
  	if (self->owner->flags & AFFILE_PIPE)
  		flags = O_RDWR | O_NOCTTY | O_NONBLOCK;
  	else
	  	flags = O_WRONLY | O_CREAT | O_APPEND |O_NOCTTY | O_NONBLOCK;
	if (do_open_file(self->expanded_fname, flags, 
	                 self->owner->uid, self->owner->gid, self->owner->perm, 
	                 self->owner->dir_uid, self->owner->dir_gid, self->owner->dir_perm,
	                 self->owner->flags & AFFILE_CREATE_DIRS, &fd)) {

		self->dest = 
			io_write(make_io_fd(cfg->backend, fd), 
				 make_pkt_buffer_ext(self->owner->sync_freq, 
				                     MAX(self->owner->super.log_fifo_size + 1, 
					                 self->owner->sync_freq)),
				 NULL);
		self->res = REMEMBER_RESOURCE(cfg->resources, &self->dest->super.super);
		self->time_reap = cfg->time_reap;
		if ((self->owner->flags & AFFILE_NO_EXPAND) == 0) {
			self->reap = io_callout(cfg->backend, cfg->time_reap, make_affile_dest_reaper(cfg, self));
			io_callout_set_drop(self->reap, 0);
		}
		/* self->attempts = 0; */
	}
	else {
		werror("Cannot open file %S for writing (%z)\n", self->expanded_fname, strerror(errno));
		return ST_FAIL | ST_QUIT;
	}
	return ST_OK | ST_GOON;

}

static void do_handle_dest_writer(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest_writer, self, c);

	if (self->dest) {
		A_WRITE_STRING(&self->dest->buffer->super, c_format("%S %S %S\n", msg->date, msg->host, msg->msg));
		if (self->reap)
			io_callout_set_timeout(self->reap, self->time_reap);
	}
	log_info_free(msg);
}

struct affile_dest_writer *make_affile_dest_writer(struct affile_dest *owner, struct ol_string *expanded_fname)
{
	NEW(affile_dest_writer, self);

	self->super.init = do_init_dest_writer;
	self->super.handler = do_handle_dest_writer;
	self->owner = owner;
	self->expanded_fname = expanded_fname;
	return self;
}

/* CLASS:
     (class
       (name affile_dest)
       (super log_dest_driver)
       (vars
         (uid . int)
	 (gid . int)
	 (perm . int)
	 (dir_uid . int)
	 (dir_gid . int)
	 (dir_perm . int)
         (flags simple int)
	 (sync_freq simple UINT32)
	 (template_fname string)
	 (writers object affile_dest_writer)
	 (cfg object syslog_config)))
*/

void affile_set_syncfreq(struct log_dest_driver *c, int syncfreq)
{
	CAST(affile_dest, self, c);

	self->sync_freq = syncfreq;
}

void affile_set_compress(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_COMPRESS, AFFILE_COMPRESS_SHIFT, enable);
	werror("Warning: File compression not implemented yet.\n");
}

void affile_set_encrypt(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_ENCRYPT, AFFILE_ENCRYPT_SHIFT, enable);
	werror("Warning: File encryption not implemented yet.\n");
}

void affile_set_owner(struct log_dest_driver *c, char *owner)
{
	CAST(affile_dest, self, c);
	struct passwd *pwd;

	pwd = getpwnam(owner);
	if (pwd)
		self->uid = pwd->pw_uid;
	else
		self->uid = atoi(owner);
}

void affile_set_group(struct log_dest_driver *c, char *group)
{
	CAST(affile_dest, self, c);
	struct group *grp;

	grp = getgrnam(group);
	if (grp)
		self->gid = grp->gr_gid;
	else
		self->gid = atoi(group);
}

void affile_set_perm(struct log_dest_driver *c, int perm)
{
	CAST(affile_dest, self, c);

	self->perm = perm;
}

void affile_set_dir_owner(struct log_dest_driver *c, char *owner)
{
	CAST(affile_dest, self, c);
	struct passwd *pwd;

	pwd = getpwnam(owner);
	if (pwd)
		self->dir_uid = pwd->pw_uid;
	else
		self->dir_uid = atoi(owner);
}

void affile_set_dir_group(struct log_dest_driver *c, char *group)
{
	CAST(affile_dest, self, c);
	struct group *grp;

	grp = getgrnam(group);
	if (grp)
		self->dir_gid = grp->gr_gid;
	else
		self->dir_gid = atoi(group);
}


void affile_set_dir_perm(struct log_dest_driver *c, int perm)
{
	CAST(affile_dest, self, c);

	self->dir_perm = perm;
}

void affile_set_create_dirs(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_CREATE_DIRS, AFFILE_CREATE_DIRS_SHIFT, enable);
}

#define MAX_MACRO_ARGS        32
#define MAX_EXPANDED_MACRO    1024

struct macro_def {
	char *name;
	int id;
};

void expand_macro(struct syslog_config *cfg, int id, char **dest, int *left, struct log_info *msg)
{
	int length = 0;
	
	switch (id) {
	case 'h': {
		UINT8 *p1 = memchr(msg->host->data, '@', msg->host->length);
		UINT8 *p2;
		int remaining;
		
		if (p1) 
			p1++; 
		else 
			p1 = msg->host->data;
                remaining = msg->host->length - (p1 - msg->host->data);
		p2 = memchr(p1, '/', remaining);
		if (p2) {
			length = MIN(p2 - p1, *left);
		}
		else {
			length = MIN(*left, msg->host->length - (p1 - msg->host->data));
		}
		memcpy(*dest, p1, length);

		break;
	}
	case 'f': {
		char *n = syslog_lookup_value(msg->pri & LOG_FACMASK, sl_facilities);
		if (n) {
			length = MIN(*left, strlen(n));
			memcpy(*dest, n, length);
		}
		else {
			length = snprintf(*dest, *left, "%x", (msg->pri & LOG_FACMASK) >> 3);
		}
		break;
	}
	case 'p': {
		if (msg->program) {
			length = MIN(*left, msg->program->length);
			memcpy(*dest, msg->program->data, length);
		}
		break;
	}
	case 'l': {
		char *n = syslog_lookup_value(msg->pri & LOG_PRIMASK, sl_levels);
		if (n) {
			length = MIN(*left, strlen(n));
			memcpy(*dest, n, length);
		}
		else {
			/* should never happen */
			length = snprintf(*dest, *left, "%d", msg->pri & LOG_PRIMASK);
		}

		break;
	}
	case 'y':
	case 'm':
	case 'd': 
	case 'H':
	case 'M':
	case 'S': {
		struct tm *tm;

		if (cfg->use_time_recvd)
			tm = localtime(&msg->recvd);
		else
			tm = localtime(&msg->stamp);

		switch (id) {
		case 'y':
			length = snprintf(*dest, *left, "%04d", tm->tm_year + 1900);
			break;
		case 'm':
			length = snprintf(*dest, *left, "%02d", tm->tm_mon + 1);
			break;
		case 'd':
			length = snprintf(*dest, *left, "%02d", tm->tm_mday);
			break;
		case 'H':
			length = snprintf(*dest, *left, "%02d", tm->tm_hour);
			break;
		case 'M':
			length = snprintf(*dest, *left, "%02d", tm->tm_min);
			break;
		case 'S':
			length = snprintf(*dest, *left, "%02d", tm->tm_sec);
			break;
		}
		break;
	}
	default:
		break;
	}
	*left -= length;
	*dest += length;
}

struct ol_string *
expand_macros(struct syslog_config *cfg, struct ol_string *template, struct log_info *msg)
{
	static struct macro_def macros[] = {
		{ "HOST", 'h' },
		{ "FACILITY", 'f' },
		{ "PRIORITY", 'l' },
		{ "LEVEL", 'l' },
		{ "PROGRAM", 'p' },
		{ "YEAR", 'y' },
		{ "MONTH", 'm' },
		{ "DAY", 'd' },
		{ "HOUR", 'H' },
		{ "MIN", 'M' },
		{ "SEC", 'S' }
	};
	char format[MAX_EXPANDED_MACRO], *format_ptr = format;
	int left = sizeof(format);
	int i, j;

	i = 0;
	while (left && (i < template->length)) {

		if (template->data[i] == '$') {
			/* beginning of a macro */
			for (j = 0; j < (sizeof(macros) / sizeof(struct macro_def)); j++) {
				if (strncmp(macros[j].name, &template->data[i + 1], strlen(macros[j].name)) == 0) {
					break;
				}
			}
			if (j == (sizeof(macros) / sizeof(struct macro_def))) {
				i++;
				while ((template->data[i] >= 'A' && 
					template->data[i] <= 'Z') ||
				       template->data[i] == '_') 
					i++;
			}
			else {
				i += strlen(macros[j].name) + 1;
				expand_macro(cfg, macros[j].id, &format_ptr, &left, msg);
			}
		}
		else {
			*format_ptr = template->data[i];
			format_ptr++;
			i++;
		}
	}
	*format_ptr = 0;
	return c_format_cstring("%z", format);
}

/* FIXME: a binary search would be nicer */
static struct affile_dest_writer *
affile_find_writer(struct affile_dest *self, 
		   struct log_info *msg)
{
	struct ol_string *expanded_fname = NULL;
	struct affile_dest_writer *p;
	UINT32 res;	

	if (self->flags & AFFILE_NO_EXPAND) {
		if (self->writers) 
			return self->writers;
		self->writers = make_affile_dest_writer(self, self->template_fname);
		res = LOG_HANDLER_INIT(self->writers, self->cfg, NULL);
		if (res & ST_QUIT)
			return NULL;
	}

	expanded_fname = expand_macros(self->cfg, self->template_fname, msg);
	p = self->writers;
	while (p && 
	       ((p->expanded_fname->length != expanded_fname->length) || 
		(memcmp(p->expanded_fname->data, expanded_fname->data, expanded_fname->length) != 0))) {
		p = p->next;
	}
	if (p) {
		ol_string_free(expanded_fname);
		return p;
	}
	else {
	
		/* allocate new writer */
		p = make_affile_dest_writer(self, expanded_fname);
		res = LOG_HANDLER_INIT(p, self->cfg, NULL);
		if (res & ST_QUIT)
			return NULL;
		p->next = self->writers;
		if (p->next) p->next->prev = p;
		self->writers = p;
		return p;
	}
}

static int 
do_init_affile_dest(struct log_handler *c, 
		    struct syslog_config *cfg, 
		    struct persistent_config *persistent)
{
	CAST(affile_dest, self, c);
	
	if (self->sync_freq == -1)
		self->sync_freq = cfg->sync_freq;
	if (self->super.log_fifo_size == -1)
		self->super.log_fifo_size = cfg->log_fifo_size;
	if (cfg->create_dirs)
		self->flags |= AFFILE_CREATE_DIRS;
	if (self->uid == -1)
		self->uid = cfg->uid;
	if (self->gid == -1)
		self->gid = cfg->gid;
	if (self->perm == -1)
		self->perm = cfg->perm;
	if (self->dir_uid == -1)
		self->dir_uid = cfg->dir_uid;
	if (self->dir_gid == -1)
		self->dir_gid = cfg->dir_gid;
	if (self->dir_perm == -1)
		self->dir_perm = cfg->dir_perm;
	
	self->cfg = cfg;
	return ST_OK | ST_GOON;
}

static void do_handle_affile_log(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest, self, c);
	struct affile_dest_writer *w;

	w = affile_find_writer(self, msg);
	if (w) {
		HANDLE_LOG(&w->super, msg);
	}
	else {
		log_info_free(msg);
	}
}

struct log_dest_driver *make_affile_dest(const char *name, int flags)
{
	NEW(affile_dest, self);
	
	self->super.super.init = do_init_affile_dest;
	self->super.super.handler = do_handle_affile_log;
	self->super.log_fifo_size = -1;
	self->uid = self->gid = -1;
	self->perm = -1;
	self->dir_uid = self->dir_gid = -1;
	self->dir_perm = -1;
	self->template_fname = c_format_cstring("%z", name);
	if (strchr(name, '$') == NULL)
		flags |= AFFILE_NO_EXPAND;
	self->flags = flags;
	self->sync_freq = -1; /* use global setting */
	return &self->super;
}
