/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999 BalaBit Computing
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: log.c,v 1.20 2000/11/29 11:21:31 bazsi Exp $
 *
 ***************************************************************************/

#include "log.h"
#include "xalloc.h"
#include "format.h"

#include <sys/time.h>
#include <syslog.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <string.h>

#define CLASS_DEFINE
#include "log.h.x"
#undef CLASS_DEFINE

static void parse_log_msg(struct log_info *lm, UINT32 length, UINT8 *data)
{
	char *src;
	int left;
	int pri;
	time_t now = time(NULL);
	char *oldsrc;
	int oldleft;
	
	src = data;
	left = length;
	
	if (left && src[0] == '<') {
		src++;
		left--;
		pri = 0;
		while (left && *src != '>') {
			if (isdigit(*src)) {
				pri = pri * 10 + ((*src) - '0');
			}
			else {
				lm->msg = c_format_cstring("unparseable log message: \"%s\"", length, data);
				lm->pri = LOG_SYSLOG | LOG_ERR;
				return;
			}
			src++;
			left--;
		}
		lm->pri = pri;
		if (left) {
			src++;
			left--;
		}
	}
	else {
		lm->pri = LOG_USER | LOG_NOTICE;
	}
	while (left && *src == ' ') {
		src++;
		left--;
	}
	if (left >= 15) {
		if (src[3] == ' ' && src[6] == ' ' && src[9] == ':' && src[12] == ':') {
			struct tm tm, *nowtm;

			lm->date = c_format_cstring("%s", 15, src);
			src += 15;
			left -= 15;

			nowtm = localtime(&now);
			memset(&tm, 0, sizeof(tm));
			strptime(lm->date->data, "%b %e %H:%M:%S", &tm);
			tm.tm_isdst = -1;
			tm.tm_year = nowtm->tm_year;
			if (tm.tm_mon > nowtm->tm_year)
				tm.tm_year--;
			lm->stamp = mktime(&tm);
		}
	}

	if (lm->date) {
		while (left && *src == ' ') {
			src++; /* skip whitespace */
			left--;
		}
		oldsrc = src;	
		oldleft = left;
		while (left && *src != ' ' && *src != ':' && *src != '[') {
			src++;
			left--;
		}
		if (left && *src == ' ') {
			/* this was a hostname */
			lm->host = c_format_cstring("%s", oldleft - left, oldsrc);
		}
		else {
			src = oldsrc;
			left = oldleft;
		}

		while (left && *src == ' ') {
			src++; /* skip whitespace */
			left--;
		}
		
		oldsrc = src;
		oldleft = left;

		while (left && *src != ':' && *src != '[') {
			src++;
			left--;
		}
		if (left) {
			lm->program = c_format_cstring("%s", oldleft - left, oldsrc);
		}
		src = oldsrc;
		left = oldleft;
	}
	else {
		oldsrc = src;
		oldleft = left;
		if ((lm->pri & LOG_FACMASK) == LOG_KERN) {
			lm->program = c_format_cstring("kernel");
		}
		else {
			while (left && *src != ' ' && *src != '[' && *src != ':' && 
			       *src != '/' && *src != ',' && *src != '<') {
				src++;
				left--;
			}
			if (left) {
				lm->program = c_format_cstring("%s", oldleft - left, oldsrc);
			}
			left = oldleft;
			src = oldsrc;
		}
		lm->stamp = now;
	}

	for (oldsrc = src, oldleft = left; oldleft >= 0; oldleft--, oldsrc++) {
		if (*oldsrc == '\n' || *oldsrc == '\r') *oldsrc = ' ';
	}
	lm->msg = c_format_cstring("%s", left, src);
}

struct log_info *log_info_use(struct log_info *msg)
{
	msg->use_cnt++;
	return msg;
}

void log_info_free(struct log_info *msg)
{
	if (--msg->use_cnt == 0) {
		ol_string_free(msg->host);
		ol_string_free(msg->program);
		ol_string_free(msg->date);
		ol_string_free(msg->msg);
		ol_space_free(msg);
	}
}

struct log_info *make_log_info(UINT32 length, UINT8 *msg)
{
	struct log_info *self;

	NEW_SPACE(self);
  	parse_log_msg(self, length, msg);
	self->use_cnt = 1;
  	self->recvd = time(NULL);
	return self;
}

struct log_info *make_internal_message(UINT32 pri, UINT32 length, UINT8 *data)
{
	struct log_info *self;

	NEW_SPACE(self);
	self->msg = c_format_cstring("syslog-ng[%i]: %s", getpid(), length, data);
	self->program = c_format_cstring("syslog-ng");
	self->stamp = self->recvd = time(NULL);
	self->pri = pri;
	self->flags = LF_INTERNAL;
	self->use_cnt = 1;
	return self;
}

struct log_info *make_mark_message(void)
{
	struct log_info *self = make_internal_message(LOG_SYSLOG | LOG_NOTICE, 12, "--- MARK ---");
	self->flags |= LF_MARK;
	return self;
}
