/***************************************************************************
 *
 * Copyright (c) 1999 Balzs Scheidler
 * Copyright (c) 1999-2001 BalaBit IT Ltd.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Inspired by nsyslog, originally written by Darren Reed.
 *
 * $Id: affile.c,v 1.52 2002/08/21 14:03:50 bazsi Exp $
 *
 ***************************************************************************/
 
/*
 * Several patches to the macro expansion function were contributed
 * by Gert Menke.
 *
 * Portions Copyright (c) 2002 by Gert Menke
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS `AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "affile.h"
#include "format.h"
#include "pkt_buffer.h"
#include "cfgfile.h"
#include "utils.h"
#include "center.h"
#include "resource.h"
#include "cfgfile.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <time.h>
#include <stdio.h>
#include "syslog-names.h"

#include "affile.c.x"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

struct ol_string *
expand_macros(struct syslog_config *cfg, struct ol_string *template, int template_escape, struct log_info *msg);


static int do_open_file(struct ol_string *name, int flags, 
                        int uid, int gid, int mode, 
                        int dir_uid, int dir_gid, int dir_mode,
                        int create_dirs, int *fd)
{
	if (strstr(name->data, "../") ||
	    strstr(name->data, "/..")) {
		werror("Spurious path %S, logfile not created\n", name);
		return 0;
	}
	if (strncmp(name->data, "/dev", 4) == 0) {
		werror("Changing permissions on special file %S\n", name);
	}

	*fd = open(name->data, flags, mode != -1 ? mode : 0600);
	if (create_dirs && *fd == -1 && errno == ENOENT) {
		/* directory does not exist */
		char *p = name->data + 1;

		p = strchr(p, '/');
		while (p) {
			struct stat st;
			*p = 0;
			if (stat(name->data, &st) == 0) {
				if (!S_ISDIR(st.st_mode))
					return 0;
			}
			else if (errno == ENOENT) {
				if (mkdir(name->data, dir_mode != -1 ? dir_mode : 0700) == -1)
					return 0;
				if (dir_uid != -1 || dir_gid != -1)
					chown(name->data, dir_uid, dir_gid);
				if (dir_mode != -1)
					chmod(name->data, dir_mode);
			}
			*p = '/';
			p = strchr(p + 1, '/');
		}
		*fd = open(name->data, flags, mode);
	}
	if (uid != -1 || gid != -1)
		chown(name->data, uid, gid);
	if (mode != -1)
		chmod(name->data, mode);
	return *fd != -1;
}

/* CLASS:
   (class 
     (name affile_source)
     (super log_source_driver)
     (vars
       (res pointer "struct resource_node")
       (flags . UINT32)
       (src object io_fd)
       (prefix pointer UINT8)
       (pad_size . UINT32)
       (name string)))
*/


static int do_init_affile_source(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(affile_source, self, c);
	int fd, flags;
  
  	if (self->flags & AFFILE_PIPE)
  		flags = O_RDWR | O_NOCTTY | O_NONBLOCK | O_LARGEFILE;
  	else
	  	flags = O_RDONLY | O_NOCTTY | O_NONBLOCK | O_LARGEFILE;
	if (do_open_file(self->name, flags, -1, -1, -1, -1, -1, -1, 0, &fd)) {
		lseek(fd, 0, SEEK_END);
		self->src = io_read(make_io_fd(cfg->backend, fd, ol_string_use(self->name)), 
			make_log_reader(0, self->prefix, cfg->log_msg_size, self->pad_size, cfg->check_hostname ? LF_CHECK_HOSTNAME : 0, c), 
			NULL);
		self->res = REMEMBER_RESOURCE(cfg->resources, &self->src->super.super);
		return ST_OK | ST_GOON;
	}
	else {
		werror("Error opening file %S for reading (%z)\n", self->name, strerror(errno));
	}
	return ST_FAIL | ST_QUIT;
}

static void do_affile_handle_line(struct log_handler *c, struct log_info *log)
{
	CAST(affile_source, self, c);
	log->flags |= LF_LOCAL;
	HANDLE_LOG(self->super.super.next, log);
}

static void do_destroy_affile_source(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persist)
{
	CAST(affile_source, self, c);
	if (self->res)
  		KILL_RESOURCE_NODE(cfg->resources, self->res);
}

struct log_source_driver *make_affile_source(const char *name, int flags, UINT8 *prefix, UINT32 pad_size)
{
	NEW(affile_source, self);
  
	self->super.super.super.init = do_init_affile_source;
	self->super.super.super.handler = do_affile_handle_line;
	self->super.super.super.destroy = do_destroy_affile_source;
	self->name = c_format_cstring("%z", name);
	self->flags = flags;
	self->prefix = prefix;
	self->pad_size = pad_size;
	return &self->super;
}

/* CLASS:
     (class
       (name affile_dest_reaper)
       (super callback)
       (vars
         (cfg object syslog_config)
         (affile_dest object affile_dest_writer)))
*/

static void do_reap_affile_dest(struct callback *c)
{
	CAST(affile_dest_reaper, self, c);

	if (self->affile_dest->prev) {
		self->affile_dest->prev->next = self->affile_dest->next;
	}
	else {
		self->affile_dest->owner->writers = self->affile_dest->next;
	}
	if (self->affile_dest->next) {
		self->affile_dest->next->prev = self->affile_dest->prev;
	}
	if (self->affile_dest && self->affile_dest->dest) {
		KILL_RESOURCE_NODE(self->cfg->resources, self->affile_dest->res);
	}
}

static struct callback *
make_affile_dest_reaper(struct syslog_config *cfg,
                        struct affile_dest_writer *affile_dest)
{
	NEW(affile_dest_reaper, self);
	
	self->super.f = do_reap_affile_dest;
	self->cfg = cfg;
	self->affile_dest = affile_dest;
	return &self->super;
}

/* CLASS:
     (class
       (name affile_dest_writer)
       (super log_handler)
       (vars
         (res pointer "struct resource_node")
         (time_reap simple UINT32)
         (owner object affile_dest)
         (dest object io_fd)
	 (reap simple "struct callout *")
         (expanded_fname string)
	 (prev object affile_dest_writer)
	 (next object affile_dest_writer)))
*/

static int
do_init_dest_writer(struct log_handler *c, struct syslog_config *cfg, struct persistent_config *persistent)
{
	CAST(affile_dest_writer, self, c);
	int fd, flags;
	
  	if (self->owner->flags & AFFILE_PIPE)
  		flags = O_RDWR | O_NOCTTY | O_NONBLOCK | O_LARGEFILE;
  	else
	  	flags = O_WRONLY | O_CREAT | O_APPEND | O_NOCTTY | O_NONBLOCK | O_LARGEFILE;

        if (self->owner->remove_if_older > 0) {
                /* remove (=unlink) if file is more than XXX days old */
                struct stat st;
                time_t now = time(NULL);

                if (stat(self->expanded_fname->data, &st) == 0) {
                        if ((now-st.st_mtime) >= ((self->owner->remove_if_older))) {
                                unlink(self->expanded_fname->data);
                        }
                }  
        }


	if (do_open_file(self->expanded_fname, flags, 
	                 self->owner->uid, self->owner->gid, self->owner->perm, 
	                 self->owner->dir_uid, self->owner->dir_gid, self->owner->dir_perm,
	                 self->owner->flags & AFFILE_CREATE_DIRS, &fd)) {

		self->dest = 
			io_write(make_io_fd(cfg->backend, fd, ol_string_use(self->expanded_fname)), 
				 make_pkt_buffer_ext(self->owner->sync_freq, 
				                     MAX(self->owner->super.log_fifo_size + 1, 
					                 self->owner->sync_freq)),
				 NULL);
		if (self->owner->flags & AFFILE_FSYNC)
			self->dest->fsync = 1;
		self->res = REMEMBER_RESOURCE(cfg->resources, &self->dest->super.super);
		self->time_reap = cfg->time_reap;
		if ((self->owner->flags & AFFILE_NO_EXPAND) == 0) {
			self->reap = io_callout(cfg->backend, cfg->time_reap, make_affile_dest_reaper(cfg, self));
			/* we want to be called on reinitialization */
			io_callout_set_drop(self->reap, 0);
		}
		/* self->attempts = 0; */
	}
	else {
		werror("Cannot open file %S for writing (%z)\n", self->expanded_fname, strerror(errno));
		return ST_FAIL | ST_QUIT;
	}
	return ST_OK | ST_GOON;

}

static void do_handle_dest_writer(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest_writer, self, c);

	if (self->dest) {
		struct ol_string *l;
		if (self->owner->template_output)
			l = expand_macros(self->owner->cfg, self->owner->template_output, self->owner->template_escape, msg);
		else
			l = c_format("%S %S %S\n", msg->date, msg->host, msg->msg);
		A_WRITE_STRING(&self->dest->buffer->super, l);
		if (self->reap)
			io_callout_set_timeout(self->reap, self->time_reap);
	}
	log_info_free(msg);
}

struct affile_dest_writer *make_affile_dest_writer(struct affile_dest *owner, struct ol_string *expanded_fname)
{
	NEW(affile_dest_writer, self);

	self->super.init = do_init_dest_writer;
	self->super.handler = do_handle_dest_writer;
	self->owner = owner;
	self->expanded_fname = expanded_fname;
	return self;
}

/* CLASS:
     (class
       (name affile_dest)
       (super log_dest_driver)
       (vars
         (uid . int)
	 (gid . int)
	 (perm . int)
	 (dir_uid . int)
	 (dir_gid . int)
	 (dir_perm . int)
         (flags simple int)
	 (remove_if_older . int)
	 (sync_freq simple UINT32)
	 (template_fname string)
	 (template_output string)
	 (template_escape . int)
	 (writers object affile_dest_writer)
	 (cfg object syslog_config)))
*/

void affile_set_syncfreq(struct log_dest_driver *c, int syncfreq)
{
	CAST(affile_dest, self, c);

	self->sync_freq = syncfreq;
}

void affile_set_fsync(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_FSYNC, AFFILE_FSYNC_SHIFT, enable);
}

void affile_set_compress(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_COMPRESS, AFFILE_COMPRESS_SHIFT, enable);
	werror("Warning: File compression not implemented yet.\n");
}

void affile_set_encrypt(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_ENCRYPT, AFFILE_ENCRYPT_SHIFT, enable);
	werror("Warning: File encryption not implemented yet.\n");
}

void affile_set_owner(struct log_dest_driver *c, char *owner)
{
	CAST(affile_dest, self, c);
	struct passwd *pwd;

	pwd = getpwnam(owner);
	if (pwd)
		self->uid = pwd->pw_uid;
	else
		self->uid = atoi(owner);
}

void affile_set_group(struct log_dest_driver *c, char *group)
{
	CAST(affile_dest, self, c);
	struct group *grp;

	grp = getgrnam(group);
	if (grp)
		self->gid = grp->gr_gid;
	else
		self->gid = atoi(group);
}

void affile_set_perm(struct log_dest_driver *c, int perm)
{
	CAST(affile_dest, self, c);

	self->perm = perm;
}

void affile_set_dir_owner(struct log_dest_driver *c, char *owner)
{
	CAST(affile_dest, self, c);
	struct passwd *pwd;

	pwd = getpwnam(owner);
	if (pwd)
		self->dir_uid = pwd->pw_uid;
	else
		self->dir_uid = atoi(owner);
}

void affile_set_dir_group(struct log_dest_driver *c, char *group)
{
	CAST(affile_dest, self, c);
	struct group *grp;

	grp = getgrnam(group);
	if (grp)
		self->dir_gid = grp->gr_gid;
	else
		self->dir_gid = atoi(group);
}


void affile_set_dir_perm(struct log_dest_driver *c, int perm)
{
	CAST(affile_dest, self, c);

	self->dir_perm = perm;
}

void affile_set_create_dirs(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->flags = set_flags(self->flags, AFFILE_CREATE_DIRS, AFFILE_CREATE_DIRS_SHIFT, enable);
}

void affile_set_file_template(struct log_dest_driver *c, char *t)
{
	CAST(affile_dest, self, c);

	self->template_output = c_format("%z", t);
}

void affile_set_template_escape(struct log_dest_driver *c, int enable)
{
	CAST(affile_dest, self, c);

	self->template_escape = enable;
}

void affile_set_remove_if_older(struct log_dest_driver *c, int interval)
{
	CAST(affile_dest, self, c);

	if (interval < 0) {
		werror("Warning: the time before removal parameter is negative, disables option\n");
		interval = -1;
	}
	self->remove_if_older = interval;
}



#define MAX_MACRO_ARGS        32
#define MAX_EXPANDED_MACRO    2048

struct macro_def {
	char *name;
	int id;
};

#define M_FACILITY 0
#define M_LEVEL    1
#define M_TAG      2

#define M_DATE     3
#define M_FULLDATE 4
#define M_ISODATE  5
#define M_YEAR     6
#define M_MONTH    7
#define M_DAY      8
#define M_HOUR     9
#define M_MIN      10
#define M_SEC      11
#define M_WEEKDAY  12
#define M_UNIXTIME 13

#define M_DATE_RECVD     14
#define M_FULLDATE_RECVD 15
#define M_ISODATE_RECVD  16
#define M_YEAR_RECVD     17
#define M_MONTH_RECVD    18
#define M_DAY_RECVD      19
#define M_HOUR_RECVD     20
#define M_MIN_RECVD      21
#define M_SEC_RECVD      22
#define M_WEEKDAY_RECVD  23
#define M_UNIXTIME_RECVD 24

#define M_DATE_STAMP     25
#define M_FULLDATE_STAMP 26
#define M_ISODATE_STAMP  27
#define M_YEAR_STAMP     28
#define M_MONTH_STAMP    29
#define M_DAY_STAMP      30
#define M_HOUR_STAMP     31
#define M_MIN_STAMP      32
#define M_SEC_STAMP      33
#define M_WEEKDAY_STAMP  34
#define M_UNIXTIME_STAMP 35

#define M_FULLHOST 36
#define M_HOST     37
#define M_FULLHOST_FROM 38
#define M_HOST_FROM  39
#define M_PROGRAM  40

#define M_MESSAGE  41

#define M_SOURCE_IP 42

int append_string(char **dest, int *left, char *str, int length, int escape)
{
	int l;
  
	if (!escape) {
		l = MIN(length, *left - 1);
		strncpy(*dest, str, l);
	}
	else {
		char *p;
      
		l = 0;
		for (p = str; length && ((*left - l) > 1); p++, length--, l++) {
			if (*p == '\'' || *p == '\"' || *p == '\\') {
				if ((*left - l) < 3)
					break;
				*(*dest + l) = '\\';
				*(*dest + l + 1) = *p;
				l++;
			}
			else {
				*(*dest + l) = *p;
			}
		}
	}
	return l;
}

void expand_macro(struct syslog_config *cfg, int id, int escape, char **dest, int *left, struct log_info *msg)
{
	int length = 0;
	
	switch (id) {
	case M_FACILITY: {
		/* facility */
		char *n = syslog_lookup_value(msg->pri & LOG_FACMASK, sl_facilities);
		if (n) {
			length = append_string(dest, left, n, strlen(n), 0);
		}
		else {
			length = snprintf(*dest, *left, "%x", (msg->pri & LOG_FACMASK) >> 3);
		}
		break;
	}
	case M_LEVEL: {
		/* level */
		char *n = syslog_lookup_value(msg->pri & LOG_PRIMASK, sl_levels);
		if (n) {
			length = append_string(dest, left, n, strlen(n), 0);
		}
		else {
			/* should never happen */
			length = snprintf(*dest, *left, "%d", msg->pri & LOG_PRIMASK);
		}

		break;
	}
	case M_TAG: {
		length = snprintf(*dest, *left, "%02x", msg->pri);
		break;
	}
	case M_SOURCE_IP: {
 		char *ip;
 		
		if (msg->saddr) {
	 		CAST(inet_address_info, addr, msg->saddr);
 		
	 		ip = inet_ntoa(addr->sa.sin_addr);
		}
		else {
			ip = "127.0.0.1";
		}
		length = append_string(dest, left, ip, strlen(ip), escape);
		break;
	}
	case M_FULLHOST_FROM:
	case M_FULLHOST: {
		struct ol_string *host = (id == M_FULLHOST ? msg->host : msg->host_from);
		/* full hostname */
		length = append_string(dest, left, host->data, host->length, escape);
		break;
	}
	case M_HOST_FROM:
	case M_HOST: {
		/* host */
		struct ol_string *host = (id == M_HOST ? msg->host : msg->host_from);
		UINT8 *p1;
		UINT8 *p2;
		int remaining;
		
		p1 = memchr(host->data, '@', host->length);
		if (p1) 
			p1++; 
		else 
			p1 = host->data;
                remaining = host->length - (p1 - host->data);
		p2 = memchr(p1, '/', remaining);
		if (p2) {
			length = MIN(p2 - p1, *left);
		}
		else {
			length = MIN(*left, host->length - (p1 - host->data));
		}
		length = append_string(dest, left, p1, length, escape);
		break;
	}
	case M_PROGRAM: {
		/* program */
		if (msg->program) {
			length = append_string(dest, left, msg->program->data, msg->program->length, escape);
		}
		break;
	}
	case M_FULLDATE_RECVD:
	case M_ISODATE_RECVD:
	case M_WEEKDAY_RECVD:
	case M_DATE_RECVD:
	case M_YEAR_RECVD:
	case M_MONTH_RECVD:
	case M_DAY_RECVD:
	case M_HOUR_RECVD:
	case M_MIN_RECVD:
	case M_SEC_RECVD:
	case M_UNIXTIME_RECVD:

	case M_FULLDATE_STAMP:
	case M_ISODATE_STAMP:
	case M_WEEKDAY_STAMP:
	case M_DATE_STAMP:
	case M_YEAR_STAMP:
	case M_MONTH_STAMP:
	case M_DAY_STAMP:
	case M_HOUR_STAMP:
	case M_MIN_STAMP:
	case M_SEC_STAMP:
	case M_UNIXTIME_STAMP:

	case M_FULLDATE:
	case M_ISODATE:
	case M_WEEKDAY:
	case M_DATE:
	case M_YEAR:
	case M_MONTH:
	case M_DAY: 
	case M_HOUR:
	case M_MIN:
	case M_SEC:
	case M_UNIXTIME: {
		/* year, month, day */
		struct tm *tm;
		time_t unixtime;

	       	switch(id) {
	       	case M_FULLDATE_RECVD:
	       	case M_ISODATE_RECVD: 
	       	case M_WEEKDAY_RECVD: 
	       	case M_DATE_RECVD:    
	       	case M_YEAR_RECVD:    
	       	case M_MONTH_RECVD:   
	       	case M_DAY_RECVD:     
	       	case M_HOUR_RECVD:    
	       	case M_MIN_RECVD:     
	       	case M_SEC_RECVD:     
	       	case M_UNIXTIME_RECVD:     
			unixtime = msg->recvd;
			break;
		case M_FULLDATE_STAMP:
		case M_ISODATE_STAMP: 
		case M_WEEKDAY_STAMP: 
		case M_DATE_STAMP:    
		case M_YEAR_STAMP:    
		case M_MONTH_STAMP:   
		case M_DAY_STAMP:     
		case M_HOUR_STAMP:    
		case M_MIN_STAMP:     
		case M_SEC_STAMP:     
		case M_UNIXTIME_STAMP:     
			unixtime = msg->stamp;
	                break;
		default:
 		        if (cfg->use_time_recvd)
				unixtime = msg->recvd;
 		        else
				unixtime = msg->stamp;
 		        break;
		}
		
		tm = localtime(&unixtime);

		switch (id) {
		case M_WEEKDAY:
 	        case M_WEEKDAY_RECVD:
 	        case M_WEEKDAY_STAMP:
	                length = strftime(*dest, *left - 1, "%a", tm);			
			break;
		case M_YEAR:
		case M_YEAR_RECVD:
 	        case M_YEAR_STAMP:
			length = snprintf(*dest, *left, "%04d", tm->tm_year + 1900);
			break;
		case M_MONTH:
 	        case M_MONTH_RECVD:
 	        case M_MONTH_STAMP:
			length = snprintf(*dest, *left, "%02d", tm->tm_mon + 1);
			break;
		case M_DAY:
 		case M_DAY_RECVD:
 	        case M_DAY_STAMP:  
			length = snprintf(*dest, *left, "%02d", tm->tm_mday);
			break;
		case M_HOUR:
		case M_HOUR_RECVD:
 	        case M_HOUR_STAMP:
			length = snprintf(*dest, *left, "%02d", tm->tm_hour);
			break;
		case M_MIN:
 		case M_MIN_RECVD:
		case M_MIN_STAMP:
			length = snprintf(*dest, *left, "%02d", tm->tm_min);
			break;
		case M_SEC:
		case M_SEC_RECVD:
		case M_SEC_STAMP:
			length = snprintf(*dest, *left, "%02d", tm->tm_sec);
			break;
		case M_ISODATE:
 		case M_ISODATE_RECVD:
		case M_ISODATE_STAMP:
	                length = strftime(*dest, *left - 1, "%Y-%m-%dT%H:%M:%S%z", tm);
	                break;
	        case M_FULLDATE:
 	        case M_FULLDATE_RECVD:
 	        case M_FULLDATE_STAMP:
	                length = strftime(*dest, *left - 1, "%Y %h %e %H:%M:%S", tm);
	        	break;
	        case M_DATE:
 	        case M_DATE_RECVD:
 	        case M_DATE_STAMP:
	                length = strftime(*dest, *left - 1, "%h %e %H:%M:%S", tm);
	                break;
	        case M_UNIXTIME:
 	        case M_UNIXTIME_RECVD:
 	        case M_UNIXTIME_STAMP:
			length = snprintf(*dest, *left, "%ld", (long) unixtime);
	                break;
		}
		break;
	}
	case M_MESSAGE: {
		/* message */
		length = append_string(dest, left, msg->msg->data, msg->msg->length, escape);
		break;
	}
	default:
		break;
	}
	*left -= length;
	*dest += length;
}

struct ol_string *
expand_macros(struct syslog_config *cfg, struct ol_string *template, int template_escape, struct log_info *msg)
{
	static struct macro_def macros[] = {
		{ "FACILITY", M_FACILITY },
		{ "PRIORITY", M_LEVEL },
		{ "LEVEL", M_LEVEL },
		{ "TAG", M_TAG },

		{ "DATE", M_DATE },
		{ "FULLDATE", M_FULLDATE },
		{ "ISODATE", M_ISODATE },
		{ "YEAR", M_YEAR },
		{ "MONTH", M_MONTH },
		{ "DAY", M_DAY },
		{ "HOUR", M_HOUR },
		{ "MIN", M_MIN },
		{ "SEC", M_SEC },
		{ "WEEKDAY", M_WEEKDAY },
		{ "UNIXTIME", M_UNIXTIME },

 		{ "R_DATE", M_DATE_RECVD },
 		{ "R_FULLDATE", M_FULLDATE_RECVD },
 		{ "R_ISODATE", M_ISODATE_RECVD },
 		{ "R_YEAR", M_YEAR_RECVD },
 		{ "R_MONTH", M_MONTH_RECVD },
 		{ "R_DAY", M_DAY_RECVD },
 		{ "R_HOUR", M_HOUR_RECVD },
 		{ "R_MIN", M_MIN_RECVD },
 		{ "R_SEC", M_SEC_RECVD },
 		{ "R_WEEKDAY", M_WEEKDAY_RECVD },
		{ "R_UNIXTIME", M_UNIXTIME_RECVD },
 
 		{ "S_DATE", M_DATE_STAMP },
 		{ "S_FULLDATE", M_FULLDATE_STAMP },
 		{ "S_ISODATE", M_ISODATE_STAMP },
 		{ "S_YEAR", M_YEAR_STAMP },
 		{ "S_MONTH", M_MONTH_STAMP },
 		{ "S_DAY", M_DAY_STAMP },
 		{ "S_HOUR", M_HOUR_STAMP },
 		{ "S_MIN", M_MIN_STAMP },
 		{ "S_SEC", M_SEC_STAMP },
 		{ "S_WEEKDAY", M_WEEKDAY_STAMP },
 		{ "S_UNIXTIME", M_UNIXTIME_STAMP },
		
		{ "HOST_FROM", M_HOST_FROM },
		{ "FULLHOST_FROM", M_FULLHOST_FROM },
		{ "HOST", M_HOST },
		{ "FULLHOST", M_FULLHOST },

		{ "PROGRAM", M_PROGRAM },
		{ "MSG", M_MESSAGE },
		{ "MESSAGE", M_MESSAGE },
		{ "SOURCEIP", M_SOURCE_IP }
	};
	char format[cfg->log_msg_size + 1], *format_ptr = format;
	int left = sizeof(format);
	int i, j;

	i = 0;
	while (left && (i < template->length)) {

		if (template->data[i] == '$') {
			/* beginning of a macro */
			for (j = 0; j < (sizeof(macros) / sizeof(struct macro_def)); j++) {
				if (strncmp(macros[j].name, &template->data[i + 1], strlen(macros[j].name)) == 0) {
					break;
				}
			}
			if (j == (sizeof(macros) / sizeof(struct macro_def))) {
				i++;
				while ((template->data[i] >= 'A' && 
					template->data[i] <= 'Z') ||
				       template->data[i] == '_') 
					i++;
			}
			else {
				i += strlen(macros[j].name) + 1;
				expand_macro(cfg, macros[j].id, template_escape, &format_ptr, &left, msg);
			}
		}
		else {
			*format_ptr = template->data[i];
			format_ptr++;
			i++;
		}
	}
	*format_ptr = 0;
	return c_format_cstring("%z", format);
}

/* FIXME: a binary search would be nicer */
static struct affile_dest_writer *
affile_find_writer(struct affile_dest *self, 
		   struct log_info *msg)
{
	struct ol_string *expanded_fname = NULL;
	struct affile_dest_writer *p;
	UINT32 res;	

	if (self->flags & AFFILE_NO_EXPAND) {
		if (!self->writers) {
			self->writers = make_affile_dest_writer(self, ol_string_use(self->template_fname));
			res = LOG_HANDLER_INIT(self->writers, self->cfg, NULL);
			if (res & ST_QUIT)
				return NULL;
		}
		return self->writers;
	}

	expanded_fname = expand_macros(self->cfg, self->template_fname, 0, msg);
	p = self->writers;
	while (p && 
	       ((p->expanded_fname->length != expanded_fname->length) || 
		(memcmp(p->expanded_fname->data, expanded_fname->data, expanded_fname->length) != 0))) {
		p = p->next;
	}
	if (p) {
		ol_string_free(expanded_fname);
		return p;
	}
	else {
	
		/* allocate new writer */
		p = make_affile_dest_writer(self, expanded_fname);
		res = LOG_HANDLER_INIT(p, self->cfg, NULL);
		if (res & ST_QUIT)
			return NULL;
		p->next = self->writers;
		if (p->next) p->next->prev = p;
		self->writers = p;
		return p;
	}
}

static int 
do_init_affile_dest(struct log_handler *c, 
		    struct syslog_config *cfg, 
		    struct persistent_config *persistent)
{
	CAST(affile_dest, self, c);
	
	if (self->sync_freq == -1)
		self->sync_freq = cfg->sync_freq;
	if (self->super.log_fifo_size == -1)
		self->super.log_fifo_size = cfg->log_fifo_size;
	if (cfg->create_dirs)
		self->flags |= AFFILE_CREATE_DIRS;
	if (self->uid == -1)
		self->uid = cfg->uid;
	if (self->gid == -1)
		self->gid = cfg->gid;
	if (self->perm == -1)
		self->perm = cfg->perm;
	if (self->dir_uid == -1)
		self->dir_uid = cfg->dir_uid;
	if (self->dir_gid == -1)
		self->dir_gid = cfg->dir_gid;
	if (self->dir_perm == -1)
		self->dir_perm = cfg->dir_perm;
	
	self->cfg = cfg;
	return ST_OK | ST_GOON;
}

static void do_handle_affile_log(struct log_handler *c, struct log_info *msg)
{
	CAST(affile_dest, self, c);
	struct affile_dest_writer *w;

	w = affile_find_writer(self, msg);
	if (w) {
		HANDLE_LOG(&w->super, msg);
	}
	else {
		log_info_free(msg);
	}
}

struct log_dest_driver *make_affile_dest(const char *name, int flags)
{
	NEW(affile_dest, self);
	
	self->super.super.init = do_init_affile_dest;
	self->super.super.handler = do_handle_affile_log;
	self->super.log_fifo_size = -1;
	self->uid = self->gid = -1;
	self->perm = -1;
	self->dir_uid = self->dir_gid = -1;
	self->dir_perm = -1;
	self->remove_if_older = -1;
	self->template_fname = c_format_cstring("%z", name);
	self->template_escape = 1;
	if (strchr(name, '$') == NULL)
		flags |= AFFILE_NO_EXPAND;
	self->flags = flags;
	self->sync_freq = -1; /* use global setting */
	return &self->super;
}
