/* 
   Copyright (c) 1995 by Cisco systems, Inc.
   All rights reserved.

   Please NOTE:  None of the TACACS code available here comes with any
   warranty or support.
*/

#include "tac_plus.h"
#include "expire.h"

/* Choose an authentication function. Return 0 if chosen, 1 if we need
   a username, -1 on failure
*/

int
choose_authen(data, type)
struct authen_data *data;
struct authen_type *type;
{
    char *cfg_passwd = NULL;
    char *name;

    switch (data->action) {
    case TAC_PLUS_AUTHEN_SENDPASS:
	switch (type->authen_type) {
	case TAC_PLUS_AUTHEN_TYPE_CHAP:
	case TAC_PLUS_AUTHEN_TYPE_PAP:
	case TAC_PLUS_AUTHEN_TYPE_ARAP:
	    /* Do we have a username? */
	    if (!(char) data->NAS_id->username[0]) {
		/* get user name */
		return (CHOOSE_GETUSER);
	    }
	    type->authen_func = sndpass_fn;
	    strcpy(type->authen_name, "sndpass_fn");
	    return (CHOOSE_OK);

	case TAC_PLUS_AUTHEN_TYPE_ASCII:
	default:
	    break;
	}
	return (CHOOSE_FAILED);

    case TAC_PLUS_AUTHEN_LOGIN:
	/* Note: if service = ppp then make sure your authentication
	   function only sends getuser and/or getpass. Any prompts will
	   be ignored */


	/* If we're enabling, we don't even need a username */
       	if (data->service == TAC_PLUS_AUTHEN_SVC_ENABLE) {
	    type->authen_func = enable_fn;
	    strcpy(type->authen_name, "enable_fn");
	    return (CHOOSE_OK);
	}

	/* Everything else requires a username */
	if (!(char) data->NAS_id->username[0]) {
	    /* get user name */
	    return (CHOOSE_GETUSER);
	}
       
	name = data->NAS_id->username;

	/* We have a username. Now we can choose an authentication
	   function based on whatever criteria we want. For now, this
	   is password "type" */

	cfg_passwd = cfg_get_login_secret(name, TAC_PLUS_RECURSE);

	/* If there is no login password for this user, see if there is 
	   a global password for her that we can use */

	if (!cfg_passwd) {
	    cfg_passwd = cfg_get_global_secret(name, TAC_PLUS_RECURSE);
	}

	if (!cfg_passwd) {
	    /* No password. Is a default password file set? */
	    char *file = cfg_get_authen_default();
	    if (file) {
		type->authen_func = default_fn;
		strcpy(type->authen_name, "default_fn");
		return (CHOOSE_OK);
	    }
	}

	if (!cfg_passwd) {
	    
	    /* No password or default. Bad news. Try the default
	       function anyway, so that the user sees a username &
	       password prompt and cannot tell whether the user really
	       exists or not */

	    report(LOG_ERR, 
		   "%s: choose_authen: No password type for %s",
		   session.peer, 
		   name ? name : "<unknown>");

	    type->authen_func = default_fn;
	    strcpy(type->authen_name, "default_fn");
	    return (CHOOSE_OK);
	}

	/* We found a configured password. What kind is it? */

	if (tac_find_substring("cleartext ", cfg_passwd) ||
	    tac_find_substring("file ", cfg_passwd) ||
	    tac_find_substring("des ", cfg_passwd)) {

	    /* A "fixed" password */
	    if (debug & DEBUG_PASSWD_FLAG)
		report(LOG_DEBUG, "pw is %s", cfg_passwd);

	    type->authen_func = default_fn;
	    strcpy(type->authen_name, "default_fn");
	    return (CHOOSE_OK);
	}	


	/* An skey password */
	if (STREQ(cfg_passwd, "skey")) {
	    if (debug & DEBUG_PASSWD_FLAG)
		report(LOG_DEBUG, "pw is skey");

#ifdef SKEY

	    type->authen_func = skey_fn;
	    strcpy(type->authen_name, "skey_fn");
	    return (CHOOSE_OK);

#else /* SKEY */

	    report(LOG_ERR, 
	   "%s: Attempting skey password %s in non skey version for %s",
		   session.peer, 
		   cfg_passwd && cfg_passwd[0] ? cfg_passwd : "<NULL>", 
		   name ? name : "<unknown>");
	    return(CHOOSE_FAILED);

#endif	/* SKEY */
	}

	/* Oops. No idea what kind of password this is. This should
	   never happen as the parser should never allow it. */
	
	report(LOG_ERR, 
	       "%s: Error cannot identify password type %s for %s",
	       session.peer, 
	       cfg_passwd && cfg_passwd[0] ? cfg_passwd : "<NULL>", 
	       name ? name : "<unknown>");

	type->authen_func = default_fn;
	strcpy(type->authen_name, "default_fn");
	return (CHOOSE_OK);

    case TAC_PLUS_AUTHEN_CHPASS:
	return(CHOOSE_FAILED);

    default:
	break;
    }
    return (CHOOSE_FAILED);
}
