#include <string.h>
#include <syslog.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <time.h>
#include <sys/param.h>
#include <utmp.h>
#include <libutil.h>
#include <tac_plus.h>
#include <tac_lib.h>

#include "fsm.h"
#include "ipcp.h"
#include "upap.h"

int taclogged_in = 0;
char tac_ipaddr[64];
static char tac_user[256], tac_port[256];

int tlogin(char *user, char *port, char *pwd, u_int32_t remote) {
    char *pkt, *p, *msg, *av;
    struct authen_reply *authen;
    struct author_reply *author;
    struct acct_reply *acct;
    ipcp_options *wo = &ipcp_wantoptions[0];
    char buf[256], task_id[256], start_time[256], timezone[256];
    size_t len;
    int ac, c;
    struct tm *tm;
    struct utmp utmp;
    extern time_t stime;

    if((port = strrchr(port, '/')) != NULL)
	++port;

    strncpy(tac_user, user, sizeof(tac_user));
    strncpy(tac_port, port, sizeof(tac_port));

    pkt = tacacs_auth_pap(tac_user, tac_port, "async", pwd);

    if(pkt == NULL) {
	syslog(LOG_WARNING, "PAP authentication failure for %s", tac_user);
	return UPAP_AUTHNAK;
    }

    authen = (struct authen_reply *)(pkt + TAC_PLUS_HDR_SIZE);

    if(authen->status != TAC_PLUS_AUTHEN_STATUS_PASS) {
	free(pkt);
	syslog(LOG_WARNING, "PAP authentication failure for %s", tac_user);
	return UPAP_AUTHNAK;
    }

    free(pkt);

    strcpy(tac_ipaddr, ip_ntoa(remote));
    sprintf(buf, "addr*%s", tac_ipaddr);

    pkt = tacacs_author(tac_user, tac_port, "async", "service=ppp", "protocol=ip", buf, NULL);

    if(pkt == NULL) {
	syslog(LOG_WARNING, "authorization failure for %s", tac_user);
	return UPAP_AUTHNAK;
    }

    author = (struct author_reply *)(pkt + TAC_PLUS_HDR_SIZE);
    len = ntohs(author->msg_len); ac = author->arg_cnt;
    p = pkt + TAC_PLUS_HDR_SIZE + TAC_AUTHOR_REPLY_FIXED_FIELDS_SIZE + ac + len;

    len = ntohs(author->data_len);
    if(len) {
	msg = tac_make_string(p, len);
	if(msg != NULL) {
	    syslog(LOG_NOTICE, "%s", msg);
	    free(msg);
	}
	p += len;
    }

    if(author->status != AUTHOR_STATUS_PASS_ADD && author->status != AUTHOR_STATUS_PASS_REPL) {
	free(pkt);
	syslog(LOG_WARNING, "authorization failure for %s", tac_user);
	return UPAP_AUTHNAK;
    }

    av = pkt + TAC_PLUS_HDR_SIZE + TAC_AUTHOR_REPLY_FIXED_FIELDS_SIZE;
    for(c = 0; c < ac; c++) {
	len = av[c];
	if(len >= sizeof(buf))
	    len = sizeof(buf) - 1;
	bcopy(p, buf, len); buf[len] = '\0';

	msg = strpbrk(buf, "=*"); if(msg != NULL) *msg++ = '\0';
	if(!strcmp(buf, "addr"))
	    strncpy(tac_ipaddr, msg, sizeof(tac_ipaddr));

	p += av[c];
    }

    free(pkt);

    sprintf(task_id, "task_id=%d", getpid());
    sprintf(start_time, "start_time=%d", stime);
    tm = localtime(&stime);
    snprintf(timezone, sizeof(timezone), "timezone=%s", tm->tm_zone);

    sprintf(buf,"addr=%s",tac_ipaddr);
    pkt = tacacs_acct(tac_user, tac_port, "async", TAC_PLUS_ACCT_FLAG_START,
             task_id, start_time, timezone, "service=ppp", buf, NULL);

    if(pkt == NULL) {
	syslog(LOG_WARNING, "accounting failure for %s", tac_user);
	return UPAP_AUTHNAK;
    }

    acct = (struct acct_reply *)(pkt + TAC_PLUS_HDR_SIZE);
    len = ntohs(acct->msg_len);
    p = pkt + TAC_PLUS_HDR_SIZE + TAC_ACCT_REPLY_FIXED_FIELDS_SIZE + len;

    len = ntohs(acct->data_len);
    if(len) {
	msg = tac_make_string(p, len);
	if(msg != NULL) {
	    syslog(LOG_NOTICE, "%s", msg);
	    free(msg);
	}
    }

    if(acct->status != TAC_PLUS_ACCT_STATUS_SUCCESS) {
	syslog(LOG_WARNING, "acct_reply->status=%d", acct->status);
	free(pkt);
	syslog(LOG_WARNING, "accounting failure for %s", tac_user);
	return UPAP_AUTHNAK;
    }

    free(pkt);

    bzero(&utmp, sizeof(utmp));
    time(&utmp.ut_time);
    strncpy(utmp.ut_name, tac_user, sizeof(utmp.ut_name));
    utmp.ut_host[0]='*';
    strncpy(utmp.ut_host+1, tac_ipaddr, sizeof(utmp.ut_host)-1);
    strncpy(utmp.ut_line, tac_port, sizeof(utmp.ut_line));
    login(&utmp);

    syslog(LOG_INFO, "user %s logged in", tac_user);
    taclogged_in = 1;
    return UPAP_AUTHACK;
}

void tlogout() {
    char buf[256], task_id[256], start_time[256], timezone[256], elapsed_time[256];
    struct tm *tm;
    char *pkt, *p, *msg;
    struct acct_reply *acct;
    size_t len;
    extern time_t stime;

    sprintf(task_id, "task_id=%d", getpid());
    sprintf(start_time, "start_time=%d", stime);
    tm = localtime(&stime);
    snprintf(timezone, sizeof(timezone), "timezone=%s", tm->tm_zone);
    sprintf(elapsed_time, "elapsed_time=%d", time((time_t *) NULL) - stime);
    sprintf(buf,"addr=%s",tac_ipaddr);
    
    pkt = tacacs_acct(tac_user, tac_port, "async", TAC_PLUS_ACCT_FLAG_STOP, task_id, start_time, timezone, "service=ppp", buf , elapsed_time, NULL);

    if(pkt != NULL) {
	acct = (struct acct_reply *)(pkt + TAC_PLUS_HDR_SIZE);
	len = ntohs(acct->msg_len);
	p = pkt + TAC_PLUS_HDR_SIZE + TAC_ACCT_REPLY_FIXED_FIELDS_SIZE + len;

	len = ntohs(acct->data_len);
	if(len) {
	    msg = tac_make_string(p, len);
	    if(msg != NULL) {
		syslog(LOG_NOTICE, "%s", msg);
		free(msg);
	    }
	}

	if(acct->status != TAC_PLUS_ACCT_STATUS_SUCCESS)
	    syslog(LOG_WARNING, "acct_reply->status=%d", acct->status);

	free(pkt);
    }

    taclogged_in = 0;
}
