#include <signal.h>
#include <stdarg.h>
#include <syslog.h>
#include <stdio.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <pwd.h>
#include <string.h>

#include "tac_plus.h"
#include "tac_lib.h"

#ifndef TMOUT
#define TMOUT 10
#endif

#ifndef CONFIG
#define CONFIG "/etc/tac_plus/tac_plus.conf"
#endif

#ifndef TRY
#define TRY 3
#endif

extern char *read_packet();

int console = 0;
int debug = 0;
int parse_only = 0;
int sendauth_only = 0;
int single = 0;
struct session session;

static int inited = 0;

static int authen_type = TAC_PLUS_AUTHEN_TYPE_ASCII;
static int authen_service = TAC_PLUS_AUTHEN_SVC_LOGIN;

static void catchup (s) {
	syslog(LOG_WARNING, "tacacs+ server not responding");
	session.aborted = 1;
    return;
}

static char *getloginname(char *buf, size_t len) {
    int ch;
    char *p;

    for (p = buf; (ch = getchar()) != '\n'; ) {
		if (ch == EOF) {
			if (ferror(stdin))
				perror(NULL);
			clearerr(stdin);
			return NULL;
		}

	if (p < buf + (len - 1))
	    *p++ = ch;
    }
	
    *p = '\0';
    return buf;
}

/*
 *  IP  (host) tacacs+    (key)  
 *  .
 */

void tacacs_server(char *host, char *key) {
    if (!inited) {
		bzero(&session, sizeof(session));
		inited++;
    }

    if (session.peer) {
		free(session.peer);
		session.peer = NULL;
    }
	
    if (session.key) {
		free(session.key);
		session.key = NULL;
    }

    session.peer = tac_strdup(host);
    session.key = tac_strdup(key);
}

/*
 *      .  1  
 * .    0.
 */

int tacacs_config() {
    FILE *cf;
    char buf[256];
    char peer[256];
    char key[256];
    char *s, *v;

    if ((cf = fopen(CONFIG, "r")) == NULL) {
		syslog(LOG_ERR, "%s: %m", CONFIG);
		return 0;
    }

    while (fgets(buf, sizeof(buf), cf)) {
		s = strtok(buf, " \t\n");
		v = strtok(NULL, " \t\n");
		if (!strcmp(s, "server")) {
			strncpy(peer, v, sizeof(peer));
			continue;
		}
		if (!strcmp(s, "key")) {
			strncpy(key, v, sizeof(key));
			continue;
		}
    }

    fclose(cf);

    tacacs_server(peer, key);
    return 1;
}

/*
 *    tacacs+ .  1  
 * .     0.
 */

static int tacacs_connect_i() {
    int f = -1;
    struct sockaddr_in s;
    void (*oldalrm)();
    int oldtimeout;

    session.aborted = 0;
    if (session.peer == NULL || session.key == NULL)
		return 0;
    
    if ((f = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		syslog(LOG_ERR, "%s: %m", "socket");
		return 0;
    }
    s.sin_addr.s_addr = htonl(INADDR_ANY);
    s.sin_len = sizeof(struct sockaddr_in);
    s.sin_family = AF_INET;
    s.sin_port = 0;

    if (bind(f, (struct sockaddr *)&s, sizeof(s)) < 0) {
		syslog(LOG_ERR, "%s: %m", "bind");
		close(f);
		return 0;
    }
    if (!inet_aton(session.peer, &s.sin_addr.s_addr)) {
		syslog(LOG_ERR, "inet_aton failed: %s", session.peer);
		close(f);
		return 0;
    }

    s.sin_len = sizeof(struct sockaddr_in);
    s.sin_family = AF_INET;
    s.sin_port = htons(49);

	oldalrm = signal(SIGALRM, catchup);
	oldtimeout = alarm(TMOUT);
	if (connect(f, (struct sockaddr *)&s, sizeof(s)) < 0) {
		syslog(LOG_ERR, "%s: %m", "connect");
		session.aborted = 1;
	}
	else {
		session.sock = f;
		session.aborted = 0;
	}
	signal(SIGALRM, oldalrm);
	alarm(oldtimeout);

    if (session.aborted) {
		close(f);
		return !session.aborted;
	}

    srandom(time(NULL));
    session.session_id = htonl(random());
    session.seq_no = 0;
    session.last_exch = time(NULL);
    return !session.aborted;
}

int tacacs_connect() {
	int i;

	for (i = 1; i <= TRY; i++)
		if (tacacs_connect_i())
			return 1;
		else
			syslog(LOG_ERR, "try %d, connect to TACACS+ server failed", i);
	return 0;
}

/*
 *  tacacs+  START  ASCII .  1
 *     0,   .
 */

int send_auth_start(char *user, char *port, char *rem) {
    u_char buf[256];
    size_t len;
    HDR *hdr = (HDR *)buf;
    struct authen_start *ask = (struct authen_start *)(buf + TAC_PLUS_HDR_SIZE);
    u_char *p = buf + TAC_PLUS_HDR_SIZE + TAC_AUTHEN_START_FIXED_FIELDS_SIZE;
    size_t ul, pl, rl;

    ul = strlen(user); pl = strlen(port); rl = strlen(rem);
    len = ul + pl + rl + TAC_AUTHEN_START_FIXED_FIELDS_SIZE;
    if (len + TAC_PLUS_HDR_SIZE >= sizeof(buf))
		return 0;

    bzero(buf, sizeof(buf));

    hdr->version = TAC_PLUS_VER_0;
    hdr->type = TAC_PLUS_AUTHEN;
    hdr->seq_no = ++session.seq_no;
    hdr->encryption = TAC_PLUS_CLEAR;
    hdr->session_id = session.session_id;
    hdr->datalength = htonl(len);

    ask->action = TAC_PLUS_AUTHEN_LOGIN;
    ask->priv_lvl = TAC_PLUS_PRIV_LVL_MIN;
    ask->authen_type = TAC_PLUS_AUTHEN_TYPE_ASCII;
    authen_type = ask->authen_type;
    ask->service = TAC_PLUS_AUTHEN_SVC_LOGIN;
    authen_service = ask->service;
    ask->user_len = ul;
    ask->port_len = pl;
    ask->rem_addr_len = rl;

    strcpy(p, user);
    p += ul;
    strcpy(p, port);
    p += pl;
    strcpy(p, rem);

    if (write_packet(buf))
		return 0;

    return 1;
}

/*
 *  tacacs+  CONTINUE  .  1 
 *    0,   .
 */

int send_auth_cont(char *arg) {
    u_char buf[256];
    size_t len;
    HDR *hdr = (HDR *)buf;
    struct authen_cont *ask = (struct authen_cont *)(buf + TAC_PLUS_HDR_SIZE);
    u_char *p = buf + TAC_PLUS_HDR_SIZE + TAC_AUTHEN_CONT_FIXED_FIELDS_SIZE;

    bzero(buf, sizeof(buf));

    strncpy(p, arg, sizeof(buf) - (TAC_PLUS_HDR_SIZE + TAC_AUTHEN_CONT_FIXED_FIELDS_SIZE + 1));
    len = strlen(p);

    hdr->version = TAC_PLUS_VER_0;
    hdr->type = TAC_PLUS_AUTHEN;
    hdr->seq_no = ++session.seq_no;
    hdr->encryption = TAC_PLUS_CLEAR;
    hdr->session_id = session.session_id;
    hdr->datalength = htonl(len + TAC_AUTHEN_CONT_FIXED_FIELDS_SIZE);

    ask->user_msg_len = htons(len);

    if (write_packet(buf))
		return 0;

    return 1;
}

/*
 *  ASCII    (user),   (port),
 *   (rem)   (pwd).  NULL,   
 *    tacacs+ ,    struct authen_reply.
 * ,   authen_reply,     free().
 */

char *tacacs_auth_ascii(char *user, char *port, char *rem, char *pwd) {
    char *pkt;
    struct authen_reply *rep;

    if (!tacacs_connect())
		return NULL;

    if (!send_auth_start(user, port, rem)) {
		close(session.sock);
		return NULL;
    }

    if (!(pkt = read_packet())) {
		close(session.sock);
		return NULL;
    }

    rep = (struct authen_reply *)(pkt + TAC_PLUS_HDR_SIZE);
    if (rep->status == TAC_PLUS_AUTHEN_STATUS_PASS) {
		close(session.sock);
		return pkt;
    }
    if (rep->status != TAC_PLUS_AUTHEN_STATUS_GETPASS) {
		close(session.sock);
		return pkt;
    }
    free(pkt);

    if (!send_auth_cont(pwd)) {
		close(session.sock);
		return NULL;
    }

    if (!(pkt = read_packet())) {
		close(session.sock);
		return NULL;
    }
    
    close(session.sock);
    return pkt;
}

/*
 *  PAP    (user),   (port),
 *   (rem)   (pwd).  NULL,   
 *    tacacs+ ,    struct authen_reply.
 * ,   authen_reply,     free().
 */

char *tacacs_auth_pap(char *user, char *port, char *rem, char *pwd) {
    size_t len, ul, pl, rl, dl;
    u_char buf[256];
    HDR *hdr = (HDR *)buf;
    struct authen_start *ask = (struct authen_start *)(buf + TAC_PLUS_HDR_SIZE);
    u_char *p = buf + TAC_PLUS_HDR_SIZE + TAC_AUTHEN_START_FIXED_FIELDS_SIZE;
    char *pkt;

    if (!tacacs_connect())
		return NULL;

    ul = strlen(user); pl = strlen(port); rl = strlen(rem); dl = strlen(pwd);
    len = ul + pl + rl + dl + TAC_AUTHEN_START_FIXED_FIELDS_SIZE;
    if (len + TAC_PLUS_HDR_SIZE >= sizeof(buf)) {
		close(session.sock);
		return NULL;
    }

    bzero(buf, sizeof(buf));

    hdr->version = TAC_PLUS_VER_1;
    hdr->type = TAC_PLUS_AUTHEN;
    hdr->seq_no = ++session.seq_no;
    hdr->encryption = TAC_PLUS_CLEAR;
    hdr->session_id = session.session_id;
    hdr->datalength = htonl(len);

    ask->action = TAC_PLUS_AUTHEN_LOGIN;
    ask->priv_lvl = TAC_PLUS_PRIV_LVL_MIN;
    ask->authen_type = TAC_PLUS_AUTHEN_TYPE_PAP;
    authen_type = ask->authen_type;
    ask->service = TAC_PLUS_AUTHEN_SVC_PPP;
    authen_service = ask->service;
    ask->user_len = ul;
    ask->port_len = pl;
    ask->rem_addr_len = rl;
    ask->data_len = dl;

    strcpy(p, user); p += ul;
    strcpy(p, port); p += pl;
    strcpy(p, rem); p += rl;
    strcpy(p, pwd);

    if (write_packet(buf)) {
		close(session.sock);
		return NULL;
    }

    if (!(pkt = read_packet())) {
		close(session.sock);
		return NULL;
    }

    close(session.sock);
    return pkt;
}

/*
 *  accounting    user   port c 
 *  rem.  action   .  
 *  (    (char *))  accounting attribut-value
 * pairs.     NULL.  NULL, 
 *      tacacs+ ,   
 * struct acct_reply. ,   acct_reply,  
 *   free().
 */

char *tacacs_acct(char *user, char *port, char *rem, int action, ...) {
    va_list ap;
    int ac, c;
    char *p;
    size_t len = 0, ul, pl, rl;
    char buf[4096];
    HDR *hdr = (HDR *)buf;
    struct acct *req = (struct acct *)(buf + TAC_PLUS_HDR_SIZE);
    u_char *av;
    char *pkt;

    if (!tacacs_connect())
		return NULL;

    va_start(ap, action);
    for (ac=0; ; ac++) {
		p = va_arg(ap, char *);
		if (p == NULL)
			break;
		len += strlen(p);
    }
    va_end(ap);

    ul = strlen(user); pl = strlen(port); rl = strlen(rem);
    len += ac + ul + pl + rl + TAC_ACCT_REQ_FIXED_FIELDS_SIZE;
    if (len + TAC_PLUS_HDR_SIZE >= sizeof(buf)) {
		close(session.sock);
		return NULL;
    }

    bzero(buf, sizeof(buf));

    hdr->version = TAC_PLUS_VER_0;
    hdr->type = TAC_PLUS_ACCT;
    hdr->seq_no = ++session.seq_no;
    hdr->encryption = TAC_PLUS_CLEAR;
    hdr->session_id = session.session_id;
    hdr->datalength = htonl(len);

    req->flags = action;
    req->authen_method = AUTHEN_METH_TACACSPLUS;
    req->priv_lvl = TAC_PLUS_PRIV_LVL_MIN;
    req->authen_type = authen_type;
    req->authen_service = authen_service;
    req->user_len = ul;
    req->port_len = pl;
    req->rem_addr_len = rl;
    req->arg_cnt = ac;

    av = (char *)(buf + TAC_PLUS_HDR_SIZE + TAC_ACCT_REQ_FIXED_FIELDS_SIZE);
    p = av + ac;
   
    strcpy(p, user); p += ul;
    strcpy(p, port); p += pl;
    strcpy(p, rem); p += rl;
    
    va_start(ap, action);
    for (c = 0; c < ac; c++) {
		strcpy(p, va_arg(ap, char *));
		p += av[c] = strlen(p);
    }
    va_end(ap);

    if (write_packet(buf)) {
		close(session.sock);
		return NULL;
    }

    if (!(pkt = read_packet())) {
		close(session.sock);
		return NULL;
    }

    close(session.sock);
    return pkt;
}

/*
 *     user   port c 
 *  rem.    rem  (   
 * (char *))  authorization attribut-value pairs.  
 *   NULL.  NULL,    
 *   tacacs+ ,    struct author_reply.
 * ,   author_reply,     free().
 */

char *tacacs_author(char *user, char *port, char *rem, ...) {
    va_list ap;
    int ac, c;
    char *p;
    size_t len = 0, ul, pl, rl;
    char buf[4096];
    HDR *hdr = (HDR *)buf;
    struct author *req = (struct author *)(buf + TAC_PLUS_HDR_SIZE);
    u_char *av;
    char *pkt;

    if (!tacacs_connect())
		return NULL;

    va_start(ap, rem);
    for (ac=0; ; ac++) {
		p = va_arg(ap, char *);
		if (p == NULL)
			break;
		len += strlen(p);
    }
    va_end(ap);

    ul = strlen(user); pl = strlen(port); rl = strlen(rem);
    len += ac + ul + pl + rl + TAC_AUTHOR_REQ_FIXED_FIELDS_SIZE;
    if (len + TAC_PLUS_HDR_SIZE >= sizeof(buf)) {
		close(session.sock);
		return NULL;
    }

    bzero(buf, sizeof(buf));

    hdr->version = TAC_PLUS_VER_0;
    hdr->type = TAC_PLUS_AUTHOR;
    hdr->seq_no = ++session.seq_no;
    hdr->encryption = TAC_PLUS_CLEAR;
    hdr->session_id = session.session_id;
    hdr->datalength = htonl(len);

    req->authen_method = AUTHEN_METH_TACACSPLUS;
    req->priv_lvl = TAC_PLUS_PRIV_LVL_MIN;
    req->authen_type = authen_type;
    req->service = authen_service;
    req->user_len = ul;
    req->port_len = pl;
    req->rem_addr_len = rl;
    req->arg_cnt = ac;

    av = (char *)(buf + TAC_PLUS_HDR_SIZE + TAC_AUTHOR_REQ_FIXED_FIELDS_SIZE);
    p = av + ac;
   
    strcpy(p, user); p += ul;
    strcpy(p, port); p += pl;
    strcpy(p, rem); p += rl;
    
    va_start(ap, rem);
    for (c = 0; c < ac; c++) {
		strcpy(p, va_arg(ap, char *));
		p += av[c] = strlen(p);
    }
    va_end(ap);

    if (write_packet(buf)) {
		close(session.sock);
		return NULL;
    }

    if (!(pkt = read_packet())) {
		close(session.sock);
		return NULL;
    }

    close(session.sock);
    return pkt;
}

/*
 *   login   user a  port,
 *    rem.   ,  1.  
 * ,  0.
 */

int tacacs_auth_login(char *user, char *port, char *rem) {
    char *pkt, *p;
    struct authen_reply *rep;
    size_t len;
    char buf[256];

    if (!tacacs_connect()) {
		fprintf(stderr, "\n%% Authenticate failed.\n");
		return 0;
    }

    if (!send_auth_start(user, port, rem)) {
		fprintf(stderr, "\n%% Authenticate failed.\n");
		close(session.sock);
		return 0;
    }

    for (;;) {
        if (!(pkt = read_packet())) {
			fprintf(stderr, "\n%% Authenticate failed.\n");
			close(session.sock);
			return 0;
        }

        rep = (struct authen_reply *)(pkt + TAC_PLUS_HDR_SIZE);
	
	len = ntohs(rep->msg_len);
	if (len) {
	    p = pkt + TAC_PLUS_HDR_SIZE + TAC_AUTHEN_REPLY_FIXED_FIELDS_SIZE;
	    write(STDOUT_FILENO, p, len);
	}

	if (rep->status == TAC_PLUS_AUTHEN_STATUS_GETDATA ||
		rep->status == TAC_PLUS_AUTHEN_STATUS_GETUSER ||
		rep->status == TAC_PLUS_AUTHEN_STATUS_GETPASS) {

	    if (rep->flags == TAC_PLUS_AUTHEN_FLAG_NOECHO)
			p = getpass("");
		else
			p = getloginname(buf, sizeof(buf));

	    free(pkt);
	    if (p == NULL) {
	        fprintf(stderr, "\n%% Authenticate failed.\n");
	        close(session.sock);
	        return 0;
		}
		
		if (!send_auth_cont(p)) {
			bzero(p, strlen(p));
	        fprintf(stderr, "\n%% Authenticate failed.\n");
	        close(session.sock);
	        return 0;
		}
		bzero(p, strlen(p));

	    continue;
	}

	printf("\n");

	if (rep->status == TAC_PLUS_AUTHEN_STATUS_PASS) {
	    free(pkt);
	    close(session.sock);
	    return 1;
	}

	if (rep->status == TAC_PLUS_AUTHEN_STATUS_FAIL) {
	    free(pkt);
	    close(session.sock);
	    return 0;
	}

	free(pkt);
	close(session.sock);
	return 0;
    }
}
