#ifndef lint
static char rcsid[] = "$Id: config.parse.c,v 1.2 92/11/03 02:43:33 genek Exp $";
#endif

/*
 * config.parse.c
 *
 *	read in the preen.config file
 *
 * Gene Kim
 * Purdue University
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STDLIBH
#include <stdlib.h>
#include <unistd.h>
#endif
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef DIRENT
# include <dirent.h>
#else
# ifndef XENIX
#  include <sys/dir.h>
# else		/* XENIX */
#  include <sys/ndir.h>
# endif		/* XENIX */
#endif	/* DIRENT */
#if (defined(SYSV) && (SYSV < 3))
# include <limits.h>
#endif	/* SVR2 */
#include <ctype.h>
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
#endif
#include "../include/list.h"
#include "../include/tripwire.h"

#if defined(SYSV) && (SYSV < 4)
#ifndef HAVE_LSTAT
#  define lstat(x,y) stat(x,y)
#endif
#endif		/* SYSV */

#if !defined(major)
#define major(x)        (((unsigned)(x)>>16)&0xffff)
#endif
#if !defined(minor)
#define minor(x)        ((x)&0xffff)
#endif

/* prototypes */
char *mktemp();
static void configfile_descend();

#ifndef L_tmpnam
# define L_tmpnam (unsigned int) MAXPATHLEN
#endif

/* global */
/*		we keep track of all the entry headers */
static struct list *prune_list = (struct list *) NULL;

/*
 * configfile_read(struct list **pp_list, struct list **pp_entry_list)
 *
 *	open the configuration file, and pulls out the {file/dir,ignore-flag}
 *	pairs.
 *
 *	(**pp_list) is pointer the head of the file list, where all the
 *	files are added to.
 */

void
configfile_read(pp_list, pp_entry_list)
    struct list **pp_list;
    struct list **pp_entry_list;
{
    FILE 	*fpin, *fpout;
    char	filename[MAXPATHLEN];
    char	ignorestring[1024];
    char	s[MAXPATHLEN+1024];
    char	configfile[MAXPATHLEN];
    char	*tmpfilename;
    char	number[20];
    int		entrynum = 0;
    int		err;

    /* don't print banner if we're in print-preprocessor mode */
    if (!printpreprocess)
	fputs("### Phase 1:   Reading configuration file\n", stderr);

    /* generate temporary file name */
    if ((tmpfilename = (char *) malloc(L_tmpnam)) == NULL) {
	perror("configfile_read: malloc()");
	exit(1);
    };
    (void) strcpy(tmpfilename, TEMPFILE_TEMPLATE);

    if ((char *) mktemp(tmpfilename) == NULL) {
	perror("database_build: mktemp()");
	exit(1);
    }

    /* generate configuration file name */
    if (specified_configfile == NULL)
	sprintf(configfile, "%s/%s", config_path, config_file);
    else
	(void) strcpy(configfile, specified_configfile);

    /* open the files */
    /*		check to see if input is just stdin */
    if (*configfile == '-' && !configfile[1]) {  /* configfile == "-" */
	fpin = stdin;
    }
    else if ((fpin = fopen(configfile, "r")) == NULL) {
	perror("configfile_read: fopen()");
	exit(1);
    }

    err = umask(077);  /* to protect the tempfile */

    if ((fpout = fopen(tmpfilename, "w+")) == NULL) {
	perror("configfile_read: fopen()");
	exit(1);
    }
    (void) umask(err);  /* return it to its former state */

    /* The following unlink accomplishes two things:
     *  1) if the program terminates, we won't leave a temp
     *     file sitting around with potentially sensitive names
     *     in it.
     *  2) the file is "hidden" while we run
     */
    if (unlink(tmpfilename) < 0) {
      	perror("configfile_read: unlink()");
	exit(1);
    }


    /*
     * pass 0: preprocess file
     *		call the yacc hook, located in y.tab.c
     */

    tw_macro_parse(configfile, fpin, fpout, (struct list **) pp_entry_list);

    if (fpin != stdin)
	(void) fclose(fpin);
    if (fflush(fpout) == EOF) {
        fputs("configfile_read: unknown error on fflush(fpout)\n", stderr);
	exit(1);
    }
    else
        rewind(fpout);

    fpin = fpout;

    /* do we just print out the file, and then exit? */
    if (printpreprocess) {
	int t;

	while ((t = getc(fpin)) != EOF)
	  putc((char) t, stdout);
	exit(0);	
    }

    /* pass 1: get all of the prune entries '!' */
    while (fgets(s, sizeof(s), fpin) != NULL) {

	int prune_mode;

	/* read in database entry */
	if ((err = sscanf(s, "%s %s", filename, ignorestring)) == 1) {
	    (void) strcpy(ignorestring, defaultignore);
	}
	else if (err != 2) {
	    fprintf(stderr, "'%s'\n", s);
	    fputs("configfile_read: parse error\n", stderr);

	    exit(1);
	}

	/* check for removeflag (preceding "!" or "=") */
	switch (*filename) {
      	case '!':
	    prune_mode = PRUNE_ALL;
	    (void) strcpy(filename, filename+1);	/* adjust name */
  	    break;
        case '=':
	    prune_mode = PRUNE_ONE;
	    (void) strcpy(filename, filename+1);	/* adjust name */
	    break;
        default:
	  continue; /* nothing */
	}


	/* check for fully qualified pathname
	 */
	if (*filename != '/') {
	    fprintf(stderr,
		"config: %s is not fully qualified!  Skipping...\n" ,
			filename);
	    /* XXX -- error handling needed here */
	    continue;
	}

	/* expand any escaped octal characters in name */
	filename_escape_expand(filename);

	/* add to local prune list */
	list_set(filename, "", 0, &prune_list);

	/* set appropriate prune flag */
	list_setflag(filename, prune_mode, &prune_list);
    }

    /* rewind the file for pass 2 */
    rewind(fpin);

    /* pass 2: build file lists */

    /* it's time for another banner */
    if (!printpreprocess)
	fputs("### Phase 2:   Generating file list\n", stderr);

    while (fgets(s, sizeof(s), fpin) != NULL) {
	int	howdeep;
	int	prunedir = 0;

	/*
	 * get {filename,ignore} pair:
	 * 	if only argument, then apply default ignore-flags
	 *
	 *	note that {ignore} used in the tripwire.config file is
	 *		different than the one stored in the database file!
	 *
	 *	humans use the [N|R|L]+/-[pinugsmc3] format.  in the database,
	 *		we use the old style where any capitalized letter
	 *		means it's to be ignored.
	 */

	/* make sure to remember that the ignorestring could be a comment! */
	if ( ((err = sscanf(s, "%s %s", filename, ignorestring)) == 1) ||
			(ignorestring[0] == '#')) {
	    (void) strcpy(ignorestring, defaultignore);
	}
	else if (err != 2) {
	    fprintf(stderr, "'%s'\nconfigfile_read: parse error\n", s);

	    exit(1);
	}

	/* skip all prune entries (we've already taken care of it) */
	if (*filename == '!')
	    continue;

	/* check for leading '=', prune after one recursion */
	else if (*filename == '=') {
	    (void) strcpy(filename, filename+1);
	    prunedir++;
	}

	/* check for fully qualified pathname
	 */
	if (*filename != '/') {
	    fprintf(stderr,
		"config: %s is not fully qualified!  Skipping...\n" ,
			filename);
	    /* XXX -- error handling needed here */
	    continue;
	}

	/* expand any escaped octal characters in name */
	filename_escape_expand(filename);

	/* pass down the priority -- based on how fully-qualified the
	 * 	entry was.
	 */
	howdeep = slash_count(filename);

	/*
	 * convert configuration-file style ignore-string to our database
	 * representation.
	 */
	ignore_configvec_to_dvec(ignorestring);

	/*
	 * add the entry to list of entry headers (used for incremental
	 * database updates.
	 */
	
	sprintf(number, "%d", entrynum);
	list_set(filename, number, 0, pp_entry_list);

	configfile_descend(filename, ignorestring, howdeep, prunedir,
					pp_list, entrynum++);
    }						/* end reading file */

    /* print out the list, if we're in a debuggin mode */
    if (debuglevel > 10)
	list_print(pp_list);

    /* clean up */
    (void) fclose(fpin);

    return;
}

/*
 * configfile_descend(char *filename, char *ignorestring, int howdeep,
 *				int prunedir, struct list **pp_list,
 *				int entrynum)
 *
 *	recurses down the specified filename.  when it finally hits a real
 *	file, it is added to the list of files.
 *	
 *	if (prunedir) is set, then we quit after one recursion.
 *
 *	this routine also resolves any multiple instances of a file by
 *	using (howdeep) as a precendence level.
 *
 *	(entrynum) is the unique entry number tag from tw.config.
 */

static void
configfile_descend (filename, ignorestring, howdeep,
				prunedir, pp_list, entrynum)
    char *filename;
    char *ignorestring;
    int howdeep;
    int prunedir;
    struct list **pp_list;
    int entrynum;
{
    struct stat statbuf;
    static int	countrecurse = 0;	/* count how many levels deep we are */
    static int	majordev, minordev;
    char t[512];
    extern int  errno;

    countrecurse++;

SPDEBUG(10)
printf("---> %d: %s\n", countrecurse, filename);

    /* check to see if it's on the prune list */
    if (list_lookup(filename, &prune_list) != NULL) {

	int flag;

	/* return only if it was a '!' directive */
	if ((flag = list_getflag(filename, &prune_list)) == PRUNE_ALL) {
	    countrecurse--;
	    return;
	}
	else if (flag == PRUNE_ONE)
	    prunedir = 1;
    }

    /* get the stat structure of the (real) file */
    if (lstat(filename, &statbuf) < 0) {
	char err[MAXPATHLEN+64];
        int real_err = errno;  /* in case sprintf clobbers the value */
		
	if (debuglevel > 10) {
	    sprintf(err, "configfile_descend: lstat(%s)", filename);
	} else {
	    sprintf(err, "%s: %s", progname, filename);
	}
	errno = real_err;
	perror(err);

	/* so we just skip it */
	countrecurse--;
	return;
    }

    /*
     * record our {major,minor} device pair if this is our first time
     * recursing.  then we check if it changes.  if it does, we've crossed
     * a filesystem, and we prune our tree.
     */
    if (countrecurse == 1) {

SPDEBUG(4)
printf("configfile_descend: r=%d: %s\n", countrecurse, filename);

	majordev = major(statbuf.st_dev);
	minordev = minor(statbuf.st_dev);
    } else {
	if (major(statbuf.st_dev) != majordev ||
					minor(statbuf.st_dev) != minordev) {

SPDEBUG(4)
printf("configfile_descend: pruning '%s' n(%d,%d) != o(%d, %d)\n", filename,
			major(statbuf.st_dev), minor(statbuf.st_dev),
			majordev, minordev);

	    countrecurse--;
	    return;
	    /* prune */
	}
    }

    /*
     * if it is a directory file, then we read in the directory entries
     * and then recurse into the directory.
     *
     * remember, check to see if it's a symbolic link.  we never traverse
     * them.
     */
    if (((statbuf.st_mode & S_IFMT) == S_IFDIR)

#if !defined(SYSV) || (SYSV > 3)
	&& !((statbuf.st_mode & S_IFMT) == S_IFLNK))
#else
	)
#endif
    {
	DIR *p_dir;

#ifdef DIRENT
	struct dirent *pd;
#else
	struct direct *pd;
#endif

	char recursefile[MAXPATHLEN];

	/* handle prunedir flag */

	/*
	 * concatenate entry number to the ignore-string
	 */

	sprintf(t, "%d %s", entrynum, ignorestring);

	/*
	 * just nix it from the list?
	 */

	list_set(filename, t, howdeep, pp_list);
	(void) list_setflag(filename, FLAG_NOOPEN, pp_list);

	/* if it's a symbolic link, make sure we flag it as such! */

#if !defined(SYSV) || (SYSV > 3)
	if ((statbuf.st_mode & S_IFMT) == S_IFLNK) {
	    (void) list_setflag(filename, FLAG_SYMLINK, pp_list);
	}
#endif

	if (prunedir) {
	    countrecurse--;
	    return;
	}

SPDEBUG(4)
fprintf(stderr, "configfile_descend: %s: it's a directory!\n", filename);

	if ((p_dir = opendir(filename)) == NULL) {
	    if (debuglevel > 10) {
		perror("configfile_descend: opendir()");
	    } else {
		char err[MAXPATHLEN+64];
		int real_errno = errno;
		
		sprintf(err, "%s: %s", progname, filename);
		errno = real_errno;
		perror(err);
	    }
	    countrecurse--;
	    return;
	}


/* broken xenix compiler returns "illegal continue" ?!? */
#ifdef XENIX
#define XCONTINUE goto XENIX_CONT
#else
#define XCONTINUE continue
#endif

	for (pd = readdir(p_dir); pd != NULL; pd = readdir(p_dir)) {
	    /* we could use strcmp in the following, but this is much faster */
	    if (pd->d_name[0] == '.') {
	      if (pd->d_name[1] == 0)    /* must be . */
		XCONTINUE;
	      else if (pd->d_name[1] == '.') {
		if (pd->d_name[2] == 0)  /* must be .. */
		  XCONTINUE;
	      }
	    }
	
SPDEBUG(4)
printf("--> descend: %s\n", pd->d_name);

	    /* build pathname of file */
	    sprintf(recursefile, "%s/%s", filename, pd->d_name);

	    /* recurse.  it'll pop right back if it is just a file */
	    configfile_descend(recursefile, ignorestring, howdeep, 0,
					pp_list, entrynum);

XENIX_CONT: ;
	
	} 					/* end foreach file */

	/* cleanup */
	closedir(p_dir);
    }						/* end if dir */
    else {

	/*
	 * concatenate entry number to the ignore-string
	 */

	sprintf(t, "%d %s", entrynum, ignorestring);

	/* add to list */
	list_set(filename, t, howdeep, pp_list);

	/*
	 * if it is a special file or device, add it to the list, but
	 * make sure we don't open it and read from it!
	 */
	switch (statbuf.st_mode & S_IFMT) {
	  case S_IFIFO:
	  case S_IFCHR:
	  case S_IFBLK:
#if !defined(SYSV) || (SYSV > 3)
	  case S_IFSOCK:
#endif
	    (void) list_setflag(filename, FLAG_NOOPEN, pp_list);
	    break;
#if !defined(SYSV) || (SYSV > 3)
	  case S_IFLNK:	/* if it's a symbolic link, make sure we flag it as such! */
	    (void) list_setflag(filename, FLAG_SYMLINK, pp_list);
#endif
	    break;
	  default:
	    break;   /* do nothing for regular files */
	}
    }						/* end else dir */

    countrecurse--;
    return;
}

