%token DIRECTIVE
%token COMMENT	
%token EOL
%token STRING
%token INCLUDE
%token DEFINE
%token UNDEF
%token IFDEF
%token IFNDEF
%token IFHOST
%token IFNHOST
%token ELSE
%token ENDIF
%token CONTENTS
%token DBASEVERSION
%token LPAREN
%token RPAREN
%token ANDAND
%token OROR
%token ECHOTHIS
%token BSLASH
%token ESCCHAR

%{
/* $Id: config.pre.y,v 1.2 92/11/03 04:49:41 genek Exp $ */

/*
 * config.y
 *
 *	tw.config preprocessor parser for yacc.
 *
 *	This implementation does an unfortunately large number of 
 *	malloc()'s and free()'s to store the lexeme values.  Although
 *	memory leaks are few, too much time is spent doing memory
 *	allocation.
 *
 *	At this point, I would argue that this is not too significant,
 *	since we only run this routine once.
 *
 * Gene Kim
 * Purdue University
 * October 5, 1992
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
#endif
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "../include/list.h"
#include "../include/tripwire.h"

#define INCLUDE_STACK_SZ 	16	/* max num of nested includes */

int yaccdebuglevel = 0;

static int linenumber = 0;

static FILE *fp_stack[INCLUDE_STACK_SZ];
static int linenumber_stack[INCLUDE_STACK_SZ];
static char *filename_stack[INCLUDE_STACK_SZ];
static int stackpointer = 0;
static int found_db_version = 0;
static struct list **pp_entry_list_global = NULL;

static char *currparsefile = NULL;

/* prototypes */
static char *string_dequote();
static void include_push();
static FILE *include_pop();

%}

%union {
    char 	*string;
    int 	val;
}

%left <string> COMMENT STRING ESCCHAR 
%token <val> IFDEF IFNDEF IFHOST IFNHOST
%type <string> word words directive colines coline else 
%type <val> if_expr host_expr
%left <val> ANDAND OROR

%start lines
%%

lines	: lines line 
	| 
	;

/* we do all of the line-emitting in this production (line) */

line	: directive EOL
	    {
		/*
		linenumber++;
		*/

		if ($1)	{ 
		    fprintf(yyout, "%s\n", $1); 
		    free($1);
		}
	    }
	| words EOL
	    {
		/*
		linenumber++; 
		*/

		if ($1)	{ 
		    fprintf(yyout, "%s\n", $1); 
		    free($1);
		}
	    }
	;


colines	: colines coline 
	    {
		/* concatenate the two terminals together */
		if ($1 == NULL) {
		    $$ = (char *) malloc((unsigned) strlen($2) + 1);
		    $$[0] = '\0';
		}
		else {
		    $$ = (char *) malloc((unsigned) 
					(strlen($1) + strlen($2)) + 2);
		    (void) strcpy($$, $1);
		    (void) strcat($$, "\n");

		    /* free up the left component */
		    free($1);
		}
		(void) strcat($$, $2);

		/* free up the right component */
		if ($2)
		    free($2);
	    }
	| 
	    {
		$$ = NULL;
	    }
	;

coline	: directive EOL 		{ $$ = $1; /* linenumber++; */}
	| words EOL 			{ $$ = $1; /* linenumber++; */}
	;

else	: ELSE colines 
	    {
		$$ = $2;
	    }
	| 
	    {
		$$ = NULL;
	    }
	;

if_expr	: LPAREN if_expr RPAREN
	    { 
		$$ = $2;
	    }
	| if_expr ANDAND if_expr
	    {
		$$ = $1 && $3;
	    }
	| if_expr OROR if_expr
	    {
		$$ = $1 || $3;
	    }
	| word
	    {
		$$ = tw_mac_ifdef($1);
	    }

host_expr: LPAREN host_expr RPAREN
	    { 
		$$ = $2;
	    }
	| host_expr ANDAND host_expr
	    {
		$$ = $1 && $3;
	    }
	| host_expr OROR host_expr
	    {
		$$ = $1 || $3;
	    }
	| word
	    {
		$$ = tw_mac_ifhost($1);
	    }

directive:	
	  DEFINE word 			{ tw_mac_define($2, ""); $$ = NULL; }
	| DEFINE word word 		{ tw_mac_define($2, $3); $$ = NULL; }
	| UNDEF word 			{ tw_mac_undef($2); $$ = NULL; }
	| IFDEF if_expr
	    {
		$1 = $2;
	    }
          EOL colines else ENDIF
	    {
		if ($1) { $$ = $5; }
		else 	{ $$ = $6; }

		/*
		linenumber++;
		*/
	    }
	| IFNDEF if_expr
	    {
		$1 = !$2;
	    }
          EOL colines else ENDIF
	    {
		if ($1) { $$ = $5; }
		else 	{ $$ = $6; }

		/*
		linenumber++;
		*/
	    }
	| IFHOST host_expr
	    {
		$1 = $2;
	    }
          EOL colines else ENDIF
	    {
		if ($1) { $$ = $5; }
		else 	{ $$ = $6; }

		/*
		linenumber++;
		*/
	    }
	| IFNHOST host_expr
	    {
		$1 = !$2;
	    }
          EOL colines else ENDIF
	    {
		if ($1) { $$ = $5; }
		else 	{ $$ = $6; }

		/*
		linenumber++;
		*/
	    }
	| INCLUDE word
	    {
		/* push a new @@include file onto the include stack */
		include_push($2, &yyin);
		$$ = NULL;

	    }
	| CONTENTS word
	    {
		char *pc = "@@contents ";

		/* record contents in list */
		list_set($2, "", 0, pp_entry_list_global);

		/* reconstruct and emit the entire string */
		$$ = (char *) malloc((unsigned) (strlen($2) + strlen(pc)) + 1);
		(void) strcpy($$, pc);
		(void) strcat($$, $2);

		/* free up the right side */
		free($2);
	    }
	| ECHOTHIS words
	    {
		printf("tw.config: echo: %s\n", $2);
	    }
	| DBASEVERSION word
	    {
		int version;

		if (sscanf($2, "%d", &version) != 1) {
		    yyerror("");
		}

		/* check if the database format is too old */
		if (version != db_version_num) {
		    fprintf(stderr, 
			"tripwire: configuration file '%s' format is too old!\n",
			version);
		    exit(1);
		}

		/* free up the right side */
		free($2);

		/* we must see one of these productions in the file */
		found_db_version = 1;

		$$ = NULL;
	    }
	;

words	: words word
	    {
		/* concatenate the two terminals together */
		if ($1 == NULL) {
		    $$ = (char *) malloc((unsigned) strlen($2) + 1);
		    $$[0] = '\0';
		}
		else {
		    $$ = (char *) malloc((unsigned) 
					(strlen($1) + strlen($2)) + 2);
		    (void) strcpy($$, $1);
		    (void) strcat($$, " ");

		    /* free up the left component */
		    free($1);
		}
		(void) strcat($$, $2);

		/* free up the right component */
		if ($2) 
		    free($2);
	    }
	|
	    {
		$$ = NULL;
	    }
	;

word	: STRING
	    {
		$$ = strcpy((char *) malloc((unsigned) strlen($1) + 1), $1);
	    }
	| DIRECTIVE STRING
	    {
		char *pc;

		/* if not defined */
		if (tw_mac_ifdef($2) == 0) {
		    fprintf(stderr, "warning: uninitialized definition at line %d in '%s'!\n(Hint: maybe a misspelled directive?)\n", linenumber,
					    currparsefile);
		    $$ = NULL;
		}
		else {
		    /* else substitute in the input stream */
		    pc = tw_mac_dereference($2);
		    assert(pc != NULL);

		    $$ = (char *) malloc((unsigned) strlen(pc) + 1);
		    (void) strcpy($$, pc);
		}
	    }
	;

	
%%

#include "lex.yy.c"

/*ARGSUSED*/
yyerror(s)
    char *s;
{
     fprintf(stderr, 
	"syntax error at line %d in configuration file '%s'!  Aborting...\n", 
		++linenumber, currparsefile);
}

/*
 * void
 * tw_macro_parse(char *filename, FILE *fpin, FILE *fpout, 
 *						struct list **pp_entry_list)
 *
 *	wrapper around yyparse(), initiailzing input and output data.
 */

void
tw_macro_parse(filename, fpin, fpout, pp_entry_list)
    char *filename;
    FILE *fpin, *fpout;
    struct list **pp_entry_list;
{
    /* set up input and output pointers */
    yyin = fpin;
    yyout = fpout;

    /* set up initial filename */
    currparsefile = filename;

    pp_entry_list_global = pp_entry_list;

    (void) yyparse();
}

/* counters odd behaviour of flex -- Simon Leinen */
#ifdef yywrap
# undef yyrap
#endif

yywrap()
{
    /* check to see if we've reached the bottom of the @@include stack */
    if (include_pop()) {
	linenumber++;
	return 0;
    }

    /* check to see if we've seen a @@dbversion line */
#ifdef FOO
    if (!found_db_version) {
	fprintf(stderr, "tripwire: database file doesn't contain version number!  (old database?)\n");
	exit(1);
    }
#endif

    /* close up parser */
    return 1;
}

/*
 * static char *
 * string_dequote(char *s)
 *
 *	remove pairs of quoted strings.
 */

static char *
string_dequote(s)
    char *s;
{
    char temp[1024];

    /* do we need to do anything? */
    if (s[0] != '"') 		{ return s; }

    (void) strncpy(temp, s+1, strlen(s) - 2);
    (void) strcpy(s, temp);

    return s;

}

/* 
 * void
 * include_push(char *filename, FILE **p_fp_old)
 *
 *	return a stdio (FILE *) pointer to the opened (filename), saving 
 *	the old (FILE *) pointer and line number on the stack.
 *
 *	returns (NULL) when we pop back to the original file.
 */

static void
include_push(filename, p_fp_old)
    char *filename;
    FILE **p_fp_old;
{
    static FILE *fp;
    char *pc;
    extern int  errno;

    /* check for stack overflow */
    if (stackpointer == INCLUDE_STACK_SZ) {
	fprintf(stderr, "Too many nested includes at line %d in file '%s'!\n", 
				linenumber, currparsefile);
	exit(1);
    }

    /* dequote the include filename */
    string_dequote(filename);

    /* save the old file pointer, filename, and linenumber on the stack */
    fp_stack[stackpointer] = *p_fp_old;

    (void) strcpy((pc = (char *) malloc((unsigned) strlen(currparsefile) + 1)), 
					currparsefile);
    filename_stack[stackpointer] = pc;

    linenumber_stack[stackpointer++] = linenumber;

    /* try opening the file */
    if ((fp = fopen(filename, "r")) == NULL) {
	if (errno == ENOENT) {
	    fprintf(stderr, "tw.config: @@include '%s' file not found in '%s'!\n",
			    filename, currparsefile);
	    exit(1);
	}
	else {
	    char msg[100];
	    sprintf(msg, "%s: fopen()", filename);
	    perror(msg);
	    exit(1);
	}
    }

    /* replace old pointer with new */
    *p_fp_old = fp;

    /* reset line number and filename */
    linenumber = 0;
    currparsefile = pc;
}

/*
 * FILE *
 * include_pop()
 *
 *	pop the last file structure off the @@include stack.
 *
 *	returns NULL when we've exhausted the stack.
 */

static FILE *
include_pop()
{
    /* check for stack underflow */
    if (stackpointer-- == 0)
	return NULL;

    /* pop off the line numbers and the stdio file pointer */
    yyin = fp_stack[stackpointer];
    linenumber = linenumber_stack[stackpointer];
    currparsefile = filename_stack[stackpointer];
    free(filename_stack[stackpointer]);

    return yyin;
}

