#ifndef lint
static char rcsid[] = "$Id: main.c,v 1.3 92/11/03 02:43:52 genek Exp $";
#endif

/************************************************************************
 *
 *   All files in the distribution of Tripwire are Copyright 1992 by the
 *   Purdue Research Foundation of Purdue University.  All rights
 *   reserved.  Some individual files in this distribution may be covered
 *   by other copyrights, as noted in their embedded comments.
 *
 *   Redistribution and use in source and binary forms are permitted
 *   provided that this entire copyright notice is duplicated in all such
 *   copies, and that any documentation, announcements, and other
 *   materials related to such distribution and use acknowledge that the
 *   software was developed at Purdue University, W. Lafayette, IN by
 *   Gene Kim and Eugene Spafford.  No charge, other than an "at-cost"
 *   distribution fee, may be charged for copies, derivations, or
 *   distributions of this material without the express written consent
 *   of the copyright holder.  Neither the name of the University nor the
 *   names of the authors may be used to endorse or promote products
 *   derived from this material without specific prior written
 *   permission.  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY
 *   EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE
 *   IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR ANY PARTICULAR
 *   PURPOSE.
 *
 ************************************************************************/

/*
 * main.c
 *
 *	main routines and global variables
 *
 * Gene Kim
 * Purdue University
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STDLIBH
#include <stdlib.h>
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/param.h>
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef MALLOCH
# include <malloc.h>
#endif
#if (defined(SYSV) && (SYSV < 3))
# include <limits.h>
#endif	/* SVR2 */
#include "../include/list.h"
#include "../include/tripwire.h"
#include "../include/patchlevel.h"


/* version information */

char *version_num = VERSION_NUM;
int db_version_num = DB_VERSION_NUM;

/******* signature functions *****************************************
 *	sig_md5_get		: MD5 by RSA
 *	sig_snefru_get		: Snefru by Xerox
 *	sig_null_get		: null
 *********************************************************************/

int (*pf_signatures [NUM_SIGS]) () = {
					SIG0FUNC,
					SIG1FUNC,
					SIG2FUNC,
					SIG3FUNC,
					SIG4FUNC,
					SIG5FUNC,
					SIG6FUNC,
					SIG7FUNC,
					SIG8FUNC,
					SIG9FUNC
				      };
char *signames[NUM_SIGS] = {
					SIG0NAME,
					SIG1NAME,
					SIG2NAME,
					SIG3NAME,
					SIG4NAME,
					SIG5NAME,
					SIG6NAME,
					SIG7NAME,
					SIG8NAME,
					SIG9NAME
				      };

char *config_file = CONFIG_FILE;
char *database_file = DATABASE_FILE;

char *database_path = DATABASE_PATH;
char *config_path = CONFIG_PATH;

char tempdatabase_file[MAXPATHLEN];

char *defaultignore = DEFAULTIGNORE;
static char *defaultignore_parsed;

char *db_record_format = DB_RECORD_FORMAT;

struct list *filelist = (struct list *) NULL;

int debuglevel = 1;
int verbosity = 0;
static int dbaseinit = 0;
int printhex = 0;
static char *updateentry = NULL;
int quiet = 0;
int printpreprocess = 0;
char *specified_dbasefile = NULL;
char *specified_configfile = NULL;
int runtimeignore = 0;

char *progname;

static void
usage()
{
    fputs("usage: tripwire [ options ... ]\n", stderr);
    fputs("\tWhere `options' are:\n", stderr);
    fputs("\t\t-initialize	Database Generation mode\n", stderr);
    fputs("\t\t-init		\n", stderr);
    fputs("\t\t-update entry	update entry (a file, directory, or \n", stderr);
    fputs("\t\t		    tw.config entry) in the database\n", stderr);
    fputs("\t\t-d dbasefile	read in database from dbasefile\n", stderr);
    fputs("\t\t		    (use `-d -' to read from stdin)\n", stderr);
    fputs("\t\t-c configfile	read in config file from configfile\n", stderr);
    fputs("\t\t		    (use `-c -' to read from stdin)\n", stderr);
    fputs("\t\t-Dvar=value	define a tw.config variable (ala @@define)\n",
	stderr);
    fputs("\t\t-Uvar		undefine a tw.config variable (ala @@undef)\n",
	stderr);
    fputs("\t\t-i #|all 	ignore the specified signature (to reduce\n", stderr);
    fputs("\t\t		    execution time)\n", stderr);
    fputs("\t\t-q		quiet mode\n", stderr);
    fputs("\t\t-v		verbose mode\n", stderr);
    fputs("\t\t-preprocess	print out preprocessed configuration file\n",
	stderr);
    fputs("\t\t-E		\n", stderr);
    fputs("\t\t-version	print version and patch information\n", stderr);
    exit(1);
}

/*
 * void
 * version()
 *
 *	print out version information, with patchlevel information.
 *	currently, there is no real correlation between the two.
 */

static void
version()
{
    fprintf(stderr, "\nTripwire version %s (patchlevel %d)\n\n", version_num,
			    PATCHLEVEL);
    fprintf(stderr, "Copyright (c) 1992, Purdue Research Foundation\n");
    fprintf(stderr, "\tBy Gene Kim, Eugene Spafford\n\n");
    exit(0);
}

int
main(argc, argv)
    int argc;
    char *argv[];
{
    int i;
    char *pc;
    char database[MAXPATHLEN];
    char mask[64];
    int fd;

    progname = argv[0];

    /* iterate through arguments */
    for (i = 1; i < argc; i++) {
	pc = argv[i];
	/* verbose mode */
	if (strcmp(pc, "-v") == 0) {
	    verbosity++;
	    continue;
	}
	/* quiet mode */
	if (strcmp(pc, "-q") == 0) {
	    quiet++;
	    continue;
	}
	/* hex mode */
	if (strcmp(pc, "-x") == 0) {
	    printhex++;
	    continue;
	}
	/* database generation mode */
	if (strcmp(pc, "-initialize") == 0 || strcmp(pc, "-init") == 0 ||
					      strcmp(pc, "-initialise") == 0) {
	    dbaseinit++;
	    continue;
	}
	/* print preprocessed configuration file */
	if ((strcmp(pc, "-preprocess") == 0) || (strcmp(pc, "-E") == 0)) {
	    printpreprocess++;
	    continue;
	}
	/* update specific database entry */
	if (strcmp(pc, "-update") == 0) {
	    /* check to see that there is an argument */
	    if ((pc = argv[++i]) == NULL) {
		usage();
	    }
	    updateentry = pc;
	    continue;
	}
	/* specify database file */
	if (strcmp(pc, "-d") == 0) {
	    /* check to see that there is an argument */
	    if ((pc = argv[++i]) == NULL) {
		usage();
	    }
	    specified_dbasefile = pc;
	    continue;
	}
	/* specify configuration file */
	if (strcmp(pc, "-c") == 0) {
	    /* check to see that there is an argument */
	    if ((pc = argv[++i]) == NULL) {
		usage();
	    }
	    specified_configfile = pc;
	    continue;
	}
	/* ignore specified signatures */
	if (strcmp(pc, "-i") == 0) {
	    int tmpflag;

	    /* check to see if there is an argument */
	    if ((pc = argv[++i]) == NULL) {
		usage();
	    }
	    if (strcmp(pc, "all") == 0) {
		runtimeignore = IGNORE_0_9;
		continue;
	    }
	    if ((sscanf(pc, "%d", &tmpflag)) != 1)
		usage();
	    runtimeignore |= (IGNORE_0 << tmpflag);
	    continue;
	}
	/* print out version information */
	if (strcmp(pc, "-version") == 0) {
	    version();
	}
	/* define (-Dfoo=bar) */
	if (strncmp(pc, "-D", 2) == 0) {
	    char key[512], value[512];

	    if (!pc[2]) {
		fputs("tripwire: -D requires an argument!\n", stderr);
		exit(1);
	    }
	    (void) string_split_ch(pc+2, key, value, '=');
	    tw_mac_define(key, value);
	    continue;
	}
	/* undef (-Ufoo) */
	if (strncmp(pc, "-U", 2) == 0) {
	    if (!pc[2]) {
		fputs("tripwire: -U requires an argument!\n", stderr);
		exit(1);
	    }
	    tw_mac_undef(pc+2);
	    continue;
	}
	usage();
    }

    /* argument sanity checking */
    /*		are two files set to read from stdin? */
    if (specified_configfile != NULL && specified_dbasefile) {
	if (strcmp(specified_configfile, "-") == 0 &&
	    strcmp(specified_dbasefile, "-") == 0) {
	    fprintf(stderr, "%s: specified database and configuration file can't be both be stdin!\n", progname);
	    exit(1);
	}
    }

    /* initialize lists */
    list_init();

    /* build hostname specific names */
    filename_hostname_expand(&config_path);
    filename_hostname_expand(&config_file);
    filename_hostname_expand(&database_path);
    filename_hostname_expand(&database_file);

    /* recompute the default ignore string (old -> new format) */
    (void) strcpy(mask, defaultignore);
    ignore_configvec_to_dvec(mask);
    defaultignore_parsed = mask;

    /* are we in database generation mode? */
    if (dbaseinit) {
	char *oldpath = database_path;
	char *newpath = database_path = "./databases";
	struct list *dbase_entry_list = (struct list *) NULL;

	/* place database in ./databases */
	database_path = newpath;

	if (mkdir(newpath, 0777) >= 0) {
	    fprintf(stderr, "### Warning:\tcreating %s directory!\n",
				newpath);
	    fprintf(stderr, "###\n");
	}

	/* generate the database */
	configfile_read(&filelist, &dbase_entry_list);
	database_build(&filelist, DBASE_PERMANENT, &dbase_entry_list);
	fprintf(stderr, "###\n");
	fprintf(stderr,
"### Warning:   Database file placed in %s/%s.\n", database_path,
							database_file);
	fprintf(stderr, "###\n");
	fprintf(stderr,
"###            Make sure to move this file file and the configuration\n");
	fprintf(stderr,
"###            to secure media!\n");
	fprintf(stderr, "###\n");
	fprintf(stderr,
"###            (Tripwire expects to find it in '%s'.)\n", oldpath);
	exit(0);
    }

    /*
     * 	make sure that database file is there!
     * 		(this is meaningless if we specified stdin "-")
     */

    if (specified_dbasefile == NULL)
	sprintf(database, "%s/%s", database_path, database_file);
    else
	(void) strcpy(database, specified_dbasefile);

    if (strcmp(database, "-") != 0) {
	if ((fd = open(database, O_RDONLY)) < 0) {
	    fprintf(stderr,
		    "%s: database file '%s/%s' does not exist!  Aborting...\n",
		    progname, database_path, database_file);
	    exit(1);
	}
	(void) close(fd);
    }

    /* are we in database update mode? */
    if (updateentry) {
	preen_update(updateentry);
	exit(0);
    }

    /* we're in integrity checking mode */
    preen_gather();
    exit(0);
    /*NOTREACHED*/
}
