/* ttywatch.c -- log output from TTYs, based on robin.c from the book
 * Linux Application Development, by Michael K. Johnson and Erik W. Troan
 *
 * Copyright  2000 Michael K. Johnson <johnsonm@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/poll.h>
#include <errno.h>
#include <fcntl.h>
#include <popt.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>             /* for strerror() */
#include <termios.h>
#include <unistd.h>

#include <glib.h>

static GSList *machines;
static GHashTable *machine_names;
static GHashTable *machine_fds;

/* configuration */
static char *logpath;
static char *configfile;
static char *pidfile;
static int intarg;
static char *chararg;
static int do_daemon;

static struct poptOption optionsTable[] = {
    { "config",  'c', POPT_ARG_STRING, &configfile, 'c',
      "config file to read before parsing other options", "path" },
    { "pidfile",  'P', POPT_ARG_STRING, &pidfile, 0,
      "file in which to store the pid", "path" },
    { "name", 'n', POPT_ARG_STRING, &chararg, 'n',
      "name of machine to monitor", "machinename" },
    { "port", 'p', POPT_ARG_STRING, &chararg, 'p',
      "pathname for port for current machine", "/dev/<foo>" },
    { "bps",  'b', POPT_ARG_INT, &intarg, 'b',
      "signaling rate for current maching in bps", "<BPS>" },
    { "logpath", 'l', POPT_ARG_STRING, &logpath, 0,
      "directory path for logfiles", "path" },
    { "daemon", 'd', POPT_ARG_NONE, &do_daemon, 0,
      "directory path for logfiles", "path" },
      POPT_AUTOHELP
    { NULL, 0, 0, NULL, 0 }
};

typedef struct machine {
	char *name;
	char *devpath;
	int dev_fd;
	int log_fd;
	speed_t speed;
	struct termios ots;
	struct termios ts;
} machine;

void warn(char *warning, machine *m) {
	char * message;
	int size;

	if (!m) abort();

	message = alloca(strlen(warning) + strlen(m->name) + 15);
	size = sprintf(message, "ttywatch[%s]: %s\n", m->name, message);
	write(STDERR_FILENO, message, size);
	if (m->log_fd >= 0) write(m->log_fd, message, size);
}

/* eeeeeek! */
void die(char *message) {
	write(STDERR_FILENO, message, strlen(message));
	exit (1);
}

/* get machine* for machine name, creating a new machine* if necessary */
machine *get_machine(char *mach) {
	machine *m;

	m = g_hash_table_lookup(machine_names, mach);
	if (m) return m;

	m = calloc(sizeof(machine), 1);
	if (!m) abort();

	m->name = strdup(mach);
	m->speed = B115200;

	/* defaults to devpath = /dev/<name> for symlinks */
	m->devpath = malloc(strlen(mach)+6);
	strcpy(m->devpath, "/dev/");
	strcat(m->devpath, mach);

	g_hash_table_insert(machine_names, mach, m);
	machines = g_slist_append(machines, m);

	return m;
}

speed_t symbolic_speed(int speednum) {
   if (speednum >= 460800) return B460800;
   if (speednum >= 230400) return B230400;
   if (speednum >= 115200) return B115200;
   if (speednum >= 57600) return B57600;
   if (speednum >= 38400) return B38400;
   if (speednum >= 19200) return B19200;
   if (speednum >= 9600) return B9600;
   if (speednum >= 4800) return B4800;
   if (speednum >= 2400) return B2400;
   if (speednum >= 1800) return B1800;
   if (speednum >= 1200) return B1200;
   if (speednum >= 600) return B600;
   if (speednum >= 300) return B300;
   if (speednum >= 200) return B200;
   if (speednum >= 150) return B150;
   if (speednum >= 134) return B134;
   if (speednum >= 110) return B110;
   if (speednum >= 75) return B75;
   return B50;
}

void setup_pidfile(void) {
	if (pidfile) {
		char pid[32]; /* more than enough for an int... */
		int pidfd;

		sprintf(pid, "%d\n", getpid());
		pidfd = open(pidfile, O_RDWR|O_CREAT|O_TRUNC, 0644);
		write(pidfd, pid, strlen(pid));
		close(pidfd);
	}
}

void delete_pidfile(void) {
	if (pidfile) {
		unlink(pidfile);
	}
}

void open_one_logfile(machine *m) {
	char *logname;

	logname = alloca(strlen(logpath)+strlen(m->name)+6);
	sprintf(logname, "%s/%s.log", logpath, m->name);
	m->log_fd = open(logname, O_RDWR|O_CREAT|O_APPEND, 0666);
	if (m->log_fd < 0)
		warn(strerror(errno), m);
}

/* open and set termios settings for a single port, storing original settings */
void setup_one_termios(gpointer data, gpointer ignore) {
	machine *m = data;

	open_one_logfile(m);

	m->dev_fd = open(m->devpath, O_RDONLY|O_NOCTTY);
	if (m->dev_fd < 0) {
		warn(strerror(errno), m);
		g_hash_table_remove(machine_names, m->name);
		machines = g_slist_remove(machines, m);
		free(m->devpath);
		free(m->name);
		free(m);
		return;
	}
	g_hash_table_insert(machine_fds, &m->dev_fd, m);

	/* modify the port configuration */
	tcgetattr(m->dev_fd, &m->ots);
	m->ts = m->ots;
	/* lots of random junk... */
	m->ts.c_lflag &= ~ICANON;
	m->ts.c_lflag &= ~(ECHO | ECHOCTL | ECHONL);
	m->ts.c_cflag |= HUPCL;
	m->ts.c_cflag &= ~CRTSCTS;
	m->ts.c_iflag &= ~(IXON | IXOFF | IXANY);
	m->ts.c_cc[VMIN] = 1;
	m->ts.c_cc[VTIME] = 0;

	/* throw away ^M (\r) characters on input */
	m->ts.c_iflag |= IGNCR;

	/* set ALL the speeds */
	cfsetospeed(&m->ts, m->speed);
	cfsetispeed(&m->ts, m->speed);

	/* Now set the modified termios settings */
	tcsetattr(m->dev_fd, TCSANOW, &m->ts);
}

/* set termios settings for all ports */
void setup_termios(void) {
	g_slist_foreach(machines, setup_one_termios, NULL);
}

/* restore termios settings for a single port, then close fds */
void cleanup_one_termios(gpointer data, gpointer ignore) {
	machine *m = data;

	tcsetattr(m->dev_fd, TCSANOW, &m->ots);
	close(m->dev_fd);
	close(m->log_fd);
}

/* restore all original terminal settings on exit */
void cleanup_termios_and_exit(int signal) {
	g_slist_foreach(machines, cleanup_one_termios, NULL);
	delete_pidfile();
	exit(0);
}

void reopen_one_logfile(gpointer data, gpointer ignore) {
	machine *m = data;

	close(m->log_fd);
	open_one_logfile(m);
}

void reopen_logfiles(int signo) {
	g_slist_foreach(machines, reopen_one_logfile, NULL);
}

struct pollfd * prepare_poll(int *nfds /*OUT*/) {
	struct pollfd *p;
	GSList *l;
	machine *m;
	int i;

	*nfds = g_slist_length(machines);
	p = calloc(sizeof(struct pollfd), *nfds);

	for (l = machines, i=0; l; l = g_slist_next(l), i++) {
		m = l->data;
		p[i].fd = m->dev_fd;
		p[i].events = POLLIN;
	}

	return p;
}


machine *parse_argument (char c, machine *m) {
	int	conf_fd = 0;
	struct stat sb;

	switch (c) {
	case 'n':
		m = get_machine(chararg);
		break;
	case 'p':
		if (!m) die ("--name or -n must come before --port or -p\n");
		if (m->devpath) free(m->devpath);
		m->devpath = strdup(chararg);
		break;
	case 'b':
		if (!m) die ("--name or -n must come before --bps or -b\n");
		m->speed = symbolic_speed(intarg);
		break;
	case 'c':
		/* read the whole config file in at once */
		conf_fd = open(configfile, O_RDONLY);
		if (conf_fd >= 0) {
			poptContext optCon;
			char	*conftext;
			char	*confline;
			char	*confnext;
			int	confc;
			char	**confv;

			fstat(conf_fd, &sb);
			conftext = malloc(sb.st_size);
			/* pretend read always works, since this is from disk :-) */
			read(conf_fd, conftext, sb.st_size);
			close(conf_fd);
			
			/* read the config file line by line */
			for (confline = conftext;
			     confline;
			     confline = confnext) {
				confnext = strchr(confline, '\n');
				if (confnext) *confnext++ = '\0';

				if (confline[0] == '#') continue;

				poptParseArgvString(confline, &confc, (const char ***)&confv);
				optCon = poptGetContext("ttywatch", confc, (const char **)confv,
							optionsTable, POPT_CONTEXT_KEEP_FIRST);
				while ((c = poptGetNextOpt(optCon)) >= 0)
					m = parse_argument (c, m);
				poptFreeContext(optCon);
			}
			free(conftext);
		}
	}

	return m;
}

int main(int argc, char **argv) {
	char    c;            /* used for argument parsing */
	struct sigaction sact;/* used to initialize the signal handler */
	poptContext optCon;   /* context for parsing command-line options */

	machine *m = NULL;

	struct pollfd *ufds;
	int nfds;

	machine_names = g_hash_table_new(g_str_hash, g_str_equal);
	machine_fds = g_hash_table_new(g_int_hash, g_int_equal);

	if (getuid())
		logpath = ".";
	else
		logpath = "/var/log/ttywatch";

	optCon = poptGetContext("ttywatch", argc, (const char **)argv,
				optionsTable, 0);
	while ((c = poptGetNextOpt(optCon)) >= 0)
		m = parse_argument (c, m);

	if (c < -1) {
		/* an error occurred during option processing */
		fprintf(stderr, "%s: %s\n", 
			poptBadOption(optCon, POPT_BADOPTION_NOALIAS),
			poptStrerror(c));
		return 1;
	}
	poptFreeContext(optCon);

	/* set everything up -- daemon() after setup to report problems */
	setup_pidfile();
	setup_termios();
	if (do_daemon)
		daemon(0, 0);

	/* set the signal handler to restore the old
	 * termios handler */
	sact.sa_handler = cleanup_termios_and_exit;
	sigaction(SIGINT, &sact, NULL);
	sigaction(SIGPIPE, &sact, NULL);
	sigaction(SIGTERM, &sact, NULL);
	sact.sa_handler = reopen_logfiles;
	sigaction(SIGHUP, &sact, NULL);

	ufds = prepare_poll(&nfds);

	do {
		int i, n;

		poll(ufds, nfds, -1);
		for (n = 0; n < nfds; n++) {
			/* if (ufds[n].revents & POLLERR) FIXME: ??????? */
			if (ufds[n].revents & POLLIN) {
#				define BUFSIZE 1024
				char    buf[BUFSIZE];
				machine *m;
				
				m = g_hash_table_lookup(machine_fds, &ufds[n].fd);
				if (!m) continue;

				i = read(m->dev_fd, buf, BUFSIZE);
				if (i >= 1) write(m->log_fd, buf, i);
			}
		}
	} while (1); /* exits through signal handler */
}
