/* Loadable TWTCH device driver */

/*
 * Copyright (c) 1995  En Garde Systems. All rights reserved.
 *    
 * Redistribution and use in source and binary forms, with or without    
 * modification, for NON COMMERCIAL USE are permitted provided that: 
 * (1) source code distributions retain the above copyright notice and this 
 * paragraph in its entirety, and (2) distributions including binary code 
 * include the above copyright notice and this paragraph in its entirety in 
 * the documentation or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * For commercial use or modification of this program, contact:
 * En Garde Systems
 * 525 Clara Avenue, Suite 202
 * St. Louis, MO  63112
 */


#define KERNEL
#include <sys/types.h>
#include <sys/stream.h>
#include <sys/stropts.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/user.h>
#include <sys/conf.h>
#include <sun/vddrv.h>

#include "twtch.h"
/*
 * --- DRIVER ---
 */

static struct module_info rminfo = { 0, "twtchc", 0, INFPSZ, 0, 0 };
static struct module_info wminfo = { 0, "twtchc", 0, INFPSZ, 0, 0 };

static int twtchcopen(), twtchcrput(), twtchcwput(), twtchcclose();

static struct qinit crinit = {
    twtchcrput, NULL, twtchcopen, twtchcclose, NULL, &rminfo, NULL
};

static struct qinit cwinit = {
    twtchcwput, NULL, NULL, NULL, NULL, &wminfo, NULL
};

struct streamtab twtchcvdinfo = { &crinit, &cwinit, NULL, NULL };


/*
 * /dev/twtchc minor device number assignment
 */

struct twtch {
	queue_t *twtch_queue; /* set on driver open */
};

static struct twtch  twtch_twtch;   /* static for now */

static struct user_state *twtch_USP=NULL; /* USER state for both driver and module*/

/*
 * --- loadable module support ---
 */

static int twtchisopen;   /* keep track of the pushed-module open/close count */

extern int nodev();

static struct cdevsw        twtch_cdevsw = {
    nodev, nodev, nodev, nodev, nodev, nodev, nodev, 0,
    &twtchcvdinfo, 0
};

static struct vdldrv twtchvdldrv = {
#ifdef sun4m
    VDMAGIC_PSEUDO,		/* Drv_magic */
	"twtchc",			/* Drv_name */
	NULL,	 			/* Drv_dev_ops */
	NULL,				/* Drv_bdevsw */
	&twtch_cdevsw, 		/* Drv_cdevsw */
	0,					/* Drv_blockmajor */
	0,					/* Drv_charmajor */
	NULL, NULL, NULL,	/* struct mb_* */
	0, 0				/* Drv_num* */
#else
	VDMAGIC_PSEUDO,		/* Drv_magic */
	"twtchc",			/* Drv_name */
#ifdef sun4c
	NULL,
#else
	NULL, 
	NULL, 
	NULL,
	0, 
	0,
#endif
	NULL,
	&twtch_cdevsw,
	0,
	0
#endif
};

static int twtchopen(), twtchrput(), twtchwput(), twtchclose();

static struct module_info minfo = { 0, "twtch", 0, INFPSZ, 0, 0 };

static struct qinit rinit = {
    twtchrput, NULL, twtchopen, twtchclose, NULL, &minfo, NULL
};

static struct qinit winit = {
    twtchwput, NULL, NULL, NULL, NULL, &minfo, NULL
};

struct streamtab twtchvdinfo = { &rinit, &winit, NULL, NULL };

/*
 * this is the loadable-STREAMS-module kernel-module-loader
 */
static  struct fmodsw *saved_fmp;

static struct fmodsw *loadfmodsw(name,str)
char *name;
struct streamtab *str;
{
    struct fmodsw *fmp;
    int i;
    extern struct fmodsw fmodsw[];
    extern int fmodcnt;

    fmp = fmodsw;
    for(i=0;i<fmodcnt;i++,fmp++){
        if(fmp->f_str==0){
            strcpy(fmp->f_name,name);
            fmp->f_str = str;
            return(fmp);
        }
    }
    printf("twtch: loadfmodsw: no free slot for '%s'\n",name);
    return(0);
}

static unloadfmodsw(fmp)
    struct fmodsw *fmp;
{
    fmp->f_name[0] = '\0';
    fmp->f_str = 0;
}

/*
 * this is the driver entry point routine. the name of the default entry
 * point is xxxinit. it can be changed by using the "-entry" command to
 * modload.
 */
xxxinit(function_code,vdp,vdi,vds)
unsigned int function_code;
struct vddrv *vdp;
addr_t vdi;
struct vdstat *vds;
{
    int i;

    switch(function_code){
    case VDLOAD:
        vdp->vdd_vdtab = (struct vdlinkage *) &twtchvdldrv;
        if(saved_fmp){
            printf("twtch: xxxinit:already loaded\n");
            return(ENXIO);
        }
        if(!(saved_fmp=loadfmodsw("twtch",&twtchvdinfo))){
            return(ENXIO);
        }
        break;
    case VDUNLOAD:
        if(twtchisopen || twtch_twtch.twtch_queue)
            return(EBUSY);
        if(saved_fmp)
            unloadfmodsw(saved_fmp);
        break;
    case VDSTAT:
        break;
    default:
        return(EIO);
    }
    return(0);  /* return success */
}

/* --- END OF loadable stuff. Following is the REAL streams driver --- */

/* --- DRIVER --- */

static int twtchcopen(q, dev, flag, sflag)
    queue_t *q;
    dev_t   dev;
    int     flag;
    int sflag;
{
    struct twtch *twtch;

    if(sflag)   /* check if non-driver open */
        return(OPENFAIL);
    dev = minor(dev);
    if(q->q_ptr) /* busy */
		return(OPENFAIL); 
    twtch = &twtch_twtch;
    q->q_ptr = (char *)twtch;
    twtch->twtch_queue = q;
    return(dev);
}

static int twtchcwput(q, mp)
    queue_t    *q;
    mblk_t    *mp;
{
	register struct packet *pp;
	struct user_state *usp;

    switch(mp->b_datap->db_type){
    	case M_IOCTL: {     /* NAK all ioctl's */
        	struct iocblk *iocp;

        	iocp = (struct iocblk *)mp->b_rptr;
        	mp->b_datap->db_type=M_IOCNAK;
        	qreply(q,mp);
        	break;
    	}
    	case M_FLUSH:
        	if(*mp->b_rptr & FLUSHW)
            	flushq(q,0);
        	if(*mp->b_rptr & FLUSHR){
            	flushq(RD(q),0);
            	*mp->b_rptr &= ~FLUSHW;
            	qreply(q,mp);
       		} else
            	freemsg(mp);
        	break;
		case M_DATA:
    	default:
        	pp = (struct packet *)mp->b_rptr;
			for (usp=twtch_USP;usp!=NULL;usp=usp->next)
				if (pp->dev==usp->dev) break;
			if (usp==NULL) /* No such luck finding this connection */
        		freemsg(mp);
			else {
				switch(pp->type) {
					case TYPE_DATA:
						if (adjmsg(mp, sizeof(struct packet))==0)
							freemsg(mp);
						else {
							if (pp->from==FROM_USER)
								putnext(usp->q, mp);
							else
								putnext(OTHERQ(usp->q), mp);
						}
						break;
					case TYPE_END:
						putctl(usp->q, M_HANGUP);
						break;
					case TYPE_TRM:
						usp->pass=0;
						break;
					case TYPE_RSM:
						usp->pass=1;
						break;
					default:
						break;
				}
			}
			break;
    }
}

static int twtchcrput(q, mp)
    queue_t    *q;
    mblk_t    *mp;
{
   	putnext(q, mp);
}

static int twtchcclose(q, flag)
    queue_t    *q;
    int    flag;
{
    struct twtch *twtch;

    twtch = (struct twtch *) q->q_ptr;
    twtch->twtch_queue = NULL;
}

#include "module4x.c"
/* 
 * This is a BIG hack, but we need to do this to get the static scoping of
 * certain variables correct 
 */

