/* Loadable STREAMS device driver and module for Solaris */

/*
 * Copyright (c) 1995  En Garde Systems. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, for NON COMMERCIAL USE are permitted provided that:
 * (1) source code distributions retain the above copyright notice and this
 * paragraph in its entirety, and (2) distributions including binary code
 * include the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * For commercial use or modification of this program, contact:
 * En Garde Systems
 * 525 Clara Avenue, Suite 202
 * St. Louis, MO  63112
 */

#define _KERNEL 
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/uio.h>
#include <sys/buf.h>
#include <sys/user.h>
#include <sys/modctl.h>
#include <sys/open.h>
#include <sys/kmem.h>
#include <sys/poll.h>
#include <sys/conf.h>
#include <sys/cmn_err.h>
#include <sys/stat.h>
#include <sys/stropts.h>
#include <sys/devops.h>
#include <sys/ddi.h>
#include <sys/sunddi.h>

#include "twtch.h"

/*
 * Solaris (SVR4) Streams Driver declarations
 */

static struct module_info rminfo = { 0, "twtchc", 0, INFPSZ, 0, 0 };
static struct module_info wminfo = { 0, "twtchc", 0, INFPSZ, 0, 0 };

static int twtchcopen(), twtchcrput(), twtchcwput(), twtchcclose();

static struct qinit crinit = {
    twtchcrput, NULL, twtchcopen, twtchcclose, NULL, &rminfo, NULL
};

static struct qinit cwinit = {
    twtchcwput, NULL, NULL, NULL, NULL, &wminfo, NULL
};

struct streamtab twtchcvdinfo = { &crinit, &cwinit, NULL, NULL };

struct twtch {
	queue_t *twtch_queue; /* set on driver open */
};

static struct twtch  twtch_twtch; 

static struct user_state *twtch_USP=NULL; /* USER state for both driver and module */

typedef struct {
	dev_info_t *dip;
} twtchc_devstate_t;

static void *twtchc_state;

/*
 *  Solaris (SVR4) Streams loadable module declarations
 */

static int twtchisopen;   /* keep track of the pushed-module open/close count */

static struct cb_ops twtchc_cb_ops = {
    nulldev,		/* open */
	nulldev,		/* close */
	nodev,			/* strategy */
	nodev,			/* print */
	nodev,			/* dump */
	nodev,			/* read */
	nodev,			/* write */
	nodev,			/* ioctl */
	nodev,			/* devmap */
	nodev,			/* memmap */
	nodev,			/* segmap */
	nochpoll,		/* poll */
	ddi_prop_op,	/* prop_op */
    &twtchcvdinfo,	/* stream */
	D_NEW 
};

static int twtchc_getinfo(), twtchc_identify(), twtchc_attach(), twtchc_detach();

static struct dev_ops twtchc_ops = {
	DEVO_REV,
	0,
	twtchc_getinfo,
	twtchc_identify,
	nulldev,
	twtchc_attach,
	twtchc_detach,
	nodev,
	&twtchc_cb_ops,
	(struct bus_ops *)0
};

extern struct mod_ops mod_driverops;

static struct modldrv modldrv = {
	&mod_driverops,
	"tty watcher driver v1.0",
	&twtchc_ops
};


static int twtchopen(), twtchrput(), twtchwput(), twtchclose();

static struct module_info minfo = { 0, "twtch", 0, INFPSZ, 0, 0 };

static struct qinit rinit = {
    twtchrput, NULL, twtchopen, twtchclose, NULL, &minfo, NULL
};

static struct qinit winit = {
    twtchwput, NULL, NULL, NULL, NULL, &minfo, NULL
};

struct streamtab twtchvdinfo = { &rinit, &winit, NULL, NULL };
/* loadable-STREAMS-module kernel-module-loader */

static struct fmodsw twtch_fmodsw = {
	"twtch",
	&twtchvdinfo,
	0
};

static struct modlstrmod modlstrmod = {
	&mod_strmodops,
	"tty watcher module v1.0",
	&twtch_fmodsw
};

static struct modlinkage modlinkage = {
	MODREV_1,
	&modldrv,
	&modlstrmod,
	0
};

int _init(void)
{
	int e;

	if ((e = ddi_soft_state_init(&twtchc_state, sizeof(twtchc_devstate_t), 1)) != 0)
		return(e);

	if ((e = mod_install(&modlinkage)) != 0) 
		ddi_soft_state_fini(&twtchc_state);

	return(e);
}

int _fini()
{
	int e;

	if ((e = mod_remove(&modlinkage)) != 0) return(e);

	ddi_soft_state_fini(&twtchc_state);
	
	return(e);
}

int _info(modinfop)
struct modinfo *modinfop;
{	
	return(mod_info(&modlinkage, modinfop));
}

static int twtchc_identify(dip)
dev_info_t *dip;
{
	if (strcmp(ddi_get_name(dip), "twtchc") == 0) 
		return(DDI_IDENTIFIED);
	else
		return(DDI_NOT_IDENTIFIED);
}

static int twtchc_attach(dip,cmd)
dev_info_t *dip;
ddi_attach_cmd_t cmd;
{
	int instance;
	twtchc_devstate_t *tsp;

	switch (cmd)
	{
		case DDI_ATTACH:
			instance = ddi_get_instance(dip);

			/* Allocate memory for our state entry */
			if (ddi_soft_state_zalloc(twtchc_state, instance) != DDI_SUCCESS)
			{	
				cmn_err(CE_CONT, "%s%d: can't allocate state!\n",
					ddi_get_name(dip), instance);
				return(DDI_FAILURE);
			}
			else
				tsp = ddi_get_soft_state(twtchc_state,instance);
				
			/* Create minor node */
			if(ddi_create_minor_node(dip,"twtchc",S_IFCHR, instance, 
				DDI_PSEUDO, 0)== DDI_FAILURE) return(DDI_FAILURE);
			ddi_report_dev(dip);	
			break;
		default:
			return(DDI_FAILURE);
			break;
	}
	return(DDI_SUCCESS);
}

twtchc_detach(dip, cmd)
dev_info_t *dip;
ddi_detach_cmd_t cmd;
{
	int instance;
	twtchc_devstate_t *tsp;

	switch(cmd)
	{
		case DDI_DETACH:
			instance = ddi_get_instance(dip);
			tsp = ddi_get_soft_state(twtchc_state,instance);
			ddi_remove_minor_node(dip, NULL);
			ddi_soft_state_free(twtchc_state, instance);
			return (DDI_SUCCESS);
			break;
		default:
			return(DDI_FAILURE);
			break;
	}
}			

twtchc_getinfo(dip,infocmd, arg, result)
dev_info_t *dip;
ddi_info_cmd_t infocmd;
void *arg;
void **result;
{
	twtchc_devstate_t *tsp;
	int error = DDI_FAILURE;
	
	switch (infocmd)
	{
		case DDI_INFO_DEVT2DEVINFO:
			if ((tsp = ddi_get_soft_state(twtchc_state, 
				getminor((dev_t) arg))) != NULL) 
			{
				*result = tsp->dip;
				error = DDI_SUCCESS;
			}
			else
				*result = NULL;
			break;

		case DDI_INFO_DEVT2INSTANCE:
			*result = (void *)getminor((dev_t) arg);
			error = DDI_SUCCESS;
			break;

		default:
			break;
	}
	return(error);
}


/* --- DRIVER --- */
static int twtchcopen(q, dev, flag, sflag, cred)
    queue_t *q;
    dev_t   *dev;
    int     flag;
    int sflag;
	cred_t *cred;
{
	minor_t m_dev;
	struct twtch *twtch;

    if(sflag)   /* check if non-driver open */
        return(OPENFAIL);
    m_dev = getminor(*dev);
    if(q->q_ptr) /* busy */
		return(OPENFAIL); 
    twtch = &twtch_twtch;
    q->q_ptr = (char *)twtch;
    twtch->twtch_queue = q;
    return(m_dev);
}

static int twtchcwput(q, mp)
    queue_t    *q;
    mblk_t    *mp;
{
	register struct packet *pp;
	struct user_state *usp;

    switch(mp->b_datap->db_type){
    case M_IOCTL: {     /* NAK all ioctl's */
        struct iocblk *iocp;

        iocp = (struct iocblk *)mp->b_rptr;
        mp->b_datap->db_type=M_IOCNAK;
        qreply(q,mp);
        break;
    }
    case M_FLUSH:
        if(*mp->b_rptr & FLUSHW)
            flushq(q,0);
        if(*mp->b_rptr & FLUSHR){
            flushq(RD(q),0);
            *mp->b_rptr &= ~FLUSHW;
            qreply(q,mp);
       } else
            freemsg(mp);
        break;
	case M_DATA:
    default:
		pp = (struct packet *)mp->b_rptr;
		for (usp=twtch_USP;usp!=NULL;usp=usp->next)
			if (pp->dev==usp->dev) break;
		if (usp == NULL) /* No such luck finding this connection */
			freemsg(mp);
		else {
			switch (pp->type) {
				case TYPE_DATA:
					if (adjmsg(mp, sizeof(struct packet))==0)
						freemsg(mp);
					else {
						if (pp->from==FROM_USER)
							putnext(usp->q,mp);
						else
							putnext(OTHERQ(usp->q), mp);
					}
					break;
				case TYPE_END:
					putctl(usp->q, M_HANGUP);
					break;
				case TYPE_TRM:
					usp->pass=0;
					break;
				case TYPE_RSM:
					usp->pass=1;
					break;
				default:
					break;
			}
		}
		break;
	}
}

static int twtchcrput(q, mp)
    queue_t    *q;
    mblk_t    *mp;
{
    putnext(q, mp);
}

static int twtchcclose(q, flag)
    queue_t    *q;
    int    flag;
{
    struct twtch *twtch;

    twtch = (struct twtch *) q->q_ptr;
    twtch->twtch_queue = NULL;
}

#include "module5x.c"
/* 
 * This is a BIG hack, but we need to do this to get the static scoping of
 * certain variables correct 
 */

