#!/usr/bin/perl
#
# viperdb.pl - Filesystem Integrity Monitor
#
# Copyright (C) 1998-2001 J-Dog <J-Dog@Resentment.org> & Peter Surda <shurdeek@panorama.sth.ac.at>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

use MD5;

# These are the only things you should need to set
$hostname=`hostname`;
$hostname =~ s/\n//g;
$configfile='/etc/viperdb.conf';
$ignorefile='/etc/viperdb.ignore';
$notifymail='Y';
$notify_email='root';
$notify_subject="ViperDB: Changes Detected on $hostname!\n";
$loglevel=2;
my $domd5 = 0;

#Grab da OS name
chomp($os_type=lc(qx[uname]));

# Detect what command line switches were passed and act accordingly
if ($ARGV[1] eq '-md5') {
	$domd5=1;
	print "MD5 Detected. Will do MD5 Digests as well...\n";
}
if ($ARGV[0] eq '-init'){
	print "Init Detected. Creating Databases...\n";
	&InitDB;
} elsif ($ARGV[0] eq '-check'||$ARGV[0] eq '-checkstrict'){
	print "Check Detected: Now Checking File Sanity...\n";
	&SysCheck;
} else {
	print "\n\nViperDB v0.9.3\n";
	print "ERROR: Unrecognized option or none given.\n";
	print "usage: ViperDB [-init|-check|-checkstrict] [-md5]\n";
	print "     -init	  Initializes the ViperDB Databases\n";
	print "     -check	Runs a system file sanity check\n";
	print "     -checkstrict   Runs a system file sanity check (protective)\n";
	print "     -md5    Will do MD5 digests as well\n";
}

sub InitDB {
	$runtype='init';
	&ParseConfigs;
	&CreateDB;
}

sub SysCheck {
	$runtype='check';
	&ParseConfigs;
	&CreateDB;
	if ($ARGV[0] eq '-checkstrict'){
		$strictmode="Y";
	}
	&Compare;
	&Cleanup;

	# At this point I re-init the databases to stop
	# changes from constantly being displayed. We have
	# displayed changes, if any, and we are now going
	# to re-create the database with the new perms.
	$runtype='init';
	print "Creating New Databases...\n";
	&CreateDB;

}

sub ParseConfigs {
	# Parse in the Ignore file
	open (IGNORE, "< $ignorefile");
		%ignorefiles;
		$ignorecount=0;
		while(<IGNORE>){
			chomp;
			$ignorefiles{$ignorecount} = $_;
			$ignorecount+=1;
		}#while
	close(IGNORE);

	#Parse in the Config file
	open (CONFIG, "< $configfile");
		%rdirs;
		$rdirscount=0;
		%configdirs;
		$configdirs_count=0;
		STARTCONFIG:
		$configline=<CONFIG>;
		chomp $configline;
		while (defined($configline)){
			if ($configline =~ /:/){
				goto STARTCONFIG;
			}

			if($configline =~ /^recursive/){
				($junk,$configline) =  split / /,$configline;
				$rdirs{$configline} = $configline;
			} else {
				$configdirs{$configline} = $configline;
			}
			$configline=<CONFIG>;
			chomp $configline;
		}
	close (CONFIG);

	$found_newdir='Y';
	while($found_newdir ne 'N'){
		$found_newdir='N';

		foreach $configline (sort keys %rdirs){
			$wd = $rdirs{$configline};
			opendir(DIRLIST, $wd) || die "can't opendir $wd: $!";
				@files = readdir(DIRLIST);
				$filecount=@files;
				$count=0;
				RESTARTLOOP:
				while($count ne $filecount){
					foreach $ignorecount (sort keys %ignorefiles){
						if ($files[$count] eq $ignorefiles{$ignorecount}){
							$count+=1;
							goto RESTARTLOOP;
						}
					}
					$file="$wd/$files[$count]";
					if(-d $file){
						$newdir='Y';
						foreach $configline (sort keys %rdirs){
							if(!defined($rdirs{$file})){
								$rdirs{$file} = $file;
								$found_newdir='Y';
							}
						}
					}
					$count+=1;
				}
			closedir(DIRLIST);
		} #foreach
	}

	foreach $configline (sort keys %rdirs){
		if(!defined($configdirs{$configline})){
			$configdirs{$configline} = $rdirs{$configline};
		}
	}

	# Debuggin Stuff
	#print "\nFinal Array\n";
	#foreach $configline (sort keys %configdirs){
	#	print "$configdirs{$configline}\n";
	#}


}

sub CreateDB {
	$md5 = new MD5;
	foreach $configline (sort keys %configdirs){
		$wd=$configdirs{$configline};
		# Set some vars up for runtype
		if ($runtype eq 'init'){
			$ViperDB=$wd . '/.ViperDB';
			# Solaris be goofy and we can't do this
			if ($os_type ne 'sunos'){
				system("chattr -iu $ViperDB 2>/dev/null 1>/dev/null");
			}#if
		} else {
			$ViperDB=$wd . '/.ViperDB.tmp';
		}#if...else

		# Read in the current dir and build the DB
		open (VIPERDB, "> $ViperDB");
			opendir(BINLIST, $wd) || die "can't opendir $wd: $!";
				@files = readdir(BINLIST);
				$filecount=@files;
				$count=0;
				RESTARTLOOP:
				while($count ne $filecount){
					foreach $ignorecount (sort keys %ignorefiles){
						if ($files[$count] eq $ignorefiles{$ignorecount}){
							$count+=1;
							goto RESTARTLOOP;
						}
					} #foreach
					$bname=$files[$count];
					$file="$wd/$bname";
					if (! -d $file){
						($dev, $ino, $mode, $nlink, $uid, $gid,$rdev, $size, $atime, $mtime, $ctime, $blksize, $blocks)= stat($file);
						my $digest = 0;
						if ($domd5 == 1) {
							$md5->reset;
							if (open (MD5F, $file)) {
								seek (MD5F, 0, 0);
								$md5->addfile(MD5F);
								$digest = uc ($md5->hexdigest());
								close (MD5F);
							}
						}
						print VIPERDB "$bname,$size,$mode,$uid,$gid,$mtime";
						if ($domd5 == 1) {
							print VIPERDB ",$digest\n";
						} else {
							print VIPERDB "\n";
						}
					}
					$count+=1;
				} #while
			closedir(BINLIST);
		close(VIPERDB);

		# Change the permissions to only allow root to read...
		chmod(0400, "$ViperDB");
		if ($runtype eq 'init') {
			if ($os_type ne 'sunos') {
				system("chattr +iu $ViperDB 2>/dev/null 1>/dev/null");
			}
		}
	}
}

sub Compare {
	foreach $configline (sort keys %configdirs){
		$mypath=$configdirs{$configline};
		$RealDB=$mypath . '/.ViperDB';
		$ChkDB=$mypath . '/.ViperDB.tmp';

		# Init some Assoc. Arrays
		%valid = ();
		%check = ();

		# Read the RealDB into an Assoc. Array
		open(A, $RealDB);
			while(<A>){
				($bname,$junk) =  split /,/,$_;
				chomp $bname;
				if(defined($bname)){
					if(!defined($valid{$bname})){
						$valid{$bname} = $_;
					} # if
				} # if
			} # while
		close (A);

		# Read the CheckDB into an Assoc. Array
		open(B, $ChkDB);
			while(<B>){
				($bname,$junk) =  split /,/,$_;
				chomp $bname;
				if(defined($bname)){
					if(! defined($check{$bname})){
						$check{$bname} = $_;
					} # if
				} # if
			} # while
		close (B);

		foreach $bname (sort keys %valid){
			$fileinfoa=$valid{$bname};
			$fileinfob=$check{$bname};
			chomp $fileinfoa;
			chomp $fileinfob;
			if($fileinfoa ne $fileinfob){
				($a_bname,$a_size,$a_mode,$a_uid,$a_gid,$a_mtime,$a_md5) = split/,/,$fileinfoa;
				($b_bname,$b_size,$b_mode,$b_uid,$b_gid,$b_mtime,$b_md5) = split/,/,$fileinfob;
				$myfile=$wd.$a_bname;
				if(! defined($b_bname)){
					print LOG "Alert - FILE DELETED: $mypath/$a_bname\n";
					$errorsummary .= "Alert - FILE DELETED: $mypath/$a_bname\n";
					$trouble++;
				} else {
					print LOG "Alert - CHANGES TO FILE: $mypath/$a_bname\n";
					$errorsummary .= "Alert - CHANGES TO FILE: $mypath/$a_bname\n";
					$trouble++;
					if ($loglevel eq 2){
						if($a_size ne $b_size){
							print LOG "Alert - SIZE: was $a_size now $b_size\n";
							$errorsummary .= "Alert - SIZE: was $a_size now $b_size\n";
						}
						if($a_uid ne $b_uid||$a_gid ne $b_gid){
							print LOG "Alert - UID/GID: was $a_uid:$a_gid now $b_uid:$b_gid\n";
							$errorsummary .= "Alert - UID/GID: was $a_uid:$a_gid now $b_uid:$b_gid\n";
							if($strictmode eq 'Y'){
								print LOG "Alert - UID/GID: Changing uid/gid of $a_bname back to $a_uid:$a_gid\n";
								$errorsummary .= "Alert - UID/GID: Changing uid/gid of $a_bname back to $a_uid:$a_gid";
								chown $a_uid, $a_gid, $myfile;
							}
						}
						if($a_perms ne $b_perms){
							print LOG "Alert - PERMS: was $a_perms now $b_perms\n";
							$errorsummary .= "Alert - PERMS: was $a_perms now $b_perms";
							if($strictmode eq 'Y'){
								print LOG "Alert - PERMS: Changing perms on $a_bname back to $a_perms\n";
								$errorsummary .= "Alert - PERMS: Changing perms on $a_bname back to $a_perms\n";
								chmod $a_perms, $myfile;
							} # if
						} # if
						if($a_mtime ne $b_mtime){
							($a_seconds, $a_minutes, $a_hours, $a_dom, $a_month, $a_year, $a_wday, $a_yday, $a_isdst) = localtime($a_mtime);
							$a_year += 1900;
							($b_seconds, $b_minutes, $b_hours, $b_dom, $b_month, $b_year, $b_wday, $b_yday, $b_isdst) = localtime($b_mtime);
							$b_year += 1900;
							$string = sprintf "Alert - MTIME: was %02i/%02i/%04i %02i:%02i:%02i now %02i/%02i/%04i %02i:%02i:%02i\n", $a_month + 1, $a_dom, $a_year, $a_hours, $a_minutes, $a_seconds, $b_month + 1, $b_dom, $b_year, $b_hours, $b_minutes, $b_seconds;
							print LOG $string;
							$errorsummary .= $string;
						} # if
						if($domd5 == 1 && $a_md5 ne $b_md5){
							print LOG "Alert - MD5: was $a_md5 now $b_md5\n";
							$errorsummary .= "Alert - MD5: was $a_md5 now $b_md5\n";
						} # if
					} #if.. loglevels
				} # if ... else
			} # if
		} # foreach

		foreach $bname ( sort keys %check ) {
			$fileinfoa=$valid{$bname};
			$fileinfob=$check{$bname};
			($a_bname,$junk) = split/,/,$fileinfoa;
			($b_bname,$junk) = split/,/,$fileinfob;
			if (! defined($a_bname) ) {
				print LOG "Alert - NEW FILE: $mypath/$b_bname\n";
				$errorsummary .= "Alert - NEW FILE: $mypath/$b_bname\n";
				$trouble=$trouble+1;
			} # if
		} # foreach

	}#foreach

	if ($trouble!=0) {
		print LOG "Info - END RUN - $trouble changes detected.";
		$errorsummary .= "Info - END RUN - $trouble changes detected.";
	} # if
	close (LOG);
	if ($notifymail eq 'Y' && $trouble != 0) {
		open (MAIL, "|mail -s '$notify_subject' $notify_email");
			print MAIL "$errorsummary";
		close(MAIL);
	} #if
} # sub

sub Cleanup {
	foreach $configline (sort keys %configdirs){
		$mypath=$configdirs{$configline};
		$tmpDB=$mypath . '/.ViperDB.tmp';
		unlink("$tmpDB");
	}
} # sub


