#!/usr/bin/perl
#
# viperdb.pl - Filesystem Integrity Monitor
#
# Copyright (C) 1998-2001 J-Dog <J-Dog@Resentment.org> & Peter Surda <shurdeek@panorama.sth.ac.at>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

use MD5;
use GDBM_File;
use POSIX;

my $hostname = `hostname`;
$hostname =~ s/\n//g;

# configfile is the only thing you should set. Other stuff is overriden by
# contents of the config file
my $configfile = '/etc/viperdb.conf';
my $dbfname = '.ViperDB';
my $lock = "/var/lock/subsys/viperdb.lock";
my $md5lock = "/var/lock/subsys/viperdb-md5.lock";
my $strict = 0;
my $init = 0;
my $doattr = 0;
my $errorsummary = "";
my %db;
my $dbopened = 0;
my $trouble = 0;
my %cdirs;
my %ignore;
my @rignore;
my $md5obj = new MD5;

# Grab da OS name
chomp($os_type=lc(qx[uname]));
#open LOG, "|cat";

&ParseConfigs;

&Lock;

# logfile is only used if defined in config file.
if (defined $logfile) {
	open LOG, ">>$logfile";
}

if ($ARGV[1] eq '-md5') {
	$domd5=1;
	print "MD5 Detected. Will do MD5 Digests as well...\n";
}
if ($ARGV[0] eq '-init'){
	print "Init Detected. Creating Databases...\n";
	$init = 1;
	&Go;
} elsif ($ARGV[0] eq '-check'||$ARGV[0] eq '-checkstrict'){
	if ($ARGV[0] eq "-checkstrict") {
		$strict = 1;
	}
	print "Check Detected: Now Checking File Sanity...\n";
	&Go;
} else {
	print "\n\nViperDB v0.9.6\n";
	print "ERROR: Unrecognized option or none given.\n";
	print "usage: ViperDB [-init|-check|-checkstrict] [-md5]\n";
	print "     -init	  Initializes the ViperDB Databases\n";
	print "     -check	Runs a system file sanity check\n";
	print "     -checkstrict   Runs a system file sanity check (protective)\n";
	print "     -md5    Will do MD5 digests as well\n";
}

&UnLock;

sub Lock
{
	my $locked = 0;
	(undef, undef, undef, undef, undef, undef, undef, undef, undef, $mtime,
	undef, undef, undef) = stat ($lock);

	if (-e $lock) {
		open LOCK, $lock;
		$tmp = <LOCK>;
		$pid = chomp $tmp;
		close LOCK;
		@tmp = `ps $pid|grep $pid`;
		if (scalar @tmp == 1 && time () - $mtime < $timelock) {
			$locked = 1;
		}
	}
	if ($domd5 && ! -e $md5lock) {
		open LOCK, ">$md5lock";
		print LOCK "$$";
		close LOCK;
	}
	if ($locked) {
		exit;
	}

	open LOCK, ">$lock";
	print LOCK "$$";
	close LOCK;

	if (-e $md5lock) {
		$domd5 = 1;
		unlink $md5lock;
	}
}

sub UnLock
{
	unlink $lock;
}

sub ParseConfigs 
{
	my $sub = "options";
	open (CONFIG, "< $configfile");
	while (<CONFIG>) {
		chomp;
		if (/\[(.+)\]/) {
			$sub = $1;
		# empty, comment or whitespace-only
		} elsif (! defined $_ or /(^\s*(#|:|;))|(^\s*$)/ ) {
			next;
		} else {
			if ($sub eq "ignore") {
				$ignore{$_} = $_;
			} elsif ($sub eq "rignore") {
				push @rignore, $_;
			} elsif ($sub eq "dirs") {
				$tmp = $_;
				if ($tmp =~ /^recursive\s*(.+)/) {
					$tmp = $1;
					$rdirs{$tmp} = $tmp;
				}
				$cdirs{$tmp} = $tmp;
			} elsif ( $sub eq "options") {
				# add some sanity check pls.
				if (/^hostname\s*=\s*(.+)/) {
					$hostname = $1;
				} elsif (/^email\s*=\s*(.+)/) {
					$email = $1;
				} elsif (/^subject\s*=\s*(.+)/) {
					$subject = $1;
					$subject =~ s/\$hostname/$hostname/g;
				} elsif (/^loglevel\s*=\s*(.+)/) {
					$loglevel = $1;
				} elsif (/^md5\s*=\s*(.+)/) {
					$domd5 = $1;
				} elsif (/^attr\s*=\s*(.+)/) {
					$doattr = $1;
				} elsif (/^lock\s*=\s*(.+)/) {
					$lock = $1;
				} elsif (/^md5lock\s*=\s*(.+)/) {
					$md5lock = $1;
				} elsif (/^timelock\s*=\s*(.+)/) {
					$timelock = $1;
				} elsif (/^dbfname\s*=\s*(.+)/) {
					$dbfname = $1;
				} elsif (/^logfile\s*=\s*(.+)/) {
					$logfile = $1;
				}
			}
		}
	}

	$found = 1;
	%ndirs = {};
	my $file;
	while ($found) {
		$found = 0;
		foreach $dir (sort values %rdirs) {
			if (!defined $dir) {
				# undef->undef gets passed
				# anyone know how to efficiently COMPLETELY delete contents of a hash
				next;
			}
			opendir (DIRLIST, $dir) || &Err ("internal", "Can't opendir $dir\n$!\nSkipping.");
				my @files = readdir(DIRLIST);
				foreach $file (@files) {
					if ($file eq "." || $file eq "..") {
						next;
					}
					# not this means when a subdir of a to be recursed directory is also
					# in the config file, its recursion will be skipped. This is a "bug"
					# in this algorithm, but can be used as a feature to exclude recusing
					# directories :-)
					# e.g. if you put
					# recursive /etc
					# /etc/whatever
					# in your config file, directories such as /etc/whatever/something
					# will be skipped.
					$fname = "$dir/$file";
					$fname =~ s/\/+/\//g;
					if (-d $fname && !exists $cdirs{$fname} &&
							! exists $ignore{"$file"}) {
						$cdirs {$fname} = $ndirs{$fname} = $fname;
						$found = 1;
					}
				}
			closedir(DIRLIST);
			%rdirs = %ndirs; # copy list of new directories
			%ndirs = {}; # reset
		}
	}

	# Debuggin Stuff
	# print "\nFinal Array\n";
	# foreach $tmp (sort keys %cdirs){
	# 	print "$cdirs{$tmp}\n";
	# }
}

sub Err
{
	my $class = uc (shift);
	my $err = shift;
	# format nicely
	$err =~ s/\n/\n                 /mg;
	$err =~ s/\n\s*$//m;

	if (($init || $loglevel ne 2) && $class ne "INTERNAL" && $class ne "CRITICAL") {
		# on $loglevel 1 only say that file changed but not how
		if ($init || $class ne "FILE CHANGED") {
			return;
		}
	}

	my ($tmp) = sprintf "%-15s: %s\n", uc $class, $err;
	printf LOG $tmp;
	$errorsummary .= $tmp;
}

sub OpenDB 
{
	my $wd = shift;
	#my $fname = "$wd/$dbfname";
	my $fname = "$dbfname";
	$dbopened == 1 && &CloseDB($wd);

	if ($strict != 1 && $os_type ne 'sunos') {
		system("chattr -iu $fname 2>/dev/null 1>/dev/null");
	}

	if ($init) {
		chmod 0600, $fname && unlink $fname;
	}
	
	# if it isn't a file, we do some checks.
	if (! -f $fname) {
		if (-e $fname) {
			$str = "";
			if ($strict == 0) {
				$str .= "Deleting.";
				chmod(0600, $fname);
				unlink $fname;
			}
			&Err ("critical", "In $wd, $dbfname isn't a file.\n$str");
		} elsif (!$init) {
			$str = "";
			if ($strict == 0) {
				$str .= "Creating.";
			}
			&Err ("suspicious", "In $wd, $dbfname doesn't exist.\n$str");
		}
	}
	# create or open.
	$dbopened = 1;
	if ($strict) {
		if (! tie (%db, 'GDBM_File', $fname, &GDBM_READER, 0400)) {
			$dbopened = 0;
			&Err ("critical", 
				"In $wd, can't tie $dbfname\nDatabase probably corrupt, suggesting deleting.");
		}
	} else {
		# if exists, chmod so we can write
		( -f $fname ) && chmod (0600, $fname);
		if (! tie (%db, 'GDBM_File', $fname, &GDBM_WRCREAT, 0600)) {
			&Err ("strange", 
				"In $wd, can't tie $dbfname\nDeleting and retrying...");
			unlink $fname;
			if (! tie (%db, 'GDBM_File', $fname, &GDBM_WRCREAT, 0600)) {
				$dbopened = 0;
				&Err ("critical", "Failed. Unable to monitor $wd.");
			}
		}
	}
}

sub CloseDB 
{
	my $wd = shift;
	my $fname = "$wd/$dbfname";
	$dbopened == 0 && return;
	
	untie %db;

	if (! -f $fname) {
		&Err ("critical", "In $wd, $dbfname isn't a file while closing.");
	} else {
		if ($strict == 0) {
			chmod(0400, $fname);
			($os_type ne 'sunos') &&
				system("chattr +iu $fname 2>/dev/null 1>/dev/null");
		}
	}
	$dbopened = 0;
}

sub GetAttr
{
	my $file = shift;
	if ($doattr) {
		return split / /, `lsattr \"$file\"`;
	} else {
		return "--------";
	}
}

sub GetMD5
{
	my $file = shift;
	if ( -f $file) {
		if ($domd5) {
			open (MD5F, $file);
			$md5obj->reset;
			seek (MD5F, 0, 0);
			$md5obj->addfile(MD5F);
			$digest = uc ($md5obj->hexdigest());
			close (MD5F);
		} else {
			$digest = "file";
		}
	} elsif ( -l $file ) {
		$digest = "symlink";
	} elsif ( -p $file ) {
		$digest = "fifo";
	} elsif ( -S $file ) {
		$digest = "socket";
	} elsif ( -b $file ) {
		$digest = "block device";
	} elsif ( -d $file ) {
		$digest = "character device";
	} else {
		$digest = "unknown";
	}
	return $digest;
}

sub FType
{
	my $tmp = shift;
	if (length $tmp == 32) {
		return "file";
	} else {
		return $tmp;
	}
}

sub ChkFile
{
	my $wd = shift;
	my $file = shift;
	my $fname = "$wd/$file";
	$fname =~ s/\/+/\//g;
	if ( -d $file ) {
		return;
	}
	foreach $ig (@rignore) {
		if ($file =~ /$ig/) {
			if (!$strict) {
				delete $db{$file};
			}
			return;
		}
	}
	if ( ! -e $file && ! -l $file) {
		&Err ("file deleted", "$fname");
		$trouble++;
		if (!$strict) {
			delete $db{$file};
		}
		return;
	}
	if ( -l $file ) {
		($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size, $atime, $mtime, $ctime, $blksize, $blocks) = lstat $file;
		$size = readlink $file;
	} else {
		($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size, $atime, $mtime, $ctime, $blksize, $blocks) = stat $file;
	}
	if (exists $db{"$file"}) {
		@olddat = split /,/,$db{"$file"};
	} else {
		@olddat = (0,0,0,0,0,0);
	}
	$attr = &GetAttr ($file);
	$digest = &GetMD5 ($file);
	if ($digest eq "file" && &FType ($olddat[6]) eq "file") {
		$digest = $olddat[6];
	}
	$newdat = "$size,$mode,$attr,$uid,$gid,$mtime,$digest";
	if (! exists $db{"$file"}) {
		&Err ("file added", "$fname");
		$trouble++;
		if (!$strict) {
			# enforce MD5 check so we avoid double complaints.
			$nft = &FType ($digest);
			if (! $domd5 && $nft eq "file") {
				$domd5 = 1;
				$digest = &GetMD5 ($file);
				$domd5 = 0;
				$newdat = "$size,$mode,$attr,$uid,$gid,$mtime,$digest";
			}
			$db{$file} = $newdat;
		}
	} elsif ($newdat ne $db{"$file"}) {
		&Err ("file changed", "$fname");
		$trouble++;
		# 0 - size
		# 1 - mode
		# 2 - attr
		# 3 - uid
		# 4 - gid
		# 5 - mtime
		# 6 - digest
		$oft = &FType ($olddat[6]);
		$nft = &FType ($digest);

		# check MD5 now so we have more precise report and in case you combine MD5
		# and non-MD5 check it doesn't complain twice
		if (! $domd5 && $nft eq "file") {
			$domd5 = 1;
			$digest = &GetMD5 ($file);
			$domd5 = 0;
			$newdat = "$size,$mode,$attr,$uid,$gid,$mtime,$digest";
		}
		if ($oft ne $nft) {
			&Err ("file type", "was $oft\nnow $nft");
		} else {
			if ($olddat[0] ne $size) {
				if ($nft eq "symlink") {
					$str = "";
					if ($strict) {
						unlink $file;
						symlink $file, $olddat[0];
						$str = ", changed back";
					}
					&Err ("symlink", "was $olddat[0]\nnow $size$str");
				} else {
					&Err ("size", "was $olddat[0]\nnow $size");
				}
			}
			if ($mode ne $olddat[1]) {
				$str = "";
				if ($strict) {
					chmod $olddat[1], $file;
					$str = ", changed back";
				}
				$omodestr = sprintf "%.4o", $olddat[1];
				$nmodestr = sprintf "%.4o", $mode;
				&Err ("perms", "were $omodestr\n now $nmodestr$str");
			}
			if ($attr ne $olddat[2]) {
				$str = "";
				#if ($strict) {
					#someone make this working with chattr 
					#$str = ", changed back";
				#}
				&Err ("attr", "was $olddat[2]\nnow $attr$str");
			}
			if ($uid ne $olddat[3] || $gid ne $olddat[4]) {
				$str = "";
				if ($strict) {
					chown $olddat[3], $olddat[4], $file;
					$str = ", changed back";
				}
				&Err ("uid/gid", "were $olddat[3]:$olddat[4]\n now $uid:$gid$str");
			}
			if ($mtime ne $olddat[5]) {
				$str1 = &strftime ("%c", localtime ($olddat[5]));
				$str2 = &strftime ("%c", localtime ($mtime));
				&Err ("mtime", "was $str1\nnow $str2");
			}
			# ne "file" means no previous data
			if ($digest ne $olddat[6] && $olddat[6] ne "file") {
				&Err ("md5", "was $olddat[6]\nnow $digest");
			}
		}
		if (!$strict) {
			$db{$file} = $newdat;
		}
	}
}

sub Go
{
	my $file;
	foreach $wd (values %cdirs) {
		chdir $wd;
		&OpenDB($wd);
		if (!opendir(DIRLIST, $wd)) {
			&Err ("internal", "Can't opendir $wd:\n$!\nSkipping.");
			next;
		}
		my @files = readdir(DIRLIST);
		FILE: foreach $file (@files) {
			(exists $ignore{$file}) && next;
			&ChkFile ($wd, $file);
		}
		close (DIRLIST);
		foreach $file (keys %db) {
			if (! -e "$file") {
				&ChkFile ($wd, $file);
			}
		}
		&CloseDB($wd);
	}
	if ($trouble) {
		$str = &strftime ("%c", localtime);
		&Err ("info", "$str\nEND RUN - $trouble changed files detected.");
	}
	close (LOG);
	if (defined $email && $trouble && ! $init) {
		open (MAIL, "|mail -s '$subject' $email");
			print MAIL "$errorsummary";
		close(MAIL);
	}
}
