/* strutil.c - string utilities 
 *
 * functions defined in this file:
 * 
 *     mstrtok(string,pattern)
 *     mstrtok_clear(string)
 *
 * strtok(string,pattern) OR strtok(s1,s2)
 *    - standard strtok wrapper around the mstrtok function
 *
 * mstrtok is a version of strtok that can be safely used to process
 * two or more strings (actually MAXSIMUL strings) at the same time -
 * the difference is that the string is specified on *all* calls ...
 * i.e. NULL is not used for the second and further calls
 *
 *
 *
 * MAXSIMUL - defines how many strings can be in the process of being
 *	    mstrtok-ed at once
 * AUTO_CLEAR - mstrtok does an mstrtok_clear itself (thereby forgetting
 *	  any details about a string) when the end of the string is reached
 *	  i.e. when NULL is going to be returned to the caller
 *	  - this is defined by DEFAULT now
 * NEED_STRTOK - define a strtok function in terms of mstrtok (for those
 *	  platforms which don't have strtok)
 * LOCAL_TEST - build the test harness for this code; also a good
 *	  place to look for example usage of these routines.
 *
 * 
 * TODO: make maximum number of active mstrtok calls dynamic
 *
 * NOTES:
 *    - strtok is not useful at detecting zero length tokens in 
 *    a parameter list ... multiple separators are silently swallowed
 *    as there are all whitespace which can reduce the usefulness in
 *    certain contexts
 *	  e.g.	  1,,3,4
 *    will be seen as three tokens "1" "3" "4" when "," is the separator
 *    - what is required is two sorts of separator ... those that are
 *    white space and those that are single separators ... I might code
 *    something for this ... after I have a think about it --tjh
 *    
 * LOCAL_TEST requires -traditional with GCC as we write to strings 
 *			that are passed to a function
 *
 * 30-Aug-90 tjh	original coding
 *
 *
 *
 * From: X/OPEN Portability Guide (Jan 1987)
 
     Strtok considers the string s1 to consist of a sequence of zero or more
     text tokens separated by spans of one or more characters from the
     separator string s2.  The first call (with pointer s1 specified) returns
     a pointer to the first character of the first token, and will have
     written a null character into s1 immediately following the token. Strtok
     saves a pointer to the first character following this null character.
     Subsequent calls made to strtok with NULL as the value of s1 continue the
     search from the last saved position, repeating the behavior described
     above.  When no more tokens can be found, strtok returns NULL.  The
     contents of the separator string s2 need not be constant between
     invocations of strtok, even when using a NULL s1.

 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* assume that we have access to the ANSI string routines ... but
 * some of us still like using the BSD (and pre-ANSI) standard 
 * routines --tjh
 */
#ifndef bzero
#define bzero(S,C) memset(S,'\0',C)
#endif
#ifndef index
#define index(S,C) strchr(S,C)
#endif
#ifndef rindex
#define rindex(S,C) strrchr(S,C)
#endif

/* maximum simultaneous mstrtok()'s active at a single point in time */
#define MAXSIMUL	20

/* mstrtok does a mstrtok_clear on returning NULL */
#define AUTO_CLEAR

/* local state variables ... */
static char *_stu_last=NULL;
static int _stu_listsize=MAXSIMUL;
static char **_stu_list;
static char **_stu_pos;

/* a form of strtok that allows multiple strings to be in the process
 * of being strtok()ed at the same time
 */
char *
mstrtok(char *cp,char *patlist)
#if 0
char *cp;		  /* string being token-ised */
char *patlist;		  /* list of token delimeters */
#endif
{
	int i,j;
	char *cpos=NULL;
	char *p;

	/* attempt some form of compatibility with strtok() */
	if (cp==NULL)
		cp=_stu_last;
	/* allocate space if not initialised */
	if (_stu_list==NULL) {
		_stu_list=(char **)calloc(1,_stu_listsize*sizeof(char *));
		_stu_pos=(char **)calloc(1,_stu_listsize*sizeof(char *));
	}
	j=(-1);
	for(i=0;i<_stu_listsize;i++)
		if ((_stu_list[i]==NULL)&&(j==-1))
			j=i;
		else if (_stu_list[i]==cp)
			break;
	/* if not found in the list then this is a new string! */
	if (i>=_stu_listsize) {
		/* if no free space in the list ... */
		if (j==-1) {
			/* TODO - this is real problem as we cannot
			 *	tell the user about this error ...
			 *	which is a real bummer!
			 */
			return NULL;
		}
		_stu_list[j]=cp;
		_stu_pos[j]=cp;
		cpos=cp;
		i=j;
	} else
		cpos=_stu_pos[i];

	/* now we have a current position in the string and a list of
	 * separator characters - off to work
	 */
	/* skip anything in patlist ... this may not be necessary */
	while((*cpos!='\0')&&(index(patlist,*cpos)!=NULL))
		cpos++;
	/* now skip until EOS or patlist character is found */
	p=cpos;
	while((*p!='\0')&&(index(patlist,*p)==NULL))
		p++;
	/* terminate string and remember position as this will be next 
	 * location to start from 
	 */
	if ((*p)!='\0') 
		*p++='\0';
	_stu_pos[i]=p;
	/* if we started on a EOS then finished finding tokens */
	if (*cpos!='\0')
		return cpos;
	else {
#ifdef AUTO_CLEAR
		/* do equivalent of mstrtok_clear() */
		_stu_list[i]=NULL;
		_stu_pos[i]=NULL;
#endif /* AUTO_CLEAR */
		return NULL;
	}
}

/* remove evidence of a string from the knowledge of mstrtok() - allows
 * the same string/buffer to be reused ...
 */
int 
mstrtok_clear(char *cp)
#if 0
char *cp;	 /* string to "forget" about */
#endif
{
	int i;

	/* no list yet means no work to do ... */
	if (_stu_list==NULL)
		return 0;
	/* otherwise attempt to find given char * in list and blank it out */
	for(i=0;i<_stu_listsize;i++)
		if (_stu_list[i]==cp) {
			_stu_list[i]=NULL;
			_stu_pos[i]=NULL;
			return 0;
		}
	return -1;
}

#ifdef NEED_STRTOK

/* strtok in terms of mstrtok ... implemented such that you can
 * still have only one strtok active ... we wouldn't want people
 * to expect too much of the standard strtok function so we must 
 * do the same things as the "real" one
 */

char *
mstrtok(cp,patlist)
char *cp;		  /* string being token-ised */
char *patlist;		  /* list of token delimeters */
{
    static char *last_str=NULL;
    char *ret;

    /* if this is not the first call then use the saved string 
     * otherwise simple remember the string for the next and subseqent
     * calls
     */
    if (cp==NULL)
	cp=last_str;
    else
	last_str=cp;
    ret=mstrtok(cp,patlist);
#ifndef AUTO_CLEAR
    if (ret==NULL)
	mstrtok_clear(cp);
#endif /* AUTO_CLEAR */
    return (ret);
}

#endif /* NEED_STRTOK */


#ifdef LOCAL_TEST
/* test harness for the strtok routines ...
 */

char *seps=" \n\t=";
char *argseps=",";
char *tests[]=
{ "  FIRST_word 1arg1,1arg2     SECOND_WORD     single_arg   ",
  "FIRST_word=1,2,3,4,5,,7,8,9, 2nds=asdasd 3rd asdasd,asdasd,asd,asd" 
};
#define N_tests	   (sizeof(tests)/sizeof(tests[0]))

char *strseps=" \n\t=,";
char *strtest="FIRST_word=1,2,3,4,5,,7,8,9, 2nds=asdasd 3rd asdasd,asdasd,asd,asd";

main(argc,argv)
int argc;
char *argv[];
{
    char *tok, *arg, *val;
    int t,i;

    /* examples for mstrtok ... */
    for(t=0;t<N_tests;t++) {
	/* get the first token which is the "key" */
	while ( (tok=mstrtok(tests[t],seps)) != NULL ) {
	    /* get the second token which is the "value" */
	    arg=mstrtok(tests[t],seps);
	    i=0;
	    while ( (val=mstrtok(arg,argseps)) != NULL ) {
		printf("key=%s val%d=%s\n",tok,i,val);
		i++;
	    }
	}
    }

    /* an example of strtok usage ... */
    tok=strtok(strtest,strseps);
    while ( tok != NULL ) {
	printf("token=%s\n",tok);
	tok=strtok(NULL,strseps);
    }

    exit(0);
}

#endif /* LOCAL_TEST */
