#!/usr/bin/perl
#
# pat_search.pl          - SHADOW Version 1.7
#                          Last changed 31 May 2001
#
#
#  Script to accept a beginning and ending time and search all the hourly
#  SHADOW gzipped raw data files for the pattern passed as a calling parameter.
#
#  Parameters are: -l SITE -s YYYYMMDDHH -e YYYYMMDDHH -p "pattern"
#  
#  Written by Bill Ralph  <RalphWD@nswc.navy.mil>
#
# Set up some variables.
#
use Getopt::Long;
use POSIX qw(strftime);
use Time::Local;
use IO::Handle;
#
#########################################################################
#
# Subroutine to return a method reference. (From "Programming Perl"
# Third Edition, p. 261.
#
sub get_method_ref {
   my ($self, $methodname) = @_;
   my $methref = sub {
      return $self->$methodname(@_);
   };
   return $methref;
}
#########################################################################
#
# See if the Compress::Zlib module is available:
#
   if (eval "require Compress::Zlib") {
      import Compress::Zlib;
      $open_sub = sub {
         $gz = gzopen($_[0], "rb");   
         $read_sub = get_method_ref($gz, 'gzread');
         $close_sub = get_method_ref($gz, 'gzclose');
         $end_of_file = get_method_ref($gz, 'gzerror');
         return $gzerrno;
      };
   } else {
      $fh = *ZIP_CMD;
      $open_sub = sub {
         my $file_name = $_[0];
         my $unzip_cmd = 'gunzip -c $file_name |';      
         return(open($fh, $unzip_cmd));
      };
      $read_sub = sub {
         my $p1 = $_[0];
         my $p2 = $_[1];
         return sysread($fh, $p1, $p2) };
      $close_sub = sub { return(close($fh)) };
      $end_of_file = sub { return eof($fh) };
   }
#
#
do "/usr/local/etc/SHADOW.conf" ||
   die("Unable to read SHADOW configuration file: /usr/local/etc/SHADOW.conf.");
#
############################################################################
#
sub usage {
        print "Usage: pat_search.pl {-n} -b YYYYMMDDHH -e YYYYMMDDHH -l SITE -p PATTERN.\n";
	exit 2;
}
#
$group_pid = getpgrp(0);
#
###############################################################################
#  Catch user interrupt signal and abort our children.
#
sub sig_catch {
   $SIG{HUP}=\&sig_catch;
   $SIG{INT}=\&sig_catch;
   kill (-TERM, $group_pid);
   exit 0;
   return;
}
#
$SIG{HUP}=\&sig_catch;
$SIG{INT}=\&sig_catch;
#
#########################################################################
#
#	Parse the parameters.
#
#
&GetOptions("n", \$no_lookup_flag, "b:s", \$beg_date, "e:s", \$end_date, 
            "l:s", \$Site, "p=s", \$pattern);
##
#
#  Check parameter validity.
#
$sdlen = length($beg_date);
if (("$Site" eq "") or ($pattern eq "") or (($sdlen > 0) and ($sdlen != 10)))
{
        usage();
}
$yr_format = "a4a2a2a2"; 
#
# Once the Site is identified from the command line,
# load the needed external Parameters.
#
unshift(@INC, "$SHADOW_PATH/sites");
die ("No such site: ${Site}.") if ( ! -e "$SHADOW_PATH/sites/${Site}.ph");
require "${Site}.ph";

#
#
#
# If no dates specified on command line, assume today.
#
if ($beg_date eq "") {
   $beg_date = strftime("%Y%m%d", localtime);
   $beg_date .= "00";
}
if ($end_date eq "") {
   $end_date = $beg_date;
}
#
#  Unpack the dates into their useful components.
#
($beg_year, $beg_mon, $beg_mday, $beg_hour) = unpack($yr_format, $beg_date);
($end_year, $end_mon, $end_mday, $end_hour) = unpack($yr_format, $end_date);
#
# Compensate for the way Perl stores months and years.
#
$beg_mon -= 1;
$beg_year -= 1900;
$end_mon -= 1;
$end_year -= 1900;
#
# Convert our dates back to time format.
#
$beg_time = timelocal(0, 0, $beg_hour, $beg_mday, $beg_mon, $beg_year);
$end_time = timelocal(0, 0, $end_hour, $end_mday, $end_mon, $end_year);
$end_time += 3600;
#
#  Set the no-lookup_flag to -n for tcpdump if it was specified on the 
#  calling parameter list, i.e. not looking up is the default.
#
$no_lookup_flag = ($no_lookup_flag) ?"-n":"";
#
# Make sure the passed filter pattern has the quotes.
#
$pattern='"'.$pattern.'"';
#
#
$SIG{INT}=\&sig_catch;
$SIG{HUP}=\&sig_catch;
#
# Loop through the appropriate raw data files.
#
$dump_it = "$TCPDUMP_CMD -S $no_lookup_flag -r - $pattern";
for ($time = $beg_time; $time < $end_time; $time += 3600) {
   @date = localtime($time);
   $subdir = strftime("%b%d", @date);
   $fname = strftime("%Y%m%d%H", @date);
   $file_name = "$ANALYZER_DIR/$subdir/tcp.${fname}.gz";
   $line_hdr = strftime("%Y/%m/%d-", @date);
   next if ( ! -e $file_name);
#
#  Fork off a child to read the gzipped data file, shove it through tcpdump
#  which will filter the data and write its output to a pipe.
#
   if ($childpid = open(CHILD, "-|")) {
#
# This is the parent process. It will read from the pipe and write the
# output to STDOUT.
#
      while (<CHILD>) {
         print $line_hdr, $_;
      }
      close(CHILD);
   } else {
#
# This is the child process. 
#
      STDOUT->autoflush(1);
      $open_sub->($file_name) || die ("Unable to open $file_name: $!\n");
      $pid = open(TCPDUMP, "|$dump_it") or 
             die("Cannot start tcpdump process.");
#
# Our output command is open, unzip a buffer load,
# and feed it to the output tcpdump command.
#
      $blksize = 32768;
      until ($end_of_file->()) {
         $read_len = $read_sub->($buf, $blksize);
         $write_len = $read_len;
         $offset = 0;
         while ($write_len ) {          # Handle partial writes.
            $written = syswrite(TCPDUMP, $buf, $write_len, $offset);
            die("System write error: $!\n")
               unless defined $written;
            $write_len -= $written;
            $offset += $written;
         }
      }
      close(TCPDUMP);
      $close_sub->();
      exit 0;
   }
}
# 
exit 0;
