/* AMAP - application mapper Copyright (c) 2002, 2003 DJ.Rev.Moon and
 * vanHauser
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.    
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *    
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "amap-inc.h"
#include "amap.h"

//-----------------------------------------

// The command line switches:
int verbose = 0, banner = 0, dump = 0, harmful = 0, one_is_enough = 0;
int SSL_reconnect = 1, RPC_reconnect = 1;
int log_to_file = 0, machine_readable = 0, resp_recv = 0, fail = 0;
int timeout_t = TIMEOUT_T;
int unrec = 1;
int tasks = TASKS;
char scantype = 't';
char *only_prot = NULL;
int mode;

// for banner printing
unsigned char banner_string[256];

// misc
char *time_string = "time";
int taskno;
int total_tasks;
int running_tasks = 0;
ll_resp *first_resp, *rpc_resp = NULL;

//-------------------------------------------

void help()
{
  printf("%s %s (c) 2003 by %s <%s> %s
Syntax: %s [-USRHbdmruv]1 [-o <file>] [-D <file>] [-t sec] [-T cons]
           [-p PROTO ] [-i <file>] [TARGET PORT [PORT] [PORT] ...]
Options:
    -i FILE   Nmap machine readable outputfile to read ports from
    -U        Ports specified on commandline are UDP (default is TCP)
    -S        Do NOT look behind an SSL port
    -R        Do NOT identify RPC service
    -H        Do NOT send application triggers marked as potentially harmful
    -u        Do NOT dump unrecognised responses (better for scripting)
    -1        Only send triggers to a port until 1st identification. Speeeeed!
    -v        verbose mode, use twice for debug (not recommended :-)
    -o FILE   Write output to file FILE
    -m        Make output to file (-o) machine-readable (colon-separated list)
    -d        Dump hex traffic (only if a response is received)
    -b        Print ascii banner of responses
    -T CONS   Amount of parallel connections to make (default %d, max %d)
    -p PROTO  Only test for applicational protocol PROTO (i.e.: ftp)
    -t SEC    Response timeout, wait longer on slow connections (default %d)
    -D FILE   Read from Definitions FILE[.trig|.resp|.rpc] instead of default
    -h        Print this shit
    TARGET PORT   The target address and port(s) to scan (additional to -i)
%s is a tool to identify application protocols on target ports.
", PROGRAM, VERSION, AUTHOR, EMAIL, RESOURCE, PROGRAM, TASKS, MAX_TASKS, TIMEOUT_T, PROGRAM);
  exit(-1);
}

//--------------------------------------------

void exit_err(char *msg)
{
  fprintf(stderr, "Error: %s\n", msg);
  exit(-1);
}

//--------------------------------------------

void warn(char *msg)
{
  fprintf(stderr, "Warning: %s, continuing...\n", msg);
}

//--------------------------------------------

int check_trig(char *trig)
{
  if (strstr(trig, "0x") == trig)
    return TRIG_HEX;
  if (trig[0] == '"')
    return TRIG_STR;

  return TRIG_UNK;
}

//--------------------------------------------

void trans_esc(char *orig)	// translates "\n" to a real line feed
{
  int i, t;
  char *walk;

/* currently only does \n, \t and \r escapes. Oh, and \\ for \ */

  while ((walk = index(orig, '\\')) != NULL) {
    t = 1;
    switch (*(walk + 1)) {
    case 'n':
      *walk++ = '\x0a';
      break;
    case 'r':
      *walk++ = '\x0d';
      break;
    case 't':
      *walk++ = '\x09';
      break;
    case '\\':
      *walk++ = '\\';
      break;
    default:
      t = 0;
      break;
    }
    if (t == 1) {
      for (i = 0; walk[i] != '\0'; i++) {
	walk[i] = walk[i + 1];
      }
    }
    orig = walk;
  }
}

//--------------------------------------------

void make_lower(char *target, int len)	// make target lowercase
{
  register int l = 0;

  for (l = 0; l < len; l++)
    if (target[l] != 0)
      target[l] = (char) tolower(target[l]);
}

//--------------------------------------------

char *amap_memstr(char *haystack, char *needle, int haystack_length, int needle_length)	// implements strstr for mem...
{
  register int i;

  if (needle_length > haystack_length)
    return NULL;

  for (i = 0; i <= haystack_length - needle_length; i++)
    if (memcmp(haystack + i, needle, needle_length) == 0)
      return (haystack + i);
  return NULL;
}

//--------------------------------------------

void write_banner_string(FILE * OUT, unsigned char *response, int length)
{
  int i = 0, j = 0;

  if (length < 1)
    return;

  while (j == 0 && i < length) {
    if (!isprint(response[i])
	&& !isspace(response[i]))
      j = 1;
    i++;
  }
  if (j) {
    fprintf(OUT, "0x");
    for (i = 0; i < length; i++) {
      fprintf(OUT, "%c",
	      response[i] / 16 > 9 ?
	      response[i] / 16 + 87 : response[i] / 16 + 48);
      fprintf(OUT, "%c",
	      response[i] % 16 >
	      9 ? response[i] % 16 + 87 : response[i] % 16 + 48);
    }
  } else {
    fprintf(OUT, "\"");
    for (i = 0; i < length; i++)
      switch (response[i]) {
      case '\t':
	fprintf(OUT, "\\t");
	break;
      case '\n':
	fprintf(OUT, "\\n");
	break;
      case '\r':
	fprintf(OUT, "\\r");
	break;
      case '\\':
	fprintf(OUT, "\\\\");
	break;
      default:
	fprintf(OUT, "%c", response[i]);
      }
    fprintf(OUT, "\"");
  }
}

//--------------------------------------------

char *print_banner_string(char *response, int length)
{
  char *pbanner;
  int i = 0;
  int len = 0;

  memset(banner_string, 0, sizeof(banner_string));
  pbanner = banner_string;

  while (len < length && i < sizeof(banner_string) - 16) {
    if (*response == '\t' || *response == '\n' || *response == '\r'
	|| (*response != 0 && isprint(*response) && *response >= ' '
	    && *response <= 'z')) {
      switch (*response) {
      case '\r':
	strcat(banner_string, "\\r");
	i += 2;
	break;
      case '\n':
	strcat(banner_string, "\\n");
	i += 2;
	break;
      case '\t':
	strcat(banner_string, "\\t");
	i += 2;
	break;
      case '\\':
	strcat(banner_string, "\\\\");
	i += 2;
	break;
      case ':':
	break;			// we filter ':'
      default:
	*(pbanner + i) = *response;
	i++;
      }
    }
    response++;
    len++;
  }

  return banner_string;
}

//--------------------------------------------

char *print_banner(char *response, int length)
{
  if (!banner)
    return "";
  else {
    char bs[sizeof(banner_string)];

    strcpy(bs, " - banner: ");
    strcat(bs, print_banner_string(response, length));
    strcpy(banner_string, bs);
    return (banner_string);
  }
}

//--------------------------------------------

void clear_block(sock_block * babble)
{
  memset(babble, 0, sizeof(sock_block));
  babble->active = 0;
  babble->timer = 0;
  strcpy(babble->ip_ad, "0.0.0.0");
  babble->port = (unsigned short int) 0;
  babble->prot = 'b';
  babble->ssl_enabled = 0;
  babble->ssl_sock = NULL;
  babble->s = -1;
  strcpy(babble->response, "");
  babble->orig_response = NULL;
  babble->response_length = 5;
  babble->trigger_ptr = NULL;
  babble->port_ptr = NULL;
  babble->target_ptr = NULL;
}

//--------------------------------------------

void delspaces(char *target)	// only needed for hex strings
{

/* spaces and tabs be gone! But only if it's not a string */
  register int l = 0;
  register int k = 0;

  while ((target[l] != '\0')) {
    if ((target[l] == ' ') || (target[l] == '\t')) {
      k = l--;
      while (target[k++] != '\0')
	target[k - 1] = target[k];
    }
    l++;
  }
}

//--------------------------------------------

int fromhex(int a, int b)
{
  if (isxdigit(a)) {
    a = tolower(a);
  } else {
    printf("char %c\n", a);
    warn("non-hex digit in hex trigger string, silly you");
    a = '0';
  }
  if (isxdigit(b)) {
    b = tolower(b);
  } else {
    printf("char %c\n", b);
    warn("non-hex digit in hex trigger string, silly you");
    b = '0';
  }

  isalpha(a) ? (a -= 87) : (a -= 48);

  isalpha(b) ? (b -= 87) : (b -= 48);

  return ((a * 16) + b);
}

//--------------------------------------------

FILE *open_amap_file(char *fnam, char *type, char *extension)
{
  char file_name[256];
  FILE *f = NULL;
  int i = 0;

  if (fnam != NULL) {
    strcpy(file_name, fnam);
    strcat(file_name, extension);
    f = fopen(file_name, "r");
  } else
    while (searchpath[i] && f == NULL) {
      strcpy(file_name, searchpath[i]);
      strcat(file_name, APP_FINGERPRINT_FILE);
      strcat(file_name, extension);
      f = fopen(file_name, "r");
      i++;
    }
  if (f == NULL)
    return NULL;
  if (verbose)
    printf("Using %s file %s\n", type, file_name);
  return f;
}

//--------------------------------------------

ll_triggers *read_triggers(char *fnam)
{
  ll_triggers *first_trig = NULL, *curr_trig = NULL;
  FILE *trigger_file = NULL;
  char *in_line;
  char *readln = NULL;
  char *walk = NULL;
  int cnt = 0;

  in_line = (char *) malloc(BUFSIZE);

  if ((trigger_file = open_amap_file(fnam, "trigger", TRIG_EXT)) == NULL)
    exit_err("Error opening trigger file\n");

  while (fgets(in_line, BUFSIZE, trigger_file) != NULL) {
    if (in_line[strlen(in_line) - 1] != '\n') {
      fprintf(stderr, "Error: %s\n", in_line);
      exit_err("Line is too long or not ending with \\n!\n");
    }
    if ((in_line[0] != '#') && (index(in_line, ':') != NULL)) {	/* we have to read this line */
      if ((only_prot == NULL)
	  ||
	  (strncasecmp
	   (only_prot, in_line,
	    (int) (index(in_line, ':') - in_line)) == 0)) {
	if (!((harmful == 1) && (strstr(in_line, ":1:") != NULL))) {
	  if (cnt == 0) {
	    curr_trig = (ll_triggers *) malloc(sizeof(ll_triggers));
	    first_trig = curr_trig;
	    cnt++;
	  } else {
	    (ll_triggers *) curr_trig->next =
	      (ll_triggers *) malloc(sizeof(ll_triggers));
	    curr_trig = (ll_triggers *) curr_trig->next;
	    cnt++;
	  }
	  if ((readln = (char *) malloc(strlen(in_line) + 1)) == NULL)
	    exit_err("malloc failed\n");
	  in_line[strlen(in_line) - 1] = '\0';
	  strcpy(readln, in_line);
	  curr_trig->prot = readln;
	  if ((walk = index(readln, ':')) == NULL) {
	    fprintf(stderr, "Error(1): invalid line in trigger file - %s\n",
		    readln);
	    exit(-1);
	  }
	  *walk = '\0';
	  walk++;
	  if (verbose > 1)
	    printf("Loading trigger information for %s ...\n",
		   curr_trig->prot);
	  curr_trig->ip_prot = ((*walk == 't')
				|| (*walk == 'u') ? *walk : 'b');
	  if ((walk = index(walk, ':')) == NULL) {
	    fprintf(stderr, "Error(2): invalid line in trigger file - %s\n",
		    readln);
	    exit(-1);
	  }
	  walk++;
	  curr_trig->dangerous = atoi(walk);
	  if ((walk = index(walk, ':')) == NULL) {
	    fprintf(stderr, "Error(3): invalid line in trigger file - %s\n",
		    readln);
	    exit(-1);
	  }
	  walk++;
	  curr_trig->trigger_string = walk;

	  if ((curr_trig->trigger_type =
	       check_trig(curr_trig->trigger_string)) == TRIG_UNK) {
	    fprintf(stderr, "Error: %s\n", walk);
	    exit_err("Incorrect trigger definition, watch what you type!\n");
	  }

	  if (check_trig(curr_trig->trigger_string) == TRIG_HEX)
	    delspaces(curr_trig->trigger_string);

	  make_lower(curr_trig->prot, strlen(curr_trig->prot));
	}
      }
    }
  }
  free(in_line);
  if (cnt == 0)
    exit_err("No triggers read. bastard, trying to trick me, eh?\n");
  return (first_trig);
}


//--------------------------------------------

char *response_index(char *str)
{
  int escaped = 0, done = 0;

  if (str == NULL)
    return NULL;

  if (strstr(str, "0x") == str) {	// string starts with "0x"
    return strpbrk(str, "^~/");
  } else {			// string starts with '"'
    str++;
    while (*str && !done) {
      switch (*str) {
      case '"':
	if (!escaped)
	  done = 1;
	else {
	  str++;
	  escaped = 0;
	}
	break;
      case '\\':
	if (!escaped)
	  escaped = 1;
	else
	  escaped = 0;
	str++;
	break;
      case 0:
	return NULL;
	break;			// not reached
      default:
	escaped = 0;
	str++;
      }
    }
    while (*str) {
      switch (*str) {
      case '^':
      case '~':
      case '/':
	if (!escaped)
	  return str;
	else {
	  str++;
	  escaped = 0;
	}
	break;
      case '\\':
	if (!escaped)
	  escaped = 1;
	else
	  escaped = 0;
	str++;
	break;
      case 0:
	return NULL;
	break;			// not reached
      default:
	escaped = 0;
	str++;
      }
    }
  }
  return NULL;			// not reached
}

//--------------------------------------------

ll_resp *read_responses(char *fnam)
{
  ll_resp *first_resp = NULL, *curr_resp = NULL;
  ll_response *curr_response, *new_response = NULL;
  FILE *resp_file = NULL;
  char *in_line;
  char *readln = NULL, *walk = NULL, *walk2;
  int cnt = 0, k, need_new_resp = 0, counter;
  char out_bytes[BUFSIZE];

  in_line = (char *) malloc(BUFSIZE);

  if ((resp_file = open_amap_file(fnam, "response", RESP_EXT)) == NULL)
    exit_err("Error opening response file\n");

  while (fgets(in_line, BUFSIZE, resp_file) != NULL) {
    if (in_line[strlen(in_line) - 1] != '\n') {
      fprintf(stderr, "Error: %s\n", in_line);
      exit_err("Line is too long or not ending with \\n!\n");
    }
    if ((strstr(in_line, "DEFAULT:") != in_line) && (in_line[0] != '#') && (strstr(in_line, ":") != 0)) {	/* we have to read this line */
      if (cnt == 0) {
	curr_resp = (ll_resp *) malloc(sizeof(ll_resp));
	first_resp = curr_resp;
	cnt++;
      } else {
	curr_resp->next = malloc(sizeof(ll_resp));
	curr_resp = (ll_resp *) curr_resp->next;
	cnt++;
      }
      memset(curr_resp, 0, sizeof(ll_resp));
      curr_resp->next = NULL;	// NULL doesnt need to be 0x00000000
      need_new_resp = 2;
      if (in_line[strlen(in_line) - 1] == '\n')
	in_line[strlen(in_line) - 1] = '\0';
      readln = malloc(strlen(in_line) + 1);
      strcpy(readln, in_line);

      if ((walk = index(readln, ':')) == NULL) {
	fprintf(stderr, "Error: %s\n", in_line);
	exit_err("No ':' found in the line, invalid response definition.\n");
      }
      *walk = '\0';
      walk++;
      curr_resp->prot = malloc(strlen(readln) + 1);
      strcpy(curr_resp->prot, readln);
      if (verbose > 1)
	printf("Loading trigger information for %s ...\n", curr_resp->prot);

      counter = 0;

      while (walk != NULL) {
	if (need_new_resp) {
	  if (need_new_resp == 2) {
	    curr_response = malloc(sizeof(ll_response));
	    curr_resp->response = curr_response;
	  } else {
	    curr_response->next = malloc(sizeof(ll_response));
	    curr_response = (ll_response *) curr_response->next;
	  }
	  memset(curr_response, 0, sizeof(ll_response));
	  curr_response->next = NULL;
	} else
	  curr_response = (ll_response *) curr_response->next;

	if (need_new_resp != 2)
	  counter++;
	curr_response->counter = counter;

	if (curr_response->search_type == 0) {
	  switch (*walk) {
	  case '/':
	    curr_response->search_type = SUB_STR;
	    break;
	  case '^':
	    curr_response->search_type = START_STR;
	    break;
	  case '~':
	    curr_response->search_type = NLSTART_STR;
	    break;
	  default:
	    fprintf(stderr, "Error: %s\n", in_line);
	    exit_err("All responses must start with either ~, / or ^!\n");
	  }
	}
	walk++;

	walk2 = response_index(walk);
	if (walk2 != NULL) {
	  need_new_resp = 0;

	  curr_response->next = malloc(sizeof(ll_response));
	  new_response = (ll_response *) curr_response->next;
	  memset(new_response, 0, sizeof(ll_response));
	  new_response->next = NULL;

	  switch (*walk2) {
	  case '/':
	    new_response->search_type = SUB_STR;
	    break;
	  case '^':
	    fprintf(stderr, "Error: %s\n", in_line);
	    exit_err
	      ("The '^' search clause may only be used as first instance. Logical, isn't it?\n");
	    break;
	  case '~':
	    new_response->search_type = NLSTART_STR;
	    break;
	  default:
	    fprintf(stderr, "Error: %s\n", in_line);
	    exit_err("All responses must start with either ~, / or ^.\n");
	  }
	  *walk2 = 0;
	} else
	  need_new_resp = 1;

	switch (check_trig(walk)) {
	case TRIG_HEX:
	  walk += 2;		// skip "0x"
	  while (*walk == ' ')
	    walk++;
	  k = 0;
	  while (*walk != 0) {
	    out_bytes[k] = fromhex(*walk, walk[1]);
	    walk += 2;
	    k++;
	    while (*walk == ' ')
	      walk++;
	  }

	  if (curr_response->search_type == NLSTART_STR) {
	    curr_response->response_length = k + 1;
	    curr_response->response_string =
	      malloc(curr_response->response_length + 1);
	    *curr_response->response_string = '\n';
	    memcpy(curr_response->response_string + 1, out_bytes, k);
	  } else {
	    curr_response->response_length = k;
	    curr_response->response_string =
	      malloc(curr_response->response_length + 1);
	    memcpy(curr_response->response_string, out_bytes,
		   curr_response->response_length);
	  }

	  curr_response->response_string[curr_response->response_length] = 0;
	  break;
	case TRIG_STR:
	  walk++;		// skip first '"'
	  while (walk[strlen(walk) - 1] == ' ')
	    walk[strlen(walk) - 1] = 0;
	  if (walk[strlen(walk) - 1] != '"') {
	    fprintf(stderr, "Error: %s\n", in_line);
	    exit_err("Strings must start and end with an \"!\n");
	  }
	  walk[strlen(walk) - 1] = 0;
	  curr_response->response_length = strlen(walk);
	  curr_response->response_string =
	    malloc(curr_response->response_length + 1);
	  strcpy(curr_response->response_string, walk);
	  break;
	default:
	  fprintf(stderr, "Error: %s\n", in_line);
	  exit_err
	    ("All response strings must either start with \"0x\" or start and end with '\"' after the search definition.\n");
	}

	make_lower(curr_response->response_string,
		   curr_response->response_length);

	walk = walk2;
      }
      free(readln);
    }
  }
  free(in_line);
  if (cnt <= 1)
    exit_err
      ("No responses read. Response file empty? Come on, try to be a *real* hacker!\n");
  return (first_resp);
}

//--------------------------------------------

ll_triggers *read_triggers_rpc(char *fnam)
{
  ll_triggers *first_trig = NULL, *curr_trig = NULL;
  FILE *trigger_file = NULL;
  char *in_line;
  char *readln = NULL, *walk = NULL;
  int cnt = 0;

  in_line = (char *) malloc(BUFSIZE);

  if ((trigger_file = open_amap_file(fnam, "rpc", RPC_EXT)) == NULL)
    exit_err("Error opening rpc file\n");

  while (fgets(in_line, BUFSIZE, trigger_file) != NULL) {
    if (in_line[strlen(in_line) - 1] != '\n') {
      fprintf(stderr, "Error: %s\n", in_line);
      exit_err("Line is too long or not ending with \\n!\n");
    }
    if ((strstr(in_line, "DEFAULT:") != in_line) && (in_line[0] != '#') && (strstr(in_line, ":") != 0)) {	/* we have to read this line */
      if (cnt == 0) {
	curr_trig = (ll_triggers *) malloc(sizeof(ll_triggers));
	first_trig = curr_trig;
	cnt++;
      } else {
	(ll_triggers *) curr_trig->next =
	  (ll_triggers *) malloc(sizeof(ll_triggers));
	curr_trig = (ll_triggers *) curr_trig->next;
	cnt++;
      }
      memset(curr_trig, 0, sizeof(ll_triggers));
      curr_trig->next = NULL;
      if (in_line[strlen(in_line) - 1] == '\n')
	in_line[strlen(in_line) - 1] = '\0';
      delspaces(in_line);
      readln = malloc(strlen(in_line) + 1);
      strcpy(readln, in_line);
      if ((walk = index(readln, ':')) == NULL) {
	fprintf(stderr, "Error: %s\n", in_line);
	exit_err("No ':' found in the line, invalid rpc definition.\n");
      }
      *walk = '\0';
      walk++;
      curr_trig->prot = malloc(strlen(readln) + 1);
      strcpy(curr_trig->prot, readln);
      if (verbose > 1)
	printf("Loading rpc information for %s ...", curr_trig->prot);
      curr_trig->trigger_string = malloc(strlen(walk) + 1);
      strcpy(curr_trig->trigger_string, walk);
      curr_trig->trigger_length = strlen(walk);
      curr_trig->dangerous = 0;
      curr_trig->trigger_type = TRIG_RPC;
      curr_trig->ip_prot = 0;
      free(readln);
      if (verbose > 1)
	printf(" with value %s\n", curr_trig->trigger_string);
    }
  }
  free(in_line);
  if (cnt <= 1)
    exit_err
      ("No rpc data read. RPC file empty? Come on, try to be a *real* hacker!\n");
  return (first_trig);
}

//--------------------------------------------

ll_ports *scan_line_4_ports(char *m, char *scanstr)
{
  ll_ports *first_port = NULL, *this_port = NULL;
  int cnt = 0;
  char *found;

  while ((found = strstr(m, scanstr)) != NULL) {
    if (cnt == 0) {
      (ll_ports *) this_port = (ll_ports *) malloc(sizeof(ll_ports));
      first_port = this_port;
      cnt++;
    } else {
      (ll_ports *) this_port->next = (ll_ports *) malloc(sizeof(ll_ports));
      this_port = (ll_ports *) this_port->next;
      cnt++;
    }
    while (*--found != ' ' && *found != 0);
    this_port->port = (unsigned short int) atoi(found);
    this_port->appl_ids = (ll_appl_id *) malloc(sizeof(ll_appl_id));
    this_port->appl_ids->appl_id = (char *) malloc(MAX_PROTO_LEN);
    strcpy(this_port->appl_ids->appl_id, "unidentified");
    this_port->appl_ids->next = NULL;
    this_port->unknown_response_length = 0;
    this_port->skip = 0;
    this_port->ssl = 0;
    this_port->rpc = 0;
    while (*found != 0 && *found++ != '/');
    while (*found != 0 && *found++ != '/');
    this_port->ip_prot = *found;
    if (verbose > 1)
      printf("DEBUG nmap port: %d\n", this_port->port);
    if (this_port->ip_prot == 0 || this_port->port == 0) {
      fprintf(stderr, "Error: invalid nmap line - %s\n", scanstr);
      exit(-1);
    }
    m = found;
  }
  return (first_port);
}

//--------------------------------------------

ll_targets *read_targets(char *filename)
{
  ll_targets *this_tar = NULL, *first_targ = NULL;
  FILE *fp_in = NULL;
  char *in_line;
  char *l = NULL, *m = NULL;
  char *walk = NULL;
  char scanstr[12];
  int k = 0, count_targets = 0;

/*This whole routine RELIES on the nmap format output. if that is broken, so is this. 
  Also, it assumes that nmap lines won't be longer that 4096 chars.*/

  in_line = (char *) malloc(BUFSIZE);

  strcpy(scanstr, "open/");
  if (scantype == 't')
    strcpy(scanstr, "open/tcp");
  if (scantype == 'u')
    strcpy(scanstr, "open/udp");
  if ((fp_in = fopen(filename, "r")) == NULL)
    return (0);

  while ((fgets(in_line, BUFSIZE, fp_in)) != NULL) {
    /* parse the line */
    l = in_line;
    k = 0;
    in_line[strlen(in_line) - 1] = '\0';
    if ((in_line[0] != '#') && (strstr(in_line, scanstr) != NULL)) {	/* skip lines starting with # and lines with no open ports */
      if (count_targets == 0) {
	this_tar = (ll_targets *) malloc(sizeof(ll_targets));
	first_targ = this_tar;
	count_targets = 1;
      } else {
	(ll_targets *) this_tar->next =
	  (ll_targets *) malloc(sizeof(ll_targets));
	this_tar = (ll_targets *) this_tar->next;
	count_targets++;
      }
      if ((walk = index(in_line, ' ')) == NULL) {
	fprintf(stderr, "Error: invalid line in nmap file - %s\n", in_line);
	exit(-1);
      }
      walk++;
      if ((m = index(walk, ' ')) == NULL) {
	fprintf(stderr, "Error: invalid line in nmap file - %s\n", in_line);
	exit(-1);
      }
      *m++ = '\0';
      this_tar->target = (char *) malloc(strlen(walk) + 1);
      strcpy(this_tar->target, walk);
      if (verbose > 1)
	fprintf(stderr, "DEBUG nmap target: %s\n", this_tar->target);
      /* Go find the open ports */
      this_tar->ports = scan_line_4_ports(m, scanstr);
      if (this_tar->ports == NULL)
	warn("huh? nmap file might be invalid");
    }
  }
  free(in_line);
  if (count_targets == 0)
    exit_err
      ("invalid nmap input file, must be created with the -oM option! *sigh*");
  return (first_targ);
}

//--------------------------------------------

int compile_line(char *trig, unsigned char *out_bytes, int type, char proto)
{
  int i;
  int j;
  int k;
  long int *ul = (long int *) out_bytes;
  char temp1[BUFSIZE];
  char *temp2 = NULL;

  memset(out_bytes, 0, BUFSIZE);
  j = strlen(trig);

  switch (type) {
  case TRIG_RPC:
// bei tcp: 80 0 0 28 muss noch davor
    if (proto == 't') {
      out_bytes[0] = 128;
      out_bytes[3] = 40;
      temp2 = out_bytes + 4;
      j = 4;
    } else {
      temp2 = out_bytes;
      j = 0;
    }

    ul = (long int *) temp2;
    *ul = htonl(strtol(trig, (char **) NULL, 10));
    ul = (long int *) (temp2 + 12);
    temp2[11] = 2;
    *ul = htonl(strtol(trig, (char **) NULL, 10));
    temp2[17] = 7;
    temp2[18] = 120;
    temp2[19] = 74;
    return (40 + j);
    break;
  case TRIG_STR:
    strcpy(temp1, trig);
    temp2 = rindex(temp1, '"');
    *temp2 = '\0';
    temp2 = index(temp1, '"');
    temp2++;
    trans_esc(temp2);
    j = sprintf(out_bytes, temp2, "");
    return (j);
    break;
  case TRIG_HEX:
    k = 0;
    for (i = 2; i < j - 1; i += 2) {
      out_bytes[k] = fromhex(trig[i], trig[i + 1]);
      k++;
    }
    return ((j - 2) / 2);
    break;
  default:
    exit_err("Memory seems to be corrupted (1)!\n");
  }
  return 0;			// not reached
}

//--------------------------------------------

RSA *ssl_temp_rsa_cb(SSL * ssl, int export, int keylength)
{
  RSA *rsa = NULL;

  if (rsa == NULL)
    rsa = RSA_generate_key(512, RSA_F4, NULL, NULL);
  return rsa;
}

//--------------------------------------------

int amap_conn(sock_block * babble)
{
  int ssl_err;
  struct sockaddr_in my_tar;
  struct in_addr addie;
  int p, t, s = -1, ret;

  SSL *ssl = NULL;
  SSL_CTX *sslContext = NULL;

  if (babble->prot == 't')
    p = SOCK_STREAM;
  else
    p = SOCK_DGRAM;
  if (babble->prot == 't')
    t = IPPROTO_TCP;
  else
    t = IPPROTO_UDP;

  if (babble->ssl_enabled == 1) {
    SSL_load_error_strings();
    SSLeay_add_ssl_algorithms();

    // context: ssl2 + ssl3 is allowed, whatever the server demands
    if ((sslContext = SSL_CTX_new(SSLv23_method())) == NULL) {
      if (verbose) {
	ssl_err = ERR_get_error();
	exit_err(ERR_error_string(ssl_err, NULL));
      }
    }
    // set the compatbility mode
    SSL_CTX_set_options(sslContext, SSL_OP_ALL);

    // we set the default verifiers and dont care for the results
    (void) SSL_CTX_set_default_verify_paths(sslContext);
    SSL_CTX_set_tmp_rsa_callback(sslContext, ssl_temp_rsa_cb);
    SSL_CTX_set_verify(sslContext, SSL_VERIFY_NONE, NULL);
  }

  if ((s = socket(AF_INET, p, t)) == -1) {
    warn("socket creation failed");
    return (-9999);
  }

  my_tar.sin_port = htons((unsigned short int) babble->port);
  my_tar.sin_family = AF_INET;
  if (inet_pton(AF_INET, babble->ip_ad, &addie) <= 0) {
    warn("inet_pton() failed");
    return (-9999);
  }
  my_tar.sin_addr.s_addr = addie.s_addr;

  ret = -1;
  fail = 0;

  while ((fail < MAX_RETRIES) && (ret < 0)) {
    if ((ret = connect(s, (struct sockaddr *) &my_tar, sizeof(my_tar))) == 0) {
      if (babble->ssl_enabled == 1) {
	if ((ssl = SSL_new(sslContext)) == NULL)
	  exit_err("Could not prepare SSL context");
	SSL_set_fd(ssl, s);
	if (SSL_connect(ssl) < 0) {
	  babble->ssl_enabled = 0;
	  // dump this connection and reconnect
	  close(s);
	  ret = -1;
	}
      }
    }

    if (ret < 0) {
      fail++;
      if (fail >= MAX_RETRIES) {
	close(s);
	return -1;
      }
      sleep(1);
    }
  }

  if (verbose > 1 && babble->ssl_enabled)
    printf("Connected to %s port %d using SSL\n", babble->ip_ad,
	   babble->port);
  else if (verbose > 1)
    printf("Connected to %s port %d\n", babble->ip_ad, babble->port);

  babble->s = s;

  if (babble->ssl_enabled)
    babble->ssl_sock = ssl;

// make this socket non-blocking! 
  if (fcntl(s, F_SETFL, O_NONBLOCK) < 0)
    fprintf(stderr, "fcntl error \n");

  return (s);
}

//--------------------------------------------

void dump_r(sock_block * babble)
{
  int i;
  int rows;
  int lastrow;
  unsigned char lastrow_data[16];
  unsigned char *q = (char *) babble->orig_response;

  printf("Response received from %s port %d %s (length %d bytes):\n",
	 babble->ip_ad, babble->port, babble->prot == 't' ? "tcp" : "udp",
	 babble->response_length);
  rows = babble->response_length / 16;
  lastrow = babble->response_length - (rows * 16);

  fflush(stdout);
  for (i = 0; i < rows; i++) {
    printf("%04hx:\t", i * 16);
    printf
      ("%02x%02x %02x%02x %02x%02x %02x%02x %02x%02x %02x%02x %02x%02x %02x%02x\n",
       q[(i * 16)], q[(i * 16) + 1], q[(i * 16) + 2], q[(i * 16) + 3],
       q[(i * 16) + 4], q[(i * 16) + 5], q[(i * 16) + 6], q[(i * 16) + 7],
       q[(i * 16) + 8], q[(i * 16) + 9], q[(i * 16) + 10], q[(i * 16) + 11],
       q[(i * 16) + 12], q[(i * 16) + 13], q[(i * 16) + 14],
       q[(i * 16) + 15]);
  }
  if (lastrow > 0) {
    memset(lastrow_data, 0, sizeof(lastrow_data));
    memcpy(lastrow_data, q + babble->response_length - lastrow, lastrow);
    printf("%04hx:\t", rows * 16);
    for (i = 0; i < lastrow; i++) {
      printf("%02x", lastrow_data[i]);
      if (i % 2 == 1)
	printf(" ");
    }
    printf("\n");
  }
  printf("ASCII:  \"%s\"\n",
	 print_banner_string(babble->orig_response, babble->response_length));
}

//--------------------------------------------

int match_response(ll_response * response, char *response_string,
		   int response_length)
{
  ll_response *resp;
  char *ptr;

  if (response_length < response->response_length)
    return 0;

  switch (response->search_type) {
  case SUB_STR:
    if ((ptr = amap_memstr
	 (response_string, response->response_string, response_length,
	  response->response_length)) != NULL) {
      if ((resp = (ll_response *) response->next) == NULL)
	return 1;
      else
	return (match_response
		(resp, ptr + response->response_length,
		 response_length - response->response_length));
    } else
      return 0;
    break;
  case START_STR:
    if (memcmp
	(response_string, response->response_string,
	 response->response_length) == 0) {
      if ((resp = (ll_response *) response->next) == NULL)
	return 1;
      else
	return (match_response
		(resp, response_string + response->response_length,
		 response_length - response->response_length));
    } else
      return 0;
    break;
  case NLSTART_STR:
    if (response->counter == 0) {
      if (memcmp
	  (response_string, response->response_string + 1,
	   response->response_length - 1) == 0) {
	if ((resp = (ll_response *) response->next) == NULL)
	  return 1;
	else
	  return (match_response
		  (resp, response_string + response->response_length,
		   response_length - response->response_length));
      }
    }
    if ((ptr = amap_memstr
	 (response_string, response->response_string, response_length,
	  response->response_length)) != NULL) {
      if ((resp = (ll_response *) response->next) == NULL)
	return 1;
      else
	return (match_response
		(resp, ptr + response->response_length,
		 response_length - response->response_length));
    } else
      return 0;
    break;
  default:
    exit_err("Memory seems to be corrupted (1)\n");
  }
  return 0;			// not reached
}

//--------------------------------------------

void add_appl_id(ll_appl_id * appl, char *str)
{
  ll_appl_id *app = appl;

  if (strcmp(app->appl_id, "unidentified") == 0) {
    strcpy(app->appl_id, str);
    return;
  }
  while (app->next != NULL)
    app = (ll_appl_id *) app->next;
  (ll_appl_id *) app->next = (ll_appl_id *) malloc(sizeof(ll_appl_id));
  app = (ll_appl_id *) app->next;
  app->appl_id = malloc(MAX_PROTO_LEN);
  app->next = NULL;
  strcpy(app->appl_id, str);
}

//--------------------------------------------

int lookup_appl_id(ll_appl_id * appl, char *str, int len)
{
  ll_appl_id *app = appl;

  if (len == 0 || strcmp(app->appl_id, "unidentified") == 0)
    return 0;
  while (app != NULL) {
    if (len > 0) {
      if (strncmp(app->appl_id, str, len) == 0)
	return 1;
    } else {
      if (strcmp(app->appl_id, str) == 0)
	return 1;
    }
    app = (ll_appl_id *) app->next;
  }
  return 0;
}

//--------------------------------------------

void lookup_response(sock_block * babble, FILE * outp_file,
		     ll_targets * tar_lst)
{
  int gotit;
  ll_resp *my_resp;
  ll_response *my_response;
  ll_ports *mapped_port;
  ll_appl_id *app;
  char *orig_response;

//  char *str1, *str2;
  int gogo;

  if (mode != SCAN_RPC) {
    babble->orig_response = malloc(babble->response_length);
    memcpy(babble->orig_response, babble->response, babble->response_length);
    orig_response = babble->orig_response;

    my_resp = first_resp;

    gotit = 0;
    mapped_port = babble->port_ptr;

    if (strncmp(mapped_port->appl_ids->appl_id, "echo", 4) != 0) {
      while (my_resp != NULL) {
	my_response = my_resp->response;

	make_lower(babble->response, babble->response_length);
	if ((gogo =
	     match_response(my_response, babble->response,
			    babble->response_length))
	  ) {
	  gotit++;
	  gogo = 0;
	  app = mapped_port->appl_ids;
/// XXX CHange in list search
	  if (lookup_appl_id((ll_appl_id *) app, my_resp->prot, -1))
	    gogo = -1;

// we dont need the following anymore

/*
*	  else {
*	    if (strlen(mapped_port->appl_ids.appl_id) > strlen(my_resp->prot)) {
*	      str1 = malloc(strlen(my_resp->prot) + 2);
*	      strcpy(str1, my_resp->prot);
*	      strcat(str1, "-");
*	      if (lookup_appl_id(mapped_port->appl_ids, str1, strlen(str1))
*		gogo = -1;
*	    }
*	     else {
*	      str2 = malloc(strlen(my_resp->prot) + 2);
*	      strcpy(str2, my_resp->prot);
*	      str1 = malloc(strlen(mapped_port->appl_ids.appl_id) + 1);
*	      strcpy(str1, mapped_port->appl_ids.appl_id);
*	      strcat(str1, "-");
*	      if (strncmp(str1, str2, strlen(str1)) == 0)
*		gogo = 1;
*	    }
*	    free(str1);
*	    free(str2);
*	  }
*/
	  if ((strcasecmp(my_resp->prot, "rpc") == 0
	       || strncasecmp(my_resp->prot, "rpc-", 4) == 0))
	    mapped_port->rpc = 1;

	  if (mode == SCAN_PLAIN && mapped_port->ip_prot == 't'
	      && (strcasecmp(my_resp->prot, "ssl") == 0
		  || strncasecmp(my_resp->prot, "ssl-", 4) == 0))
	    mapped_port->ssl = 1;

	  if (gogo >= 0) {
//if (gogo == 1 || (strcmp(mapped_port->appl_ids.appl_id, "unidentified")) == 0)
	    add_appl_id(app, my_resp->prot);
// XXX Change: add to appl_ids list! IMMER!!! also if() raus.
//            strcpy(mapped_port->appl_ids->appl_id, my_resp->prot);
	    if (log_to_file) {
	      if (machine_readable) {
		fprintf(outp_file, "%s:%d:%s:%s:%s:%s:", babble->ip_ad,
			babble->port, babble->prot == 't' ? "tcp" : "udp",
			babble->ssl_enabled == 1 ? "SSL" : "", my_resp->prot,
			print_banner_string(orig_response,
					    babble->response_length));
		write_banner_string(outp_file, orig_response,
				    babble->response_length);
		fprintf(outp_file, "\n");
	      } else
		fprintf(outp_file,
			"Protocol on IP %s port %d %s matches %s%s%s\n",
			babble->ip_ad, babble->port,
			babble->prot == 't' ? "tcp" : "udp", my_resp->prot,
			babble->ssl_enabled == 1 ? " over SSL" : " ",
			print_banner(orig_response, babble->response_length));
	    }
	    if (dump)
	      dump_r(babble);
	    printf("Protocol on IP %s port %d %s matches %s%s%s\n",
		   babble->ip_ad, babble->port,
		   babble->prot == 't' ? "tcp" : "udp", my_resp->prot,
		   babble->ssl_enabled == 1 ? " over SSL" : " ",
		   print_banner(orig_response, babble->response_length));
	  }
	}
	my_resp = (ll_resp *) my_resp->next;
      }
      if (gotit == 0 && babble->response_length == 4
	  && strcmp(mapped_port->appl_ids->appl_id, "unidentified") == 0) {
	gotit++;
	strcpy(mapped_port->appl_ids->appl_id, time_string);
	if (log_to_file) {
	  if (machine_readable) {
	    fprintf(outp_file, "%s:%d:%s:%s:%s:%s:", babble->ip_ad,
		    babble->port, babble->prot == 't' ? "tcp" : "udp",
		    babble->ssl_enabled == 1 ? "SSL" : "",
		    mapped_port->appl_ids->appl_id,
		    print_banner_string(orig_response,
					babble->response_length));
	    write_banner_string(outp_file, orig_response,
				babble->response_length);
	    fprintf(outp_file, "\n");
	  } else
	    fprintf(outp_file,
		    "Protocol on IP %s port %d %s matches %s%s%s\n",
		    babble->ip_ad, babble->port,
		    babble->prot == 't' ? "tcp" : "udp",
		    mapped_port->appl_ids->appl_id,
		    babble->ssl_enabled == 1 ? " over SSL" : " ",
		    print_banner(orig_response, babble->response_length));
	}
	if (dump)
	  dump_r(babble);
	printf("Protocol on IP %s port %d %s matches %s%s%s\n",
	       babble->ip_ad, babble->port,
	       babble->prot == 't' ? "tcp" : "udp",
	       mapped_port->appl_ids->appl_id,
	       babble->ssl_enabled == 1 ? " over SSL" : " ",
	       print_banner(orig_response, babble->response_length));
      }
    }

    if (gotit == 0 && unrec == 1 && babble->response_length > 0 &&
	strcmp(mapped_port->appl_ids->appl_id, "echo") != 0 &&
	(mapped_port->unknown_response_length == 0 ||
	 mapped_port->unknown_response_length != babble->response_length ||
	 memcmp(mapped_port->unknown_response, babble->response,
		babble->response_length) != 0)
      ) {
      if (unrec == 1) {
	printf
	  ("\nI received an unrecognized response from %s %s port %d%s Please send us this and the application name + version to %s\n",
	   babble->ip_ad, babble->prot == 't' ? "tcp" : "udp", babble->port,
	   babble->ssl_enabled == 1 ? " when connecting with SSL." : ".",
	   EMAIL);
	dump_r(babble);
      }
      mapped_port->unknown_response_length = babble->response_length;
      mapped_port->unknown_response = malloc(babble->response_length + 1);
      memset(mapped_port->unknown_response, 0, babble->response_length + 1);
      memcpy(mapped_port->unknown_response, babble->orig_response,
	     babble->response_length);
    }
  } else {
    if (babble->response_length == 32 || babble->response_length == 36) {
      char rpcname[128] = "rpc-";
      int i;

      strcat(rpcname, babble->trigger_ptr->prot);
      strcat(rpcname, "-v");
      i = strlen(rpcname);
      rpcname[i] = (char) babble->response[babble->response_length - 1] + 48;
      rpcname[i + 1] = 0;
      babble->port_ptr->skip = 1;
      strcpy(babble->port_ptr->appl_ids->appl_id, rpcname);
      printf("Protocol on IP %s port %d %s matches %s%s\n",
	     babble->ip_ad, babble->port,
	     babble->prot == 't' ? "tcp" : "udp", rpcname,
	     babble->ssl_enabled == 1 ? " over SSL" : "");
      if (log_to_file) {
	if (machine_readable) {
	  fprintf(outp_file, "%s:%d:%s:%s:%s::\n", babble->ip_ad,
		  babble->port, babble->prot == 't' ? "tcp" : "udp",
		  babble->ssl_enabled == 1 ? "SSL" : "", rpcname);
	} else
	  fprintf(outp_file,
		  "Protocol on IP %s port %d %s matches %s%s%s\n",
		  babble->ip_ad, babble->port,
		  babble->prot == 't' ? "tcp" : "udp", rpcname,
		  babble->ssl_enabled == 1 ? " over SSL" : " ",
		  print_banner(babble->response, babble->response_length));
      }
    }
  }
}

//--------------------------------------------

void read_response(sock_block * babble, ll_targets * s_target,
		   ll_targets * targ_todo, ll_ports * port_todo,
		   FILE * outp_file)
{
  int i;
  int rlen;
  char rline[BUFSIZE];
  time_t t;

  for (i = 0; i < tasks; i++) {

    // initialise these vars each time!!!!!
    memset(rline, 0, sizeof(rline));
    rlen = -1;

    if (babble[i].active == 1) {	// OK, have something for us yet?
      // if we have a few tasks only, fast networks will fuck us up
      if (tasks < TASKS)
	sleep(1);
      errno = 0;
      if (babble[i].ssl_enabled) {
	rlen = SSL_read(babble[i].ssl_sock, rline, sizeof(rline));
      } else {
	rlen = recv(babble[i].s, rline, sizeof(rline), 0);
      }
      if (rlen > 0) {		// apparently, we received some shit!
	// set some stuff
	resp_recv++;
	babble[i].response_length = rlen;
	memcpy(&(babble[i].response), rline, rlen);
	babble[i].response[rlen] = '\0';
	// Lookup the response and print
	lookup_response(&(babble[i]), outp_file, s_target);
	// OK, this socket is done, free it up for new use.
	close(babble[i].s);
	running_tasks--;
	clear_block(&(babble[i]));
      } else {
	// check to see it this socket hasn't outlived it's poor little life
	if (errno == ECONNREFUSED && babble[i].prot == 'u') {
	  ll_targets *tmpt = babble[i].target_ptr;
	  ll_ports *tmpp = babble[i].port_ptr;

	  printf
	    ("Couldn't connect to %s port %d on %s. Service crashed after scanning?\n",
	     tmpp->ip_prot == 't' ? "tcp" : "udp", tmpp->port, tmpt->target);
	  tmpp->skip = 1;
	}
	t = time(NULL);
	if ((t - babble[i].timer) > timeout_t) {
	  // Bye bye, you're history!
	  if (verbose > 1)
	    printf("No response received from %s port %d on %s.\n",
		   port_todo->ip_prot == 't' ? "tcp" : "udp",
		   port_todo->port, targ_todo->target);
	  running_tasks--;
	  close(babble[i].s);
	  clear_block(&(babble[i]));
	}
      }
    }
  }
}

//--------------------------------------------

int do_scan(int scanmode, FILE * outp_file, ll_targets * s_target,
	    ll_triggers * s_trigger)
{

  ll_targets *targ_todo;
  ll_ports *port_todo;
  ll_triggers *trigger_todo;

  sock_block babble[MAX_TASKS];
  int i;
  int len_t;
  int ready_for_next;
  char compiled_t[BUFSIZE];

  mode = scanmode;

/* initialise all socket blocks */
  for (i = 0; i < MAX_TASKS; i++)
    clear_block(&(babble[i]));

/* Count tasks */
  targ_todo = s_target;
  total_tasks = 0;

  while (targ_todo != NULL) {
    if (verbose > 1)
      printf("Target: %s - ", targ_todo->target);
    port_todo = (ll_ports *) targ_todo->ports;
    while (port_todo != NULL) {
      if (verbose > 1)
	printf("%d ", port_todo->port);
      trigger_todo = s_trigger;
      while (trigger_todo != NULL
	     && (mode == SCAN_PLAIN ||
		 (mode == SCAN_SSL && port_todo->ssl) ||
		 (mode == SCAN_RPC && port_todo->rpc)
	     )
	) {
	/* now see if this task is appropriate */
	if (!
	    (((trigger_todo->ip_prot == 't') && (port_todo->ip_prot == 'u'))
	     || ((trigger_todo->ip_prot == 'u')
		 && (port_todo->ip_prot == 't'))))
	  total_tasks++;
	trigger_todo = (ll_triggers *) trigger_todo->next;
      }
      if ((mode == SCAN_SSL && port_todo->ssl)
	  || (mode == SCAN_RPC && port_todo->rpc)) {
	strcpy(port_todo->appl_ids->appl_id, "unidentified");
	port_todo->appl_ids->next = NULL;
      }
      port_todo = (ll_ports *) port_todo->next;
    }
    targ_todo = (ll_targets *) targ_todo->next;
    if (verbose > 1)
      printf("\n");
  }

  if (verbose) {
    printf("Total amount of tasks to perform in ");
    switch (mode) {
    case SCAN_PLAIN:
      printf("plain connect");
      break;
    case SCAN_SSL:
      printf("SSL connect");
      break;
    case SCAN_RPC:
      printf("RPC connect");
      break;
    default:
      exit_err("unknown scan connect mode");
    }
    printf(" mode: %d\n", total_tasks);
  }
  if (total_tasks < 1)
    return -1;
  if (total_tasks < tasks)
    tasks = total_tasks;

/* Here comes the parallelising using non-blocking sockets */

/* again, loop through the tasks, but now not for counting, but for
   execution  */

  trigger_todo = s_trigger;

  while (trigger_todo != NULL) {
    targ_todo = s_target;
    while (targ_todo != NULL) {
      port_todo = (ll_ports *) targ_todo->ports;
      while (port_todo != NULL) {
	/* now see if this task is appropriate */
	if (port_todo->skip != 1 &&
	    (mode == SCAN_PLAIN ||
	     (mode == SCAN_SSL && port_todo->ssl) ||
	     (mode == SCAN_RPC && port_todo->rpc)
	    ) &&
	    (one_is_enough == 0
	     || strcmp(port_todo->appl_ids->appl_id, "unidentified") == 0)
	    &&
	    (!((trigger_todo->ip_prot == 't') && (port_todo->ip_prot == 'u'))
	     || ((trigger_todo->ip_prot == 'u')
		 && (port_todo->ip_prot == 't')))) {
	  /* Main loop */
	  // Check for available sock_block
	  ready_for_next = 0;
	  for (i = 0; i < tasks; i++) {
	    if (babble[i].active == 0) {
	      ready_for_next = 1;
	      break;
	    }
	  }
	  //wait for sock_block to become available
	  while (ready_for_next == 0) {
	    read_response(babble, s_target, targ_todo, port_todo, outp_file);
	    // re-check to see if block has become available
	    for (i = 0; i < tasks; i++) {
	      if (babble[i].active == 0) {
		ready_for_next = 1;	// this one is free
		break;
	      }
	    }
	  }
	  // then fill it in
	  clear_block(&(babble[i]));	// shouldn't be necessary, but why not?
	  strcpy(babble[i].ip_ad, targ_todo->target);
	  babble[i].port = port_todo->port;
	  babble[i].prot = port_todo->ip_prot;
	  babble[i].trigger_ptr = (ll_triggers *) trigger_todo;
	  babble[i].port_ptr = (ll_ports *) port_todo;
	  babble[i].target_ptr = (ll_targets *) targ_todo;

	  if (mode != SCAN_PLAIN && port_todo->ssl)
	    babble[i].ssl_enabled = 1;

	  switch (amap_conn(&(babble[i]))) {
	  case -9999:
	    break;
	  case -1:
	    printf
	      ("Couldn't connect to %s port %d on %s. Service crashed after scanning?\n",
	       port_todo->ip_prot == 't' ? "tcp" : "udp", port_todo->port,
	       targ_todo->target);
	    port_todo->skip = 1;
	    break;
	  default:
	    babble[i].active = 1;
	    running_tasks++;
	    babble[i].timer = time(NULL);
	    len_t =
	      compile_line(trigger_todo->trigger_string, compiled_t,
			   trigger_todo->trigger_type, babble[i].prot);
	    if (babble[i].ssl_enabled) {
	      SSL_write(babble[i].ssl_sock, compiled_t, len_t);
	    } else {
	      write(babble[i].s, compiled_t, len_t);
	    }
	  }

	  // and read responses, but wait for a small amount of time for responses to become available!!
	  read_response(babble, s_target, targ_todo, port_todo, outp_file);
	}			// end of main loop
	port_todo = (ll_ports *) port_todo->next;
      }
      targ_todo = (ll_targets *) targ_todo->next;
    }
    trigger_todo = (ll_triggers *) trigger_todo->next;
  }

  if (verbose)
    printf("%d responses received in total for %d tasks.\n", resp_recv,
	   total_tasks);

  return 0;
}

//--------------------------------------------

int do_scan_wrapper(ll_targets * s_target, ll_triggers * s_trigger,
		    ll_triggers * r_trigger, char *logfile)
{
  ll_targets *targ_todo;
  ll_ports *port_todo;
  struct stat tmpstat;
  FILE *outp_file = NULL;
  char old_logfile[256];
  char *today = NULL;
  time_t t;
  int up;

  t = time(NULL);
  today = ctime(&t);
  today[strlen(today) - 1] = '\0';

  if (logfile != NULL) {
    log_to_file = 1;
    if (stat(logfile, &tmpstat) == 0) {
      warn("output file already exists. Moving to <logfile>.old ");
      strcpy(old_logfile, logfile);
      strcat(old_logfile, ".old");
      rename(logfile, old_logfile);
    }
    if ((outp_file = fopen(logfile, "w")) == 0)
      exit_err("Couldn't open outputfile for writing\n");
  }

  if (log_to_file) {
    (void) setvbuf(outp_file, NULL, _IONBF, 0);
    if (machine_readable)
      fprintf(outp_file, "# ");
    fprintf(outp_file, "%s %s started at %s\n", PROGRAM, VERSION, today);
    if (machine_readable)
      fprintf(outp_file,
	      "# IPADDRESS:PORT:PROTOCOL:SSL:IDENTIFICATION:PRINTABLE_BANNER:FULL_BANNER\n");
  }

  printf("%s %s started at %s, stand back and keep children away\n",
	 PROGRAM, VERSION, today);

  if (do_scan(SCAN_PLAIN, outp_file, s_target, s_trigger) < 0)
    exit_err("Nothing to do. Quitting. Why did you wake me up? Bastard.\n");
  if (SSL_reconnect)
    do_scan(SCAN_SSL, outp_file, s_target, s_trigger);
  if (RPC_reconnect)
    do_scan(SCAN_RPC, outp_file, s_target, r_trigger);

  targ_todo = s_target;
  up = 0;
  printf("Unidentified ports: ");
  if (log_to_file && !machine_readable)
    fprintf(outp_file, "Unidentified ports: ");
  while (targ_todo != NULL) {
    port_todo = (ll_ports *) targ_todo->ports;
    while (port_todo != NULL) {
      if (strcmp(port_todo->appl_ids->appl_id, "unidentified") == 0) {
	up++;
	printf("%d/%s ", port_todo->port,
	       port_todo->ip_prot == 't' ? "tcp" : "udp");
	if (log_to_file) {
	  if (machine_readable) {
	    fprintf(outp_file, "%s:%d:%s::unidentified:%s:",
		    targ_todo->target, port_todo->port,
		    port_todo->ip_prot == 't' ? "tcp" : "udp",
		    print_banner_string(port_todo->unknown_response,
					port_todo->unknown_response_length));
	    write_banner_string(outp_file, port_todo->unknown_response,
				port_todo->unknown_response_length);
	    fprintf(outp_file, "\n");
	  } else
	    fprintf(outp_file, "%d/%s ", port_todo->port,
		    port_todo->ip_prot == 't' ? "tcp" : "udp");
	}
      }
      port_todo = (ll_ports *) port_todo->next;
    }
    targ_todo = (ll_targets *) targ_todo->next;
  }

  if (up == 0) {
    if (log_to_file && !machine_readable)
      fprintf(outp_file, "none.\n");
    printf("none.\n");
  } else {
    if (log_to_file && !machine_readable)
      fprintf(outp_file, "(total %d).\n", up);
    printf("(total %d).\n", up);
  }

/* All done, clean up */

  t = time(NULL);
  today = ctime(&t);
  today[strlen(today) - 1] = '\0';

  printf("%s %s ended at %s\n", PROGRAM, VERSION, today);
  if (log_to_file) {
    if (machine_readable)
      fprintf(outp_file, "# ");
    fprintf(outp_file, "%s %s ended at %s\n", PROGRAM, VERSION, today);
    fclose(outp_file);
  }

  return 0;
}

//--------------------------------------------

int main(int argc, char *argv[])
{
  int i;
  int have_targetsfile = 0;
  int count_targets = 0;
  ll_targets *first_targ = NULL;
  ll_ports *first_port;
  ll_triggers *first_trig, *rpc_trig = NULL;
  struct in_addr in;
  char out[16];

  char *nmap_infile = NULL, *output_file = NULL;	// otherwise logfiles with weird filenames will be created ...
  char *deffile = NULL;

  memset(banner_string, 0, sizeof(banner_string));

  (void) setvbuf(stdout, NULL, _IONBF, 0);

  while ((i = getopt(argc, argv, "1SRbuvdhmHi:s:T:p:o:D:t:U")) >= 0) {
    switch (i) {
    case '1':
      one_is_enough = 1;
      break;
    case 'd':
      dump = 1;
      break;
    case 'v':
      verbose++;
      break;
    case 'S':
      SSL_reconnect = 0;
      break;
    case 'R':
      RPC_reconnect = 0;
      break;
    case 'b':
      banner = 1;
      break;
    case 'm':
      machine_readable = 1;
      break;
    case 'H':
      harmful = 1;
      break;
    case 'u':
      unrec = 0;
      break;
    case 'h':
      help();
      break;
    case 'T':
      tasks = atoi(optarg);
      break;
    case 'i':
      nmap_infile = optarg;
      have_targetsfile = 1;
      break;
    case 'U':
      scantype = 'u';
      break;
    case 's':
      scantype = tolower(*optarg);
      break;
    case 'p':
      only_prot = optarg;
      break;
    case 'o':
      output_file = optarg;
      break;
    case 'D':
      deffile = optarg;
      break;
    case 't':
      timeout_t = atoi(optarg);
      break;
    default:
      fprintf(stderr, "Error: unknown option -%c\n", i);
      help();
    }
  }

  if (machine_readable && output_file == NULL)
    exit_err("Option -m specified, however not logfile (with -o FILE)\n");

  if (tasks > MAX_TASKS)
    exit_err
      ("Too many tasks man, lighten up! Ever *tried* to read the help output, hmm?\n");

  if ((optind + 2 > argc) && (have_targetsfile != 1))
    help();

  if (optind + 2 <= argc) {
    if (inet_addr(argv[optind]) == -1) {
      struct hostent *target;

      if ((target = gethostbyname(argv[optind])) == NULL)
	exit_err("Error resolving target, ever heard of DNS and IP?\n");
      memcpy(&in, target->h_addr, target->h_length);
      //argv[optind] = inet_ntoa(in); //deprecated
      argv[optind] =
	(char *) inet_ntop(AF_INET, &in, (char *) &out, sizeof(out));
    }
    if ((first_targ = (ll_targets *) malloc(sizeof(ll_targets))) == NULL)
      exit_err("Malloc failed\n");
    if ((first_port = (ll_ports *) malloc(sizeof(ll_ports))) == NULL)
      exit_err("Malloc failed\n");
    (ll_ports *) first_targ->ports = first_port;
    first_targ->target = malloc(strlen(argv[optind]) + 1);
    strcpy(first_targ->target, argv[optind]);
    first_targ->next = NULL;
    first_port->port = atoi(argv[optind + 1]);
    if (scantype == 'b')
      exit_err
	("You want me to *guess* whether you mean tcp or udp? Come on, give me at least a -sT or -sU....");
    first_port->ip_prot = scantype;
    first_port->unknown_response_length = 0;
    first_port->appl_ids = (ll_appl_id *) malloc(sizeof(ll_appl_id));
    first_port->appl_ids->appl_id = malloc(MAX_PROTO_LEN);
    strcpy(first_port->appl_ids->appl_id, "unidentified");
    first_port->appl_ids->next = NULL;
    first_port->skip = 0;
    first_port->ssl = 0;
    first_port->rpc = 0;
    first_port->next = NULL;
    count_targets = 1;

    if (first_port->port <= 0)
      exit_err
	("Thats not a valid port which you defined on the command line!");

    i = optind + 2;

    while (i < argc) {
      (ll_ports *) first_port->next = (ll_ports *) malloc(sizeof(ll_ports));
      first_port = (ll_ports *) first_port->next;
      first_port->port = (unsigned short int) atoi(argv[i]);
      first_port->ip_prot = scantype;
      first_port->appl_ids = (ll_appl_id *) malloc(sizeof(ll_appl_id));
      first_port->appl_ids->appl_id = (char *) malloc(MAX_PROTO_LEN);
      strcpy(first_port->appl_ids->appl_id, "unidentified");
      first_port->appl_ids->next = NULL;
      first_port->unknown_response_length = 0;
      first_port->ssl = 0;
      first_port->rpc = 0;
      first_port->skip = 0;
      first_port->next = NULL;
      if (first_port->port <= 0)
	exit_err
	  ("Thats not a valid port which you defined on the command line!");
      i++;
    }
  }

  if (have_targetsfile == 1) {
    /* Read in the targets file */
    if (count_targets == 1) {
      if (((ll_targets *) first_targ->next =
	   (ll_targets *) read_targets(nmap_infile)) == NULL)
	exit_err
	  ("No targets read. Empty or maybe incorrect protocol type? Anyway, I'm going home.\n");
    } else if ((first_targ = read_targets(nmap_infile)) == NULL)
      exit_err
	("No targets read. Empty or maybe incorrect protocol type? Anyway, I'm going home.\n");
  }

  /* Read in the triggers and the responses */
  if ((first_trig = read_triggers(deffile)) == 0)
    exit_err
      ("Couldn't read the triggers file (not that I am illiterate, but ...)\n");
  if ((first_resp = read_responses(deffile)) == 0)
    exit_err
      ("Couldn't read the responses file (not that I am illiterate, but ...)\n");

  if (RPC_reconnect)
    if ((rpc_trig = read_triggers_rpc(deffile)) == 0) {
        warn("Couldn't read the rpc trigger file, disabling RPC identification");
        RPC_reconnect = 0;
    }

  return (do_scan_wrapper(first_targ, first_trig, rpc_trig, output_file));
}
