#	digoutany.awk	- DIG answer output parsing script
#
#	Extract only answer records from DIG results and output them in
#	this format:
#		* exactly one RR per line (no continuation lines)
#		* fields are space-separated
#		* first field is Record Type all in uppercase
#		* second and remaining fields are Data
#		* TXT records and UINFO records have double-quotes surrounding
#		  the data field, since data portion can have unknown number
#		  of spaces
#	Even the many fields of SOA and WKS records are printed all on one line.
#
#	Example Output Lines:
#		PTR host.dom.ain.
#		A 1.2.3.4
#		MX 100 mailhost.dom.ain.
#		MX 105 mailhost2.dom.ain.
#		CNAME someotherhost.dom.ain.
#		TXT "text record description could be anything right here"
#		SOA gw.home.vix.com. postmaster.home.vix.com. 1996122301 3600 1800 604800 3600
#		UINFO "Paul Balyoz +1 602-555-1212"
#
#	Exit status:	If no answer records are found, we exit with status 1,
#			otherwise exit with status 0.
#
# Possible internal "status" values:
#	none	have not seen the ANSWERS section of DiG output yet.
#	now	currently parsing ANSWERS lines.
#	never	we're past the ANSWERS section, will not parse any more.
#
# RESTRICTIONS
#  * Assumes only SOA and WKS RRs ever have parentheses continuation lines.
#  * Assumes SOA and WKS always have parentheses continuation lines.
#

BEGIN {
	numrecs = 0;
	status = "none";
}

NF==0	{ next; }

/^;/ && status=="now" {
	status = "never";
	next;
}

$1==";;" && $2=="ANSWERS:" && status=="none" {
	status = "now";
	insoa = 0;
	inwks = 0;
	next;
}

status=="now" {
	numrecs++;

# If we aren't currently inside a multiline record, parse this record.
# If it's not the start of a multiline record, then just print it;
# else remember origin and mailaddr (for SOA) or ip and protocol (for WKS).

	if (insoa == 0 && inwks == 0) {
		if ($3 == "SOA") {
			origin = $4;
			mailaddr = $5;
			valuelist = "";
			insoa = 1;

		} else if ($3 == "WKS") {
			ip = $4;
			protocol = $5;
			wkslist = "";
			inwks = 1;

		} else {
			str = "";
			for (i=4; i<NF; i++)
				str = str $i " ";
			str = str $NF;
			# add double-quotes around TXT and UINFO data if missing:
			if (($3 == "TXT" || $3 == "UINFO") && $4 !~ /^"/)
				print $3 " \"" str "\"";
			else
				print $3 " " str;
		}

# If we are inside an SOA record, add the number on this line to our list
# of values.  If this line also has an end-parenthesis then it's the last
# line of the SOA record so we need to print everything.

	} else if (insoa == 1) {
		valuelist = valuelist " " $1;
		if ($2 == ")") {
			insoa = 0;
			print "SOA " origin " " mailaddr valuelist;
		}

# If we are inside an WKS record, add the entries on this line to our list
# of wks's.  If this line also has an end-parenthesis then it's the last
# line of the WKS record so we need to print everything.

	} else if (inwks == 1) {
		for (i=1; i<NF; i++)
			wkslist = wkslist " " $i;
		if ($NF != ")") {
			wkslist = wkslist " " $NF;
		} else {
			inwks = 0;
			print "WKS " ip " " protocol wkslist;
		}
	}
}

# After all records have been dealt with, if none were printed then return
# an error status; otherwise return a normal exit status.

END {
	if (numrecs < 1)
		exit 1;
	exit 0;
}
