/*
 * gensubnetlist
 *
 * by Paul Balyoz
 *
 * Generate list of all possible subnet addresses given a network, netmask,
 * and subnetmask.
 */

/* Header files we need (plus definitions that should be in there) */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
char *strtok();

/* Function macros */

#define isnum(c)	((c)>='0' && (c)<='9')

/* Functions defined below */

unsigned long ip2long(), makemask();
void printip();

/* Global variables */

unsigned long octetmask[] = {
	0x000000FF,
	0x0000FF00,
	0x00FF0000,
	0xFF000000
};
unsigned long subbits[32], numbits;
char *thiscommand;



/*
 *	MAIN
 */

main (argc, argv)
int argc;
char *argv[];
{
	unsigned long i, j, network, netmask, subnetmask, maskdiff;
	char *cp;

/* Set thiscommand so it can be used by other functions below */

	thiscommand = argv[0];

/* Must give us all 3 arguments */

	if (argc != 4) {
		fprintf (stderr, "usage: %s network netmask subnetmask\n",
				thiscommand);
		printf ("ERROR\n");
		exit (1);
	}

/* Make sure all 3 arguments are purely numeric digits and periods */

	for (i=1; i<=3; i++) {
		for (cp=argv[i]; *cp; cp++) {
			if (!isnum(*cp) && *cp != '.') {
				fprintf (stderr, "%s: %s is not a dotted-quad\
 IP address.\n",
						thiscommand, argv[i]);
				printf ("ERROR\n");
				exit (1);
			}
		}
	}

/* Process the 3 arguments into long integers */

	network = ip2long (argv[1]);
	netmask = ip2long (argv[2]);
	subnetmask = ip2long (argv[3]);

	if (netmask == subnetmask) {
		fprintf (stderr, "%s: netmask and subnetmask are the same.\n",
				thiscommand);
		printf ("ERROR\n");
		exit (1);
	}

/* Find the difference in netmasks and make sure there aren't any bits
   turned on in netmask that aren't on in subnetmask. */

	maskdiff = netmask ^ subnetmask;

	if ((netmask&maskdiff) != 0L) {
		fprintf (stderr, "%s: netmask has bits turned on that are off\
 in subnetmask.\n",
				thiscommand);
		printf ("ERROR\n");
		exit (1);
	}

/* Generate a list of what bits are turned on in maskdiff by putting their
   bit position numbers in a sequential array named subbits[]. */

	for (i=numbits=0; i<32; i++) {
		if (maskdiff&(1<<i))
			subbits[numbits++] = i;
	}

/* Strip off all host bits from the "network" address just in case */

	network &= netmask;

/* Loop through all possible configurations for the bits in maskdiff and
   print all IP addresses of subnets within network this way.  Avoid printing
   the all-bits-off and all-bits-on mask combinations. */

	for (i=1; i < (1<<numbits)-1; i++) {
		printip (network|makemask(i));
	}

/* All worked well. */

	exit (0);
}



/*
 * ip2long	Convert a string containing a dotted-quad IP address into
 *		a long integer value.
 */

unsigned long
ip2long (str)
char *str;
{
	unsigned long i, octet, ip = 0;
	char *cp, arg[256];

/* Duplicate the string passed in so we can trash it with strtok() */

	strncpy (arg, str, sizeof(arg)-1);
	arg[sizeof(arg)-1] = '\0';

/* Position string-pointer to the first octet in the string */

	cp = strtok (arg, ".");

/* Loop through 4 octets in the string to process them */

	for (i=4; i>0; i--) {

/* If any octet string is blank, they didn't give is a real IP type thing */

		if (cp == NULL) {
			fprintf (stderr, "%s: %s is not a dotted-quad\
 IP address.\n",
					thiscommand, str);
			printf ("ERROR\n");
			exit (1);
		}

/* Loop through all characters in this string octet and convert to numeric */

		octet = 0;
		while (*cp) {
			if (!isnum(*cp)) {
				fprintf (stderr, "%s: %s is not a dotted-quad\
 IP address.\n",
						thiscommand, str);
				printf ("ERROR\n");
				exit (1);
			}
			octet = octet*10 + *cp-'0';
			cp++;
		}

/* Make sure we have a legal octet */

		if (octet > 255) {
				fprintf (stderr, "%s: %s has an octet that is\
 greater than 255.\n",
						thiscommand, str);
				printf ("ERROR\n");
				exit (1);
		}

/* Add this octet at the correction position within the IP address */

		ip += octet*(1<<((i-1)*8));

/* Position string-pointer to the next octet in the string */

		cp = strtok (NULL, ".");
	}

/* Return the answer */

	return ip;
}



/*
 * makemask	Convert an index number into a mask given the maskdiff.
 *		No check is made to be sure that the index number lies within
 *		the capabilities of the mask, so don't pass in bad numbers!
 */

unsigned long
makemask (index)
unsigned long index;
{
	int j;
	unsigned long answ = 0;

/* Loop through all bits in maskdiff, extract each bit from index and
   shift it up into its proper bit position according to maskdiff. */

	for (j=0; j<numbits; j++) {
		answ += (((index&(1<<j))?1:0) << subbits[j]);
	}

	return answ;
}



/*
 * printip	Print a long integer network IP address in standard
 *		dot-separated, decimal, four-octet format.
 */

void
printip (ip)
unsigned long ip;
{
	int i;

	for (i=4; i>0; i--) {
		printf ("%d%c", (ip&octetmask[i-1]) >> ((i-1)*8),
				(i>1) ? '.' : '\n');
	}
}
