/* filterrules
 * Copyright (C) 1999 Herve Schauer Consultants and Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: matrix.c,v 1.16 1999/09/21 15:34:43 renaud Exp $
 *
 * Author : Renaud Deraison <Renaud.Deraison@hsc.fr>
 *
 */
 

#include <includes.h>
#include "matrix.h"

/*
 * Since we send many packets, we have to find an handy yet 
 * efficient way to store the results.
 *
 * A matrix (an array of arrays) is convenient, because it will
 * allow us to sort the results according to two rules :
 * the destination port and the source port (in the case of udp and
 * tcp), the type and code (in the case of icmp), the total length
 * and offset (in the case of fragmented packets) and so on...
 *
 * The source is in Y and the destination in X :
 *
 *
 *  			1  2  3  4  5  6  7  8  9  10 
 *		    22  0  1  1  1  1  1  0  0  1   1
 *  		  1025  0 258 0  0  0 257 0  0  0   1
 * 
 * 
 * An element of the matrix may contain more informations
 * than simply a 'reachable / unreachable' boolean value. See
 * routers.c for details
 *
 * Major drawback : if the crazy user wants to test all the TCP
 * combinations _ever_ (that is 65535*65535 packets sent), then
 * we will need something like 4GB of memory to store the data...
 *
 */
 
 
 
/*
 * new_matrix :
 *
 * dst, src : Arrays to source and destination ports
 *   x,   y : will be filled with the actual height and width
 *	      of the matrix
 *
 * returns : an empty matrix
 */
matrix new_matrix(u_short * dst, u_short * src, u_short *x, u_short *y)
{
 int a = 0;
 int b = 0;
 matrix ret;
 int i;
 
 /* compute the heigth and width of our 
  * matrix
  */
 while(src[a])a++;
 while(dst[b])b++;
 ret = malloc(a*sizeof(*ret));
 bzero(ret, a*sizeof(*ret));
 for(i=0;i<a;i++){
 	ret[i] = malloc(b*(sizeof(**ret)));
	if(!ret[i])
	 {
	 printf("Could not allocate memory for the matrix\n");
	 printf("Try to use reasonable test rules\n");
	 exit(1);
	 }
	bzero(ret[i], b);
	}
	
 /* store the heigth and width in *x and *y  */	
 *y = a;
 *x = b;
 return(ret);
}

/*
 * Set an element of the matrix
 *
 * mat     : a matrix initialized with new_matrix()
 * value   : the value of the element
 * src,dst : arrays to the ports
 * sport   : the source port (the Y)
 * dport   : the dest port (the X)
 */
void matrix_set_element(matrix mat, 
			u_short value,
			u_short * src, 
			u_short * dst, 
			u_short sport,
			u_short dport)
{
 int i = 0, j = 0;
#ifdef DEBUG
 printf("(%d,%d) -> %d\n", sport, dport, value);
#endif

 /* find the location of the element in the matrix */
 while((src[i]!=sport)&&src[i])i++;
 while((dst[j]!=dport)&&dst[j])j++;
 if(!src[i] || !dst[j])return;
 /* and set it to 'value' */
 mat[i][j] = value;
}

/* 
 * matrix_elements
 *
 * Returns the elements of the row <row> that have the
 * value <value>
 *
 * Exemple :
 * Let the matrix:
 *
 *  			21	22	25	110
 *		1024    0	2	1	1
 *		  22	0	1	2	4
 *
 * A call to matrix_elements(mat, 1, dstports, 0, MatrixX,&acce)
 * will return the string : "25,110"
 *
 * A call to matrix_elements(mat, .., 1, ....) will return 
 * the string "22"
 *
 *
 * Arguments :
 *		mat   : a matrix initialized with new_matrix()
 *		value : search the elements of value <value>
 *		dst   : array of destination ports
 *		row   : the row we are interested in (the Y)
 *		x     : width of the matrix
 *		pol   : policy (accept or deny). May be modified
 *			by the function if it is nonsense
 *
 * Returns :
 *	a string or chars.
 */
char* matrix_elements(matrix mat,
		        u_short value,
			u_short * dst,
			int row, u_short x,
			int * pol,
			int translate)
{
 u_short * r = mat[row];
 u_short ** pairs;
 int num_pairs = 1;
 int i=0,j=0;
 int c_min, c_max;
 char *ret;
 int num = 0, num_ok = 0;
 int acc = 1;
 u_short * ports_ok = malloc((x+1)*(sizeof(u_short)));
 bzero(ports_ok, (1+x)*sizeof(u_short));
 
 /* compute the number of elements that matches */
 while(i<x)
 {
  num++;
  if(r[i]==value)num_ok++;
  i++;
 }
 
 /* 
  * should we consider the elements that match or that
  * DONT match  ?
  *
  * if the calling function orders us a policy, then we just
  * obey. Else, we will use the policy that matches the less
  * (because it's often the most interesting)
  */
  
 if(pol)
 {
 if(*pol!=POL_AUTO)acc = *pol;
 else if(2*num_ok<num)acc=POL_ACCEPT;   /* only consider those which are open */
 else acc=POL_DENY;		 	/* only consider those which are closed */
 
 if(value == PKT_PASSED)		/* in the case of passed packets, */
 {					/* then correct nonsenses 	  */
 if(!num_ok)acc=POL_DENY;
 if(num_ok == num)acc=POL_ACCEPT;
 }
 *pol = acc;
 
 }
 i=0;
 
 /*
  * make the list of elements that match
  */
 while(i<x)
 {
  int v = acc==POL_ACCEPT?(r[i]==value):!(r[i]==value);
  if(v)
  {
   ports_ok[j] = dst[i];
   num_ok++;
   j++;
  }
  num++;i++;
 }

 /* the calling function may want us to translate the ports.
  * this is mainly useful for ICMP
  */
 for(i=0;ports_ok[i];i++) ports_ok[i]+=translate;
 

 /*
  * Ready to make the pairs. 
  *
  * ' What are the pairs ? ' you say. Well, Imagine how unhappy
  * you'd be if you had a line like :
  *		deny tcp from any 22 to any 1,2,3,4,5,6,7,....,1024
  *
  * Instead, we make up pairs, so that 
  * 1,2,3,4,10,11,12,13,14,20,21,22 becomes :
  *		1-4,10-14,20-22
  *
  * (easier to read, huh ? :)
  *
  */
 
 /* 
  * current minimum and current maximum
  */
 c_min = ports_ok[0];
 c_max = c_min;
 pairs = malloc(x*sizeof(char*));
 bzero(pairs, x*sizeof(char*));

 /* go through our list of open ports */
 for(i=1;ports_ok[i];i++)
 {
#ifdef DEBUG
  printf("p:%d, min %d max %d\n",	ports_ok[i],
  					c_min,c_max);
#endif
  if(ports_ok[i]!=(c_max+1))
  {
   /* 
    * gotta change the number of pairs, because the new
    * max is not equal to the old maximum plus 1.
    */
   pairs[num_pairs-1] = malloc(2);
   pairs[num_pairs-1][0] = c_min;
   pairs[num_pairs-1][1] = c_max;
   num_pairs++;
   c_min = c_max = ports_ok[i];
   }
  else c_max = ports_ok[i];
  }
#ifdef DEBUG
 printf("min : %d max : %d\n", c_min, c_max);
#endif
 /* last pair */
 pairs[num_pairs-1] = malloc(2);
 pairs[num_pairs-1][0] = c_min;
 pairs[num_pairs-1][1] = c_max; 
 free(ports_ok);
#ifdef DEBUG
 for(i=0;i<num_pairs;i++)
 {
  printf("%d-%d\n", pairs[i][0], pairs[i][1]);
 }
#endif

 /*
  * now, we'll return a string containing all those
  * pairs, not the array itself. 
  */
 ret = malloc(num_pairs*20);
 bzero(ret, num_pairs*20);
 for(i=0;i<num_pairs;i++)
 {
  if((pairs[i][0])!=(pairs[i][1]))
  {
   sprintf(ret, "%s%d-%d,",ret, pairs[i][0],
   				pairs[i][1]);
  }
  else sprintf(ret, "%s%d,",ret, pairs[i][0]);
  
  free(pairs[i]);
 }
 free(pairs);
#ifdef DEBUG
 printf("ret : %s\n", ret);
#endif
 ret[strlen(ret)-1]=0;
 realloc(ret, strlen(ret)+1);
 return(ret);
}
 
