/* filterrules
 * Copyright (C) 1999 Herve Schauer Consultants and Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * $Id: tcp.c,v 1.18 1999/09/21 15:34:43 renaud Exp $
 *
 * Author : Renaud Deraison <deraison@nessus.org>
 *
 */
 
#include <includes.h>
#include <libnet.h>
#include "matrix.h"
#include "forgers.h"
#include "utils.h"
#include "tests.h"
#include "source_route.h"
#include "ttls.h"

#ifndef TH_FLAGS
#define TH_FLAGS (TH_SYN|TH_ACK|TH_RST|TH_PUSH|TH_FIN|TH_URG)
#endif
/*
 *				TCP
 *
 * The TCP packets are sent, and the replies are stored in our global
 * matrix, with the source ports in Y and the dest port in X.
 *
 *
 *
 */

extern int RawSoc;
extern int Policy;
extern matrix Matrix;
extern u_short MatrixX;
extern int Remote;
/*
 * TCP private functions
 */


/*
 * sub_str2tcp_flag : will convert a symbolic name ("SYN")
 * to its numeric value
 */
static int sub_str2tcp_flag(str)
	char * str;
{
 if(!strncmp(str, "SYN",3))return(TH_SYN);
 if(!strncmp(str, "ACK",3))return(TH_ACK);
 if(!strncmp(str, "RST",3))return(TH_RST);
 if(!strncmp(str, "PUSH",4))return(TH_PUSH);
 if(!strncmp(str, "FIN",3))return(TH_FIN);
 if(!strncmp(str, "URG",3))return(TH_URG);
 if(!strncmp(str, "FLAGS",5))return(TH_FLAGS); /* everything */
 return(atoi(str));
}

/*
 * str2tcp_flag converts a string of symbolic ORed values, such
 * as 'SYN|ACK' to the represented numeric value
 */
int str2tcp_flag(str)
	char * str;
{
 char *d;
 int ret = 0;
 while((d=strchr(str, '|'))){
 	ret = ret | sub_str2tcp_flag(str);
	str = d+1;
	}
 ret= ret | sub_str2tcp_flag(str);
 return(ret);
}


/*
 * converts a numeric value to a symbolic string
 */
char * tcp_flag2str(flag)
 int flag;
{
 static char ret[256];
 int f = 0;
 bzero(ret, 256);
 if(flag & TH_SYN){sprintf(ret, " SYN");f++;}
 if(flag & TH_ACK){sprintf(ret, "%s%cACK", ret, f?'|':' ');f++;}
 if(flag & TH_RST){sprintf(ret, "%s%cRST", ret, f?'|':' ');f++;}
 if(flag & TH_PUSH){sprintf(ret, "%s%cPUSH", ret, f?'|':' ');f++;}
 if(flag & TH_FIN){sprintf(ret, "%s%cFIN", ret, f?'|':' ');f++;}
 if(flag & TH_URG){sprintf(ret, "%s%cURG", ret, f?'|':' ');f++;}
 return(ret);
}


/*
 * Returns the filter that will make the slave only send us
 * the interesting packets.
 */
char * tcp_filter(test,y)
 struct test * test;
 int y;
{
 char * asc_src = (char*)strdup(inet_ntoa(test->src));
 char * asc_dst = (char*)strdup(inet_ntoa(test->dst));
 char * ret = malloc(strlen(asc_src)+strlen(asc_dst)+200);
 u_short * sport = (u_short *)test->tcp_sport;
 
 sprintf(ret, "FILTER tcp and (src host %s and dst host %s and src port %d)",
 			asc_src,asc_dst,sport[y]);
 free(asc_src);
 free(asc_dst);			
 realloc(ret, strlen(ret)+1);
 return(ret);
}




/*
 * parse the tcp options in the config file
 */
struct test * tcp_parser(str, test)
 char * str;
 struct test * test;
{
 char *d = (char*)strchr(str, ':');
 test->proto = IPPROTO_TCP;
 test->num_data = 0;
 if(d)d[0]=0;
 test->tcp_sport=(long)getpts(str,0);
 test->num_data++;
 str = d+1;
 d = (char*)strchr(str, ':');
 if(d)d[0]='\0';
 test->num_data++;
 test->tcp_dport=(long)getpts(str,0);
 str = d+1;
 d = strchr(str, ':');
 if(d)d[0]=0;
 test->tcp_flag = str2tcp_flag(str);
 test->soc = RawSoc;
 if(d)source_route_parser(test, d);
 else test->source_route = 0;
 return(test);
}



/*
 * store a received TCP packet in the appropriate place
 * of the matrix
 */
void tcp_receiver(test, pkt)
 struct test * test;
 u_char * pkt;
{
 struct libnet_ip_hdr * ip = (struct libnet_ip_hdr*)(pkt);
 struct libnet_tcp_hdr * tcp =(struct libnet_tcp_hdr*)(pkt+ip->ip_hl*4);
 matrix_set_element(Matrix,PKT_PASSED, 
   			(u_short*)test->tcp_sport, 
   			(u_short*)test->tcp_dport, 
   			ntohs(tcp->th_sport),
			ntohs(tcp->th_dport));
}



/*
 * forge and send a TCP packet
 */
int tcp_sender(test, x,y)
 struct test * test;
 int x,y;
{
 u_char * buf;
 struct libnet_ip_hdr * ip;
 u_short * sport, * dport;
 int ttl;
 
 if(libnet_init_packet(IP_H+TCP_H+40, &buf)<0)
 {
  printf("Could not allocate memory for a packet\n");
  exit(1);
 }
 
 
 sport = (u_short *)test->tcp_sport;
 dport = (u_short *)test->tcp_dport;
 ttl = get_ttl(test->dst);
 libnet_build_ip(TCP_H, 0, 1234, 0, ttl, IPPROTO_TCP,test->src.s_addr, 
 	  test->dst.s_addr, NULL, 0, buf);
 ip = (struct libnet_ip_hdr*)buf;
 
 libnet_build_tcp(sport[y],dport[x], 1234,4321,test->tcp_flag,
 	   4096, 0, NULL, 0, buf + IP_H);
 libnet_do_checksum(buf, IPPROTO_TCP, TCP_H); 	 	   
 if(test->source_route)insert_source_route(test, buf);    
 libnet_do_checksum(buf, IPPROTO_TCP, TCP_H);
 libnet_write_ip(test->soc, buf, TCP_H + (ip->ip_hl*4));
 free(buf);
 return(0);
}


/*
 * display the results of the test in a human readable
 * form
 */
char * tcp_display(test,x)
 struct test * test;
 int x;
{
 char * ret = malloc(2048);
 char * asc_src = strdup(inet_ntoa(test->src));
 char * asc_dst = strdup(inet_ntoa(test->dst));
 char * flags = tcp_flag2str(test->tcp_flag);
 u_short * sport = (u_short *)test->tcp_sport;
 char * ports;
 

 ports = matrix_elements(Matrix,PKT_PASSED,
 			 	(u_short*)test->tcp_dport,
				x,MatrixX,&test->policy,0);
 						
 
 sprintf(ret, "tcp from %s %d with flags%s to %s %s",
 		asc_src,sport[x],flags,
		asc_dst,ports);
		
		
 if(test->source_route)
 {
  struct in_addr * addr = (struct in_addr*)test->source_route;
  char * r = malloc(strlen(ret)+2000);
  int i = 0;
  sprintf(r,"%s\nSource routing :\n",ret);
  free(ret);
  while(addr[i].s_addr){
  	sprintf(r, "%s\t%d. %s\n", r, i, inet_ntoa(addr[i++]));
	}
  ret = r;
  }
 /* free(ports); */
 free(asc_src);
 free(asc_dst);
 realloc(ret, strlen(ret)+1);
 return(ret);
}		

void tcp_router_err(test, pkt, index, err,code)
 struct test * test;
 u_char * pkt;
 int index;
 int err;
{
 struct libnet_ip_hdr * ip = (struct libnet_ip_hdr *)pkt;
 struct libnet_tcp_hdr * tcp = (struct libnet_tcp_hdr*)
 				((u_char*)ip+ip->ip_hl*4);
 int sport, dport;
 
 sport = ntohs(tcp->th_sport);
 dport = ntohs(tcp->th_dport);
 if(err==3)	/* ICMP unreach */
  matrix_set_element(Matrix, (index<<8)|(code+2),
 			(u_short*)test->tcp_sport,
			(u_short*)test->tcp_dport,
			sport, dport);
 else if(err==11) /* ICMP time exceed */
 
   matrix_set_element(Matrix, 1,
 			(u_short*)test->tcp_sport,
			(u_short*)test->tcp_dport,
			sport, dport);
}			
