/*
 * Ghost Port Scan is distributed under the GPL
 *
 * DISCLAIMER: standard
 *
 */

// standard librairies
#include<ctype.h>
#include<stdio.h>
#include<string.h>
#include<stdlib.h>
#include<unistd.h>
#include<signal.h>
// network librairies
#include<libnet.h>
#include<pcap.h>
#include<net/if_arp.h>
#include<netinet/ether.h>
#include<netinet/ip.h>
#include<netinet/tcp.h>
// custom librairies
#include "services.h"
#include "fwrd.h"


#define GPS_VERSION "0.5.0"
#define STR_IP_LEN 16

int ppid, nbopt=0, nbpkt=0, scan_type=TH_SYN, rand_seed, serv_scan=1, nbports;
u_int16_t tmp, *tab, fport=1, lport=1024;
u_int32_t *src_ip, dst_ip=0, nb_src_ip=0;
struct pcap *descr;
struct libnet_link_int *interface;
services_t *st;
extern char *optarg;
fwrd_elt *tab_fwrd;

void usage(char *cmd);
void error(char *msg, int ret);
void signalhandle();
void dummy_sighandler();
void packets_generator(int scan_speed, int ttw);
void exit_thread();
void fin_scan_disp(u_int16_t pt);
int  search_tab(u_int16_t pt);
int  search_src_ip(u_int32_t ip);

int main(int argc, char **argv)
{
	char str_src_ip[STR_IP_LEN], str_dst_ip[STR_IP_LEN];
	char errbuf[(LIBNET_ERRBUF_SIZE<PCAP_ERRBUF_SIZE) ? PCAP_ERRBUF_SIZE : LIBNET_ERRBUF_SIZE];
	char *eth, *ptr, *ptr1, *ptr2;
	const u_char *packet;
	int npid, scan_speed=0, cmd_parse, ttw=5, i, j;
	struct pcap_pkthdr phdr;
	struct ether_header *ehdr;
	struct ether_arp *ahdr, duplic;
	struct ip *ihdr;
	struct tcphdr *thdr;
	ppid=getpid();
	if(!(src_ip=(u_int32_t *)malloc(sizeof(u_int32_t))))
		error("malloc: memory allocation failed", -2);
	// load known services in memory
	st=services_load();

	signal(SIGHUP, signalhandle);
	signal(SIGINT, signalhandle);
	signal(SIGQUIT, signalhandle);
	signal(SIGKILL, signalhandle);
	signal(SIGTERM, signalhandle);

	fprintf(stdout, "Ghost Port Scan version %s by whitehat@altern.org (gps.sourceforge.net)\n", GPS_VERSION);

	// take a look to the arguments:
	if(argc<3) usage(argv[0]);
	while((cmd_parse=getopt(argc, argv, "s:d:p:k:t:r:w:"))!=EOF)
	{
		switch(cmd_parse)
		{
			case 's':
				// get the src_ip list
				ptr1=optarg;
				do
				{
					for(ptr=ptr1; *ptr && !isdigit(*ptr); ptr++);
					if(!*ptr) error("Invalid list of src_ip", -11);
					for(ptr1=ptr2=ptr; *ptr1 && *ptr1!=',' && *ptr1!='-'; ptr1++)
						if(*ptr1=='.') ptr2=ptr1;
					switch(*ptr1)
					{
						case '-':
						case ',':
						case 0:
							if((i=ptr1-ptr)>=STR_IP_LEN)
								error("Invalid list of src_ip", -11);
							strncpy(str_src_ip, ptr, i);
							str_src_ip[i]='\0';
							if(!(src_ip=(u_int32_t *)realloc(src_ip, (nb_src_ip+1)*sizeof(u_int32_t))))
								error("realloc: memory allocation failed", -2);
							if((src_ip[nb_src_ip++]=libnet_name_resolve(str_src_ip, LIBNET_RESOLVE))==-1)
								libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: invalid src_ip address %s\n", str_src_ip);
							if(!*ptr1 || *ptr1==',') break;
							if(!isdigit(*(++ptr1))) error("Invalid list of src_ip", -11);
							if((j=atoi(ptr1)-atoi(++ptr2))<0) error("Invalid list of src_ip", -11);
							if(!(src_ip=(u_int32_t *)realloc(src_ip, (nb_src_ip+j)*sizeof(u_int32_t))))
								error("realloc: memory allocation failed", -2);
							for(i=0; i<j; i++) src_ip[nb_src_ip+i]=src_ip[nb_src_ip-1]+ntohl(i+1);
							nb_src_ip+=j;
							for(; *ptr1 && *ptr1!=','; ptr1++);
							break;
					}
				} while(*ptr1);
				break;
			case 'd':
				// dst_ip
				strncpy(str_dst_ip, optarg, STR_IP_LEN - 1);
				str_dst_ip[STR_IP_LEN - 1]='\0';
				if((dst_ip=libnet_name_resolve(str_dst_ip, LIBNET_RESOLVE))==-1)
				libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: invalid dst_ip address: %s\n", str_dst_ip);
				break;
			case 'p':
				// first_port and last_port
				if(!(fport=atoi(optarg))) error("atoi: invalid first port", -3);
				for(ptr=optarg; *ptr && *ptr!='-'; ptr++);
				if(!(lport=atoi(++ptr))) error("atoi: invalid last port", -3);
				if(lport<fport) error("Invalid first and last ports", -4);
				break;
			case 'k':
				// scan known ports
				if(optarg[0]=='0') serv_scan=0;
				else serv_scan=1;
				break;
			case 't':
				// scan_type
				if(!strncmp("syn", optarg, 3)) scan_type=TH_SYN;
				else if(!strncmp("fin", optarg, 3)) scan_type=TH_FIN;
				else if(!strncmp("ack", optarg, 3)) scan_type=TH_ACK;
				else if(!strncmp("xmas", optarg, 4)) scan_type=TH_FIN|TH_URG|TH_PUSH;
				else if(!strncmp("null", optarg, 4)) scan_type=0;
				else if(!strncmp("rand", optarg, 4)) scan_type=-1;
				else if(!strncmp("fwrd", optarg, 4)) scan_type=-2;
				else error("Invalid scan_type", -5);
				break;
			case 'r':
				// scan_speed
				if(!strncmp("paranoid", optarg, 8)) scan_speed=500000;
				else if(!strncmp("polite", optarg, 6)) scan_speed=100000;
				else if(!strncmp("normal", optarg, 6)) scan_speed=50000;
				else if(!strncmp("aggressive", optarg, 10)) scan_speed=10000;
				else if(!strncmp("insane", optarg, 6)) scan_speed=0;
				else error("Invalid scan_speed", -6);
				break;
			case 'w':
				ttw=atoi(optarg);
				if(!ttw) ttw=5;
				break;
			case '?':
			default:
				usage(argv[0]);
		}
	}
	if(!nb_src_ip || !dst_ip) usage(argv[0]);

	// allocate memory for data storage
	nbports=lport-fport+1;
	if(st && serv_scan)
	{
		for(i=0; st->tab[i].port<fport && i<st->nb; i++) nbports++;
		for(i=st->nb-1; st->tab[i].port>lport && i>=0; i--) nbports++;
	}
	if(!(tab=(u_int16_t *)malloc(nbports * sizeof(u_int16_t))))
		error("malloc: memory allocation failed", -2);
	if(scan_type==-2)
		if(!(tab_fwrd=(fwrd_elt *)malloc(nb_src_ip*nbports*sizeof(fwrd_elt))))
			error("malloc: memory allocation failed", -2);

	// get an ethernet device name
	if(!(eth=pcap_lookupdev(errbuf)))
		error(errbuf, -7);
	fprintf(stdout, "Network device: %s\n", eth);

	// seed the pseudo-random number generator
	rand_seed=libnet_seed_prand();

	// clone the thread:
	// old thread: sniffer
	// new thread: packet generator
	if((npid=fork())==-1) error("fork: thread duplication failed", -8);
	if(!npid) packets_generator(scan_speed, ttw);

	// open a raw link interface to send spoofed ARP packets if needed (ARP poison)
	if(!(interface=libnet_open_link_interface(eth, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed (need to be root ?)\n");

	// open network device in promiscuous mode
	if(!(descr=pcap_open_live(eth, BUFSIZ, 1, -1, errbuf)))
		error(errbuf, -9);

	// loop until all packets are sent
	while(1)
	{
		//grab a packet from network
		if(!(packet=pcap_next(descr, &phdr)))
		{
			libnet_error(LIBNET_ERR_WARNING, "Error during packet grab\n");
			continue;
		}
		ehdr=(struct ether_header *)packet;
		// packet filter
		switch(ntohs(ehdr->ether_type))
		{
			case ETHERTYPE_ARP: // ARP packet
				ahdr=(struct ether_arp *)((u_char *)ehdr + ETHER_HDR_LEN);
				// break if it is not an ARP request
				if(ntohs(ahdr->ea_hdr.ar_op)!=ARPOP_REQUEST) break;
				// break if the ARP request do not concern
				// the host we pretend to be
				if(search_src_ip(*((u_int32_t *)&ahdr->arp_tpa))==-1) break;
				// we have to reply this ARP request
				duplic=*ahdr;
				ahdr->ea_hdr.ar_op=ntohs(ARPOP_REPLY);
				// build the spoofed ARP reply
				for(i=0; i<ETH_ALEN; i++) // exchange ARP addresses
				{
					ahdr->arp_sha[i]=ehdr->ether_shost[i]=(u_int8_t)libnet_get_prand(rand_seed);
					ahdr->arp_tha[i]=ehdr->ether_dhost[i]=duplic.arp_sha[i];
				}
				for(i=0; i<4; i++) // exchange IP adresses
				{
					ahdr->arp_spa[i]=duplic.arp_tpa[i];
					ahdr->arp_tpa[i]=duplic.arp_spa[i];
				}
				// send the fake ARP reply
				if(libnet_write_link_layer(interface, eth, (u_char *)packet, ETHER_HDR_LEN + sizeof(struct ether_arp))==-1)
					libnet_error(LIBNET_ERR_WARNING, "libnet_write_link layer failed\n");
				break;
			case ETHERTYPE_IP: // IP packet
				ihdr=(struct ip *)((u_char *)ehdr + ETHER_HDR_LEN);
				// break if the packet do not come from the target host
				// or if it is not addressed to the hosts we pretend to be
				if(ihdr->ip_src.s_addr!=dst_ip || search_src_ip(ihdr->ip_dst.s_addr)==-1) break;
				// break if it is not a TCP packet
				if(ihdr->ip_p!=IPPROTO_TCP) break;
				thdr=(struct tcphdr *)((u_char *)ihdr + 4 * ihdr->ip_hl);
				// test packet flags
				i=0;
				switch(scan_type)
				{
					case TH_SYN:			// SYN scan
						if(thdr->th_flags!=(TH_SYN|TH_ACK)) i=1;
						break;
					case TH_FIN|TH_URG|TH_PUSH:	// Xmas tree scan
					case TH_FIN:			// FIN scan
					case 0:				// Null scan
					case TH_ACK:			// ACK scan
					case -1:			// rand scan
						if(!(thdr->th_flags & TH_RST)) i=1;
						break;
					case -2:			// fwrd scan
						if(thdr->th_flags & TH_RST) i=1;
						// we have a matching packet, see if we already have it
						for(j=0; j<nbopt && (tab_fwrd[j].ip.s_addr!=ihdr->ip_dst.s_addr || tab_fwrd[j].port!=thdr->th_sport); j++);
						if(j<nbopt) break;
						tab_fwrd[nbopt].ip.s_addr=ihdr->ip_dst.s_addr;
						tab_fwrd[nbopt++].port=thdr->th_sport;
						nbpkt++;
						break;
				}
				if(i) break;
				// we have a matching packet, see if we already have it
				i=0;
				while(i<nbopt && tab[i]!=thdr->th_sport) i++;
				if(i==nbopt) tab[nbopt++]=thdr->th_sport;
				nbpkt++;
				break;
		}
	}
	return(0);
}

void signalhandle()
{
	int known, i, j;
	fwrd_elt tmp_fwrd;

	signal(SIGHUP, dummy_sighandler);
	signal(SIGINT, dummy_sighandler);
	signal(SIGQUIT, dummy_sighandler);
	signal(SIGKILL, dummy_sighandler);
	signal(SIGTERM, dummy_sighandler);

	switch(scan_type)
	{
		case -2: // fwrd scan
			// order and handle byte swap
			for(i=0; i<nbopt; i++)
			{
				tab_fwrd[i].port=ntohs(tab_fwrd[i].port);
				tab_fwrd[i].ip.s_addr=ntohl(tab_fwrd[i].ip.s_addr);
			}
			for(i=0; i<nbopt; i++)
				for(j=i+1; j<nbopt; j++)
					if(tab_fwrd[i].ip.s_addr>tab_fwrd[j].ip.s_addr || (tab_fwrd[i].ip.s_addr==tab_fwrd[j].ip.s_addr && tab_fwrd[i].port>tab_fwrd[j].port))
					{
						tmp_fwrd=tab_fwrd[i];
						tab_fwrd[i]=tab_fwrd[j];
						tab_fwrd[j]=tmp_fwrd;
					}
			for(i=0; i<nbopt; i++) tab_fwrd[i].ip.s_addr=ntohl(tab_fwrd[i].ip.s_addr);
			// output
			if(nbopt==nb_src_ip*nbports)
				fprintf(stdout, "All scanned ports are not filtered for all specified IP addresses\n");
			else
				for(i=0; i<nbopt;)
				{
					fprintf(stdout, "Ports: ");
					for(j=i; j<nbopt && tab_fwrd[j].ip.s_addr==tab_fwrd[i].ip.s_addr; j++)
						fprintf(stdout, "%u, ", tab_fwrd[j].port);
					fprintf(stdout, "are not filtered for %s\n", inet_ntoa(tab_fwrd[i].ip));
					i=j;
				}
			break;

		default:
			// order and handle byte swap
			for(i=0; i<nbopt; i++) tab[i]=ntohs(tab[i]);
			for(i=0; i<nbopt; i++)
				for(j=i+1; j<nbopt; j++)
					if(tab[i]>tab[j])
					{
						tmp=tab[i];
						tab[i]=tab[j];
						tab[j]=tmp;
					}

			// output
			switch(scan_type)
			{
				case TH_SYN: // print results for SYN scan
					if(!nbopt) fprintf(stdout, "No open ports found\n");
					else
					{
						for(i=0; i<nbopt; i++)
						{
							if((known=services_known(tab[i], st))==-1)
								fprintf(stdout, "Port %u is open\n", tab[i]);
							else // port service detail
								fprintf(stdout, "Port %u: %s (%s) is open\n", tab[i], st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
						}
					}
					break;
				case TH_FIN:			// print results for FIN scan,
				case TH_FIN|TH_URG|TH_PUSH:	// for Xmas tree scan,
				case 0:				// for Null scan,
				case -1:			// for rand scan
					if(nbopt==nbports) // no open ports found
						fprintf(stdout, "No open ports found with a FIN/XmasTree/Null scan: try a SYN scan for better results\n");
					else
					{
						if(st && serv_scan)
							for(i=0; i<st->nb && st->tab[i].port<fport; i++)
								fin_scan_disp(st->tab[i].port);
						for(tmp=fport; tmp<=lport; tmp++)
							fin_scan_disp(tmp);
						if(st && serv_scan)
						{				
							for(i=0; i<st->nb && st->tab[i].port<=lport; i++);
							for(; i<st->nb; i++) fin_scan_disp(st->tab[i].port);
						}
					}
					break;
				case TH_ACK: // ACK scan
					if(nbopt==nbports)
						fprintf(stdout, "All scanned ports are not filtered\n");
					else if(nbopt<nbports/2)
					{
						fprintf(stdout, "Ports: ");
						for(i=0; i<nbopt; i++) fprintf(stdout, "%u, ", tab[i]);
						fprintf(stdout, "are not filtered\n");
					}
					else
					{
						fprintf(stdout, "Ports: ");
						if(st && serv_scan)
							for(i=0; i<st->nb && st->tab[i].port<fport; i++)
								if(search_tab(st->tab[i].port)==-1)
									fprintf(stdout, "%u, ", st->tab[i].port);
						for(tmp=fport; tmp<=lport; tmp++)
							if(search_tab(tmp)==-1)
								fprintf(stdout, "%u, ", tmp);
						if(st && serv_scan)
						{
							for(i=0; i<st->nb && st->tab[i].port<=lport; i++);
							for(; i<st->nb; i++)
								if(search_tab(st->tab[i].port)==-1)
									fprintf(stdout, "%u, ", st->tab[i].port);
						}
						fprintf(stdout, "are filtered\n");
					}
					break;
			}
			break;
	}
	fprintf(stdout, "Grabbed %d matching packets\n", nbpkt);

	// free & close everything
	if(scan_type==-2) free(tab_fwrd);
	free(tab);
	free(src_ip);
	pcap_close(descr);
	if(libnet_close_link_interface(interface)==-1)
		libnet_error(LIBNET_ERR_WARNING, "libnet_close_link_interface failed\n");
	exit(0);
}

void fin_scan_disp(u_int16_t pt)
{
	int known;
	if(search_tab(pt)==-1)
	{
		if((known=services_known(pt, st))==-1)
			fprintf(stdout, "Port %u is open or filtered\n", pt);
		else // port service detail
			fprintf(stdout, "Port %u: %s (%s) is open or filtered\n", pt, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
	}
}

int search_tab(u_int16_t pt)
{
	int m, i=0, j=nbopt-1;
	do {
		m=(i+j)/2;
		if(tab[m]==pt) return(m);
		if(tab[m]<pt) i=m+1;
		else j=m-1;
	} while (i<=j);
	return(-1);
}

int search_src_ip(u_int32_t ip)
{
	int i;
	for(i=0; i<nb_src_ip; i++)
		if(src_ip[i]==ip) return(i);
	return(-1);
}

#define NB_RAND_FLAGS 8

void packets_generator(int scan_speed, int ttw)
{
	int seq, offset, raw_sock, i, j, th_flags=scan_type, i_ip=0;
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	char **packets;
	struct libnet_arena arena, *ptr_arena=&arena;
	u_int32_t rsrc_ip=src_ip[0];

	signal(SIGHUP, exit_thread);
	signal(SIGINT, exit_thread);
	signal(SIGQUIT, exit_thread);
	signal(SIGKILL, exit_thread);
	signal(SIGTERM, exit_thread);

	//wait for the sniffer to be ready
	sleep(1);

	// open a raw IP socket
	if((raw_sock=libnet_open_raw_sock(IPPROTO_RAW))==-1) exit_thread();

	// set up TCP flags for fwrd scan
	if(scan_type==-2) th_flags=TH_ACK;

	do
	{
		// randomize ports order
		for(i=0; i<(lport-fport+1); i++) tab[i]=i + fport;
		if(st && serv_scan)
		{
			for(j=0; st->tab[j].port<fport && j<st->nb; j++) tab[i++]=st->tab[j].port;
			for(j=st->nb-1; st->tab[j].port>lport && j>=0; j--) tab[i++]=st->tab[j].port;
		}
		for(i=0; i<nbports; i++)
		{
			offset=(nbports * libnet_get_prand(rand_seed)) / 255;
			tmp=tab[i];
			tab[i]=tab[offset];
			tab[offset]=tmp;
		}

		// allocate memory for packets
		if(!(packets=(char **)malloc(nbports * sizeof(char **)))) exit_thread();
		if(libnet_init_packet_arena(&ptr_arena, nbports, LIBNET_PACKET)==-1) exit_thread();

		// open a raw IP socket
//		if((raw_sock=libnet_open_raw_sock(IPPROTO_RAW))==-1) exit_thread();

		// build and send packets
		for(i=0; i<nbports; i++)
		{
			if(!(packets[i]=libnet_next_packet_from_arena(&ptr_arena, LIBNET_PACKET))) continue;

			// calculate the pseudo-random seq number
			seq=1;
			for(j=0; j<4; j++) seq*=libnet_get_prand(rand_seed);

			// get a random flags for rand_scan
			if(scan_type==-1)
				th_flags=rflags[libnet_get_prand(rand_seed)%NB_RAND_FLAGS];

			if(scan_type!=-2)
			{
				// get a random src_ip for !fwrd scan
				if(nb_src_ip==1) rsrc_ip=src_ip[0];
				else rsrc_ip=src_ip[libnet_get_prand(rand_seed)%nb_src_ip];
			}

			// build an IP datagram with a spoofed src_ip
			libnet_build_ip(LIBNET_TCP_H, IPTOS_LOWDELAY,libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), 0, 64, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets[i]);
			libnet_build_tcp(libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), tab[i], seq, 0, th_flags, 1024, 0, NULL, 0, packets[i] + LIBNET_IP_H);
			// if the checksum is ok, send it
			if(libnet_do_checksum(packets[i], IPPROTO_TCP, LIBNET_TCP_H)!=-1)
				libnet_write_ip(raw_sock, packets[i], LIBNET_PACKET);
			// wait a bit
			usleep(scan_speed);
		}

		// close and free
//		libnet_close_raw_sock(raw_sock);
		libnet_destroy_packet_arena(&ptr_arena);
		free(packets);

		// get next IP for fwrd scan
		rsrc_ip=src_ip[++i_ip];

	} while(i_ip<nb_src_ip && scan_type==-2);

	libnet_close_raw_sock(raw_sock);

	// wait for the sniffer to grab the maximum of packets
	// (I assume that some packets may be lost, and it is
	//  the cleanest way I found to end the threads)
	sleep(ttw);
	exit_thread();
}

void exit_thread()
{
	signal(SIGHUP, dummy_sighandler);
	signal(SIGINT, dummy_sighandler);
	signal(SIGQUIT, dummy_sighandler);
	signal(SIGKILL, dummy_sighandler);
	signal(SIGTERM, dummy_sighandler);

	free(tab);
	free(src_ip);
	if(ppid==getppid()) kill(ppid, SIGHUP);
	exit(0);
}

void usage(char *cmd)
{
fprintf(stderr, "Usage: %s -s src_ip[,src_ip_1[,src_ip_2-4..]] -d dst_ip [-t scan_type]
	[-r scan_speed] [-w ttw] [-p first_port-last_port] [-k 0 | 1]\n\
 -s src_ip[,src_ip_1..]: list of IP addresses of the hosts we pretend to be\n\
 -d dst_ip.............: target host's IP\n\
 -t scan_type..........: 'stealth' scan mode (default: syn)\n\
 			 (syn | xmas | null | fin | ack | rand | fwrd)\n\
 -r scan_speed.........: packets rate (default: insane)\n\
 			 (insane | aggressive | normal | polite | paranoid)\n\
 -w ttw (sec)..........: time to wait for the sniffer to grab
			 enough packets (default: 5 - nice for LAN)\n\
 -p first-last ports...: ports range to scan (default: 1-1024)\n\
 -k 0 | 1..............: scan well-known ports (default: 1)\n", cmd);
exit(-1);
}

void error(char *msg, int ret)
{
	fprintf(stderr, "%s\n", msg);
	exit(ret);
}

void dummy_sighandler()
{
	return;
}

