/*
 * Ghost Port Scan is distributed under the GPL (GNU Public Licence)
 *
 * Homepage: gps.sourceforge.net
 * Contact.: whitehat@altern.org
 *
 */

// standard librairies
#include<ctype.h>
#include<stdio.h>
#include<string.h>
#include<stdlib.h>
#include<unistd.h>
#include<signal.h>
#include<sys/types.h>
#include<sys/socket.h>
// network librairies
#include<libnet.h>
#include<pcap.h>
#include "pcap-int.h"
#include<net/if_arp.h>
#include<netinet/if_ether.h>
#include<netinet/ip.h>
#include<netinet/tcp.h>
// custom librairies
#include "services.h"
#include "gps.h"
#include "gateway.h"


int nbopt=0, nbpkt=0, scan_type=SYN_SCAN, rand_seed, serv_scan=1, nbports, sockets[2];
int max_retries, max_timeout, ping_port=80, fragment=NO_FRAG, spoofing=MAC_SPOOF;
u_int16_t *tab_response, fport=1, lport=1024;
u_int32_t dst_ip=0, nb_src_ip=0, net_ip, netmask;
struct pcap *descr;
struct libnet_link_int *interface;
services_t *st;
extern char *optarg;
fwrd_elt *tab_fwrd;
arp_elt *src_ip;
port_elt *tab_port;
struct in_addr tmp_inet;
u_char dst_mac[ETH_ALEN];
char errbuf[GPS_ERRBUF_SIZE], *eth=NULL;


void main_sig(int sig)
{
	printf("main: dead %d\n", sig);
	exit(0);
}
void pg_sig(int sig)
{
	printf("pg: dead %d\n", sig);
	exit(0);
}

int main(int argc, char **argv)
{
	char str_src_ip[STR_IP_LEN], str_dst_ip[BUFSIZ];
	char *ptr, *ptr1, *ptr2;
	const u_char *packet;
	u_int16_t port;
	u_int32_t ip;
	int npid, scan_speed=0, cmd_parse, i, j, k, gotcha[2], known, read_ok=1, i_ip=0, timeout=0, verbose=0;
	struct pcap_pkthdr phdr;
	struct ether_header *ehdr;
	struct ether_arp *ahdr, duplic;
	struct ip *ihdr;
	struct tcphdr *thdr;
	struct timeval tv, current;
	sock_elt sckelt;
	fwrd_elt tmp_fwrd;


for(i=0; i<17; i++) signal(i, main_sig);


	// initialize some vars
	rand_seed=libnet_seed_prand();
	if(!(src_ip=(arp_elt *)malloc(sizeof(arp_elt)))) error(NULL, -2);
	// load known services in memory
	st=services_load();

	fprintf(stdout, "Ghost Port Scan version %s by whitehat@altern.org (gps.sourceforge.net)\n", GPS_VERSION);

	// take a look to the arguments:
	if(argc<3) usage(argv[0]);
	while((cmd_parse=getopt(argc, argv, "s:d:p:k:t:r:e:vf:i:S:"))!=EOF)
	{
		switch(cmd_parse)
		{
			case 's':
				// get the src_ip list
				ptr1=optarg;
				do
				{
					for(ptr=ptr1; *ptr && !isdigit(*ptr); ptr++);
					if(!*ptr) error("Invalid list of src_ip", -1);
					for(ptr1=ptr2=ptr; *ptr1 && *ptr1!=',' && *ptr1!='-'; ptr1++)
						if(*ptr1=='.') ptr2=ptr1;
					switch(*ptr1)
					{
						case '-':
						case ',':
						case 0:
							if((i=ptr1-ptr)>=STR_IP_LEN)
								error("Invalid list of src_ip", -1);
							strncpy(str_src_ip, ptr, i);
							str_src_ip[i]='\0';
							if(!(src_ip=(arp_elt *)realloc(src_ip, (nb_src_ip+1)*sizeof(arp_elt))))
								error(NULL, -2);
							if((src_ip[nb_src_ip].ip=libnet_name_resolve(str_src_ip, LIBNET_RESOLVE))==-1)
								libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: invalid src_ip address %s\n", str_src_ip);
							get_ea(src_ip[nb_src_ip++].ea);
							if(!*ptr1 || *ptr1==',') break;
							if(!isdigit(*(++ptr1))) error("Invalid list of src_ip", -1);
							if((j=atoi(ptr1)-atoi(++ptr2))<0) error("Invalid list of src_ip", -1);
							if(!(src_ip=(arp_elt *)realloc(src_ip, (nb_src_ip+j)*sizeof(arp_elt))))
								error(NULL, -2);
							for(i=0; i<j; i++)
							{
								src_ip[nb_src_ip+i].ip=src_ip[nb_src_ip-1].ip+ntohl(i+1);
								get_ea(src_ip[nb_src_ip+i].ea);
							}
							nb_src_ip+=j;
							for(; *ptr1 && *ptr1!=','; ptr1++);
							break;
					}
				} while(*ptr1);
				break;
			case 'd':
				// dst_ip
				strncpy(str_dst_ip, optarg, BUFSIZ - 1);
				str_dst_ip[BUFSIZ-1]='\0';
				if((dst_ip=libnet_name_resolve(str_dst_ip, LIBNET_RESOLVE))==-1)
					libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: invalid target: %s\n", str_dst_ip);
				break;
			case 'p':
				// first_port and last_port
				if(!(fport=atoi(optarg))) error("atoi: invalid first port", -1);
				for(ptr=optarg; *ptr && *ptr!='-'; ptr++);
				if(!(lport=atoi(++ptr))) error("atoi: invalid last port", -1);
				if(lport<fport) error("Invalid first and last ports", -1);
				serv_scan=0;
				break;
			case 'k':
				// scan known ports
				if(optarg[0]=='0') serv_scan=0;
				else serv_scan=1;
				break;
			case 't':
				// scan_type
				     if(!strncmp("syn",  optarg, 3)) scan_type=SYN_SCAN;
				else if(!strncmp("fin",  optarg, 3)) scan_type=FIN_SCAN;
				else if(!strncmp("ack",  optarg, 3)) scan_type=ACK_SCAN;
				else if(!strncmp("xmas", optarg, 4)) scan_type=XMAS_SCAN;
				else if(!strncmp("null", optarg, 4)) scan_type=NULL_SCAN;
				else if(!strncmp("rand", optarg, 4)) scan_type=RAND_SCAN;
				else if(!strncmp("fwrd", optarg, 4)) scan_type=FWRD_SCAN;
				else error("Invalid scan_type", -1);
				break;
			case 'r':
				// scan_speed
				     if(!strncmp("paranoid",   optarg, 8 )) scan_speed=500000;
				else if(!strncmp("polite",     optarg, 6 )) scan_speed=100000;
				else if(!strncmp("normal",     optarg, 6 )) scan_speed=50000;
				else if(!strncmp("aggressive", optarg, 10)) scan_speed=10000;
				else if(!strncmp("insane",     optarg, 6 )) scan_speed=0;
				else error("Invalid scan_speed", -1);
				break;
			case 'e':
				if(!(ping_port=atoi(optarg)))
				{
					libnet_error(LIBNET_ERR_WARNING, "Invalid ping_port (using port 80)\n");
					ping_port=80;
				}
				break;
			case 'v':
				verbose++;
				break;
			case 'f':
				switch(optarg[0])
				{
					case 't':
					case 'T':
						fragment=TINY_FRAG;
						break;
					case 'o':
					case 'O':
						fragment=OVERLAP_FRAG;
						if(spoofing!=MAC_SPOOF) error("MAC spoofing is required for fragment overlapping", -1);
						break;
					default:
						usage(argv[0]);
						break;
				}
				break;
			case 'i':
				eth=optarg;
				break;
			case 'S':
				if(!strncmp("ip", optarg, 2))
				{
					if(fragment==OVERLAP_FRAG) error("MAC spoofing is required for fragment overlapping", -1);
					else spoofing=IP_SPOOF;
				}
				else if(!strncmp("mac", optarg, 3)) spoofing=MAC_SPOOF;
				break;
			case '?':
			default:
				usage(argv[0]);
		}
	}
	if(!nb_src_ip || !dst_ip) usage(argv[0]);

	// define the number of retries
	switch(scan_type)
	{
		case SYN_SCAN:
		case ACK_SCAN:
		case FWRD_SCAN:
			max_retries=5;
			break;
		default:
			max_retries=10;
			break;
	}


	// allocate memory for data storage and initialize data
	gotcha[1]=0;
	nbports=lport-fport+1;
	if(st && serv_scan)
	{
		for(i=0; st->tab[i].port<fport && i<st->nb; i++) nbports++;
		for(i=st->nb-1; st->tab[i].port>lport && i>=0; i--) nbports++;
	}
	if(!(tab_port=(port_elt *)malloc(nbports*sizeof(port_elt)))) error(NULL, -2);
	// setup some packet parameters (port, src_ip, seq..)
	i=0;
	if(st && serv_scan) for(; st->tab[i].port<fport && i<st->nb; i++) init_port(i, st->tab[i].port);
	for(j=0; j<lport-fport+1; j++) init_port(i++, (u_int16_t)(j+fport));
	if(st && serv_scan) for(j=st->nb-1, i=nbports-1; st->tab[j].port>lport && j>=0; j--, i--) init_port(i, st->tab[j].port);


	// get an ethernet device name, if none is specified
	if(!eth) if(!(eth=pcap_lookupdev(errbuf))) error(errbuf, -1);
	fprintf(stdout, "Network device: %s\n", eth);


	// if MAC spoofing is activated, get the MAC address of
	// the host or gateway to which send the ethernet trames
	if(spoofing==MAC_SPOOF)
	{
		if(verbose) fprintf(stdout, "Searching for the destination MAC address\n");
		// get the local nework IP, and the netmask
		if(pcap_lookupnet(eth, &net_ip, &netmask, errbuf))
			libnet_error(LIBNET_ERR_FATAL, "pcap_lookupnet failed (%s)\n", errbuf);
		// see if we can reach the host directly
		if(IS_ON_LOCAL_NET(dst_ip, netmask, net_ip))
		{
			if(get_mac(FARP, dst_ip, dst_mac, eth)==-1)
				if(send_ARP_request(eth, &src_ip[0].ip, &dst_ip, src_ip[0].ea, dst_mac)==-1)
					libnet_error(LIBNET_ERR_FATAL, "Not able to get the MAC address of host: %s\n", libnet_host_lookup(dst_ip, LIBNET_RESOLVE));
		}
		else
		{
			// this is not very smartly handled, as far as get_gateway_ip returns
			// the first gateway IP it finds in FROUTE (/proc/net/route)
			if((ip=get_gateway_ip(FROUTE))==-1)
				libnet_error(LIBNET_ERR_FATAL, "get_gateway_ip: not able to find a gateway\n");
			if(get_mac(FARP, ip, dst_mac, eth)==-1)
				if(send_ARP_request(eth, &src_ip[0].ip, &ip, src_ip[0].ea, dst_mac)==-1)
					libnet_error(LIBNET_ERR_FATAL, "Not able to get the MAC address of host: %s\n", libnet_host_lookup(ip, LIBNET_RESOLVE));
		}
		if(verbose>2)
		{
			fprintf(stdout, "Destination MAC address: ");
			for(i=0; i<ETH_ALEN; i++) fprintf(stdout, "%2x%c", dst_mac[i], (i==ETH_ALEN-1)?'\n':':');
		}
	}


	// clone the thread and setup a communication link between parent and child
	// parent thread: sniffer
	// child  thread: packet generator
	if(socketpair(AF_UNIX, SOCK_STREAM, 0, sockets)<0) error("socketpair: creation failed", -1);
	if((npid=fork())==-1) error("fork: thread duplication failed", -1);
	if(!npid) // child
	{
for(i=0; i<20; i++) signal(i, pg_sig);
		close(sockets[0]);
		switch(spoofing)
		{
			case IP_SPOOF:
				ip_packet_generator(scan_speed);
				break;
			case MAC_SPOOF:
				ethernet_packet_generator(scan_speed);
				break;
		}
	}
	close(sockets[1]);

	// allocate memory for data
	if(!(tab_response=(u_int16_t *)malloc(nbports*sizeof(u_int16_t)))) error(NULL, -2);
	if(scan_type==FWRD_SCAN) if(!(tab_fwrd=(fwrd_elt *)malloc(nb_src_ip*nbports*sizeof(fwrd_elt)))) error(NULL, -2);

	// open a raw link interface to send spoofed ARP replies if needed (something like ARP poisoning)
	if(!(interface=libnet_open_link_interface(eth, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed (%s)\n", errbuf);

	// open network device in promiscuous mode
	if(!(descr=pcap_open_live(eth, 65536, 1, -1, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "pcap_open_live failed (%s)\n", errbuf);


	// inform the child process that the sniffer is ready
	i=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && i++<3);
	if(i>2) error("write: operation on socket failed", -1);

	// unblock the pcap socket
	unblock_socket(descr->fd);

	// evaluate timeout value
	if(verbose>1) fprintf(stdout, "Evaluating a timeout value by TCP pinging port %d\n", ping_port);
	i=k=0;
	do
	{
		if(read_ok)
		{
			if(read(sockets[0], &sckelt, sizeof(sock_elt))>=0)
			{
				read_ok=0;
				gettimeofday(&tv, NULL);
			}
			else continue;
		}
		if((packet=pcap_next(descr, &phdr)))
		{
			ehdr=(struct ether_header *)packet;
			switch(ntohs(ehdr->ether_type))
			{
				case ETHERTYPE_ARP: // ARP packet
					// send a fake ARP reply if needed
					ahdr=(struct ether_arp *)((u_char *)ehdr + ETHER_HDR_LEN);
					if(ntohs(ahdr->ea_hdr.ar_op)!=ARPOP_REQUEST) break;
					if((j=search_src_ip(*((u_int32_t *)ahdr->arp_tpa)))==-1) break;
					duplic=*ahdr;
					ahdr->ea_hdr.ar_op=ntohs(ARPOP_REPLY);
					for(k=0; k<ETH_ALEN; k++)
					{
						ahdr->arp_sha[k]=ehdr->ether_shost[k]=(u_int8_t)src_ip[j].ea[k];
						ahdr->arp_tha[k]=ehdr->ether_dhost[k]=duplic.arp_sha[k];
					}
					for(k=0; k<4; k++)
					{
						ahdr->arp_spa[k]=duplic.arp_tpa[k];
						ahdr->arp_tpa[k]=duplic.arp_spa[k];
					}
					if(libnet_write_link_layer(interface, eth, (u_char *)packet, ETHER_HDR_LEN + sizeof(struct ether_arp))==-1)
						libnet_error(LIBNET_ERR_WARNING, "libnet_write_link layer failed\n");
					break;
				case ETHERTYPE_IP: // IP datagram
					ihdr=(struct ip *)((u_char *)ehdr + ETHER_HDR_LEN);
					if(ihdr->ip_src.s_addr!=dst_ip || search_src_ip(ihdr->ip_dst.s_addr)==-1 || ihdr->ip_p!=IPPROTO_TCP || (ihdr->ip_off & !IP_DF)) break;
					thdr=(struct tcphdr *)((u_char *)ihdr + 4 * ihdr->ip_hl);
					if(!(thdr->th_flags & TH_RST)) break;
					gettimeofday(&current, NULL);
					i++;
					timeout+=TIMEVAL_MSEC_DIFF(current, sckelt.tv);
					if(verbose>2) fprintf(stdout, "Time to get a RST back: %d ms\n", timeout);
					if(i<3)
					{
						k=0; // reset the timeout counter
						sckelt.i=1;
						j=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && j++<3);
						if(j<3) read_ok=1;
						else error("write: operation on socket failed", -1);
					}
					break;
			}
		}

		// handle timeout
		if(i<3)
		{
			gettimeofday(&current, NULL);
			if(TIMEVAL_MSEC_DIFF(current, tv)>PING_TIMEOUT)
			{
				if(++k>3) error("Too much timeouts during the timeout evaluation", -1);
				sckelt.i=1;
				j=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && j++<3);
				if(j<3) read_ok=1;
				else error("write: operation on socket failed", -1);
			}
		}
	} while(i<3);
	max_timeout=(TIMEOUT_FACTOR*timeout)/i;
	// inform the packet generator that timeout evaluation is over
	sckelt.i=i=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && i++<3);
	if(i>2) error("write: operation on socket failed", -1);

	if(verbose>0) fprintf(stdout, "Timeout value: %d ms\n", max_timeout);


	// unblock sockets[0]
	unblock_socket(sockets[0]);


	// loop until all packets have been sent and all replies received
	while(sckelt.i!=-2)
	{
		// grab a packet from network
		if((packet=pcap_next(descr, &phdr)))
		{
			// packet filter
			ehdr=(struct ether_header *)packet;
			switch(ntohs(ehdr->ether_type))
			{
				case ETHERTYPE_ARP: // ARP packet
					ahdr=(struct ether_arp *)((u_char *)ehdr + ETHER_HDR_LEN);
					// break if it is not an ARP request
					if(ntohs(ahdr->ea_hdr.ar_op)!=ARPOP_REQUEST) break;
					// break if the ARP request do not concern
					// the hosts we pretend to be
					if((j=search_src_ip(*((u_int32_t *)ahdr->arp_tpa)))==-1) break;
					// we have to reply this ARP request
					duplic=*ahdr;
					ahdr->ea_hdr.ar_op=ntohs(ARPOP_REPLY);
					// build the spoofed ARP reply
					for(i=0; i<ETH_ALEN; i++) // exchange ARP addresses
					{
						ahdr->arp_sha[i]=ehdr->ether_shost[i]=(u_int8_t)src_ip[j].ea[i];
						ahdr->arp_tha[i]=ehdr->ether_dhost[i]=duplic.arp_sha[i];
					}
					for(i=0; i<4; i++) // exchange IP adresses
					{
						ahdr->arp_spa[i]=duplic.arp_tpa[i];
						ahdr->arp_tpa[i]=duplic.arp_spa[i];
					}
					// send the fake ARP reply
					if(libnet_write_link_layer(interface, eth, (u_char *)packet, ETHER_HDR_LEN + sizeof(struct ether_arp))==-1)
						libnet_error(LIBNET_ERR_WARNING, "libnet_write_link layer failed\n");
					break;
				case ETHERTYPE_IP: // IP packet
					ihdr=(struct ip *)((u_char *)ehdr + ETHER_HDR_LEN);
					// break if the packet do not come from the target host
					// or if it is not addressed to the hosts we pretend to be
					// or if it is fragmented, or if it is not a TCP packet
					if(ihdr->ip_src.s_addr!=dst_ip || ((scan_type!=FWRD_SCAN) ? (search_src_ip(ihdr->ip_dst.s_addr)==-1) : (ihdr->ip_dst.s_addr!=src_ip[i_ip].ip)) || (ihdr->ip_off & !IP_DF) || ihdr->ip_p!=IPPROTO_TCP) break;
					thdr=(struct tcphdr *)((u_char *)ihdr + 4 * ihdr->ip_hl);
					// test packet flags
					i=0;
					switch(scan_type)
					{
						case SYN_SCAN:			// SYN scan
							if(thdr->th_flags!=(TH_SYN|TH_ACK)) i=1;
							if(thdr->th_flags & TH_RST)
								if((j=search_port(ntohs(thdr->th_sport)))!=-1)
									tab_port[j].gotcha=1;
							break;
						case ACK_SCAN:			// ACK scan
						case FIN_SCAN:			// FIN scan
						case XMAS_SCAN:			// Xmas Tree scan
						case NULL_SCAN:			// NULL scan
						case RAND_SCAN:			// RAND scan
							if(!(thdr->th_flags & TH_RST)) i=1;
							break;
						case FWRD_SCAN:			// FWRD scan
							if(thdr->th_flags & TH_RST) i=1;
							// we have a matching packet, see if we already have it
							if(search_fwrd(ihdr->ip_dst.s_addr, thdr->th_sport)==-1)
							{
								tab_fwrd[nbopt].ip.s_addr=ihdr->ip_dst.s_addr;
								tab_fwrd[nbopt++].port=thdr->th_sport;
								if((j=search_port(ntohs(thdr->th_sport)))!=-1) tab_port[j].gotcha=1;
							}
							nbpkt++;
							break;
					}
					if(i) break;
					// we have a matching packet, see if we already have it
					if((i=search_port(ntohs(thdr->th_sport)))!=-1)
					{
						if(!tab_port[i].gotcha)
						{
							tab_port[i].gotcha=1;
							tab_response[nbopt++]=thdr->th_sport;
						}
					}
					nbpkt++;
					break;
			}
		}

		if(read(sockets[0], &sckelt, sizeof(sock_elt))>=0) switch(sckelt.i)
		{
			case -1: // first blast over
				if(verbose>0)
				{
					fprintf(stdout, "First blast over");
					if(scan_type==FWRD_SCAN)
					{
						tmp_inet.s_addr=src_ip[i_ip].ip;
						fprintf(stdout, " for %s", inet_ntoa(tmp_inet));
					}
					fprintf(stdout, "\n");
				}
				break;
			case -2: // all packets have been sent
				continue;
			case -3: // ready to send packets from next IP (FWRD mode)
				break;
			default:
				tab_port[sckelt.i].tv=sckelt.tv;
				tab_port[sckelt.i].nbretries++;
				tab_port[sckelt.i].wait=0;
				break;
		}

		// handle timeout
		if(!gotcha[1])
		{
			gotcha[0]=1;
			gettimeofday(&tv, NULL);
			for(i=0; i<nbports; i++)
			{
				if(!tab_port[i].gotcha)
				{
					if(tab_port[i].nbretries>max_retries) tab_port[i].gotcha=1;
					else
					{
						gotcha[0]=0;
						if(!tab_port[i].wait)
						{
							if(TIMEVAL_MSEC_DIFF(tv, tab_port[i].tv)>max_timeout)
							{
								sckelt.i=i;
								j=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && j++<3);
								tab_port[i].wait=1;
							}
						}
					}
				}
			}

			if(gotcha[0]) // all packets have been received; inform the child thread
			{
				if(verbose>0)
				{
					fprintf(stdout, "All packets");
					if(scan_type==FWRD_SCAN)
					{
						tmp_inet.s_addr=src_ip[i_ip].ip;
						fprintf(stdout, " for %s", inet_ntoa(tmp_inet));
					}
					fprintf(stdout, " have been received\n");
				}
				gotcha[1]=1;
				sckelt.i=-1;
				i=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && i++<3);
				while(sckelt.i>-2) read(sockets[0], &sckelt, sizeof(sock_elt));
			}
		}

		// if FWRD mode, reset port related data
		if(sckelt.i==-3 && scan_type==FWRD_SCAN)
		{
			i_ip++;
			gotcha[0]=gotcha[1]=0;
			for(i=0; i<nbports; i++)
				tab_port[i].nbretries=tab_port[i].gotcha=tab_port[i].wait=0;
		}
	}
	close(sockets[0]);

	if(verbose>0) fprintf(stdout, "Computing results..\n");

	// results:
	switch(scan_type)
	{
		case FWRD_SCAN: // fwrd scan
			// order and handle byte swap
			for(i=0; i<nbopt; i++)
			{
				tab_fwrd[i].port=ntohs(tab_fwrd[i].port);
				tab_fwrd[i].ip.s_addr=ntohl(tab_fwrd[i].ip.s_addr);
			}
			for(i=0; i<nbopt; i++)
				for(j=i+1; j<nbopt; j++)
					if(tab_fwrd[i].ip.s_addr>tab_fwrd[j].ip.s_addr || (tab_fwrd[i].ip.s_addr==tab_fwrd[j].ip.s_addr && tab_fwrd[i].port>tab_fwrd[j].port))
					{
						tmp_fwrd=tab_fwrd[i];
						tab_fwrd[i]=tab_fwrd[j];
						tab_fwrd[j]=tmp_fwrd;
					}
			for(i=0; i<nbopt; i++) tab_fwrd[i].ip.s_addr=ntohl(tab_fwrd[i].ip.s_addr);
			// output
			if(nbopt==nb_src_ip*nbports)
				fprintf(stdout, "None of the scanned ports are filtered for all specified IP addresses\n");
			else
				for(i=0; i<nbopt;)
				{
					for(j=i; j<nbopt && tab_fwrd[j].ip.s_addr==tab_fwrd[i].ip.s_addr; j++);
					if((j-i)==nbports) fprintf(stdout, "None of the scanned ports are filtered for %s\n", inet_ntoa(tab_fwrd[i].ip));
					else if(j-i<nbports/2)
					{
						fprintf(stdout, "Ports: ");
						for(j=i; j<nbopt && tab_fwrd[j].ip.s_addr==tab_fwrd[i].ip.s_addr; j++)
							fprintf(stdout, "%u, ", tab_fwrd[j].port);
						fprintf(stdout, "\b\b are not filtered for %s\n", inet_ntoa(tab_fwrd[i].ip));
					}
					else
					{
						fprintf(stdout, "Ports: ");
						if(st && serv_scan)
							for(k=0; k<st->nb && st->tab[k].port<fport; k++)
								if(search_fwrd(tab_fwrd[i].ip.s_addr, st->tab[i].port)==-1)
									fprintf(stdout, "%u, ", st->tab[k].port);
						for(port=fport; port<=lport; port++)
							if(search_fwrd(tab_fwrd[k].ip.s_addr, port)==-1)
								fprintf(stdout, "%u, ", port);
						if(st && serv_scan)
						{
							for(k=0; k<st->nb && st->tab[k].port<=lport; k++);
							for(; k<st->nb; k++)
								if(search_fwrd(tab_fwrd[i].ip.s_addr, st->tab[k].port)==-1)
									fprintf(stdout, "%u, ", st->tab[k].port);
						}
						fprintf(stdout, "\b\b are filtered for %s\n", inet_ntoa(tab_fwrd[i].ip));
					}
					i=j;
				}
			break;

		default:
			// order and handle byte swap
			for(i=0; i<nbopt; i++) tab_response[i]=ntohs(tab_response[i]);
			for(i=0; i<nbopt; i++)
				for(j=i+1; j<nbopt; j++)
					if(tab_response[i]>tab_response[j])
					{
						port=tab_response[i];
						tab_response[i]=tab_response[j];
						tab_response[j]=port;
					}

			// output
			switch(scan_type)
			{
				case SYN_SCAN: // print results for SYN scan
					if(!nbopt) fprintf(stdout, "No open ports found\n");
					else
					{
						for(i=0; i<nbopt; i++)
						{
							if((known=services_known(tab_response[i], st))==-1)
								fprintf(stdout, "Port %u is open\n", tab_response[i]);
							else // port service detail
								fprintf(stdout, "Port %u: %s (%s) is open\n", tab_response[i], st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
						}
					}
					break;
				case FIN_SCAN:			// print results for FIN scan,
				case XMAS_SCAN:			// for Xmas tree scan,
				case NULL_SCAN:			// for Null scan,
				case RAND_SCAN:		// for rand scan
					if(nbopt==nbports) // no open ports found
						fprintf(stdout, "No open ports found\n");
					else
					{
						if(st && serv_scan)
							for(i=0; i<st->nb && st->tab[i].port<fport; i++)
								fin_scan_disp(st->tab[i].port);
						for(port=fport; port<=lport; port++)
							fin_scan_disp(port);
						if(st && serv_scan)
						{				
							for(i=0; i<st->nb && st->tab[i].port<=lport; i++);
							for(; i<st->nb; i++) fin_scan_disp(st->tab[i].port);
						}
					}
					break;
				case TH_ACK: // ACK scan
					if(!nbopt)
						fprintf(stdout, "All scanned ports are filtered\n");
					else if(nbopt==nbports)
						fprintf(stdout, "None of the scanned ports are filtered\n");
					else if(nbopt<nbports/2)
					{
						fprintf(stdout, "Ports: ");
						for(i=0; i<nbopt; i++) fprintf(stdout, "%u, ", tab_response[i]);
						fprintf(stdout, "\b\b are not filtered\n");
					}
					else
					{
						fprintf(stdout, "Ports: ");
						if(st && serv_scan)
							for(i=0; i<st->nb && st->tab[i].port<fport; i++)
								if(search_tab(st->tab[i].port)==-1)
									fprintf(stdout, "%u, ", st->tab[i].port);
						for(port=fport; port<=lport; port++)
							if(search_tab(port)==-1)
								fprintf(stdout, "%u, ", port);
						if(st && serv_scan)
						{
							for(i=0; i<st->nb && st->tab[i].port<=lport; i++);
							for(; i<st->nb; i++)
								if(search_tab(st->tab[i].port)==-1)
									fprintf(stdout, "%u, ", st->tab[i].port);
						}
						fprintf(stdout, "\b\b are filtered\n");
					}
					break;
			}
			break;
	}
	fprintf(stdout, "%d ports scanned\nGrabbed %d matching packets\n", nbports, nbpkt);

	// free & close everything
	if(scan_type==FWRD_SCAN) free(tab_fwrd);
	free(tab_response);
	free(tab_port);
	free(src_ip);
	pcap_close(descr);
	if(libnet_close_link_interface(interface)==-1)
		libnet_error(LIBNET_ERR_WARNING, "libnet_close_link_interface failed\n");
	exit(0);
}

void get_ea(char *ea)
{
	int i;
	for(i=0; i<ETH_ALEN; i++) ea[i]=(u_char)libnet_get_prand(rand_seed);
}

void init_port(int i, u_int16_t port)
{
	int j;
	tab_port[i].ip=(scan_type==FWRD_SCAN) ? 0 : src_ip[(nb_src_ip==1) ? 0 : (libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed))%nb_src_ip].ip;
	tab_port[i].port=port;
	do { tab_port[i].src_port=libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed); } while(!tab_port[i].src_port);
	do { tab_port[i].seq=1; for(j=0; j<4; j++) tab_port[i].seq*=libnet_get_prand(rand_seed); } while(!tab_port[i].seq);
	do { tab_port[i].ack=1; for(j=0; j<4; j++) tab_port[i].ack*=libnet_get_prand(rand_seed); } while(!tab_port[i].ack);
	do { tab_port[i].ip_id=libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed); } while(!tab_port[i].ip_id);
	tab_port[i].nbretries=0;
	tab_port[i].tv.tv_sec=tab_port[i].tv.tv_usec=0;
	tab_port[i].gotcha=0;
	tab_port[i].wait=1;
}

int search_port(u_int16_t port)
{
	int i=0, j=nbports-1, m;
	do {
		m=(i+j)/2;
		if(tab_port[m].port==port) return(m);
		if(tab_port[m].port<port) i=m+1;
		else j=m-1;
	} while (i<=j);
	return(-1);
}

int search_fwrd(u_int32_t ip, u_int16_t port)
{
	int i;
	for(i=0; i<nbopt && (tab_fwrd[i].ip.s_addr!=ip || tab_fwrd[i].port!=port); i++);
	if(i<nbopt) return(i);
	else return(-1);
}

void fin_scan_disp(u_int16_t pt)
{
	int known;
	if(search_tab(pt)==-1)
	{
		if((known=services_known(pt, st))==-1)
			fprintf(stdout, "Port %u is open or filtered\n", pt);
		else // port service detail
			fprintf(stdout, "Port %u: %s (%s) is open or filtered\n", pt, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
	}
}

int search_tab(u_int16_t pt)
{
	int m, i=0, j=nbopt-1;
	do {
		m=(i+j)/2;
		if(tab_response[m]==pt) return(m);
		if(tab_response[m]<pt) i=m+1;
		else j=m-1;
	} while (i<=j);
	return(-1);
}

int search_src_ip(u_int32_t ip)
{
	int i;
	for(i=0; i<nb_src_ip; i++)
		if(src_ip[i].ip==ip) return(i);
	return(-1);
}

void unblock_socket(int fd)
{
	int options=O_NONBLOCK | fcntl(fd, F_GETFL);
	fcntl(fd, F_SETFL, options);
}


#define NB_RAND_FLAGS 8

void ip_packet_generator(int scan_speed)
{
	int raw_sock, i, j, k, th_flags=scan_type, i_ip=0, offset;
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	char *packets=NULL, *frag_packets=NULL;
	u_char *packet;
	u_int32_t rsrc_ip=src_ip[0].ip;
	sock_elt sckelt;
	port_elt tmp_port;
	u_int16_t *tab_switch_o2r=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	u_int16_t *tab_switch_r2o=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	if(!tab_switch_o2r||!tab_switch_r2o) error(NULL, -2);
	for(i=0; i<nbports; i++) tab_switch_r2o[i]=i;

	//wait for the sniffer to be ready
	i=0; while(read(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);

	// open a raw IP socket
	if((raw_sock=libnet_open_raw_sock(IPPROTO_RAW))==-1)
		libnet_error(LIBNET_ERR_FATAL, "libnet_open_raw_sock failed\n");

	// set up TCP flags for fwrd scan
	if(scan_type==FWRD_SCAN) th_flags=TH_ACK;


	// send some packets to evaluate a timeout value
	libnet_init_packet(GPS_PACKET, &packet);
	if(!packet) error(NULL, -2);
	do
	{
		libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), 0, 255, IPPROTO_TCP, src_ip[0].ip, dst_ip, NULL, 0, packet);
		libnet_build_tcp(libnet_get_prand(rand_seed)+1024, ping_port, 65535*libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), 65535*libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), TH_ACK, 1024, 0, NULL, 0, packet+GPS_IP_H);
		if(libnet_do_checksum(packet, IPPROTO_TCP, GPS_TCP_H)==-1)
			libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
		gettimeofday(&sckelt.tv, NULL);
		i=0; while(libnet_write_ip(raw_sock, packet, GPS_PACKET)<GPS_PACKET && i++<3);
		i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
		i=0; while(read(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
	} while(sckelt.i);
	libnet_destroy_packet(&packet);

	// allocate memory for packets
	if(!(packets=(char *)malloc(nbports*GPS_PACKET))) error(NULL, -2);
	if(fragment!=NO_FRAG) if(!(frag_packets=(char *)malloc(nbports*GPS_PACKET))) error(NULL, -2);
	do
	{
		// randomize port scan order.
		// Port order is randomized for the packet generator
		// and ordered for the sniffer; tab_switch_r2o/o2r
		// are used to translate randomized offset to ordered
		// offset, and vice-versa
		for(i=0; i<nbports; i++)
		{
			tmp_port=tab_port[i];
			j=(nbports*libnet_get_prand(rand_seed))/255;
			tab_port[i]=tab_port[j];
			tab_port[j]=tmp_port;
			k=tab_switch_r2o[i];
			tab_switch_r2o[i]=tab_switch_r2o[j];
			tab_switch_r2o[j]=k;
		}
		for(i=0; i<nbports; i++) tab_switch_o2r[tab_switch_r2o[i]]=i;


		// build and send packets
		offset=0;
		for(i=0; i<nbports; i++)
		{
			// get random flags for RAND scan
			if(scan_type==RAND_SCAN)
				th_flags=rflags[libnet_get_prand(rand_seed)%NB_RAND_FLAGS];

			// get an IP for ! FWRD scan
			if(scan_type!=FWRD_SCAN) rsrc_ip=tab_port[i].ip;

			switch(fragment)
			{
				case NO_FRAG:
					// build an IP datagram with a spoofed src_ip
					libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, tab_port[i].ip_id, 0, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_IP_H);
					// if the checksum is ok, send it
					if(libnet_do_checksum(packets+offset, IPPROTO_TCP, GPS_TCP_H)!=-1)
						libnet_write_ip(raw_sock, packets+offset, GPS_PACKET);
					break;
				case TINY_FRAG: // build two tiny fragments
					// build the first fragment
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_IP_H);
					libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[i].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
					if(libnet_do_checksum(packets+offset, IPPROTO_TCP, GPS_TCP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// build the second fragment
					memcpy(frag_packets+offset+GPS_IP_H, packets+offset+GPS_PACKET-4, 4);
					libnet_build_ip(GPS_TCP_H-16, IPTOS_LOWDELAY, tab_port[i].ip_id, 2 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset);
					// send them
					libnet_write_ip(raw_sock, frag_packets+offset, GPS_PACKET-16);
					libnet_write_ip(raw_sock, packets+offset, GPS_PACKET-4);
					break;
			}

			// inform the sniffer that the packet is sent
			sckelt.i=tab_switch_r2o[i];
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

			// wait a bit
			usleep(scan_speed);

			offset+=GPS_PACKET;
		}

		// inform the sniffer that the first blast is over
		sckelt.i=-1;
		j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);


		// resend dead packets (given by the sniffer)
		while(1)
		{
			if(read(sockets[1], &sckelt, sizeof(sock_elt))<0) continue;
			if(sckelt.i==-1) break; // break if we received all replies
			// the packet(s) is/are already made, just send it/them
			offset=tab_switch_o2r[sckelt.i]*GPS_PACKET;
			switch(fragment)
			{
				case NO_FRAG:
					libnet_write_ip(raw_sock, packets+offset, GPS_PACKET);
					break;
				case TINY_FRAG:
					libnet_write_ip(raw_sock, frag_packets+offset, GPS_PACKET-16);
					libnet_write_ip(raw_sock, packets+offset, GPS_PACKET-4);
					break;
			}
			// inform the sniffer that the packet(s) is/are sent
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);
			usleep(scan_speed);
		}


		i_ip++;
		if(scan_type==FWRD_SCAN && nb_src_ip>1 && i_ip<nb_src_ip)
		{
			// next IP
			rsrc_ip=src_ip[i_ip].ip;
			// inform the sniffer that we do it again
			sckelt.i=-3;
			i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
		}
	} while(i_ip<nb_src_ip && scan_type==FWRD_SCAN);

	// inform the sniffer that all is over
	sckelt.i=-2;
	i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

	// close and free
	close(sockets[1]);
	libnet_close_raw_sock(raw_sock);
	free(packets);
	if(fragment!=NO_FRAG) free(frag_packets);
	free(tab_switch_o2r);
	free(tab_switch_r2o);
	free(tab_port);
	free(src_ip);

	exit(0);
}


void ethernet_packet_generator(int scan_speed)
{
	int i, j, k, th_flags=scan_type, i_ip=0, offset;
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	char *packets=NULL, *frag_packets=NULL;
	u_char *packet, *src_mac=src_ip[0].ea;
	struct tcphdr *thdr;
	u_int32_t rsrc_ip=src_ip[0].ip;
	sock_elt sckelt;
	port_elt tmp_port;
	u_int16_t *tab_switch_o2r=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	u_int16_t *tab_switch_r2o=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	if(!tab_switch_o2r||!tab_switch_r2o) error(NULL, -2);
	for(i=0; i<nbports; i++) tab_switch_r2o[i]=i;

	//wait for the sniffer to be ready
	i=0; while(read(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);

	// open a link layer interface
	if(!(interface=libnet_open_link_interface(eth, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed (%s)\n", errbuf);

	// set up TCP flags for fwrd scan
	if(scan_type==FWRD_SCAN) th_flags=TH_ACK;


	// send some packets to evaluate a timeout value
	libnet_init_packet(GPS_TRAME, &packet);
	if(!packet) error(NULL, -2);
	do
	{
		libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), 0, 255, IPPROTO_TCP, src_ip[0].ip, dst_ip, NULL, 0, packet+GPS_ETH_H);
		libnet_build_tcp(libnet_get_prand(rand_seed)+1024, ping_port, 65535*libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), 65535*libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed), TH_ACK, 1024, 0, NULL, 0, packet+GPS_IP_H+GPS_ETH_H);
		if(libnet_do_checksum(packet+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packet+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
			libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
		libnet_build_ethernet(dst_mac, src_ip[0].ea, ETHERTYPE_IP, NULL, 0, packet);
		gettimeofday(&sckelt.tv, NULL);
		i=0; while(libnet_write_link_layer(interface, eth, packet, GPS_TRAME)<GPS_TRAME && i++<3);
		i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
		i=0; while(read(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
	} while(sckelt.i);
	libnet_destroy_packet(&packet);

	// allocate memory for packets
	if(!(packets=(char *)malloc(nbports*GPS_TRAME))) error(NULL, -2);
	if(fragment!=NO_FRAG) if(!(frag_packets=(char *)malloc(nbports*GPS_TRAME))) error(NULL, -2);
	do
	{
		// randomize port scan order.
		// Port order is randomized for the packet generator
		// and ordered for the sniffer; tab_switch_r2o/o2r
		// are used to translate randomized offset to ordered
		// offset, and vice-versa
		for(i=0; i<nbports; i++)
		{
			tmp_port=tab_port[i];
			j=(nbports*libnet_get_prand(rand_seed))/255;
			tab_port[i]=tab_port[j];
			tab_port[j]=tmp_port;
			k=tab_switch_r2o[i];
			tab_switch_r2o[i]=tab_switch_r2o[j];
			tab_switch_r2o[j]=k;
		}
		for(i=0; i<nbports; i++) tab_switch_o2r[tab_switch_r2o[i]]=i;


		// build and send packets
		offset=0;
		for(i=0; i<nbports; i++)
		{
			// get random flags for RAND scan
			if(scan_type==RAND_SCAN)
				th_flags=rflags[libnet_get_prand(rand_seed)%NB_RAND_FLAGS];

			// get an IP and a MAC address for !FWRD scan
			if(scan_type!=FWRD_SCAN)
			{
				rsrc_ip=tab_port[i].ip;
				src_mac=src_ip[search_src_ip(rsrc_ip)].ea;
			}

			switch(fragment)
			{
				case NO_FRAG:
					// build an IP datagram with a spoofed src_ip
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
					libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, tab_port[i].ip_id, 0, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
					// if the checksums are ok, send it
					if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)!=-1 && libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)!=-1)
						libnet_write_link_layer(interface, eth, packets+offset, GPS_TRAME);
					break;
				case TINY_FRAG: // build two tiny fragments
					// build the first fragment
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
					libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[i].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
					if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// build the second fragment
					memcpy(frag_packets+offset+GPS_ETH_H+GPS_IP_H, packets+offset+GPS_TRAME-4, 4);
					libnet_build_ip(GPS_TCP_H-16, IPTOS_LOWDELAY, tab_port[i].ip_id, 2 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, frag_packets+offset);
					if(libnet_do_checksum(frag_packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// send them
					libnet_write_link_layer(interface, eth, frag_packets+offset, GPS_TRAME-16);
					libnet_write_link_layer(interface, eth, packets+offset, GPS_TRAME-4);
					break;
				case OVERLAP_FRAG: // NOTE: most of recent OSes drop IP
						   // datagrams with PROTO=TCP and OFFSET=1
					// build two fragments,
					// the first fragment has TCP flags set to 0 (offset 0)
					// the second has the real TCP flags (offset 1).
					// The second frag overlaps the first at packet reconstruction
					// (the TCP flags are overwitten)
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
					libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[i].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
					if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					memcpy(frag_packets+offset+GPS_ETH_H+GPS_IP_H, packets+offset+GPS_ETH_H+GPS_TRAME-12, 12);
					libnet_build_ip(GPS_TCP_H-8, IPTOS_LOWDELAY, tab_port[i].ip_id, 1 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, frag_packets+offset);
					if(libnet_do_checksum(frag_packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// reset the first frag's TCP flags
					thdr=(struct tcphdr *)(packets+offset+GPS_ETH_H+GPS_IP_H);
					thdr->th_flags=TH_NULL;
					// send them
					libnet_write_link_layer(interface, eth, packets+offset, GPS_TRAME-4);
					libnet_write_link_layer(interface, eth, frag_packets+offset, GPS_TRAME-8);
					break;
			}

			// inform the sniffer that the packet is sent
			sckelt.i=tab_switch_r2o[i];
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

			// wait a bit
			usleep(scan_speed);

			offset+=GPS_TRAME;
		}

		// inform the sniffer that the first blast is over
		sckelt.i=-1;
		j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);


		// resend dead packets (given by the sniffer)
		while(1)
		{
			if(read(sockets[1], &sckelt, sizeof(sock_elt))<0) continue;
			if(sckelt.i==-1) break; // break if we received all replies
			// the packet(s) is/are already made, just send it/them
			offset=tab_switch_o2r[sckelt.i]*GPS_TRAME;
			switch(fragment)
			{
				case NO_FRAG:
					libnet_write_link_layer(interface, eth, packets+offset, GPS_TRAME);
					break;
				case TINY_FRAG:
					libnet_write_link_layer(interface, eth, frag_packets+offset, GPS_TRAME-16);
					libnet_write_link_layer(interface, eth, packets+offset, GPS_TRAME-4);
					break;
				case OVERLAP_FRAG:
					libnet_write_link_layer(interface, eth, packets+offset, GPS_TRAME-4);
					libnet_write_link_layer(interface, eth, frag_packets+offset, GPS_TRAME-8);
					break;
			}
			// inform the sniffer that the packet(s) is/are sent
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);
			usleep(scan_speed);
		}


		i_ip++;
		if(scan_type==FWRD_SCAN && nb_src_ip>1 && i_ip<nb_src_ip)
		{
			// next IP and MAC
			rsrc_ip=src_ip[i_ip].ip;
			src_mac=src_ip[i_ip].ea;
			// inform the sniffer that we do it again
			sckelt.i=-3;
			i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
		}
	} while(i_ip<nb_src_ip && scan_type==FWRD_SCAN);

	// inform the sniffer that all is over
	sckelt.i=-2;
	i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

	// close and free
	close(sockets[1]);
	libnet_close_link_interface(interface);
	free(packets);
	if(fragment!=NO_FRAG) free(frag_packets);
	free(tab_switch_o2r);
	free(tab_switch_r2o);
	free(tab_port);
	free(src_ip);

	exit(0);
}


void usage(char *cmd)
{
fprintf(stderr, "Usage: %s -s src_ip[,src_ip_1[,src_ip_2-4..]] -d target [-t scan_type] [-v]\n\
	   [-r scan_speed] [-p first_port-last_port] [-k 0 | 1] [-e ping_port]\n\
	   [-f t | o] [-i interface] [-S mac | ip]\n\
 -s src_ip[,src_ip_1..]: list of IP addresses of the hosts we pretend to be\n\
 -d target.............: target host's IP/name\n\
 -t scan_type..........: stealth scan mode (default: syn)\n\
 			 (syn | xmas | null | fin | ack | rand | fwrd)\n\
 -r scan_speed.........: packet rate (default: insane)\n\
 			 (insane | aggressive | normal | polite | paranoid)\n\
 -p first-last ports...: port range to scan (default: 1-1024)\n\
 -k 0 | 1..............: scan well-known ports (default: 1)\n\
 -e ping_port..........: target port for a TCP ping (default: 80)\n\
 -v....................: verbose (use twice for more verbose)\n\
 -f t | o..............: fragment IP datagrams (default: no frag)\n\
 			 (t: tiny frags | o: frag overlapping)\n\
 -i interface..........: ethernet interface to use\n\
 -S mac | ip...........: spoofing level (IP or ethernet/MAC; default: mac)\n", cmd);
exit(-1);
}

void error(char *msg, int ret)
{
	switch(ret)
	{
		case -2:
			fprintf(stderr, "malloc: memory allocation failed\n");
			exit(-2);
		default:
			fprintf(stderr, "%s\n", msg);
			exit(ret);
	}
}

