/*
 * Ghost Port Scan is distributed under terms of the GPL (see COPYING)
 *
 * Homepage: gps.sourceforge.net
 * Contact.: whitehat@altern.org
 *
 */

// standard librairies
#include<ctype.h>
#include<stdio.h>
#include<string.h>
#include<stdlib.h>
#include<unistd.h>
#include<signal.h>
#include<sys/types.h>
#include<sys/socket.h>
// network librairies
#include<libnet.h>
#include<pcap.h>
#include "pcap-int.h"
// custom librairies
#include "gps.h"
#include "services.h"
#include "mac-arp.h"
#include "timeout.h"
#include "fwrd.h"
#include "route.h"
#include "src-ip.h"


int nbopt=0, nbports, nb_src_ip=0, rand_seed, sockets[2];
int ping_port=80, fragment=NO_FRAG, spoofing=MAC_SPOOF, scan_type=SYN_SCAN;
u_int32_t dst_ip=0;
services_t *st=NULL;
u_int16_t *tab_response=NULL;
fwrd_elt *tab_fwrd=NULL;
arp_elt *src_ip=NULL;
port_elt *tab_port=NULL;
devlist *dl=NULL;
netdev dev;
struct pcap *descr=NULL;
u_char dst_mac[GPS_ETH_ALEN];
char errbuf[GPS_ERRBUF_SIZE];
extern char *optarg;


// signal handling
void immune2sig()
{
	int i;
	for(i=0; i<SIGCHLD; i++) signal(i, SIG_IGN);
}
void common_sig(int sig)
{
	fprintf(stderr, "): caught signal %d - exiting\n", sig);
	exit(sig);
}
void main_sig(int sig)
{
	immune2sig();
	if(descr) pcap_close(descr);
	fprintf(stderr, "Sniffer (parent");
	switch(sig)
	{
		case SIGCHLD:
			fprintf(stderr, "): death of packet injector - exiting\n");
			break;
		default:
			common_sig(sig);
	}
	exit(sig);
}
void pg_sig(int sig)
{
	immune2sig();
	fprintf(stderr, "Packet injector (child");
	common_sig(sig);
}


int main(int argc, char **argv)
{
	char str_ip[BUFSIZ], *ptr1, *ptr2;
	const u_char *packet;
	u_int16_t port, fport=1, lport=1024;
	u_int32_t ip, net_ip, netmask;
	long ip_diff;
	int npid, scan_speed=0, cmd_parse, i, j, gotcha[2], known, onlocalnet, localhost;
	int i_ip=0, max_retries, max_timeout=0, verbose=0, nbpkt=0, serv_scan=1;
	struct libnet_link_int *interface=NULL;
	struct pcap_pkthdr phdr;
	struct libnet_ethernet_hdr *ehdr;
	struct libnet_arp_hdr *ahdr, duplic;
	struct libnet_ip_hdr *ihdr;
	struct libnet_tcp_hdr *thdr;
	struct timeval tv;
	sock_elt sckelt;
	netdev *nd;


	for(i=0; i<=SIGCHLD; i++) signal(i, main_sig);


	// initialize data
	rand_seed=libnet_seed_prand();
	dev.name[0]='\0';
	// load known services in memory
	st=services_load();

	fprintf(stdout, "Ghost Port Scan version %s by whitehat@altern.org (gps.sourceforge.net)\n", GPS_VERSION);

	// take a look to the arguments:
	if(argc<3) usage(argv[0]);
	while((cmd_parse=getopt(argc, argv, "s:d:p:k:t:r:e:vf:i:S:"))!=EOF)
	{
		switch(cmd_parse)
		{
			case 's':
				// get the src_ip list
				ptr1=optarg;
				j=0; // indicates if we are handling a range of IPs (xxx.xxx.xxx.xxx/yyy.yyy.yyy.yyy)
				while(*ptr1) {
					for(ptr2=ptr1; *ptr1 && *ptr1!=',' && *ptr1!='/'; ptr1++);
					if((i=ptr1-ptr2)>=BUFSIZ) error("Source host name too long", -1);
					strncpy(str_ip, ptr2, i);
					str_ip[i]='\0';
					if((ip=libnet_name_resolve(str_ip, LIBNET_RESOLVE))==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: not able to resolve: %s\n", str_ip);
					if(j) { // if an IP range is specified
						j=0;
						ip_diff=ntohl(ip)-ntohl(src_ip[nb_src_ip-1].ip);
						if(ip_diff<0) error("Invalid source IP addresses range", -1);
						if(!(src_ip=(arp_elt *)realloc(src_ip, (ip_diff+nb_src_ip)*sizeof(arp_elt)))) error(NULL, -2);
						for(i=0; i<ip_diff; i++) {
							src_ip[nb_src_ip+i].ip=src_ip[nb_src_ip-1].ip+ntohl(i+1);
							get_ea(src_ip[nb_src_ip+i].ea, rand_seed);
						}
						nb_src_ip+=ip_diff;
					} else {
						if(!(src_ip=(arp_elt *)realloc(src_ip, (nb_src_ip+1)*sizeof(arp_elt)))) error(NULL, -2);
						src_ip[nb_src_ip].ip=ip;
						get_ea(src_ip[nb_src_ip++].ea, rand_seed);
					}
					if(!*ptr1) break;
					if(*ptr1=='/') j=1;
					ptr1++;
				}
				break;
			case 'd':
				// target host
				strncpy(str_ip, optarg, BUFSIZ-1);
				str_ip[BUFSIZ-1]='\0';
				if((dst_ip=libnet_name_resolve(str_ip, LIBNET_RESOLVE))==-1)
					libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: not able to reslove: %s\n", str_ip);
				break;
			case 'p':
				// first_port and last_port
				if(!(fport=atoi(optarg))) error("atoi: invalid first port", -1);
				for(ptr1=optarg; isdigit(*ptr1); ptr1++);
				switch(*ptr1)
				{
					case 0:
						lport=fport;
						break;
					case '-':
						if(!(lport=atoi(++ptr1))) error("atoi: invalid last port", -1);
						if(lport<fport) error("Invalid first and last ports", -1);
						break;
					default:
						libnet_error(LIBNET_ERR_FATAL, "Invalid argument: %s\n", optarg);
				}
				serv_scan=0;
				break;
			case 'k':
				// scan known ports
				if(optarg[0]=='0') serv_scan=0;
				else serv_scan=1;
				break;
			case 't':
				// scan_type
				     if(!strncmp("syn",  optarg, 3)) scan_type=SYN_SCAN;
				else if(!strncmp("fin",  optarg, 3)) scan_type=FIN_SCAN;
				else if(!strncmp("ack",  optarg, 3)) scan_type=ACK_SCAN;
				else if(!strncmp("xmas", optarg, 4)) scan_type=XMAS_SCAN;
				else if(!strncmp("null", optarg, 4)) scan_type=NULL_SCAN;
				else if(!strncmp("rand", optarg, 4)) scan_type=RAND_SCAN;
				else if(!strncmp("fwrd", optarg, 4)) scan_type=FWRD_SCAN;
				else error("Invalid scan_type", -1);
				break;
			case 'r':
				// scan_speed
				     if(!strncmp("paranoid",   optarg, 8 )) scan_speed=500000;
				else if(!strncmp("polite",     optarg, 6 )) scan_speed=100000;
				else if(!strncmp("normal",     optarg, 6 )) scan_speed=50000;
				else if(!strncmp("aggressive", optarg, 10)) scan_speed=10000;
				else if(!strncmp("insane",     optarg, 6 )) scan_speed=0;
				else error("Invalid scan_speed", -1);
				break;
			case 'e':
				if(!(ping_port=atoi(optarg))) {
					libnet_error(LIBNET_ERR_WARNING, "Invalid ping_port (using port 80)\n");
					ping_port=80;
				}
				break;
			case 'v':
				verbose++;
				break;
			case 'f':
				switch(optarg[0]) {
					case 't':
					case 'T':
						fragment=TINY_FRAG;
						break;
					case 'o':
					case 'O':
						fragment=OVERLAP_FRAG;
						if(spoofing!=MAC_SPOOF)
						{ spoofing=IP_SPOOF; libnet_error(LIBNET_ERR_WARNING, "Fragment overlapping requires spoofing at physical/link level\n"); }
						break;
					default:
						usage(argv[0]);
				}
				break;
			case 'i':
				strncpy(dev.name, optarg, IFNAMSIZ);
				dev.name[IFNAMSIZ-1]='\0';
				if(!strncmp("lo", dev.name, 2) && spoofing==MAC_SPOOF) {
					libnet_error(LIBNET_ERR_WARNING, "Scanning on loopback requires spoofing at IP level\n");
					spoofing=IP_SPOOF;
				}
				break;
			case 'S':
				if(!strncmp("ip", optarg, 2)) {
					if(fragment==OVERLAP_FRAG)
						{ spoofing=MAC_SPOOF; libnet_error(LIBNET_ERR_WARNING, "Fragment overlapping requires spoofing at physical/link level\n"); }
					else spoofing=IP_SPOOF;
				} else {
					spoofing=MAC_SPOOF;
					if(dev.name[0]) if(!strncmp("lo", dev.name, 2))
						{ spoofing=IP_SPOOF; libnet_error(LIBNET_ERR_WARNING, "Scanning on loopback requires spoofing at IP level\n"); }
				}
				break;
			case '?':
			default:
				usage(argv[0]);
		}
	}
	if(!dst_ip) usage(argv[0]);

	// define the number of retries
	switch(scan_type) {
		case SYN_SCAN:
		case ACK_SCAN:
		case FWRD_SCAN:
			max_retries=5;
			break;
		default:
			max_retries=10;
	}


	// guess if we'll scan localhost
	if(!(dl=get_localdevs())) libnet_error(LIBNET_ERR_FATAL, "get_localdevs: not able to get the network device list\n");
	localhost=islocalhost(dl, dst_ip);

	// get a network device name, if none is given
	if(!dev.name[0]) {
		if(localhost) {
			strcpy(dev.name, "lo");
			if(spoofing!=IP_SPOOF) {
				libnet_error(LIBNET_ERR_WARNING, "Scanning your localhost requires spoofing at IP level\n");
				spoofing=IP_SPOOF;
			}
			if(fragment==OVERLAP_FRAG) {
				libnet_error(LIBNET_ERR_WARNING, "Fragment overlapping requires spoofing at physical level, and it is not compliant with scaning through loopback\n");
				fragment=NO_FRAG;
			}
		} else {
			if((nd=get_suitable_dev(dl, dst_ip))) dev=*nd;
			else if((ptr1=pcap_lookupdev(errbuf))) {
				strncpy(dev.name, ptr1, IFNAMSIZ);
				dev.name[IFNAMSIZ-1]='\0';
			}
			else libnet_error(LIBNET_ERR_FATAL, "All attemps to get a suitable network device failed. You should try the -i option to specify the device to use.\n");
			if(strncmp("et", dev.name, 2) && strncmp("lo", dev.name, 2))
				error("Device %s is not yet supported by GPS.", -1);
		}
	}
	fprintf(stdout, "Network device: %s\n", dev.name);


	// if no source IPs have been specified, search for unused IPs
	if(!src_ip) {
		if(localhost) error("You must specify a source IP list if you want to scan your localhost", -1);
		if(verbose) fprintf(stdout, "Looking for IP addresses which are not used on the LAN..\n");
		src_ip=get_src_ip_list(&dev, ping_port, rand_seed, &nb_src_ip, dst_ip);
	}

	
	// allocate memory for data storage and initialize data
	gotcha[1]=0;
	nbports=lport-fport+1;
	if(st && serv_scan) {
		for(i=0; st->tab[i].port<fport && i<st->nb; i++) nbports++;
		for(i=st->nb-1; st->tab[i].port>lport && i>=0; i--) nbports++;
	}
	if(!(tab_port=(port_elt *)malloc(nbports*sizeof(port_elt)))) error(NULL, -2);
	// setup some packet parameters (port, src_ip, seq..)
	i=0;
	if(st && serv_scan) for(; st->tab[i].port<fport && i<st->nb; i++) init_port(i, st->tab[i].port);
	for(j=0; j<lport-fport+1; j++) init_port(i++, (u_int16_t)(j+fport));
	if(st && serv_scan) for(j=st->nb-1, i=nbports-1; st->tab[j].port>lport && j>=0; j--, i--) init_port(i, st->tab[j].port);


	// get the local nework IP, and the netmask
	if(pcap_lookupnet(dev.name, &net_ip, &netmask, errbuf))
		libnet_error(LIBNET_ERR_FATAL, "pcap_lookupnet failed (%s)\n", errbuf);
	// see if we can reach the host directly
	onlocalnet=IS_ON_LOCAL_NET(dst_ip, netmask, net_ip);

	// if MAC spoofing is activated, get the MAC address of
	// the host or gateway to which send the ethernet trames
	if(spoofing==MAC_SPOOF) {
		if(verbose) fprintf(stdout, "Searching for the destination MAC address..\n");
		if(onlocalnet) {
			if(get_mac(dst_ip, dst_mac, dev.name)==-1)
				if(send_ARP_request(dev.name, &src_ip[0].ip, &dst_ip, src_ip[0].ea, dst_mac)==-1)
					libnet_error(LIBNET_ERR_FATAL, "Not able to get the MAC address of host: %s\n", libnet_host_lookup(dst_ip, LIBNET_RESOLVE));
		} else {
			if((ip=get_gateway_ip(dst_ip))==-1)
				libnet_error(LIBNET_ERR_FATAL, "get_gateway_ip: not able to find a suitable gateway\n");
			if(get_mac(ip, dst_mac, dev.name)==-1)
				if(send_ARP_request(dev.name, &src_ip[0].ip, &ip, src_ip[0].ea, dst_mac)==-1)
					libnet_error(LIBNET_ERR_FATAL, "Not able to get the MAC address of the gateway: %s. You should add the -Sip option to fix the problem.\n", libnet_host_lookup(ip, LIBNET_RESOLVE));
		}
		if(verbose>1) {
			fprintf(stdout, "Destination MAC address: ");
			for(i=0; i<GPS_ETH_ALEN; i++) fprintf(stdout, "%x%c", dst_mac[i], (i==GPS_ETH_ALEN-1)?'\n':':');
		}
	}


	// get a timeout value
	if(verbose) fprintf(stdout, "Evaluating a timeout value..\n");
	if(localhost) max_timeout=DEFAULT_TIMEOUT_LOOPB;
	else switch(spoofing)
	{
		case IP_SPOOF:
			max_timeout=ip_eval_timeout(dev.name, rand_seed, verbose, src_ip[0].ip, dst_ip, ping_port, src_ip[0].ea);
			break;
		case MAC_SPOOF:
			max_timeout=ethernet_eval_timeout(dev.name, rand_seed, verbose, src_ip[0].ip, dst_ip, ping_port, src_ip[0].ea, dst_mac);
			break;
	}
	if(max_timeout==-1) {
		if(onlocalnet) max_timeout=DEFAULT_TIMEOUT_LOCAL;
		else max_timeout=DEFAULT_TIMEOUT_DIST;
		libnet_error(LIBNET_ERR_WARNING, "Timeout evaluation failure. Are we talking to a dead host ?\n");
		fprintf(stderr, "Using the default timeout value: %d ms\n", max_timeout);
	}


	// clone the thread and setup a communication link between parent and child
	// parent thread: sniffer
	// child  thread: packet generator
	if(socketpair(AF_UNIX, SOCK_STREAM, 0, sockets)<0) error("socketpair: creation failed", -1);
	if((npid=fork())==-1) error("fork: thread duplication failed", -1);
	if(!npid) { // child
		for(i=0; i<=SIGCHLD; i++) signal(i, pg_sig);
		close(sockets[0]);
		switch(spoofing) 
		{
			case IP_SPOOF:
				ip_packet_generator(scan_speed);
				break;
			case MAC_SPOOF:
				ethernet_packet_generator(scan_speed);
				break;
		}
	}
	close(sockets[1]);

	// allocate memory for data
	if(scan_type!=FWRD_SCAN) {
		if(!(tab_response=(u_int16_t *)malloc(nbports*sizeof(u_int16_t)))) error(NULL, -2);
	} else {
		if(!(tab_fwrd=(fwrd_elt *)malloc(nb_src_ip*sizeof(fwrd_elt)))) error(NULL, -2);
		for(i=0; i<nb_src_ip; i++) init_fwrd_elt(&tab_fwrd[i], src_ip[i].ip);
	}

	// open a raw link interface to send spoofed ARP replies if needed (something like ARP poisoning)
	if(!localhost)
		if(!(interface=libnet_open_link_interface(dev.name, errbuf)))
			libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed (%s)\n", errbuf);

	// open network device in promiscuous mode
	if(!(descr=pcap_open_live(dev.name, 65536, 1, -1, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "pcap_open_live failed (%s)\n", errbuf);


	// inform the child process that the sniffer is ready
	i=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && i++<3);
	if(i>2) error("write: operation on socket failed", -1);

	// unblock sockets
	unblock_socket(descr->fd);
	unblock_socket(sockets[0]);

	if(verbose) fprintf(stdout, "Starting port scan..\n");

	// loop until all packets have been sent and all replies received
	while(sckelt.i!=-2)
	{
		// grab a packet from network
		if((packet=pcap_next(descr, &phdr)))
		{
			// packet filter
			if(localhost) // scaning localhost through lo
			{
				ihdr=(struct libnet_ip_hdr *)(packet+4);
				if(ihdr->ip_src.s_addr==dst_ip && ihdr->ip_dst.s_addr==dst_ip && !(ihdr->ip_off & ~ntohs(IP_DF)) && ihdr->ip_p==IPPROTO_TCP)
				{
					thdr=(struct libnet_tcp_hdr *)((u_char *)ihdr+4*ihdr->ip_hl);
					// test packet flags
					i=0;
					switch(scan_type)
					{
						case SYN_SCAN:			// SYN scan
							if(thdr->th_flags!=(TH_SYN|TH_ACK)) i=1;
							if(thdr->th_flags & TH_RST)
								if((j=search_port(ntohs(thdr->th_sport)))!=-1) {
									tab_port[j].gotcha=1;
									nbpkt++;
								}
							break;
						case ACK_SCAN:			// ACK scan
						case FIN_SCAN:			// FIN scan
						case XMAS_SCAN:			// Xmas Tree scan
						case NULL_SCAN:			// NULL scan
						case RAND_SCAN:			// RAND scan
							if(!(thdr->th_flags & TH_RST)) i=1;
							break;
						case FWRD_SCAN:			// FWRD scan
							if(thdr->th_flags & TH_RST) i=1;
							// we have a matching packet, see if we already have it
							port=ntohs(thdr->th_sport);
							if(search_fwrd_port(&tab_fwrd[i_ip], port)==-1)
							{
								nbopt++;
								add_fwrd_port(&tab_fwrd[i_ip], port);
								if((j=search_port(port))!=-1) tab_port[j].gotcha=1;
							}
							nbpkt++;
							break;
					}
					if(!i)
					{
						// we have a matching packet, see if we already have it
						port=ntohs(thdr->th_sport);
						if((i=search_port(port))!=-1)
						{
							if(!tab_port[i].gotcha)
							{
								tab_port[i].gotcha=1;
								tab_response[nbopt++]=port;
							}
						}
						nbpkt++;
					}
				}
			}
			else
			{
				ehdr=(struct libnet_ethernet_hdr *)packet;
				switch(ntohs(ehdr->ether_type))
				{
					case ETHERTYPE_ARP: // ARP packet
						ahdr=(struct libnet_arp_hdr *)((u_char *)ehdr + ETHER_HDR_LEN);
						// break if it is not an ARP request
						if(ntohs(ahdr->ar_op)!=ARPOP_REQUEST) break;
						// break if the ARP request do not concern
						// the hosts we pretend to be
						if((j=search_src_ip(*((u_int32_t *)ahdr->ar_tpa)))==-1) break;
						// we have to reply this ARP request
						duplic=*ahdr;
						ahdr->ar_op=ntohs(ARPOP_REPLY);
						// build the spoofed ARP reply
						for(i=0; i<GPS_ETH_ALEN; i++) // exchange ARP addresses
						{
							ahdr->ar_sha[i]=ehdr->ether_shost[i]=(u_int8_t)src_ip[j].ea[i];
							ahdr->ar_tha[i]=ehdr->ether_dhost[i]=duplic.ar_sha[i];
						}
						for(i=0; i<4; i++) // exchange IP adresses
						{
							ahdr->ar_spa[i]=duplic.ar_tpa[i];
							ahdr->ar_tpa[i]=duplic.ar_spa[i];
						}
						// send the fake ARP reply
						if(libnet_write_link_layer(interface, dev.name, (u_char *)packet, ETHER_HDR_LEN + sizeof(struct libnet_arp_hdr))==-1)
							libnet_error(LIBNET_ERR_WARNING, "libnet_write_link layer failed\n");
						break;
					case ETHERTYPE_IP: // IP packet
						ihdr=(struct libnet_ip_hdr *)((u_char *)ehdr + ETHER_HDR_LEN);
						// break if the packet do not come from the target host
						// or if it is not addressed to the hosts we pretend to be
						// or if it is fragmented, or if it is not a TCP packet
						if(ihdr->ip_src.s_addr!=dst_ip || ((scan_type!=FWRD_SCAN) ? (search_src_ip(ihdr->ip_dst.s_addr)==-1) : (ihdr->ip_dst.s_addr!=src_ip[i_ip].ip)) || (ihdr->ip_off & ~ntohs(IP_DF)) || ihdr->ip_p!=IPPROTO_TCP) break;
						thdr=(struct libnet_tcp_hdr *)((u_char *)ihdr + 4 * ihdr->ip_hl);
						// test packet flags
						i=0;
						switch(scan_type)
						{
							case SYN_SCAN:			// SYN scan
								if(thdr->th_flags!=(TH_SYN|TH_ACK)) i=1;
								if(thdr->th_flags & TH_RST)
									if((j=search_port(ntohs(thdr->th_sport)))!=-1)
										tab_port[j].gotcha=1;
								break;
							case ACK_SCAN:			// ACK scan
							case FIN_SCAN:			// FIN scan
							case XMAS_SCAN:			// Xmas Tree scan
							case NULL_SCAN:			// NULL scan
							case RAND_SCAN:			// RAND scan
								if(!(thdr->th_flags & TH_RST)) i=1;
								break;
							case FWRD_SCAN:			// FWRD scan
								if(thdr->th_flags & TH_RST) i=1;
								// we have a matching packet, see if we already have it
								port=ntohs(thdr->th_sport);
								if(search_fwrd_port(&tab_fwrd[i_ip], port)==-1)
								{
									nbopt++;
									add_fwrd_port(&tab_fwrd[i_ip], port);
									if((j=search_port(port))!=-1) tab_port[j].gotcha=1;
								}
								nbpkt++;
								break;
						}
						if(i) break;
						// we have a matching packet, see if we already have it
						port=ntohs(thdr->th_sport);
						if((i=search_port(port))!=-1)
						{
							if(!tab_port[i].gotcha)
							{
								tab_port[i].gotcha=1;
								tab_response[nbopt++]=port;
							}
						}
						nbpkt++;
						break;
				}
			}
		}

		if(read(sockets[0], &sckelt, sizeof(sock_elt))>=0) switch(sckelt.i)
		{
			case -1: // first blast over
				if(verbose)
				{
					fprintf(stdout, "First blast over");
					if(scan_type==FWRD_SCAN)
						fprintf(stdout, " for %s", libnet_host_lookup(src_ip[i_ip].ip, LIBNET_RESOLVE));
					fprintf(stdout, "\n");
				}
				break;
			case -2: // all packets have been sent
				signal(SIGCHLD, SIG_IGN);
				continue;
			case -3: // ready to send packets from next IP (FWRD mode)
				break;
			default:
				tab_port[sckelt.i].tv=sckelt.tv;
				tab_port[sckelt.i].nbretries++;
				tab_port[sckelt.i].wait=0;
				break;
		}

		// handle timeout
		if(!gotcha[1])
		{
			gotcha[0]=1;
			gettimeofday(&tv, NULL);
			for(i=0; i<nbports; i++)
			{
				if(!tab_port[i].gotcha)
				{
					if(tab_port[i].nbretries>max_retries) {
						tab_port[i].gotcha=1;
						if(localhost && scan_type==SYN_SCAN)
							tab_response[nbopt++]=tab_port[i].port;
					}
					else
					{
						gotcha[0]=0;
						if(!tab_port[i].wait)
						{
							if(TIMEVAL_MSEC_DIFF(tv, tab_port[i].tv)>max_timeout)
							{
								sckelt.i=i;
								j=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && j++<3);
								tab_port[i].wait=1;
							}
						}
					}
				}
			}

			if(gotcha[0]) // all packets have been received; inform the child thread
			{
				if(verbose)
				{
					fprintf(stdout, "All packets");
					if(scan_type==FWRD_SCAN)
						fprintf(stdout, " for %s", libnet_host_lookup(src_ip[i_ip].ip, LIBNET_RESOLVE));
					fprintf(stdout, " have been received\n");
				}
				gotcha[1]=1;
				sckelt.i=-1;
				i=0; while(write(sockets[0], &sckelt, sizeof(sock_elt))<0 && i++<3);
				while(sckelt.i>-2) read(sockets[0], &sckelt, sizeof(sock_elt));
				if(sckelt.i==-2) signal(SIGCHLD, SIG_IGN);
			}
		}

		// if FWRD mode, reset port related data
		if(sckelt.i==-3 && scan_type==FWRD_SCAN)
		{
			i_ip++;
			gotcha[0]=gotcha[1]=0;
			for(i=0; i<nbports; i++)
				tab_port[i].nbretries=tab_port[i].gotcha=tab_port[i].wait=0;
		}
	}
	close(sockets[0]);

	if(verbose>0) fprintf(stdout, "Computing results..\n");

	// results:
	switch(scan_type)
	{
		case FWRD_SCAN: // fwrd scan
			if(nbopt==nb_src_ip*nbports)
				fprintf(stdout, "None of the scanned ports are filtered for all specified IP addresses\n");
			else
				for(i=0; i<nb_src_ip; i++)
				{
					if(!tab_fwrd[i].nb_ports)
						fprintf(stdout, "Each scanned port is filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
					else if(tab_fwrd[i].nb_ports==nbports)
						fprintf(stdout, "Each scanned port is _not_ filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
					else
					{
						fprintf(stdout, "Ports: ");
						if(tab_fwrd[i].nb_ports<=nbports/2)
						{
							fwrd_print_port_list(&tab_fwrd[i], "%d, ");
							fprintf(stdout, "\b\b are _not_ filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
						}
						else
						{
							fwrd_print_reverse_port_list(&tab_fwrd[i], "%d, ", tab_port, nbports);
							fprintf(stdout, "\b\b are filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
						}
					}
				}
			break;
		default:
			// order
			for(i=0; i<nbopt; i++)
				for(j=i+1; j<nbopt; j++)
					if(tab_response[i]>tab_response[j])
					{
						port=tab_response[i];
						tab_response[i]=tab_response[j];
						tab_response[j]=port;
					}

			// output
			switch(scan_type)
			{
				case SYN_SCAN: // print results for SYN scan
					if(!nbopt) fprintf(stdout, "No open ports found\n");
					else
					{
						for(i=0; i<nbopt; i++)
						{
							if((known=services_known(tab_response[i], st))==-1)
								fprintf(stdout, "Port %5u is open\n", tab_response[i]);
							else // port service detail
								fprintf(stdout, "Port %5u: %s (%s) is open\n", tab_response[i], st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
						}
					}
					break;
				case FIN_SCAN:			// print results for FIN scan,
				case XMAS_SCAN:			// for Xmas tree scan,
				case NULL_SCAN:			// for Null scan,
				case RAND_SCAN:			// for rand scan
					if(nbopt==nbports) // no open ports found
						fprintf(stdout, "No open ports found\n");
					else if(!nbopt)
						fprintf(stdout, "All scanned ports seem to be open\n");
					else reverse_scan_disp();
					break;
				case TH_ACK: // ACK scan
					if(!nbopt)
						fprintf(stdout, "All scanned ports are filtered\n");
					else if(nbopt==nbports)
						fprintf(stdout, "None of the scanned ports are filtered\n");
					else
					{
						fprintf(stdout, "Ports: ");
						if(nbopt<nbports/2)
						{
							for(i=0; i<nbopt; i++) fprintf(stdout, "%5u, ", tab_response[i]);
							fprintf(stdout, "\b\b are _not_ filtered\n");
						}
						else
						{
							for(i=j=0; i<nbports; i++)
							{
								if(tab_port[i].port<tab_response[j])
									fprintf(stdout, "%d, ", tab_port[i].port);
								else j++;
							}
							for(; i<nbports; i++)
								fprintf(stdout, "%5d, ", tab_port[i].port);
							fprintf(stdout, "\b\b are filtered\n");
						}
					}
					break;
			}
			break;
	}
	fprintf(stdout, "%d ports scanned\nGrabbed %d matching packets\n", nbports, nbpkt);

	// free & close everything
	if(scan_type==FWRD_SCAN)
	{
		for(i=0; i<nb_src_ip; i++)
			free_fwrd_port_list(&tab_fwrd[i]);
		free(tab_fwrd);
	}
	else free(tab_response);
	if(dl) free_devlist(dl);
	free(tab_port);
	free(src_ip);
	pcap_close(descr);
	if(!localhost)
		if(libnet_close_link_interface(interface)==-1)
			libnet_error(LIBNET_ERR_WARNING, "libnet_close_link_interface failed\n");
	return(0);
}

void get_ea(char *ea, int rs)
{
	int i;
	for(i=0; i<GPS_ETH_ALEN; i++) ea[i]=(u_char)libnet_get_prand(rs);
}

void init_port(int i, u_int16_t port)
{
	int j;
	tab_port[i].ip=(scan_type==FWRD_SCAN) ? 0 : src_ip[(nb_src_ip==1) ? 0 : (libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed))%nb_src_ip].ip;
	tab_port[i].port=port;
	do { tab_port[i].src_port=libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed); } while(!tab_port[i].src_port);
	do { tab_port[i].seq=1; for(j=0; j<4; j++) tab_port[i].seq*=libnet_get_prand(rand_seed); } while(!tab_port[i].seq);
	do { tab_port[i].ack=1; for(j=0; j<4; j++) tab_port[i].ack*=libnet_get_prand(rand_seed); } while(!tab_port[i].ack);
	do { tab_port[i].ip_id=libnet_get_prand(rand_seed)*libnet_get_prand(rand_seed); } while(!tab_port[i].ip_id);
	tab_port[i].nbretries=0;
	tab_port[i].tv.tv_sec=tab_port[i].tv.tv_usec=0;
	tab_port[i].gotcha=0;
	tab_port[i].wait=1;
}

int search_port(u_int16_t port)
{
	int i=0, j=nbports-1, m;
	do {
		m=(i+j)/2;
		if(tab_port[m].port==port) return(m);
		if(tab_port[m].port<port) i=m+1;
		else j=m-1;
	} while (i<=j);
	return(-1);
}

void reverse_scan_disp()
{
	int i, j, known;

	for(i=j=0; i<nbports && j<nbopt; i++)
	{
		if(tab_port[i].port<tab_response[j])
		{
			if((known=services_known(tab_port[i].port, st))==-1)
				fprintf(stdout, "Port %5d  is open or filtered\n", tab_port[i].port);
			else fprintf(stdout, "Port %5d: %s (%s) is open (or filtered)\n", tab_port[i].port, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
		}
		else j++;
	}
	for(; i<nbports; i++)
	{
		if((known=services_known(tab_port[i].port, st))==-1)
			fprintf(stdout, "%5d\n", tab_port[i].port);
		else fprintf(stdout, "Port %5d: %s (%s) is open (or fileterd)\n", tab_port[i].port, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "-");
	}
}

int search_src_ip(u_int32_t ip)
{
	int i;
	for(i=0; i<nb_src_ip; i++)
		if(src_ip[i].ip==ip) return(i);
	return(-1);
}

void unblock_socket(int fd)
{
	int options=O_NONBLOCK | fcntl(fd, F_GETFL);
	fcntl(fd, F_SETFL, options);
}

void randomize_port_order(u_int16_t *tab_switch_o2r, u_int16_t *tab_switch_r2o)
{
	int i, j, k;
	port_elt tmp_port;

	// randomize port order.
	// Port order is randomized for the packet generator
	// and ordered for the sniffer; tab_switch_r2o/o2r
	// are used to translate randomized offset to ordered
	// offset, and vice-versa
	for(i=0; i<nbports; i++)
	{
		tmp_port=tab_port[i];
		j=(nbports*libnet_get_prand(rand_seed))/255;
		tab_port[i]=tab_port[j];
		tab_port[j]=tmp_port;
		k=tab_switch_r2o[i];
		tab_switch_r2o[i]=tab_switch_r2o[j];
		tab_switch_r2o[j]=k;
	}
	for(i=0; i<nbports; i++) tab_switch_o2r[tab_switch_r2o[i]]=i;
}

#define NB_RAND_FLAGS 8

void ip_packet_generator(int scan_speed)
{
	int raw_sock, i, j, th_flags=scan_type, i_ip=0, offset;
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	char *packets=NULL, *frag_packets=NULL;
	u_int32_t rsrc_ip=src_ip[0].ip;
	sock_elt sckelt;
	u_int16_t *tab_switch_o2r=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	u_int16_t *tab_switch_r2o=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	if(!tab_switch_o2r||!tab_switch_r2o) error(NULL, -2);
	for(i=0; i<nbports; i++) tab_switch_r2o[i]=i;

	//wait for the sniffer to be ready
	i=0; while(read(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);

	// open a raw IP socket
	if((raw_sock=libnet_open_raw_sock(IPPROTO_RAW))==-1)
		libnet_error(LIBNET_ERR_FATAL, "libnet_open_raw_sock failed\n");

	// set up TCP flags for fwrd scan
	if(scan_type==FWRD_SCAN) th_flags=TH_ACK;

	// allocate memory for packets
	if(!(packets=(char *)malloc(nbports*GPS_PACKET))) error(NULL, -2);
	if(fragment!=NO_FRAG) if(!(frag_packets=(char *)malloc(nbports*GPS_PACKET))) error(NULL, -2);
	do
	{
		// randomize port scan order.
		randomize_port_order(tab_switch_o2r, tab_switch_r2o);

		// build and send packets
		offset=0;
		for(i=0; i<nbports; i++)
		{
			// get random flags for RAND scan
			if(scan_type==RAND_SCAN)
				th_flags=rflags[libnet_get_prand(rand_seed)%NB_RAND_FLAGS];

			// get an IP for ! FWRD scan
			if(scan_type!=FWRD_SCAN) rsrc_ip=tab_port[i].ip;

			switch(fragment)
			{
				case NO_FRAG:
					// build an IP datagram with a spoofed src_ip
					libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, tab_port[i].ip_id, 0, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_IP_H);
					// if the checksum is ok, send it
					if(libnet_do_checksum(packets+offset, IPPROTO_TCP, GPS_TCP_H)!=-1)
						libnet_write_ip(raw_sock, packets+offset, GPS_PACKET);
					break;
				case TINY_FRAG: // build two tiny fragments
					// build the first fragment
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_IP_H);
					libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[i].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
					if(libnet_do_checksum(packets+offset, IPPROTO_TCP, GPS_TCP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// build the second fragment
					memcpy(frag_packets+offset+GPS_IP_H, packets+offset+GPS_PACKET-4, 4);
					libnet_build_ip(GPS_TCP_H-16, IPTOS_LOWDELAY, tab_port[i].ip_id, 2 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset);
					// send them
					libnet_write_ip(raw_sock, frag_packets+offset, GPS_PACKET-16);
					libnet_write_ip(raw_sock, packets+offset, GPS_PACKET-4);
					break;
			}

			// inform the sniffer that the packet is sent
			sckelt.i=tab_switch_r2o[i];
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

			// wait a bit
			usleep(scan_speed);

			offset+=GPS_PACKET;
		}

		// inform the sniffer that the first blast is over
		sckelt.i=-1;
		j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);


		// resend dead packets (given by the sniffer)
		while(1)
		{
			if(read(sockets[1], &sckelt, sizeof(sock_elt))<0) continue;
			if(sckelt.i==-1) break; // break if we received all replies
			// the packet(s) is/are already made, just send it/them
			offset=tab_switch_o2r[sckelt.i]*GPS_PACKET;
			switch(fragment)
			{
				case NO_FRAG:
					libnet_write_ip(raw_sock, packets+offset, GPS_PACKET);
					break;
				case TINY_FRAG:
					libnet_write_ip(raw_sock, frag_packets+offset, GPS_PACKET-16);
					libnet_write_ip(raw_sock, packets+offset, GPS_PACKET-4);
					break;
			}
			// inform the sniffer that the packet(s) is/are sent
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);
			usleep(scan_speed);
		}


		i_ip++;
		if(scan_type==FWRD_SCAN && nb_src_ip>1 && i_ip<nb_src_ip)
		{
			// next IP
			rsrc_ip=src_ip[i_ip].ip;
			// inform the sniffer that we do it again
			sckelt.i=-3;
			i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
		}
	} while(i_ip<nb_src_ip && scan_type==FWRD_SCAN);

	// inform the sniffer that all is over
	sckelt.i=-2;
	i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

	// close and free
	close(sockets[1]);
	libnet_close_raw_sock(raw_sock);
	free(packets);
	if(fragment!=NO_FRAG) free(frag_packets);
	free(tab_switch_o2r);
	free(tab_switch_r2o);
	free(tab_port);
	free(src_ip);
	if(dl) free_devlist(dl);

	sleep(1);
	exit(0);
}


void ethernet_packet_generator(int scan_speed)
{
	int i, j, th_flags=scan_type, i_ip=0, offset;
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	char *packets=NULL, *frag_packets=NULL;
	u_char *src_mac=src_ip[0].ea;
	struct libnet_tcp_hdr *thdr;
	struct libnet_link_int *interface;
	u_int32_t rsrc_ip=src_ip[0].ip;
	sock_elt sckelt;
	u_int16_t *tab_switch_o2r=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	u_int16_t *tab_switch_r2o=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	if(!tab_switch_o2r||!tab_switch_r2o) error(NULL, -2);
	for(i=0; i<nbports; i++) tab_switch_r2o[i]=i;

	//wait for the sniffer to be ready
	i=0; while(read(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);

	// open a link layer interface
	if(!(interface=libnet_open_link_interface(dev.name, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed (%s)\n", errbuf);

	// set up TCP flags for fwrd scan
	if(scan_type==FWRD_SCAN) th_flags=TH_ACK;

	// allocate memory for packets
	if(!(packets=(char *)malloc(nbports*GPS_TRAME))) error(NULL, -2);
	if(fragment!=NO_FRAG) if(!(frag_packets=(char *)malloc(nbports*GPS_TRAME))) error(NULL, -2);
	do
	{
		// randomize port scan order.
		randomize_port_order(tab_switch_o2r, tab_switch_r2o);

		// build and send packets
		offset=0;
		for(i=0; i<nbports; i++)
		{
			// get random flags for RAND scan
			if(scan_type==RAND_SCAN)
				th_flags=rflags[libnet_get_prand(rand_seed)%NB_RAND_FLAGS];

			// get an IP and a MAC address for !FWRD scan
			if(scan_type!=FWRD_SCAN)
			{
				rsrc_ip=tab_port[i].ip;
				src_mac=src_ip[search_src_ip(rsrc_ip)].ea;
			}

			switch(fragment)
			{
				case NO_FRAG:
					// build an IP datagram with a spoofed src_ip
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
					libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, tab_port[i].ip_id, 0, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
					// if the checksums are ok, send it
					if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)!=-1 && libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)!=-1)
						libnet_write_link_layer(interface, dev.name, packets+offset, GPS_TRAME);
					break;
				case TINY_FRAG: // build two tiny fragments
					// build the first fragment
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
					libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[i].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
					if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// build the second fragment
					memcpy(frag_packets+offset+GPS_ETH_H+GPS_IP_H, packets+offset+GPS_TRAME-4, 4);
					libnet_build_ip(GPS_TCP_H-16, IPTOS_LOWDELAY, tab_port[i].ip_id, 2 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, frag_packets+offset);
					if(libnet_do_checksum(frag_packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// send them
					libnet_write_link_layer(interface, dev.name, frag_packets+offset, GPS_TRAME-16);
					libnet_write_link_layer(interface, dev.name, packets+offset, GPS_TRAME-4);
					break;
				case OVERLAP_FRAG: // NOTE: most of recent OSes drop IP
						   // datagrams with PROTO=TCP and OFFSET=1
					// build two fragments,
					// the first fragment has TCP flags set to 0 (offset 0)
					// the second has the real TCP flags (offset 1).
					// The second frag overlaps the first at packet reconstruction
					// (the TCP flags are overwitten)
					libnet_build_tcp(tab_port[i].src_port, tab_port[i].port, tab_port[i].seq, tab_port[i].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
					libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[i].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
					if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					memcpy(frag_packets+offset+GPS_ETH_H+GPS_IP_H, packets+offset+GPS_ETH_H+GPS_TRAME-12, 12);
					libnet_build_ip(GPS_TCP_H-8, IPTOS_LOWDELAY, tab_port[i].ip_id, 1 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset+GPS_ETH_H);
					libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, frag_packets+offset);
					if(libnet_do_checksum(frag_packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
					// reset the first frag's TCP flags
					thdr=(struct libnet_tcp_hdr *)(packets+offset+GPS_ETH_H+GPS_IP_H);
					thdr->th_flags=TH_NULL;
					// send them
					libnet_write_link_layer(interface, dev.name, packets+offset, GPS_TRAME-4);
					libnet_write_link_layer(interface, dev.name, frag_packets+offset, GPS_TRAME-8);
					break;
			}

			// inform the sniffer that the packet is sent
			sckelt.i=tab_switch_r2o[i];
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

			// wait a bit
			usleep(scan_speed);

			offset+=GPS_TRAME;
		}

		// inform the sniffer that the first blast is over
		sckelt.i=-1;
		j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);


		// resend dead packets (given by the sniffer)
		while(1)
		{
			if(read(sockets[1], &sckelt, sizeof(sock_elt))<0) continue;
			if(sckelt.i==-1) break; // break if we received all replies
			// the packet(s) is/are already made, just send it/them
			offset=tab_switch_o2r[sckelt.i]*GPS_TRAME;
			switch(fragment)
			{
				case NO_FRAG:
					libnet_write_link_layer(interface, dev.name, packets+offset, GPS_TRAME);
					break;
				case TINY_FRAG:
					libnet_write_link_layer(interface, dev.name, frag_packets+offset, GPS_TRAME-16);
					libnet_write_link_layer(interface, dev.name, packets+offset, GPS_TRAME-4);
					break;
				case OVERLAP_FRAG:
					libnet_write_link_layer(interface, dev.name, packets+offset, GPS_TRAME-4);
					libnet_write_link_layer(interface, dev.name, frag_packets+offset, GPS_TRAME-8);
					break;
			}
			// inform the sniffer that the packet(s) is/are sent
			gettimeofday(&sckelt.tv, NULL);
			j=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);
			usleep(scan_speed);
		}


		i_ip++;
		if(scan_type==FWRD_SCAN && nb_src_ip>1 && i_ip<nb_src_ip)
		{
			// next IP and MAC
			rsrc_ip=src_ip[i_ip].ip;
			src_mac=src_ip[i_ip].ea;
			// inform the sniffer that we do it again
			sckelt.i=-3;
			i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && i++<3);
		}
	} while(i_ip<nb_src_ip && scan_type==FWRD_SCAN);

	// inform the sniffer that all is over
	sckelt.i=-2;
	i=0; while(write(sockets[1], &sckelt, sizeof(sock_elt))<0 && j++<3);

	// close and free
	close(sockets[1]);
	libnet_close_link_interface(interface);
	free(packets);
	if(fragment!=NO_FRAG) free(frag_packets);
	free(tab_switch_o2r);
	free(tab_switch_r2o);
	free(tab_port);
	free(src_ip);
	if(dl) free_devlist(dl);

	sleep(1);
	exit(0);
}


void usage(char *cmd)
{
fprintf(stderr, "Usage: %s -s host1[,host2/host3..] -d target [-t scan_type] [-v]\n\
	   [-r scan_speed] [-p first_port-last_port] [-k 0 | 1]\n\
	   [-e ping_port]  [-f t | o] [-i interface] [-S mac | ip]\n\
 -s host1[,host2/host3]: list of hosts we pretend to be\n\
			 (use '/' to specify IP ranges)\n\
 -d target.............: target host's IP/name\n\
 -t scan_type..........: stealth scan mode (default: syn)\n\
 			 (syn | xmas | null | fin | ack | rand | fwrd)\n\
 -r scan_speed.........: packet rate (default: insane)\n\
 			 (insane | aggressive | normal | polite | paranoid)\n\
 -p first-last ports...: port range to scan (default: 1-1024)\n\
 -k 0 | 1..............: scan well-known ports (default: 1)\n\
 -e ping_port..........: target port for a TCP ping (default: 80)\n\
 -v....................: verbose (use twice for more verbose)\n\
 -f t | o..............: fragment IP datagrams (default: no frag)\n\
 			 (t: tiny frags | o: frag overlapping)\n\
 -i interface..........: network interface to use\n\
 -S mac | ip...........: spoofing level (IP or ethernet/MAC; default: mac)\n", cmd);
exit(-1);
}

void error(char *msg, int ret)
{
	switch(ret)
	{
		case -2:
			fprintf(stderr, "malloc/realloc: memory allocation failed\n");
			exit(-2);
		default:
			fprintf(stderr, "%s\n", msg);
			exit(ret);
	}
}

