/*
 * Ghost Port Scan is distributed under the GPL (see COPYING)
 *
 * Homepage: gps.sourceforge.net
 * Contact.: whitehat@altern.org
 *
 */

// standard librairies
#include<ctype.h>
#include<stdio.h>
#include<errno.h>
#include<string.h>
#include<stdlib.h>
#include<unistd.h>
#include<signal.h>
#include<sys/types.h>
#include<sys/socket.h>
#ifdef HAVE_LINUX
#include<linux/version.h>
#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c)	(((a)<<16)|((b)<<8)|(c))
#endif
#endif
// network librairies
#include<libnet.h>
#ifndef PCAP_SUBDIR
#include<pcap.h>
#else
#include<pcap/pcap.h>
#endif
#include "pcap-int.h"
// custom librairies
#include "gps.h"
#include "services.h"
#include "mac-arp.h"
#include "timeout.h"
#include "fwrd.h"
#include "route.h"
#include "src-ip.h"


char errbuf[GPS_ERRBUF_SIZE];
netdev dev;
u_char dst_mac[GPS_ETH_ALEN];
int sockets[2], nbopt=0, nbports, nb_src_ip=0;
int ping_port=80, fragment=NO_FRAG, spoofing=MAC_SPOOF, scan_type=SYN_SCAN;
int scan_speed=0, current_timeout=0, max_timeout=0, window_size=BASIC_WINDOW;
int min_window=MIN_WINDOW, max_retries=MAX_RETRIES;
u_int32_t dst_ip=0;
services_t	*st=NULL;
struct fwrd_elt	*tab_fwrd=NULL;
struct arp_elt	*src_ip=NULL;
struct port_elt	*tab_port=NULL;
devlist		*netdevlist=NULL;
struct pcap	*descr=NULL;
extern char	*optarg;



// signal handling
void immune2sig() {
	int i; for(i=0; i<=SIGCHLD; i++) signal(i, SIG_IGN);
}
void common_sig(int sig) {
	fprintf(stderr, "): caught signal %d - exiting\n", sig); exit(sig);
}
void main_sig(int sig) {
	immune2sig();
	if(descr) pcap_close(descr);
	fprintf(stderr, "Sniffer (parent");
	switch(sig) {
		case SIGCHLD:
			fprintf(stderr, "): death of packet injector - exiting\n");
			break;
		default:
			common_sig(sig);
	}
	exit(sig);
}
void pg_sig(int sig) {
	immune2sig();
	fprintf(stderr, "Packet injector (child");
	common_sig(sig);
}


int main(int argc, char **argv)
{
	char str_ip[BUFSIZ];
	struct libnet_arp_hdr duplic;
	struct pcap_pkthdr phdr;
	struct sigaction what2do;
	long int ip_diff;
	char *ptr1, *ptr2;
	u_int32_t ip, net_ip, netmask;
	int npid, cmd_parse, i, j, known, onlocalnet, localhost, counter;
	int current_ip=0, verbose=0, nbpkt=0, serv_scan=1, link_offset=0;
	struct libnet_link_int		*interface=NULL;
	struct libnet_ethernet_hdr	*ehdr=NULL;
	struct libnet_arp_hdr		*ahdr=NULL;
	struct libnet_ip_hdr		*ihdr=NULL;
	struct libnet_tcp_hdr		*thdr=NULL;
	struct libnet_udp_hdr		*uhdr=NULL;
	struct libnet_icmp_hdr		*icmphdr=NULL;
	struct gotcha_port *gotcha_port;
	netdev *nd;
	const u_char *packet;
	u_int16_t port, fport=1, lport=1024;

	// setup signal handling
	memset(&what2do, 0, sizeof(struct sigaction));
	what2do.sa_handler=main_sig;
	sigfillset(&what2do.sa_mask);
	for(i=2; i<=SIGCHLD; i++) sigaction(i, &what2do, NULL);


	// initialize data
	gps_seed_prand();
	dev.name[0]='\0';
	// load known services in memory
	st=services_load();

	fprintf(stdout, "Ghost Port Scan version %s by whitehat@altern.org (gps.sourceforge.net)\n", GPS_VERSION);

	// take a look to the arguments:
	if(argc<3) usage(argv[0]);
	while((cmd_parse=getopt(argc, argv, "s:d:p:k:t:r:e:vf:i:S:w:"))!=EOF) {
		switch(cmd_parse) {
			case 's':
				// get the src_ip list (in host byte order)
				ptr1=optarg;
				j=0; // indicates if we are handling a range of IPs (xxx.xxx.xxx.xxx/yyy.yyy.yyy.yyy)
				while(*ptr1) {
					for(ptr2=ptr1; *ptr1 && *ptr1!=',' && *ptr1!='/'; ptr1++);
					if((i=ptr1-ptr2)>=BUFSIZ) libnet_error(LIBNET_ERR_FATAL, "Source host name too long\n");
					strncpy(str_ip, ptr2, i);
					str_ip[i]='\0';
					if((ip=libnet_name_resolve(str_ip, LIBNET_RESOLVE))==-1)
						libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: not able to resolve: %s\n", str_ip);
					ip=ntohl(ip);
					if(j) { // if an IP range is specified
						j=0;
						ip_diff=ip-src_ip[nb_src_ip-1].ip;
						if(ip_diff<0) libnet_error(LIBNET_ERR_FATAL, "Invalid source IP addresses range\n");
						if(!(src_ip=(struct arp_elt *)realloc(src_ip, (ip_diff+nb_src_ip)*sizeof(struct arp_elt)))) { perror("realloc"); exit(-1); }
						for(i=0; i<ip_diff; i++) {
							src_ip[nb_src_ip+i].ip=src_ip[nb_src_ip-1].ip+(i+1);
							get_ea(src_ip[nb_src_ip+i].ea);
						}
						nb_src_ip+=ip_diff;
					} else {
						if(!(src_ip=(struct arp_elt *)realloc(src_ip, (nb_src_ip+1)*sizeof(struct arp_elt)))) { perror("realloc"); exit(-1); }
						src_ip[nb_src_ip].ip=ip;
						get_ea(src_ip[nb_src_ip++].ea);
					}
					if(!*ptr1) break;
					if(*ptr1=='/') j=1;
					ptr1++;
				}
				break;
			case 'd':
				// target host
				strncpy(str_ip, optarg, BUFSIZ-1);
				str_ip[BUFSIZ-1]='\0';
				if((dst_ip=libnet_name_resolve(str_ip, LIBNET_RESOLVE))==-1)
					libnet_error(LIBNET_ERR_FATAL, "libnet_name_resolve: not able to reslove: %s\n", str_ip);
				break;
			case 'p':
				// first_port and last_port
				if(!(fport=atoi(optarg))) libnet_error(LIBNET_ERR_FATAL, "atoi: invalid first port\n");
				for(ptr1=optarg; isdigit(*ptr1); ptr1++);
				switch(*ptr1) {
					case 0:
						lport=fport;
						break;
					case '-':
						if(!(lport=atoi(++ptr1))) libnet_error(LIBNET_ERR_FATAL, "atoi: invalid last port\n");
						if(lport<fport) libnet_error(LIBNET_ERR_FATAL, "Invalid first and last ports\n");
						break;
					default:
						libnet_error(LIBNET_ERR_FATAL, "Invalid argument: %s\n", optarg);
				}
				serv_scan=0;
				break;
			case 'k':
				// scan known ports
				if(optarg[0]=='0') serv_scan=0;
				else serv_scan=1;
				break;
			case 't':
				// scan_type
				     if(!strncmp("syn",  optarg, 3)) scan_type=SYN_SCAN;
				else if(!strncmp("fin",  optarg, 3)) scan_type=FIN_SCAN;
				else if(!strncmp("ack",  optarg, 3)) scan_type=ACK_SCAN;
				else if(!strncmp("udp",  optarg, 3)) scan_type=UDP_SCAN;
				else if(!strncmp("xmas", optarg, 4)) scan_type=XMAS_SCAN;
				else if(!strncmp("null", optarg, 4)) scan_type=NULL_SCAN;
				else if(!strncmp("rand", optarg, 4)) scan_type=RAND_SCAN;
				else if(!strncmp("fwrd", optarg, 4)) scan_type=FWRD_SCAN;
				else libnet_error(LIBNET_ERR_FATAL, "Invalid scan_type\n");
				break;
			case 'r':
				// scan_speed
				     if(!strncmp("paranoid",   optarg, 8 )) scan_speed=500000;
				else if(!strncmp("polite",     optarg, 6 )) scan_speed=100000;
				else if(!strncmp("normal",     optarg, 6 )) scan_speed=5000;
				else if(!strncmp("aggressive", optarg, 10)) scan_speed=1000;
				else if(!strncmp("insane",     optarg, 6 )) scan_speed=0;
				else libnet_error(LIBNET_ERR_FATAL, "Invalid scan_speed\n");
				break;
			case 'e':
				if(!(ping_port=atoi(optarg))) {
					libnet_error(LIBNET_ERR_WARNING, "Invalid ping_port (using port 80)\n");
					ping_port=80;
				}
				break;
			case 'v':
				verbose++;
				break;
			case 'f':
				switch(optarg[0]) {
					case 't':
					case 'T':
						fragment=TINY_FRAG;
						break;
					case 'o':
					case 'O':
						fragment=OVERLAP_FRAG;
						if(spoofing!=MAC_SPOOF)
							{ spoofing=IP_SPOOF; libnet_error(LIBNET_ERR_WARNING, "Fragment overlapping requires spoofing at physical/link level\n"); }
						break;
					default:
						usage(argv[0]);
				}
				break;
			case 'i':
				strncpy(dev.name, optarg, IFNAMSIZ);
				dev.name[IFNAMSIZ-1]='\0';
				if(!strncmp("lo", dev.name, 2) && spoofing==MAC_SPOOF) {
					libnet_error(LIBNET_ERR_WARNING, "Scanning on loopback requires spoofing at IP level\n");
					spoofing=IP_SPOOF;
				}
				break;
			case 'S':
				if(!strncmp("ip", optarg, 2)) {
					if(fragment==OVERLAP_FRAG)
						{ spoofing=MAC_SPOOF; libnet_error(LIBNET_ERR_WARNING, "Fragment overlapping requires spoofing at physical/link level\n"); }
					else spoofing=IP_SPOOF;
				} else {
					spoofing=MAC_SPOOF;
					if(dev.name[0]) if(!strncmp("lo", dev.name, 2))
						{ spoofing=IP_SPOOF; libnet_error(LIBNET_ERR_WARNING, "Scanning on loopback requires spoofing at IP level\n"); }
				}
				break;
			case 'w':
				if(!(window_size=atoi(optarg)))
					libnet_error(LIBNET_ERR_WARNING, "Invalid window size (auto-using %d as the window size)\n", window_size=BASIC_WINDOW);
				break;
			case '?':
			default:
				usage(argv[0]);
		}
	}
	if(!dst_ip) usage(argv[0]);


	// guess if we'll scan localhost
	if(!(netdevlist=get_localdevs())) libnet_error(LIBNET_ERR_FATAL, "get_localdevs: not able to get the network device list\n");
	localhost=islocalhost(netdevlist, dst_ip);

	// handle some particularities to have scans working against localhost
	if(localhost) {
#ifdef HAVE_BSD
		link_offset=4;
		if(src_ip) free(src_ip);
		if((src_ip=(struct arp_elt *)malloc(sizeof(struct arp_elt)))) {
			src_ip[0].ip=ntohl(dst_ip);
			nb_src_ip=1;
		} else { perror("malloc"); exit(-1); }
#else
		if(LINUX_VERSION_CODE<KERNEL_VERSION(2,2,0)) {
			link_offset=14;
			if(src_ip) free(src_ip);
			if((src_ip=(struct arp_elt *)malloc(sizeof(struct arp_elt)))) {
				src_ip[0].ip=ntohl(dst_ip);
				nb_src_ip=1;
			} else { perror("malloc"); exit(-1); }
		} else if(LINUX_VERSION_CODE<KERNEL_VERSION(2,4,0))
			link_offset=4;
		else	link_offset=14;
#endif
		if(scan_type==UDP_SCAN) {
			if(!src_ip) {
				if((src_ip=(struct arp_elt *)malloc(sizeof(struct arp_elt)))) {
					src_ip[0].ip=ntohl(dst_ip);
					nb_src_ip=1;
				} else { perror("malloc"); exit(-1); }
			} else if(!islocalhost(netdevlist, src_ip[0].ip)) {
				libnet_error(LIBNET_ERR_WARNING, "Performing a UDP scan against localhost requires to use a local IP as the source IP (auto-using destination IP as source IP)\n");
				free(src_ip);
				if((src_ip=(struct arp_elt *)malloc(sizeof(struct arp_elt)))) {
					src_ip[0].ip=ntohl(dst_ip);
					nb_src_ip=1;
				} else { perror("malloc"); exit(-1); }
			}
		}
	}

	// get a network device name, if none is given by the user
	if(!dev.name[0]) {
		if(localhost) {
#ifdef HAVE_LINUX
			strcpy(dev.name, "lo");
#else
			strcpy(dev.name, "lo0");
#endif
			if(spoofing!=IP_SPOOF) {
				libnet_error(LIBNET_ERR_WARNING, "Scanning localhost requires spoofing at IP level (auto-switching to IP spoofing)\n");
				spoofing=IP_SPOOF;
			}
			if(fragment==OVERLAP_FRAG) {
				libnet_error(LIBNET_ERR_WARNING, "Fragment overlapping requires spoofing at physical level, and it is not compliant with scaning through loopback (auto-switching to no fragmentation)\n");
				fragment=NO_FRAG;
			}
		} else {
			if((nd=get_suitable_dev(netdevlist, dst_ip))) dev=*nd;
			else if((ptr1=pcap_lookupdev(errbuf))) {
				strncpy(dev.name, ptr1, IFNAMSIZ);
				dev.name[IFNAMSIZ-1]='\0';
			}
			else libnet_error(LIBNET_ERR_FATAL, "All attemps to get a suitable network device failed. You should try the -i option to specify the device to use.\n");
#ifdef HAVE_BSD
			if(strncmp("ed", dev.name, 2) && strncmp("lo", dev.name, 2) && strncmp("ne", dev.name, 2))
#else
			if(strncmp("et", dev.name, 2) && strncmp("lo", dev.name, 2))
#endif
				libnet_error(LIBNET_ERR_FATAL, "Device %s is not yet supported by GPS. If it is still an ethernet or loopback device, please mail me to correct this.\n", dev.name);
		}
	}
	fprintf(stdout, "Network device: %s\n", dev.name);


	// if no source IPs have been specified, search for unused IPs on the LAN
	if(!src_ip) {
		if(localhost) libnet_error(LIBNET_ERR_FATAL, "You must specify a source IP list if you want to scan localhost");
		if(verbose) fprintf(stdout, "Looking for IP addresses which are not in use on the LAN..\n");
		src_ip=get_src_ip_list(&dev, ping_port, &nb_src_ip, dst_ip);
	}
	if(!nb_src_ip) libnet_error(LIBNET_ERR_FATAL, "all checked source IPs seem to be in use. In order to prevent a massive DoS, please specify the source IPs to use\n");
	// order the IP list and translate it to network byte order
	qsort(src_ip, nb_src_ip, sizeof(struct arp_elt), (void *)&compare_arp_elt);
	for(i=0; i<nb_src_ip; i++) src_ip[i].ip=ntohl(src_ip[i].ip);
	
	
	// allocate memory for data storage and initialize data
	nbports=lport-fport+1;
	if(st && serv_scan) {
		for(i=0; st->tab[i].port<fport && i<st->nb; i++) nbports++;
		for(i=st->nb-1; st->tab[i].port>lport && i>=0; i--) nbports++;
	}
	if(!(gotcha_port=(struct gotcha_port *)malloc(nbports*sizeof(struct gotcha_port)))) { perror("malloc"); exit(-1); }
	memset(gotcha_port, 0, nbports*sizeof(struct gotcha_port));
	// setup some packet parameters (port, src_ip, seq..)
	i=0;
	if(serv_scan) for(; st->tab[i].port<fport && i<st->nb; i++) gotcha_port[i].port=st->tab[i].port;
	for(j=0; j<lport-fport+1; j++) gotcha_port[i++].port=(u_int16_t)j+fport;
	if(serv_scan) for(j=st->nb-1, i=nbports-1; st->tab[j].port>lport && j>=0; j--, i--) gotcha_port[i].port=st->tab[j].port;


	// get the local nework IP, and the netmask
	if(pcap_lookupnet(dev.name, &net_ip, &netmask, errbuf))
		libnet_error(LIBNET_ERR_FATAL, "pcap_lookupnet failed (%s)\n", errbuf);
	// see if we can reach the host directly
	onlocalnet=IS_ON_LOCAL_NET(dst_ip, netmask, net_ip);

	// if MAC spoofing is activated, get the MAC address of the
	// host or gateway to which the ethernet trames will be sent
	if(spoofing==MAC_SPOOF) {
		if(verbose) fprintf(stdout, "Searching for the destination MAC address..\n");
		if(onlocalnet) {
			if(get_mac(dst_ip, dst_mac, dev.name)==-1)
				if(send_ARP_request(dev.name, &src_ip[0].ip, &dst_ip, src_ip[0].ea, dst_mac)==-1)
					libnet_error(LIBNET_ERR_FATAL, "Not able to get the MAC address of host: %s. Try the -Sip option to enable spoofing at IP level.\n", libnet_host_lookup(dst_ip, LIBNET_RESOLVE));
		} else {
			if((ip=get_gateway_ip(dst_ip))==-1)
				libnet_error(LIBNET_ERR_FATAL, "get_gateway_ip: not able to find a suitable gateway. Try the -Sip option to enable spoofing at IP level.\n");
			if(get_mac(ip, dst_mac, dev.name)==-1)
				if(send_ARP_request(dev.name, &src_ip[0].ip, &ip, src_ip[0].ea, dst_mac)==-1)
					libnet_error(LIBNET_ERR_FATAL, "Not able to get the MAC address of the gateway: %s. Try the -Sip option to enable spoofing at IP level.\n", libnet_host_lookup(ip, LIBNET_RESOLVE));
		}
		if(verbose>1) {
			fprintf(stdout, "Destination MAC address: ");
			for(i=0; i<GPS_ETH_ALEN; i++) fprintf(stdout, "%x%c", dst_mac[i], (i!=GPS_ETH_ALEN-1)?':':'\n');
		}
	}


	// get a timeout value
	if(verbose) fprintf(stdout, "Evaluating a timeout value..\n");
	if(localhost) {
		current_timeout=DEFAULT_TIMEOUT_LOOPB;
		max_timeout=MAX_TIMEOUT_LOOPB;
		if(verbose) fprintf(stdout, "Timeout value: %d ms\n", current_timeout);
	} else switch(spoofing) {
		case IP_SPOOF:
			current_timeout=ip_eval_timeout(dev.name, verbose, src_ip[0].ip, dst_ip, ping_port, src_ip[0].ea);
			if(onlocalnet) {
				if(current_timeout==-1 || current_timeout>DEFAULT_TIMEOUT_LOCAL<<1)
					current_timeout=DEFAULT_TIMEOUT_LOCAL<<1;
				max_timeout=MAX_TIMEOUT_LOCAL;
			} else {
				if(current_timeout==-1 || current_timeout>DEFAULT_TIMEOUT_DIST<<1)
					current_timeout=DEFAULT_TIMEOUT_DIST<<1;
				max_timeout=MAX_TIMEOUT_DIST;
			}
			break;
		case MAC_SPOOF:
			current_timeout=ethernet_eval_timeout(dev.name, verbose, src_ip[0].ip, dst_ip, ping_port, src_ip[0].ea, dst_mac);
			if(onlocalnet) {
				if(current_timeout==-1 || current_timeout>DEFAULT_TIMEOUT_LOCAL<<1)
					current_timeout=DEFAULT_TIMEOUT_LOCAL<<1;
				max_timeout=MAX_TIMEOUT_LOCAL;
			} else {
				if(current_timeout==-1 || current_timeout>DEFAULT_TIMEOUT_DIST<<1)
					current_timeout=DEFAULT_TIMEOUT_DIST<<1;
				max_timeout=MAX_TIMEOUT_DIST;
			}
			break;
	}
	switch(scan_type) {
		case UDP_SCAN:
			current_timeout<<=1;
			break;
		case FWRD_SCAN:
		case ACK_SCAN:
			current_timeout>>=1;
			max_timeout>>=1;
			min_window=MIN_WINDOW_FW;
			max_retries=MAX_RETRIES_FW;
			break;
	}


	// clone the thread and setup a communication link between parent and child
	// parent thread: sniffer
	// child  thread: packet injector
	if(socketpair(AF_UNIX, SOCK_DGRAM, 0, sockets)<0) { perror("socketpair"); exit(-1); }
	if((npid=fork())==-1) { perror("fork"); exit(-1); }
	if(!npid) { // child
		// setup signal handling
		what2do.sa_handler=pg_sig;
		for(i=0; i<=SIGCHLD; i++) sigaction(i, &what2do, NULL);
		// initialize the port related data
		if(!(tab_port=(struct port_elt *)malloc(nbports*sizeof(struct port_elt)))) { perror("child: malloc"); exit(-1); }
		for(i=0; i<nbports; i++) init_port(i, gotcha_port[i].port);
		// close and free what we don't need
		close(sockets[0]);
		free(gotcha_port);
		if(netdevlist) free_devlist(netdevlist);
		// call the suitable packet injector
		switch(spoofing) {
			case IP_SPOOF:
				ip_packet_generator();
				break;
			case MAC_SPOOF:
				ethernet_packet_generator();
				break;
		}
	}
	close(sockets[1]);

	// allocate memory for data storage
	if(scan_type==FWRD_SCAN) {
		if(!(tab_fwrd=(struct fwrd_elt *)malloc(nb_src_ip*sizeof(struct fwrd_elt)))) { perror("malloc"); exit(-1); }
		for(i=0; i<nb_src_ip; i++) init_fwrd_elt(&tab_fwrd[i], src_ip[i].ip);
	}

	// open a raw link interface to send spoofed ARP replies if needed (something like ARP poisoning)
	if(!localhost)
		if(!(interface=libnet_open_link_interface(dev.name, errbuf)))
			libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed (%s)\n", errbuf);

	// open the network device in promiscuous mode
	if(!(descr=pcap_open_live(dev.name, 256, 1, 0, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "pcap_open_live failed (%s)\n", errbuf);


	// inform the child process that the sniffer is ready
	i=0; if(write(sockets[0], &i, sizeof(int))<0) { perror("write"); exit(-1); }

	// unblock the sniffing socket
	unblock_socket(descr->fd);

	if(localhost && scan_type==SYN_SCAN) nbopt=nbports;

	if(verbose) fprintf(stdout, "Starting the port scan..\n");

	// loop until all packets have been sent and all replies received
	do {
		// grab packets from network
		for(counter=0; counter<10 && (packet=pcap_next(descr, &phdr)); counter++) {
			// packet filter
			if(localhost) { // scanning localhost through loopback
				ihdr=(struct libnet_ip_hdr *)(packet+link_offset);
				if(ihdr->ip_src.s_addr!=dst_ip) continue;
				switch(ihdr->ip_p) {
					case IPPROTO_TCP:
						if(ihdr->ip_dst.s_addr!=dst_ip) break;
						thdr=(struct libnet_tcp_hdr *)((u_char *)ihdr+(ihdr->ip_hl<<2));
						// test packet flags
						switch(scan_type) {
							case SYN_SCAN:			// SYN scan
								if(thdr->th_flags & TH_RST) {
									// check if we already get this port
									if((j=search_port(gotcha_port, ntohs(thdr->th_sport)))!=-1)
										if(!gotcha_port[j].gotcha) {
											nbopt--;
											gotcha_port[j].gotcha=GPS_PORT_CLOSED;
											// inform the packet injector we get this port
											if(write(sockets[0], &j, sizeof(int))<0) { perror("write"); exit(-1); }
										}
									nbpkt++;
								}
								break;
							case ACK_SCAN:			// ACK scan
							case FIN_SCAN:			// FIN scan
							case XMAS_SCAN:			// Xmas Tree scan
							case NULL_SCAN:			// NULL scan
							case RAND_SCAN:			// RAND scan
								if(thdr->th_flags & TH_RST) {
									// we have a matching packet, see if we already have it
									if((i=search_port(gotcha_port, ntohs(thdr->th_sport)))!=-1)
										if(!gotcha_port[i].gotcha) {
											gotcha_port[i].gotcha=GPS_PORT_UNFILTERED;
											nbopt++;
											// inform the packet injector we get this port
											if(write(sockets[0], &i, sizeof(int))<0) { perror("write"); exit(-1); }
										}
									nbpkt++;
								}
								break;
							case FWRD_SCAN:			// FWRD scan
								if(!(thdr->th_flags & TH_RST)) break;
								// we have a matching packet, see if we already have it
								port=ntohs(thdr->th_sport);
								if((j=search_port(gotcha_port, port))!=-1)
									if(!gotcha_port[j].gotcha) {
										nbopt++;
										add_fwrd_port(&tab_fwrd[current_ip], port);
										gotcha_port[j].gotcha=GPS_PORT_UNFILTERED;
										// inform the packet injector we get this port
										if(write(sockets[0], &j, sizeof(int))<0) { perror("write"); exit(-1); }
									}
								nbpkt++;
								break;
						}
						break;
					case IPPROTO_ICMP:
						// ICMP packets are cheked only if we are
						// doing a UDP scan. GPS sniffs the "UDP port
						// unreachable" ICMP reply to know which ports
						// are closed.
						if(scan_type!=UDP_SCAN || ihdr->ip_dst.s_addr!=src_ip[current_ip].ip) break;
						icmphdr=(struct libnet_icmp_hdr *)((u_char *)ihdr+(ihdr->ip_hl<<2));
						if(icmphdr->icmp_type!=ICMP_UNREACH || icmphdr->icmp_code!=ICMP_UNREACH_PORT) break;
						uhdr=(struct libnet_udp_hdr *)(icmphdr->icmp_data+(((struct libnet_ip_hdr *)icmphdr->icmp_data)->ip_hl<<2));
						port=ntohs(uhdr->uh_dport);
						if((i=search_port(gotcha_port, port))!=-1)
							if(!gotcha_port[i].gotcha) {
								gotcha_port[i].gotcha=GPS_PORT_CLOSED;
								nbopt++;
								if(write(sockets[0], &i, sizeof(int))<0) { perror("write"); exit(-1); }
							}
						nbpkt++;
						break;
				}
			} else { // !localhost
				// we grabbed an ethernet trame, lets analyse it
				ehdr=(struct libnet_ethernet_hdr *)packet;
				// lets start with the ethernet header
				switch(ntohs(ehdr->ether_type)) {
					case ETHERTYPE_ARP: // ARP packet
						ahdr=(struct libnet_arp_hdr *)((u_char *)ehdr + GPS_ETH_H);
						// break if it is not an ARP request
						if(ntohs(ahdr->ar_op)!=ARPOP_REQUEST) break;
						// break if the ARP request do not concern
						// the hosts we pretend to be
						if((j=search_src_ip(*((u_int32_t *)ahdr->ar_tpa)))==-1) break;
						// we have to reply this ARP request
						duplic=*ahdr;
						ahdr->ar_op=ntohs(ARPOP_REPLY);
						// build the spoofed ARP reply
						for(i=0; i<GPS_ETH_ALEN; i++) { // exchange ARP addresses
							ahdr->ar_sha[i]=ehdr->ether_shost[i]=(u_int8_t)src_ip[j].ea[i];
							ahdr->ar_tha[i]=ehdr->ether_dhost[i]=duplic.ar_sha[i];
						}
						for(i=0; i<4; i++) { // exchange IP adresses
							ahdr->ar_spa[i]=duplic.ar_tpa[i];
							ahdr->ar_tpa[i]=duplic.ar_spa[i];
						}
						// send the fake ARP reply
						if(libnet_write_link_layer(interface, dev.name, (u_char *)packet, GPS_ETH_H + sizeof(struct libnet_arp_hdr))==-1)
							libnet_error(LIBNET_ERR_WARNING, "libnet_write_link layer failed\n");
						break;
					case ETHERTYPE_IP: // IP packet
						ihdr=(struct libnet_ip_hdr *)((u_char *)ehdr + GPS_ETH_H);
						// break if the packet does not come from the target host
						// or if it is not addressed to the hosts we pretend to be
						// or if it is fragmented, or if it is not a TCP packet
						if(ihdr->ip_src.s_addr!=dst_ip || ((scan_type!=FWRD_SCAN) ? (search_src_ip(ihdr->ip_dst.s_addr)==-1) : (ihdr->ip_dst.s_addr!=src_ip[current_ip].ip)) || (ihdr->ip_off & ~ntohs(IP_DF))) break;
						switch(ihdr->ip_p) {
							case IPPROTO_TCP:
								thdr=(struct libnet_tcp_hdr *)((u_char *)ihdr+(ihdr->ip_hl<<2));
								// test segment flags
								switch(scan_type) {
									case SYN_SCAN:			// SYN scan
										if(thdr->th_flags & TH_RST)			i=GPS_PORT_CLOSED;
										else if(thdr->th_flags & (TH_SYN|TH_ACK))	i=GPS_PORT_OPEN;
										else break;
										if((j=search_port(gotcha_port, ntohs(thdr->th_sport)))!=-1)
											if(!gotcha_port[j].gotcha) {
												gotcha_port[j].gotcha=i;
												if(i==GPS_PORT_OPEN) nbopt++;
												// inform the packet injector we get this port
												if(write(sockets[0], &j, sizeof(int))<0) { perror("write"); exit(-1); }
											}
										nbpkt++;
										break;
									case ACK_SCAN:			// ACK scan
									case FIN_SCAN:			// FIN scan
									case XMAS_SCAN:			// Xmas Tree scan
									case NULL_SCAN:			// NULL scan
									case RAND_SCAN:			// RAND scan
										if(thdr->th_flags & TH_RST) {
											// we have a matching packet, see if we already have it
											if((i=search_port(gotcha_port, ntohs(thdr->th_sport)))!=-1)
												if(!gotcha_port[i].gotcha) {
													gotcha_port[i].gotcha=GPS_PORT_CLOSED;
													nbopt++;
													// inform the packet injector we get this port
													if(write(sockets[0], &i, sizeof(int))<0) { perror("write"); exit(-1); }
												}
											nbpkt++;
										}
										break;
									case FWRD_SCAN:			// FWRD scan
										if(!(thdr->th_flags & TH_RST)) break;
										// we have a matching packet, see if we already have it
										port=ntohs(thdr->th_sport);
										if((j=search_port(gotcha_port, port))!=-1)
											if(!gotcha_port[j].gotcha) {
												nbopt++;
												add_fwrd_port(&tab_fwrd[current_ip], port);
												gotcha_port[j].gotcha=GPS_PORT_UNFILTERED;
												// inform the packet injector we get this port
												if(write(sockets[0], &j, sizeof(int))<0) { perror("write"); exit (-1); }
											}
										nbpkt++;
										break;
								}
								break;
							case IPPROTO_ICMP:
								// ICMP packets are cheked only if we are
								// doing a UDP scan. GPS sniffs the "UDP port
								// unreachable" ICMP reply to know which ports
								// are closed.
								if(scan_type!=UDP_SCAN) break;
								icmphdr=(struct libnet_icmp_hdr *)((u_char *)ihdr+(ihdr->ip_hl<<2));
								if(icmphdr->icmp_type!=ICMP_UNREACH || icmphdr->icmp_code!=ICMP_UNREACH_PORT) break;
								uhdr=(struct libnet_udp_hdr *)(icmphdr->icmp_data+(((struct libnet_ip_hdr *)icmphdr->icmp_data)->ip_hl<<2));
								port=ntohs(uhdr->uh_dport);
								if((i=search_port(gotcha_port, port))!=-1)
									if(!gotcha_port[i].gotcha) {
										gotcha_port[i].gotcha=GPS_PORT_CLOSED;
										nbopt++;
										if(write(sockets[0], &i, sizeof(int))<0) { perror("write"); exit(-1); }
									}
								nbpkt++;
								break;
						}
				}
			}
		}

		// get the information sent by the packet injector (through the socket)
		unblock_socket(sockets[0]);
		if(read(sockets[0], &i, sizeof(int))>0)
			switch(i) {
				case GPS_1ST_BLAST_OVER: // first blast over
					if(verbose) {
						fprintf(stdout, "First blast over");
						if(scan_type==FWRD_SCAN)
							fprintf(stdout, " for %s", libnet_host_lookup(src_ip[current_ip].ip, LIBNET_RESOLVE));
						fprintf(stdout, "\n");
					}
					break;
				case GPS_SCAN_OVER: // all packets have been sent
					signal(SIGCHLD, SIG_IGN);
					continue;
				case GPS_NEXT_IP: // ready to send packets from next IP (FWRD mode)
					// if FWRD mode, reset port related data
					if(scan_type==FWRD_SCAN) {
						if(verbose) fprintf(stdout, "Scan done for %s; trying next IP..\n", libnet_host_lookup(src_ip[current_ip].ip, LIBNET_RESOLVE));
						current_ip++;
						for(j=0; j<nbports; j++) gotcha_port[j].gotcha=0;
					}
					break;
			}
		block_socket(sockets[0]);

	} while(i!=GPS_SCAN_OVER);

	close(sockets[0]);

	if(verbose) fprintf(stdout, "Computing results..\n");


	// RESULTS COMPUTING AND DISPLAYING
	switch(scan_type) {
		case FWRD_SCAN: // fwrd scan
			if(nbopt==nb_src_ip*nbports)
				fprintf(stdout, "None of the scanned ports are filtered for all specified IP addresses\n");
			else
				for(i=0; i<nb_src_ip; i++) {
					if(!tab_fwrd[i].nb_ports)
						fprintf(stdout, "Each scanned port is filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
					else if(tab_fwrd[i].nb_ports==nbports)
						fprintf(stdout, "Each scanned port is _not_ filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
					else {
						qsort(tab_fwrd[i].port_list, tab_fwrd[i].nb_ports, sizeof(u_int16_t), (void *)&fwrd_compare);
						fprintf(stdout, "Ports: ");
						if(tab_fwrd[i].nb_ports<=(nbports>>1)) {
							fwrd_print_port_list(&tab_fwrd[i], "%d, ");
							fprintf(stdout, "\b\b are _not_ filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
						} else {
							fwrd_print_reverse_port_list(&tab_fwrd[i], "%d, ", gotcha_port, nbports);
							fprintf(stdout, "\b\b are filtered for %s\n", libnet_host_lookup(tab_fwrd[i].ip, LIBNET_RESOLVE));
						}
					}
				}
			break;
		case SYN_SCAN: // print results for SYN scan
			if(!nbopt) fprintf(stdout, "No open ports found\n");
			else {
				if(localhost) {
					for(i=0; i<nbports; i++) {
						if(gotcha_port[i].gotcha==GPS_PORT_CLOSED) continue;
						if((known=services_known(gotcha_port[i].port, st))==-1)
							fprintf(stdout, "Port %5u  is open\n", gotcha_port[i].port);
						else // service detail
							fprintf(stdout, "Port %5u: %s (%s) is open\n", gotcha_port[i].port, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "no description");
					}
				} else {
					for(i=0; i<nbports; i++) {
						if(gotcha_port[i].gotcha==GPS_PORT_OPEN) {
							if((known=services_known(gotcha_port[i].port, st))==-1)
								fprintf(stdout, "Port %5u  is open\n", gotcha_port[i].port);
							else // service detail
								fprintf(stdout, "Port %5u: %s (%s) is open\n", gotcha_port[i].port, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "no description");
						}
					}
				}
			}
			break;
		case UDP_SCAN:
			if(verbose) fprintf(stdout, "UDP ports:\n");
		case FIN_SCAN:			// print results for FIN scan,
		case XMAS_SCAN:			// for Xmas tree scan,
		case NULL_SCAN:			// for Null scan,
		case RAND_SCAN:			// for rand scan
			if(nbopt==nbports) // no open ports found
				fprintf(stdout, "No open ports found\n");
			else if(!nbopt)
				fprintf(stdout, "All scanned ports seem to be open\n");
			else for(i=0; i<nbports; i++)
				if(gotcha_port[i].gotcha!=GPS_PORT_CLOSED) {
					if((known=services_known(gotcha_port[i].port, st))==-1)
						fprintf(stdout, "Port %5u  is open\n", gotcha_port[i].port);
					else // service detail
						fprintf(stdout, "Port %5u: %s (%s) is open\n", gotcha_port[i].port, st->tab[known].service, (st->tab[known].comment[0]) ? st->tab[known].comment : "no description");
				}
			break;
		case ACK_SCAN: // ACK scan
			if(!nbopt)
				fprintf(stdout, "All scanned ports are filtered\n");
			else if(nbopt==nbports)
				fprintf(stdout, "None of the scanned ports are filtered\n");
			else {
				fprintf(stdout, "Ports: ");
				if(nbopt<(nbports>>1)) {
					for(i=0; i<nbports; i++) if(gotcha_port[i].gotcha==GPS_PORT_UNFILTERED) fprintf(stdout, "%u, ", gotcha_port[i].port);
					fprintf(stdout, "\b\b are _not_ filtered\n");
				} else {
					for(i=0; i<nbports; i++) if(gotcha_port[i].gotcha!=GPS_PORT_UNFILTERED) fprintf(stdout, "%u, ", gotcha_port[i].port);
					fprintf(stdout, "\b\b are filtered\n");
				}
			}
		break;
	}
	fprintf(stdout, "%d ports scanned\nGrabbed %d matching packets\n", nbports, nbpkt);

	// free & close everything
	if(scan_type==FWRD_SCAN) {
		for(i=0; i<nb_src_ip; i++) free_fwrd_port_list(&tab_fwrd[i]);
		free(tab_fwrd);
	}
	if(netdevlist) free_devlist(netdevlist);
	free(gotcha_port);
	free(src_ip);
	pcap_close(descr);
	if(!localhost)
		if(libnet_close_link_interface(interface)==-1)
			libnet_error(LIBNET_ERR_WARNING, "libnet_close_link_interface failed\n");
	return(0);
}

void get_ea(char *ea)
{
	int i;
	for(i=0; i<GPS_ETH_ALEN; i++) ea[i]=(u_char)gps_get_prand();
}

struct port_list_elt *add_port_to_list(struct port_list_elt *pl, int offset, int *nb_packets_being_handled)
{
	struct port_list_elt *tmp_ple, *list_ptr;

	if(!(tmp_ple=(struct port_list_elt *)malloc(sizeof(struct port_list_elt))))
		{ perror("add_port_to_list: malloc"); exit(-1); }

	tmp_ple->offset=offset;
	tmp_ple->next=NULL;

	(*nb_packets_being_handled)++;

	// add the element at the end of the list
	if(pl) {
		for(list_ptr=pl; list_ptr->next; list_ptr=list_ptr->next);
		list_ptr->next=tmp_ple;
		tmp_ple->prec=list_ptr;
		return(pl);
	} else {
		tmp_ple->prec=NULL;
		return(tmp_ple);
	}
}

struct port_list_elt *del_port_from_list(struct port_list_elt *pl, int offset, int *nb_packets_being_handled)
{
	struct port_list_elt *tmp_ple, *prec, *next;

	// search for the matching element in the list
	for(tmp_ple=pl; tmp_ple; tmp_ple=tmp_ple->next)
		if(tmp_ple->offset==offset) {
			(*nb_packets_being_handled)--;
			if(tmp_ple==pl) { // check if it is the head of the list
				if((next=tmp_ple->next)) next->prec=NULL;
				free(tmp_ple);
				return(next);
			} else {
				prec=tmp_ple->prec;
				next=tmp_ple->next;
				prec->next=next;
				if(next) next->prec=prec;
				free(tmp_ple);
				return(pl);
			}
		}

	// if we did't find it, return
	return(pl);
}

void init_port(int i, u_int16_t port)
{
	memset(&tab_port[i], 0, sizeof(struct port_elt));
	tab_port[i].ip=(scan_type==FWRD_SCAN) ? 0 : src_ip[(nb_src_ip==1) ? 0 : gps_get_prand()%nb_src_ip].ip;
	tab_port[i].dst_port=port;
	tab_port[i].src_port=1024|gps_get_prand();
	tab_port[i].seq=34234^gps_get_prand();
	tab_port[i].ack=94885^gps_get_prand();
	tab_port[i].ip_id=643^gps_get_prand();
}

int search_port(struct gotcha_port *tab, u_int16_t port)
{
	int i=0, j=nbports-1, m;
	do {
		m=(i+j)>>1;
		if(tab[m].port==port) return(m);
		if(tab[m].port<port) i=m+1;
		else j=m-1;
	} while (i<=j);
	return(-1);
}

int search_src_ip(u_int32_t ip)
{
	u_int32_t tmp_ip;
	int i=0, j=nb_src_ip-1, m;
	ip=ntohl(ip);
	do {
		tmp_ip=ntohl(src_ip[m=(i+j)>>1].ip);
		if(tmp_ip==ip) return(m);
		if(tmp_ip<ip) i=m+1;
		else j=m-1;
	} while (i<=j);
	return(-1);
}

int compare_arp_elt(struct arp_elt *addr1, struct arp_elt *addr2)
{
	return(addr1->ip-addr2->ip);
}

void unblock_socket(int fd)
{
	int options=O_NONBLOCK | fcntl(fd, F_GETFL);
	if(fcntl(fd, F_SETFL, options)==-1) { perror("unblock_socket: fcntl"); exit(-1); }
}

void block_socket(int fd)
{
	int options=(~O_NONBLOCK) & fcntl(fd, F_GETFL);
	if(fcntl(fd, F_SETFL, options)==-1) { perror("block_socket: fcntl"); exit(-1); }
}

void randomize_port_order(u_int16_t *tab_switch_o2r, u_int16_t *tab_switch_r2o)
{
	struct port_elt tmp_port;
	int i, j, tmp_ord;

	// randomize port order.
	// Port order is randomized for the packet generator
	// and ordered for the sniffer; tab_switch_r2o/o2r
	// are used to translate randomized offset to ordered
	// offset, and vice-versa
	for(i=0; i<nbports; i++) {
		j=gps_get_prand()%nbports;
		tmp_port=tab_port[i];
		tmp_ord=tab_switch_r2o[i];
		tab_port[i]=tab_port[j];
		tab_switch_r2o[i]=tab_switch_r2o[j];
		tab_port[j]=tmp_port;
		tab_switch_r2o[j]=tmp_ord;
	}
	for(i=0; i<nbports; i++) tab_switch_o2r[tab_switch_r2o[i]]=i;
}

void ip_timeout_handling(int sd, int raw_sock, int *nb_packets_being_handled, struct port_list_elt **packets_being_handled_list, u_int16_t *tab_switch_o2r, char *packets, char *frag_packets, int packet_length)
{
	struct port_list_elt *list_ptr;
	struct timeval current_time;
	int i, to_offset, counter, timeout_counter=0;

	// read on the socket connected to the sniffer, to get
	// the ports from which we get some responses back
	for(counter=0; counter<10 && read(sd, &i, sizeof(int))>0; counter++) {
		timeout_counter--;
		// we receive a response: decrement the number of packets being handled
		// and delete this port from the list used to check timed out packets
		*packets_being_handled_list=del_port_from_list(*packets_being_handled_list, tab_switch_o2r[i], nb_packets_being_handled);
	}

	// TIMEOUT HANDLING:
	// get the current time
	gettimeofday(&current_time, NULL);
	// for each packet being handled, check if
	// it is timed out, and if so, send it again
	list_ptr=*packets_being_handled_list;
	while(list_ptr) {
		i=list_ptr->offset;
		list_ptr=list_ptr->next;
		if(TIMEVAL_MSEC_DIFF(current_time, tab_port[i].tv)>current_timeout) {
			// check if we didn't send it too many times
			if(tab_port[i].nbretries<max_retries) {
				// if not, send the packet again
				// the packet is already made, just send it
				to_offset=i*packet_length;
				switch(fragment) {
					case NO_FRAG:
						if(libnet_write_ip(raw_sock, packets+to_offset, packet_length)<0) {
							//perror("ip_timeout_handling: libnet_write_ip");
							window_size--;
						}
						break;
					case TINY_FRAG:
						if(libnet_write_ip(raw_sock, frag_packets+to_offset, packet_length-16)<0 || \
						   libnet_write_ip(raw_sock, packets+to_offset, packet_length-4)<0) {
							//perror("ip_timeout_handling: libnet_write_ip");
							window_size--;
						}
						break;
				}
				// update the sending time and the
				// number of retries for this port
				gettimeofday(&tab_port[i].tv, NULL);
				tab_port[i].nbretries++;
				timeout_counter++;
				// wait a bit if necessary
				if(scan_speed) usleep(scan_speed);
			} else
				// we get no response at all, delete it from the list
				*packets_being_handled_list=del_port_from_list(*packets_being_handled_list, i, nb_packets_being_handled);
		}
	}
	if(timeout_counter) {
		window_size-=timeout_counter>>4;
		current_timeout+=timeout_counter>>3;
		if(--window_size<min_window) window_size=min_window;
		if(++current_timeout>max_timeout) current_timeout=max_timeout;
	}
}

#define NB_RAND_FLAGS 8

void ip_packet_generator()
{
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	int raw_sock, i, th_flags=scan_type, current_ip=0, offset, packet_length;
	int nb_packets_being_handled, counter;
	char *packets=NULL, *frag_packets=NULL;
	struct port_list_elt *packets_being_handled_list;
	u_int32_t rsrc_ip=src_ip[0].ip, current_port;
	u_int16_t *tab_switch_o2r=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	u_int16_t *tab_switch_r2o=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	if(!(tab_switch_o2r && tab_switch_r2o)) { perror("ip_packet_generator: malloc"); exit(-1); }
	for(i=0; i<nbports; i++) tab_switch_r2o[i]=i;


	// wait for the sniffer to be ready
	if(read(sockets[1], &i, sizeof(int))<0) { perror("ip_packet_generator: read"); exit(-1); }

	// unblock the socket connected to the sniffer
	unblock_socket(sockets[1]);

	// open a raw IP socket for IP datagrams injection
	if((raw_sock=libnet_open_raw_sock(IPPROTO_RAW))==-1)
		libnet_error(LIBNET_ERR_FATAL, "ip_packet_generator: libnet_open_raw_sock failed\n");

	// set up TCP flags for fwrd scan
	if(scan_type==FWRD_SCAN) th_flags=TH_ACK;

	// determine the packet length
	packet_length=(scan_type!=UDP_SCAN) ? GPS_TCP_PACKET : GPS_UDP_PACKET;

	// allocate memory to store the packets we'll build
	if(!(packets=(char *)malloc(nbports*packet_length))) { perror("ip_packet_generator: malloc"); exit(-1); }
	if(fragment!=NO_FRAG) if(!(frag_packets=(char *)malloc(nbports*packet_length))) { perror("ip_pcaket_generator: malloc"); }

	do { // for each IP (FWRD mode)

		// randomize port order
		randomize_port_order(tab_switch_o2r, tab_switch_r2o);

		// build and send all packets
		offset=0;
		current_port=0;
		nb_packets_being_handled=0;
		packets_being_handled_list=NULL;
		// while all the packets have not been sent
		while(current_port<nbports) {
			// while the emission window is not full
			// and all the packets have not been sent
			while(nb_packets_being_handled<window_size && current_port<nbports) {
				for(counter=0; counter<8 && current_port<nbports; counter++) {
					if(scan_type!=UDP_SCAN) {
						// get random TCP flags for RAND scan
						if(scan_type==RAND_SCAN)
							th_flags=rflags[gps_get_prand()%NB_RAND_FLAGS];

						// get a source IP for ! FWRD scan
						if(scan_type!=FWRD_SCAN) rsrc_ip=tab_port[current_port].ip;

						// build the packet for the current port
						switch(fragment) {
							case NO_FRAG:
								// build an IP datagram with a spoofed src_ip
								libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 0, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
								libnet_build_tcp(tab_port[current_port].src_port, tab_port[current_port].dst_port, tab_port[current_port].seq, tab_port[current_port].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_IP_H);
								// if the checksum is ok, send it
								if(libnet_do_checksum(packets+offset, IPPROTO_TCP, GPS_TCP_H)!=-1)
									if(libnet_write_ip(raw_sock, packets+offset, packet_length)<0) {
										//perror("ip_packet_generator: libnet_write_ip");
										window_size--;
									}
								break;
							case TINY_FRAG: // build two tiny fragments
								// build the first fragment
								libnet_build_tcp(tab_port[current_port].src_port, tab_port[current_port].dst_port, tab_port[current_port].seq, tab_port[current_port].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_IP_H);
								libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[current_port].ip_id, IP_MF /*more frags*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
								if(libnet_do_checksum(packets+offset, IPPROTO_TCP, GPS_TCP_H)==-1)
									libnet_error(LIBNET_ERR_FATAL, "ip_packet_generator: libnet_do_checksum failed\n");
								// build the second fragment
								memcpy(frag_packets+offset+GPS_IP_H, (packets+offset)+(packet_length-4), 4);
								libnet_build_ip(GPS_TCP_H-16, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 2 /*offset*/, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset);
								// send them
								if(libnet_write_ip(raw_sock, frag_packets+offset, packet_length-16)<0 || \
								   libnet_write_ip(raw_sock, packets+offset, packet_length-4)<0) {
									//perror("ip_packet_generator: libnet_write_ip");
									window_size--;
								}
								break;
						}
					} else { // case of UDP scan
						// build a UDP datagram with a spoofed source IP
						libnet_build_ip(GPS_UDP_H, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 0, 255, IPPROTO_UDP, rsrc_ip, dst_ip, NULL, 0, packets+offset);
						libnet_build_udp(tab_port[current_port].src_port, tab_port[current_port].dst_port, NULL, 0, packets+offset+GPS_IP_H);
						// if the checksum is ok, send it
						if(libnet_do_checksum(packets+offset, IPPROTO_UDP, GPS_UDP_H)!=-1)
							if(libnet_write_ip(raw_sock, packets+offset, packet_length)<0) {
								//perror("ip_packet_generator: libnet_write_ip");
								window_size--;
							}
					}

					// setup the sending time for timeout hanling
					gettimeofday(&tab_port[current_port].tv, NULL);

					// add the port to the list of ports being handled (used later to handle timeout)
					packets_being_handled_list=add_port_to_list(packets_being_handled_list, current_port, &nb_packets_being_handled);

					// try the next port
					current_port++;

					// increment the offset (which indicates where
					// to write (in memory) the next datagram)
					offset+=packet_length;

					// wait a bit if necessary
					if(scan_speed) usleep(scan_speed);
				}

				// handle the packets we received and the timed out ones
				ip_timeout_handling(sockets[1], raw_sock, &nb_packets_being_handled, &packets_being_handled_list, tab_switch_o2r, packets, frag_packets, packet_length);
			}
			// check if we break the loop because the emission
			// window is full or if we sent all packets.
			if(current_port>=nbports) {
				// if we sent all packets, inform the
				// sniffer that the first blast is over
				i=GPS_1ST_BLAST_OVER;
				if(write(sockets[1], &i, sizeof(int))<0) { perror("ip_packet_generator: write"); exit(-1); }
				// while all responses have not been
				// received, check timed out packets
				while(nb_packets_being_handled>0)
					ip_timeout_handling(sockets[1], raw_sock, &nb_packets_being_handled, &packets_being_handled_list, tab_switch_o2r, packets, frag_packets, packet_length);
			} else
				// while the emission window is not 20% empty,
				// wait for responses and handle timeout
				while(nb_packets_being_handled>=(int)((float)window_size*.8f))
					ip_timeout_handling(sockets[1], raw_sock, &nb_packets_being_handled, &packets_being_handled_list, tab_switch_o2r, packets, frag_packets, packet_length);
		}

		// try next IP if the FWRD mode is activated
		current_ip++;
		if(scan_type==FWRD_SCAN && current_ip<nb_src_ip) {
			// next IP
			rsrc_ip=src_ip[current_ip].ip;
			// inform the sniffer that we do it again
			i=GPS_NEXT_IP;
			if(write(sockets[1], &i, sizeof(int))<0) { perror("ip_packet_generator: write"); exit(-1); }
			// clear the socket queue
			while(read(sockets[1], &i, sizeof(int))>=0);
		}
	} while(current_ip<nb_src_ip && scan_type==FWRD_SCAN);

	// inform the sniffer that all is over
	i=GPS_SCAN_OVER;
	if(write(sockets[1], &i, sizeof(int))<0) { perror("ip_packet_generator: write"); exit(-1); }

	// close and free
	libnet_close_raw_sock(raw_sock);
	free(packets);
	if(fragment!=NO_FRAG) free(frag_packets);
	free(tab_switch_o2r);
	free(tab_switch_r2o);
	free(tab_port);
	free(src_ip);

	sleep(1);
	close(sockets[1]);
	exit(0);
}

void ethernet_timeout_handling(int sd, struct libnet_link_int *interface, int *nb_packets_being_handled, struct port_list_elt **packets_being_handled_list, u_int16_t *tab_switch_o2r, char *packets, char *frag_packets, int packet_length)
{
	struct port_list_elt *list_ptr;
	struct timeval current_time;
	int i, to_offset, counter, timeout_counter=0;

	// read on the socket connected to the sniffer, to get
	// the ports from which we get some responses back
	for(counter=0; counter<20 && read(sd, &i, sizeof(int))>0; counter++) {
		timeout_counter--;
		// we receive a response: decrement the number of packets being handled
		// and delete this port from the list used to check timed out packets
		*packets_being_handled_list=del_port_from_list(*packets_being_handled_list, tab_switch_o2r[i], nb_packets_being_handled);;
	}

	// TIMEOUT HANDLING:
	// get the current time
	gettimeofday(&current_time, NULL);
	// for each packet being handled, check if
	// it is timed out, and if so, send it again
	list_ptr=*packets_being_handled_list;
	while(list_ptr) {
		i=list_ptr->offset;
		list_ptr=list_ptr->next;
		if(TIMEVAL_MSEC_DIFF(current_time, tab_port[i].tv)>current_timeout) {
			// check if we didn't send it too many times
			if(tab_port[i].nbretries<max_retries) {
				// if not, send the packet again
				// the packet is already made, just send it
				to_offset=i*packet_length;
				switch(fragment) {
					case NO_FRAG:
						if(libnet_write_link_layer(interface, dev.name, packets+to_offset, packet_length)<0) {
							//perror("ethernet_timeout_handling: libnet_write_link_layer");
							window_size--;
						}
						break;
					case TINY_FRAG:
						if(libnet_write_link_layer(interface, dev.name, frag_packets+to_offset, packet_length-16)<0 || \
						   libnet_write_link_layer(interface, dev.name, packets+to_offset, packet_length-4)<0) {
							//perror("ethernet_timeout_handling: libnet_write_link_layer");
							window_size--;
						}
						break;
					case OVERLAP_FRAG:
						if(libnet_write_link_layer(interface, dev.name, packets+to_offset, packet_length-4)<0 || \
						   libnet_write_link_layer(interface, dev.name, frag_packets+to_offset, packet_length-8)<0) {
							//perror("ethernet_timeout_handling: libnet_write_link_layer");
							window_size--;
						}
						break;
				}
				// update the sending time and the
				// number of retries for this port
				gettimeofday(&tab_port[i].tv, NULL);
				tab_port[i].nbretries++;
				timeout_counter++;
				// wait a bit if necessary
				if(scan_speed) usleep(scan_speed);
			} else {
				// we get no response at all, delete it from the list
				*packets_being_handled_list=del_port_from_list(*packets_being_handled_list, i, nb_packets_being_handled);
			}
		}
	}
	if(timeout_counter) {
		window_size-=timeout_counter>>4;
		current_timeout+=timeout_counter>>3;
		if(--window_size<min_window)  window_size=min_window;
		if(++current_timeout>max_timeout) current_timeout=max_timeout;
	}
}

void ethernet_packet_generator()
{
	int rflags[NB_RAND_FLAGS]={0, TH_FIN, TH_PUSH, TH_URG, TH_FIN|TH_PUSH, TH_FIN|TH_URG, TH_PUSH|TH_URG, TH_FIN|TH_PUSH|TH_URG};
	struct libnet_link_int *interface=NULL;
	int i, th_flags=scan_type, current_ip=0, offset, packet_length;
	int nb_packets_being_handled, counter;
	char *packets=NULL, *frag_packets=NULL;
	struct port_list_elt *packets_being_handled_list;
	struct libnet_tcp_hdr *thdr;
	u_char *src_mac=src_ip[0].ea;
	u_int32_t rsrc_ip=src_ip[0].ip, current_port;
	u_int16_t *tab_switch_o2r=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	u_int16_t *tab_switch_r2o=(u_int16_t *)malloc(nbports*sizeof(u_int16_t));
	if(!(tab_switch_o2r && tab_switch_r2o)) { perror("ethernet_packet_generator: malloc"); exit(-1); }
	for(i=0; i<nbports; i++) tab_switch_r2o[i]=i;


	// wait for the sniffer to be ready
	if(read(sockets[1], &i, sizeof(int))<0) { perror("ethernet_packet_generator: read"); exit(-1); }

	// unblock the socket connected to the sniffer
	unblock_socket(sockets[1]);

	// open a link layer interface for ethernet trame injecting
	if(!(interface=libnet_open_link_interface(dev.name, errbuf)))
		libnet_error(LIBNET_ERR_FATAL, "ethernet_packet_generator: libnet_open_link_interface failed (%s)\n", errbuf);

	// set up TCP flags for fwrd scan
	if(scan_type==FWRD_SCAN) th_flags=TH_ACK;

	// determine the packet length
	packet_length=(scan_type!=UDP_SCAN) ? GPS_TCP_TRAME : GPS_UDP_TRAME;

	// allocate memory to store the packets we'll build
	if(!(packets=(char *)malloc(nbports*packet_length))) { perror("ethernet_packet_generator: malloc"); exit(-1); }
	if(fragment!=NO_FRAG) if(!(frag_packets=(char *)malloc(nbports*packet_length))) { perror("ethernet_pcaket_generator: malloc"); }
	do { // for each IP (FWRD mode)

		// randomize port order
		randomize_port_order(tab_switch_o2r, tab_switch_r2o);

		// build and send all packets
		offset=0;
		current_port=0;
		nb_packets_being_handled=0;
		packets_being_handled_list=NULL;
		// while all the packets have not been sent
		while(current_port<nbports) {
			// while the emission window is not full
			// and all the packets have not been sent
			while(nb_packets_being_handled<window_size && current_port<nbports) {
				for(counter=0; counter<8 && current_port<nbports; counter++) {
					if(scan_type!=UDP_SCAN) {
						// get random TCP flags for RAND scan
						if(scan_type==RAND_SCAN)
							th_flags=rflags[gps_get_prand()%NB_RAND_FLAGS];

						// get a source IP and a MAC for ! FWRD scan
						if(scan_type!=FWRD_SCAN) {
							rsrc_ip=tab_port[current_port].ip;
							src_mac=src_ip[search_src_ip(rsrc_ip)].ea;
						}

						// build the packet for the current port
						switch(fragment) {
							case NO_FRAG:
								// build an IP datagram with a spoofed src_ip
								libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
								libnet_build_ip(GPS_TCP_H, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 0, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
								libnet_build_tcp(tab_port[current_port].src_port, tab_port[current_port].dst_port, tab_port[current_port].seq, tab_port[current_port].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
								// if the checksums are ok, send it
								if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)!=-1 && libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)!=-1)
									if(libnet_write_link_layer(interface, dev.name, packets+offset, packet_length)<0) {
										//perror("ethernet_packet_generator: libnet_write_link_layer");
										window_size--;
									}
								break;
							case TINY_FRAG: // build two tiny fragments
								// build the first fragment
								libnet_build_tcp(tab_port[current_port].src_port, tab_port[current_port].dst_port, tab_port[current_port].seq, tab_port[current_port].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
								libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[current_port].ip_id, IP_MF, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
								libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
								if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
									libnet_error(LIBNET_ERR_FATAL, "ethernet_packet_generator: libnet_do_checksum failed\n");
								// build the second fragment
								memcpy(frag_packets+offset+GPS_ETH_H+GPS_IP_H, (packets+offset)+(packet_length-4), 4);
								libnet_build_ip(GPS_TCP_H-16, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 2, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset+GPS_ETH_H);
								libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, frag_packets+offset);
								if(libnet_do_checksum(frag_packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
									libnet_error(LIBNET_ERR_FATAL, "ethernet_packet_generator: libnet_do_checksum failed\n");
								// send them
								if(libnet_write_link_layer(interface, dev.name, frag_packets+offset, packet_length-16)<0 ||\
								   libnet_write_link_layer(interface, dev.name, packets+offset, packet_length-4)<0) {
									//perror("ethernet_packet_generator: libnet_write_link_layer");
									window_size--;
								}
								break;
							case OVERLAP_FRAG: // NOTE: most of recent OSes drop IP
									   // datagrams with PROTO=TCP and OFFSET=1
								// build two fragments,
								// the first fragment has TCP flags set to 0 (offset 0)
								// the second has the real TCP flags (offset 1).
								// The second frag overlaps the first at packet reconstruction
								// (the TCP flags are overwitten)
								libnet_build_tcp(tab_port[current_port].src_port, tab_port[current_port].dst_port, tab_port[current_port].seq, tab_port[current_port].ack, th_flags, 1024, 0, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
								libnet_build_ip(GPS_TCP_H-4, IPTOS_LOWDELAY, tab_port[current_port].ip_id, IP_MF, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
								libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
								if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_TCP, GPS_TCP_H)==-1 || libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
									libnet_error(LIBNET_ERR_FATAL, "ethernet_packet_generator: libnet_do_checksum failed\n");
								memcpy(frag_packets+offset+GPS_ETH_H+GPS_IP_H, (packets+offset)+(packet_length+GPS_ETH_H-12), 12);
								libnet_build_ip(GPS_TCP_H-8, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 1, 255, IPPROTO_TCP, rsrc_ip, dst_ip, NULL, 0, frag_packets+offset+GPS_ETH_H);
								libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, frag_packets+offset);
								if(libnet_do_checksum(frag_packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)==-1)
									libnet_error(LIBNET_ERR_FATAL, "ethernet_packet_generator: libnet_do_checksum failed\n");
								// reset the first frag's TCP flags
								thdr=(struct libnet_tcp_hdr *)(packets+offset+GPS_ETH_H+GPS_IP_H);
								thdr->th_flags=TH_NULL;
								// send them
								if(libnet_write_link_layer(interface, dev.name, packets+offset, packet_length-4)<0 || \
								   libnet_write_link_layer(interface, dev.name, frag_packets+offset, packet_length-8)<0) {
									//perror("ethernet_packet_generator: libnet_write_link_layer");
									window_size--;
								}
								break;
						}
					} else { // case of UDP scan
						// build a UDP trame with a spoofed source IP and a spoofed MAC address
						libnet_build_ethernet(dst_mac, src_mac, ETHERTYPE_IP, NULL, 0, packets+offset);
						libnet_build_ip(GPS_UDP_H, IPTOS_LOWDELAY, tab_port[current_port].ip_id, 0, 255, IPPROTO_UDP, rsrc_ip, dst_ip, NULL, 0, packets+offset+GPS_ETH_H);
						libnet_build_udp(tab_port[current_port].src_port, tab_port[current_port].dst_port, NULL, 0, packets+offset+GPS_ETH_H+GPS_IP_H);
						// if the checksum is ok, send it
						if(libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_UDP, GPS_UDP_H)!=-1 && libnet_do_checksum(packets+offset+GPS_ETH_H, IPPROTO_IP, GPS_IP_H)!=-1)
							if(libnet_write_link_layer(interface, dev.name, packets+offset, packet_length)>0) {
								//perror("ethernet_packet_generator: libnet_write_link_layer");
								window_size--;
							}
					}

					// setup the sending time for timeout hanling
					gettimeofday(&tab_port[current_port].tv, NULL);

					// add the port to the list of ports being handled (used later to handle timeout)
					packets_being_handled_list=add_port_to_list(packets_being_handled_list, current_port, &nb_packets_being_handled);

					// try the next port
					current_port++;

					// increment the offset (which indicates where
					// to write (in memory) the next datagram)
					offset+=packet_length;

					// wait a bit if necessary
					if(scan_speed) usleep(scan_speed);
				}

				// handle the packets we received and the timed out ones
				ethernet_timeout_handling(sockets[1], interface, &nb_packets_being_handled, &packets_being_handled_list, tab_switch_o2r, packets, frag_packets, packet_length);
			}
			// check if we break the loop because the emission
			// window is full or if we sent all packets.
			if(current_port>=nbports) {
				// if we sent all packets, inform the
				// sniffer that the first blast is over
				i=GPS_1ST_BLAST_OVER;
				if(write(sockets[1], &i, sizeof(int))<0) { perror("ethernet_packet_generator: write"); exit(-1); }
				// while all responses have not been
				// received, check timed out packets
				while(nb_packets_being_handled>0)
					ethernet_timeout_handling(sockets[1], interface, &nb_packets_being_handled, &packets_being_handled_list, tab_switch_o2r, packets, frag_packets, packet_length);
			} else
				// while the emission window is not 20% empty,
				// wait for responses and handle timeout
				while(nb_packets_being_handled>=(int)((float)window_size*.8f))
					ethernet_timeout_handling(sockets[1], interface, &nb_packets_being_handled, &packets_being_handled_list, tab_switch_o2r, packets, frag_packets, packet_length);
		}

		// try next IP if the FWRD mode is activated
		current_ip++;
		if(scan_type==FWRD_SCAN && current_ip<nb_src_ip) {
			// next IP and MAC
			rsrc_ip=src_ip[current_ip].ip;
			src_mac=src_ip[current_ip].ea;
			// inform the sniffer that we do it again
			i=GPS_NEXT_IP;
			if(write(sockets[1], &i, sizeof(int))<0) { perror("ethernet_packet_generator: write"); exit(-1); }
			// clear the socket queue
			while(read(sockets[1], &i, sizeof(int))>=0);
		}
	} while(current_ip<nb_src_ip && scan_type==FWRD_SCAN);

	// inform the sniffer that all is over
	i=GPS_SCAN_OVER;
	if(write(sockets[1], &i, sizeof(int))<0) { perror("ethernet_packet_generator: write"); exit(-1); }

	// close and free
	libnet_close_link_interface(interface);
	free(packets);
	if(fragment!=NO_FRAG) free(frag_packets);
	free(tab_switch_o2r);
	free(tab_switch_r2o);
	free(tab_port);
	free(src_ip);

	sleep(1);
	close(sockets[1]);
	exit(0);
}

void usage(char *cmd)
{
fprintf(stderr, "Usage: %s -d target [-s host1[,host2/host3..]] [-t scan_type]\n\
	   [-v] [-r scan_speed] [-p first_port-last_port] [-k 0 | 1]\n\
	   [-e ping_port]  [-f t | o] [-i interface] [-S mac | ip]\n\
	   [-w window_size]\n\
 -d target.............: target host's IP/name\n\
 -s host1[,host2/host3]: list of hosts we pretend to be\n\
			 (use '/' to specify IP ranges)\n\
 -t scan_type..........: stealth scan mode (default: syn)\n\
 			 (syn | xmas | null | fin | ack | rand | fwrd)\n\
 -r scan_speed.........: packet rate (default: insane)\n\
 			 (insane | aggressive | normal | polite | paranoid)\n\
 -p first-last ports...: port range to scan (default: 1-1024)\n\
 -k 0 | 1..............: scan well-known ports (default: 1)\n\
 -e ping_port..........: target port for a TCP ping (default: 80)\n\
 -v....................: verbose (use twice for more verbose)\n\
 -f t | o..............: fragment IP datagrams (default: no frag)\n\
 			 (t: tiny frags | o: frag overlapping)\n\
 -i interface..........: network interface to use\n\
 -S mac | ip...........: spoofing level (IP or ethernet/MAC; default: mac)\n\
 -w window_size........: size of the emission window (default: %d packets)\n", cmd, BASIC_WINDOW);
exit(-1);
}


