/*
 * Copyright (C) 1998 Hewlett-Packard Company
 */

/* 
 * some common data types used in netperf3. some of these come from
 * "Threadtime" by Scott Norton and Mark Dipasquale - for instance, 
 * the barrier stuff
 */

#ifdef USE_THREADS
#include <pthread.h>
#endif
#include <sys/mman.h>
#include <unistd.h>
#include <limits.h>
#include <sys/time.h>

#include "nettypes.h"

#define DEFAULT_CONTROL_PORT (int16_t)56821

#define MAX_CONTROL_MSG_LEN 4096

#define NETPERF_MAX_CPUS 64

/* this is not the "on the wire" format of a control message, but the
   "in the host" format. when a message is read from the control
   socket it will first be read byte-at-a-time until the length is in,
   then the rest will be read-in in bigger chunks. then the message is
   sscanf'd to pull the version, update, fix, test name and
   test-specific data. it will be necessary to make sure there is no
   overrunning of buffers lest that open a security hole.
 */
typedef struct control_msg {
  uint32_t length;  
  uint32_t version;
  uint32_t update;
  uint32_t fix;
  char     *test_specific_data;
} control_msg_t;
  
typedef struct ring_elt {
  struct ring_elt *next;  /* next element in the ring */
  char *buffer_base;      /* in case we have to free it at somepoint */
  char *buffer_ptr;       /* the aligned and offset pointer */
} ring_elt_t;

#define NETPERF_MAX_THREADS 64

/* the pthread version of the following comes from "Threadtime" page
   121, some slight  modifications have been made to the types
 */

typedef struct barrier_struct {
#ifdef USE_THREADS
  int32_t              valid;
  pthread_mutex_t      mutex;
  pthread_cond_t       cv;
  int32_t              barrier_val;
  int32_t              blocked_threads;
  int32_t              predicate;
#else /* USE_THREADS */
  msemaphore           semaphore;
  msemaphore           inner_semaphore;
  int32_t              valid;
  int32_t              barrier_val;
  int32_t              blocked_threads;
  int32_t              predicate;
  pid_t                waiting_pids[NETPERF_MAX_THREADS];
#endif /* USE_THREADS */
} barrier_t;

#define NETPERF_ARGS_MAX 64
#define NETPERF_HOST_MAX 255
#define NETPERF_TEST_MAX 32
#define NETPERF_PWRD_MAX  16
#define NETPERF_USER_MAX 8
#define NETPERF_FILE_MAX 255

/* these control the size of the test-specific areas in the test_t
   structure. test suites should hae an assert or some other check to
   make sure that the structures they overlay upon this space are not
   larger, or some nasty data corruption problems may arise. these
   units are bytes. raj 2/98
   */ 
#define NETPERF_MAX_SETTINGS 2048
#define NETPERF_MAX_RESULTS  4096

/* These are all things that we want both the main and the worker
   thread to be able to access, so it is generally best to not have
   things like string pointers and such embedded lest they get
   allocated in the context of a worker process and we try to read it
   in the context of the main process. "True" threads would not have
   this issue, nor would this issue exist if I wrote a shared memory
   allocator; this is supposed to be a networking test - not an
   excercise in writing memory allocators :) since these are in a
   shared space, it might be a good idea to make sure that they are at
   least padded-out to the next cache line, and perhaps best to pad
   them by a page. that is something that we will need to work on in
   the future raj 2/98 */
   
typedef struct test_struct {
  void      *next;          /* the next thread's test struct */
  void      *head;          /* pointer to the head struct */ 
#ifdef USE_THREADS
  pthread_t tid;            /* the thread id of the thread running
			       this test */
#else /* USE_THREADS */
  pid_t     tid;            /* the process id of the process running
			       this test */
#endif /* USE_THREADS */
  int32_t  thread_num;     /* our own "thread" identifier */

  /* I think that having these things be  pointers is OK - they are 
     read-only really, which means that when the memory gets copied in
     a call to fork, the "right" thing should happen. if we ever have
     the worker threads play with these in an other than read-only
     way, we will have to revisit this. raj 2/98 */
  int       argc;           /* the number of arguments going to this
			       thread */
  char      *argv[NETPERF_ARGS_MAX];   /* I cannot imagine there ever
					  being more than 64 arguments
					  being passed into a thread -
					  the famous last words of
					  rick jones on january 21st,
					  1998 :) */

  char      remote_host[NETPERF_HOST_MAX];   /* the name of the target
						of the test */
  char      *setup_message; /* I'm pretty sure that having this as a
			       pointer is OK, I'd rather not have a 4K
			       structure embedded in here just to pass
			       the initial settings to the "RECV" test
			       in netserver - when that passing
			       happens, we will have either already
			       forked a child, or will have started a
			       thread. */
  uint16_t  control_port;   /* the port number for the control
			       connection */
  int       control_sock;   /* the control connection socket */

  int32_t   times_up;       /* this is set to one for every test_t
			       when the test time has expired. I use a
			       per-thread value here rather than a
			       "global" because it will be checked
			       every time through the loop and I don't
			       want it bouncing from processor data
			       cache to processor data cache. */
  /* "global" test parms. while these are "global" I have put them
     into a per-thread structure because at some point, I may want
     a more heterogeneous test environment. raj 1/98 */
  char       test_name[NETPERF_TEST_MAX];     /* the name of the test
						 eg TCP_STREAM... */ 
  uint32_t   test_num;      /* the "number" of the test */

  char       fill_file[PATH_MAX];     /* the name of a file from which
					 to take  data for our send
					 buffers */ 
  FILE      *where;         /* where any logging output should go */
  
  int32_t        test_length;    /* how long will the test run */
  float          remote_cpu_rate;
  int32_t        remote_cpu_usage;
  float          local_cpu_rate;
  int32_t        local_cpu_usage;
  int32_t        local_cpu_method;

  /* these are for the elapsed time and the CPU utilization
     calculations */
  struct timeval time1;
  struct timeval time2;
  struct timezone tz;
  float          elapsed_time;

  int32_t local_cpu_num;
  int32_t remote_cpu_num;

  uint32_t       local_send_align;
  uint32_t       local_recv_align;
  uint32_t       local_send_offset;
  uint32_t       local_recv_offset;
  uint32_t       remote_send_align;
  uint32_t       remote_recv_align;
  uint32_t       remote_send_offset;
  uint32_t       remote_recv_offset;

  /* how many buffers do we use for sending and receiving data? */
  int32_t       send_width;
  int32_t       recv_width;

  /* some of the output controls */
  int32_t       verbosity;       /* -1 database, 0 terse, 1 short, 2 ... */
  int32_t       print_headers; /* */
  char          format_units;

  /* I wonder if there is a better way to declare this? */

  /* the test-specific parms. we would like to make sure all this
     stuff is properly aligned. there may be a better way to do this -
     I am not sure that this will give me the complete alignment that
     I may need, but for now it seems OK. raj 2/98 */
  double test_specific_settings[NETPERF_MAX_SETTINGS/sizeof(double)];

  /* the results of the test */
  double      test_specific_results[NETPERF_MAX_RESULTS/sizeof(double)];
} test_t;

typedef void (*FuncWithArg) (test_t *);

/* if you want to add tests to netperf, you will need to include their
   nmonics in this list of enums - at least until I learn a better
   way. raj 1/98 be sure to add to the end */

enum test_nums { 
  /* miscelaneous tests range 1-15 */
  VERSION = 1,     /* the version check */
  LOC_CPU = 2,     /* local CPU calibration */
  REM_CPU = 3,     /* remote CPU calibration */
  SHUTDOWN = 4,    /* not a "real" test */
  NETPERF_GO = 5,  /* a start message for some cases */
  ERROR = 15,      /* not really a test, but a response from the
		      netserver which will contain some description of
		      what went wrong */
  MISC_LAST = 15,  /* placeholder */

  /* BSD Socket/TCP/UDP tests range 16-31*/
  TCP_STREAM = 16, /* basic bandwidth outbound */
  TCP_MAERTS,      /* basic bandwidth inbound */
  TCP_RR,          /* basic request/response */
  TCP_CRR,         /* outbound connect/req/res */
  TCP_CC,          /* connect/close (netperf connects) */
  TCP_ARR,         /* inbound accept/req/res */
  TCP_TRR,         /* transactional TCP */
  UDP_STREAM,      /* outbound bandwidth */
  UDP_MAERTS,      /* inbound bandwith */
  UDP_RR,          /* request/response */
  BSD_LAST = 31,   /* placeholder */

  /* XTI TCP/UDP tests range 32-47 */
  XTI_TCP_STREAM = 32, /* basic bandwidth outbound */
  XTI_TCP_MAERTS,      /* basic bandwidth inbound */
  XTI_TCP_RR,          /* basic request/response */
  XTI_TCP_CRR,         /* outbound connect/req/res */
  XTI_TCP_ARR,         /* inbound accept/req/res */
  XTI_UDP_STREAM,      /* outbound bandwidth */
  XTI_UDP_MAERTS,      /* inbound bandwith */
  XTI_UDP_RR,          /* request/response */
  XTI_LAST = 47,       /* placeholder */

  /* DLPI tests range 48-61 */
  DLPI_LAST = 61,

  /* DNS tests range 62-77 */
  DNS_RR = 62,
  DNS_LAST = 77,

  /* FTP tests rage 78-85 */
  FTP_DOWNLOAD = 78,
  FTP_DOWNLOAD_RESULT = 79,
  FTP_LAST = 85,

};
/* If you add more tests than this, you will  need to up this value to
   something larger */  
#define  TEST_LAST  255

enum test_funcs {
  SCAN = 1,
  SEND = 2,
  RECV = 3,
  PRINT = 4
};

#define  NUM_FUNC   4

enum cpu_methods {
  PSTAT = 1,
  LOOPER = 2
};

/* some "global" formats for messages 
 ccr = cpu calibration rate
 sba = send buffer alignment
 rba = recv buffer alignment
 sbo = send buffer offset
 rbo = recv buffer offset
 srw = send ring width
 rrw = recv ring width
*/

#define NETPERF_BUF_SETTINGS \
"ccr %g sba %d rba %d sbo %d rbo %d srw %d rrw %d"

#define NETPERF_GO_FMT \
"GO! GO! GO!"

#define BARRIER_VALID 546731

/* this is the format string for a control message header. it goes
   length, version update fix testnum */ 
#define CONTROL_HEADER_FORMAT "%4d %4d %4d %4d %4d "

/* this is the size, in bytes, of the control header in ASCII */
#define CONTROL_HEADER_SIZE 25

/* this is used when we dump the control header */
#define CONTROL_HEADER_FMT \
"Header              :%.25s\n"

/* netlib globals that need visibility outside netlib3.c */
#ifndef NETLIB

extern void *allocate_shared_memory(uint32_t size, char *file);
extern void init_test_globals(test_t *test);
extern ring_elt_t *allocate_buffer_ring(int32_t width, 
					int32_t size,
					int32_t alignment,
					int32_t offset,
					char    *fill_file);

extern void send_control_message(const int control_sock, 
		     const int32_t test_num,
		     const size_t len, 
		     const char *message);

extern int32_t recv_control_message(int control_sock, 
		     int wait,
		     size_t len, 
		     char *message);

extern void start_worker(test_t *test, 
			 void *start_func(),
			 void *start_arg,
			 int detached);

/* synchronization stuff */
extern barrier_t *netperf_barrier;
extern barrier_t *netperf_stop_barrier;

#ifndef USE_THREADS
/* this is used by the timer code in the process-based tests */
extern test_t    *netlib3_test;
#endif /* USE_THREADS */

extern void sub_add_timeval(struct timeval t1,
			    struct timeval t2,
			    struct timeval *t3);

extern double calc_thruput(test_t *test, double units);

/* the "switch table" */
extern FuncWithArg netperf_switch[TEST_LAST][NUM_FUNC];

/* globals for the confidence intervals */
extern uint32_t iteration_max;
extern uint32_t iteration_min;
extern int32_t  confidence_level;
extern double   interval;

/* globals for CPU utilization measurement */
extern uint32_t shell_num_cpus;
extern float    local_cpu_rate;
extern int32_t  local_cpu_usage;

extern void cpu_start(test_t *test);
extern void cpu_stop(test_t *test);
extern void calibrate_local_cpu(test_t *test);

/* control debugging output */
extern int32_t  debug;
extern FILE *where;

#endif /* NETLIB */
