#ifdef DO_DNS
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>

extern char *strdup(const char *);

#include "netlib3.h"
#include "netperf3.h"
#include "nettest3_dns.h"

const char dns_usage[] = "\n\
Usage: netperf [global options] -- [test options] \n\
\n\
DNS Test Options:\n\
    -f filename       Specify the request source file\n\
                        Default: /tmp/netperf_dns_requests\n\
    -h                Display this text\n\
\n\
For those options taking two parms, at least one must be specified;\n\
specifying one value without a comma will set both parms to that\n\
value, specifying a value with a leading comma will set just the second\n\
parm, a value with a trailing comma will set just the first. To set\n\
each parm to unique values, specify both and separate them with a\n\
comma.\n"; 

void
print_dns_usage()
{
  printf("%s",dns_usage);
  exit(1);
}

static dns_request_t *
allocate_dns_request_ring(test_t *test)
{
  dns_request_t *first_link = NULL;
  dns_request_t *temp_link  = NULL;
  dns_request_t *prev_link;

  char temp_name[NETPERF_HOST_MAX];
  char temp_class[64];
  char temp_type[64];
  char temp_success[64];

  long count,i;

  dns_test_settings *settings;
  FILE *request_source;

  settings = (dns_test_settings *)&(test->test_specific_settings);


  if (debug) {
    fprintf(where,
	    "Allocating a request ring with requests from %s\n",
	    settings->request_file);
    fflush(where);
  }

  request_source = (FILE *)fopen(settings->request_file,"r");
  if (request_source == (FILE *)NULL) {
    fprintf(where,
	    "Could not open requested fill file: errno %d",
	    errno);
    fflush(where);
    exit(-1);
  }

  prev_link = NULL;
  count = 0;
  
  while ((fscanf(request_source,
		 "%s %s %s %d\n",
		 temp_name,
		 temp_class,
		 temp_type,
		 temp_success) != EOF)) {
    count++;
    temp_link = (dns_request_t *)malloc(sizeof(dns_request_t));
    if (temp_link == NULL) {
      fprintf(where,
	      "allocate_dns_request_ring malloc failure errno %d\n",
	      errno);
      fflush(where);
      exit(-1);
    }
    if (first_link == NULL) {
      first_link = temp_link;
    }
    temp_link->class = C_IN; /* yeah, I'm hardcoding at the moment */
    temp_link->type  = T_A;  /* ditto */
    temp_link->success = 0;  /* again */
    temp_link->name = NULL;
    temp_link->name = strdup(temp_name);
    if (temp_link->name == NULL) {
      fprintf(where,
	      "allocate_dns_request_ring strdup failure errno %d\n",
	      errno);
      fflush(where);
      exit(-1);
    }
    temp_link->next = prev_link;
    prev_link = temp_link;
  }
  first_link->next = temp_link;

  /* I suspect we will never try to run through more than 32 bits
     worth of RR's... */
  settings->num_rr = (int32_t) count;

  /* we really do not want all the reqeustors requesting the same RR's
     in the same order. full randomization is not exactly something I
     would like to do, but I can at least have each start in a
     pseudo-random spot in the request chain. raj 3/98 */

  /* knuth only knows if that is a decent seed. we assume that srandom
     and random are threadsafe - may not be a good assumption... raj
     3/98 */
  srandom((unsigned)test->thread_num * (unsigned)getpid());
  count = random() % count;

printf("Thread %d is cycling through %d requests for first_request\n",
       test->thread_num,
       count);

  for (i = 0; i < count; i++) {
    first_link = first_link->next;
  }
  
  return(first_link); 
}

void
dump_dns_settings_i(dns_test_settings *settings)
{
  if (settings == NULL) {
    fprintf(where,"Cannot dump settings from a null pointer\n");
    exit(-1);
  }

  fprintf(where,"\trequest_file %s\n",settings->request_file);
  fprintf(where,"\ttest_dest %s\n",settings->test_dest);
  fprintf(where,"\tnum_rr %s\n",settings->num_rr);
}

void
dump_dns_settings(test_t *test)
{
  /* at some point, a better destination should be found */
  fprintf(where,
	  "Dumping the contents for dns_test_settings for thread %d\n",
	  test->thread_num);
  dump_dns_settings_i((dns_test_settings *)&(test->test_specific_settings));
}

static void
dump_dns_results_i(dns_test_results *results)
{
  if (results == NULL) {
    fprintf(where,"Cannot dump results from a null pointer\n");
    exit(-1);
  }
  fprintf(where,
	  "Final Settings\n");
  dump_dns_settings_i((dns_test_settings *)&(results->final_settings));

  fprintf(where,
	  "Final Results\n");
  fprintf(where,"\tnum_trans %d\n",results->num_trans);
  fprintf(where,"\telapsed_time %g\n",results->elapsed_time);
}

void
dump_dns_results(test_t *test)
{
  /* at some point, a better destination should be found */
  fprintf(where,
	  "Dumping the contents of dns_test_results for thread %d\n",
	  test->thread_num);
  dump_dns_results_i((dns_test_results *)test->test_specific_results);

}

void
init_dns_settings(dns_test_settings *settings)
{
  gethostname(settings->test_dest,NETPERF_HOST_MAX);
  strcpy(settings->request_file,"/opt/netperf/dns_requests");
}

/* given the nature of netperf3, and the DNS_RR test, the "netperf"
   side of the test is really rather passive - it is presumed to be
   running on the SUT. the "remotes" running the netserver side are
   the ones doing the real work here. so, the "send" and "recv" tests
   are really somewhat misnamed. raj 3/98 */

void
send_dns_rr(test_t *test) 
{

  char test_message[MAX_CONTROL_MSG_LEN];

  dns_test_settings *settings;
  dns_test_settings *final_settings;
  dns_test_results  *results;

  int test_num;

  settings = (dns_test_settings *)&(test->test_specific_settings);

  results = (dns_test_results *)&(test->test_specific_results);

  final_settings = (dns_test_settings *)&(results->final_settings);

  if (debug) {
    fprintf(where,
	    "Sending a DNS_RR test settings %p results %p final %p\n",
	    settings,
	    results,
	    final_settings);
    fflush(where);
  }

  if (debug > 1) {
    dump_bsd_settings(test);
  }

  /* if the user requested CPU utilization measurements, we want to
     get the CPU measurement infrastructure started and warmed-up. */

  calibrate_local_cpu(test);

  /* now that we are ready on our side, lets tell the remote what we
     want to do. admittedly, this is a triffle ugly - i guess that if
     I wanted to get fancy, I could define a parser for the format
     string that would automagically pick the proper variables, but
     that is a luxury. REMEMBER, we are sending the *remote* the
     settings it needs to know. raj 2/98 */

  sprintf(test_message,
	  DNS_RR_REQ,
	  test->test_length,
	  settings->num_rr,
	  settings->request_file,
	  settings->test_dest);

  send_control_message(test->control_sock,
		       DNS_RR,
		       strlen(test_message),
		       test_message);

  
  /* once the remote has chewed on our request, he will send us a
     response one of these days, we may set things up so we send the
     list of RR's to request through the ocntrol cosket, but until
     then we will assume that the request file is present on each
     client. raj 3/98 */

  test_num = recv_control_message(test->control_sock,
				  0, /* we want the select */
				  sizeof(test_message),
				  test_message);

  if (test_num != DNS_RR) {
    /* something went wrong */
    fprintf(where,
	    "Remote error: %s\n",test_message);
    fflush(where);
    exit(-1);
  }

  /* we now wait for everyone else to say that they are ready */
  barrier_wait(netperf_barrier);

  /* now that we are through the start barrier, send the go message to
     the remote and then sleep for the expected test length before
     looking for the results. raj 3/98 */

  sprintf(test_message,
	  NETPERF_GO_FMT);

  send_control_message(test->control_sock,
		       NETPERF_GO,
		       strlen(test_message),
		       test_message);

  sleep(test->test_length);

  test_num = recv_control_message(test->control_sock,
				  0, /* we want the select */
				  sizeof(test_message),
				  test_message);

  if (test_num != DNS_RR) {
    /* something went wrong */
    fprintf(where,
	    "Remote error: %s\n",test_message);
    fflush(where);
    exit(-1);
  }

  /* let us see what the results were */

  sscanf(test_message,
	 DNS_RR_RES,
	 &(results->num_trans),
	 &(results->elapsed_time));

}

void
recv_dns_rr(test_t *test)
{

  dns_test_settings *settings;
  dns_test_settings *final_settings;
  dns_test_results *results;

  dns_request_t *dns_request; /* a loop of requests we run through */
  dns_request_t *first_request;

  unsigned char server_response[1024]; /* should be enough for now */
  int           server_response_length = sizeof(server_response);

  char reply_message[MAX_CONTROL_MSG_LEN];

  int test_num;

  settings = (dns_test_settings *)&(test->test_specific_settings);

  results = (dns_test_results *)&(test->test_specific_results);

  final_settings = (dns_test_settings *)&(results->final_settings);

  /* pull the information from the test message */
  sscanf(test->setup_message,
	 DNS_RR_REQ,
	 &(test->test_length),
	 &(settings->num_rr),
	 &(settings->request_file),
	 &(settings->test_dest));

  if (debug) {
    fprintf(where,
	    "Receiving a DNS_RR test\nMy initial message was |%s|\n",
	    test->setup_message);
    fprintf(where," test length is %d\n",test->test_length);
    dump_dns_settings(test);
    fflush(where);
  }

  /* once we have figure-out how to go behind the back of the systems
     resolver library, or we include a library of our own, we will
     convert the test_dest to an IP address */

  first_request = dns_request = allocate_dns_request_ring(test);

  if (debug) {
    fprintf(where,
	    "recv_dns_rr: allocated request_ring with %d entries\n",
	    settings->num_rr);
    fflush(where);
  }

  /* OK, we are all set and ready to go - tell the requestor so he can
     go through the barrier and give us the go message */

   sprintf(reply_message,
	   DNS_RR_RSP,
	   final_settings->num_rr,
	   final_settings->request_file,
	   final_settings->test_dest);
 
   send_control_message(test->control_sock,
			DNS_RR,
			strlen(reply_message),
			reply_message);

   /* sometime soon, we should receive a GO message. use the flavor of
      call that hangs on a select just in case */

  test_num = recv_control_message(test->control_sock,
				  0, /* we want the select */
				  sizeof(reply_message),
				  reply_message);

  if (test_num != NETPERF_GO) {
    /* something went wrong */
    fprintf(where,
	    "Remote error: %s\n",reply_message);
    fflush(where);
    exit(-1);
  }

  
  start_timer(test);
  cpu_start(test);

  while (!test->times_up) {

    if (debug > 1) {
      fprintf(where,
	      "requesting resolution of %s class %d type %d\n",
	      dns_request->name,
	      dns_request->class,
	      dns_request->type);
      fflush(where);
    }

    if (res_query(dns_request->name,
		  dns_request->class,
		  dns_request->type,
		  server_response,
		  server_response_length) == -1) {
      fprintf(where,
	      "recv_dns_rr: res_query failled for %s. errno %d\n",
	      dns_request->name,
	      errno);
      fflush(where);
    }
    /* time to move-on to the next request */
    dns_request = dns_request->next;
    results->num_trans++;
    if (dns_request == first_request) {
      fprintf(where,"wrapped at num_trans %d after %s\n",
	      results->num_trans,
	      dns_request->name);
      fflush(where);
    }
  }

  cpu_stop(test);

  results->elapsed_time = test->elapsed_time;

  sprintf(reply_message,
	  DNS_RR_RES,
	  results->num_trans,
	  results->elapsed_time);

  send_control_message(test->control_sock,
		       DNS_RR,
		       strlen(reply_message),
		       reply_message);

}

void
scan_dns_args(test_t *test)
{
  /* it is quite important to remember that getopt may not be
     thread-safe, so we want to make certain that this routine is only
     called from the main thread. that will avoid any thread-safe
     issues when we compile -DUSE_THREADS. raj 3/98 */

  extern char *optarg;

  int		c;
  
  dns_test_settings  *settings;

  char	
    arg1[BUFSIZ],  /* argument holders		*/
    arg2[BUFSIZ];
  
  struct hostent *hp;

  assert((test != NULL));

  /* it would be a good idea to make sure that there is enough space
     for our test-specific settings. this would have been nicer as a
     compile time check, but I cannot figure-out how to do that. raj
     3/98 */

  assert((sizeof(dns_test_settings) < NETPERF_MAX_SETTINGS));

  settings = (dns_test_settings *)test->test_specific_settings;

  /* make sure things are nicely initialized */
  init_dns_settings(settings);

  /* make sure there is enough space for the results too */
  assert((sizeof(dns_test_results) < NETPERF_MAX_RESULTS));

  /* Go through all the command line arguments and break them */
  /* out. For those options that take two parms, specifying only */
  /* the first will set both to that value. Specifying only the */
  /* second will leave the first untouched. To change only the */
  /* first, use the form "first," (see the routine break_args.. */
  
  while ((c= getopt(test->argc, test->argv, DNS_ARGS)) != EOF) {
    switch (c) {
    case '?':	
    case 'h':
      print_dns_usage();
      exit(1);
    case 'f':
      /* this is the name of the log file from which we pick what to */
      /* request */
      strncpy(settings->request_file,optarg,PATH_MAX);

      if (debug) {
	fprintf(where,"Will take DNS names from %s\n",optarg);
	fflush(where);
      }
      
      break;
    case 'H':
      /* someone wishes to specify the DNS server as being different */
      /* from the local endpoint of the control connection */

      strncpy(settings->test_dest,optarg,NETPERF_HOST_MAX);
      break;
    };
  }
}

void
print_dns_rr(test_t *test)
{

  double throughput;

  dns_test_results *results;
  dns_test_settings *final_settings;

  results = (dns_test_results *)&(test->test_specific_results);
  final_settings = (dns_test_settings *)&(results->final_settings);

  if (debug > 1) {
    dump_dns_results(test);
  }

  if (debug) {
    fprintf(where,
	    "num_trans %d dest %s time %g\n",
	    results->num_trans,
	    final_settings->test_dest,
	    results->elapsed_time);
    fflush(where);
  }

  throughput = (double)results->num_trans /
    (double)results->elapsed_time;

  printf("Thread %d, elapsed time %f tput %f\n",
	 test->thread_num,
	 results->elapsed_time,
	 throughput);
}
  
void
nettest3_dns_init()
{
  /* stick out fuction pointers into the apropriate places in the
     netperf_switch table */

  if (debug) {
    fprintf(where,"Initializing the DNS Tests\n");
    fflush(where);
  }

  netperf_switch[DNS_RR][SCAN] = scan_dns_args;
  netperf_switch[DNS_RR][SEND] = send_dns_rr;
  netperf_switch[DNS_RR][RECV] = recv_dns_rr;
  netperf_switch[DNS_RR][PRINT] = print_dns_rr;
}
#endif /* DO_DNS */
