#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

extern char *strdup(const char *);

#include "netlib3.h"
#include "netperf3.h"
#include "nettest3_ftp.h"

#include "../libftpio/ftpio.h"

const char ftp_usage[] = "\n\
Usage: netperf [global options] -- [test options] \n\
\n\
FTP Test Options:\n\
    -D                Set TCP_NODELAY on the FTP client side\n\
    -h                Display this text\n\
    -H name|ip        Override the global -H as the test destination\n\
    -L name|ip        Explicitly set the source IP address\n\
    -m bytes          Set the send size (FTP_UPLOAD)\n\
    -M bytes          Set the recv size (FTP_DOWNLOAD)\n\
    -p password       Override the default FTP login password (netperf@)\n\
    -s send[,recv]    Set the FTP client socket send/recv buffer sizes\n\
    -u username       Override the default FTP login user (ftp)\n\
    -w bytes          Set the FTP client socket watermarks\n\
    -W workload       Take commands/settings from specified file\n\
\n\
For those options taking two parms, at least one must be specified;\n\
specifying one value without a comma will set both parms to that\n\
value, specifying a value with a leading comma will set just the second\n\
parm, a value with a trailing comma will set just the first. To set\n\
each parm to unique values, specify both and separate them with a\n\
comma.\n"; 

void
print_ftp_usage()
{

  printf("%s",ftp_usage);
  exit(1);

}

static ftp_request_t *
allocate_ftp_request_ring_nofile(test_t *test)
{
  ftp_request_t *temp_request  = NULL;

  ftp_test_settings *settings;

  settings = (ftp_test_settings *)&(test->test_specific_settings);

  if (debug) {
    fprintf(where,
	    "Generating a request ring from internal and command settings\n");
    fflush(where);
  }

  exit(1);

}

static ftp_request_t *
get_next_request(test_t *test,ftp_request_t *curr_request)
{
  ftp_request_t *temp_link;
  ftp_test_settings *settings;

  settings = (ftp_test_settings *)&(test->test_specific_settings);

  temp_link = (ftp_request_t *)((char *)curr_request +
				curr_request->next_offset);
  if (temp_link >= (ftp_request_t *)(settings->mmap_max_addr)) {
    temp_link = (ftp_request_t *)(settings->mmap_base_addr);
  }

  return temp_link;
}

static ftp_request_t *
allocate_ftp_request_ring(test_t *test)
{
  ftp_request_t *temp_link  = NULL;

  long count,i;

  ftp_test_settings *settings;
  int input;
  struct stat statbuf;

  settings = (ftp_test_settings *)&(test->test_specific_settings);

  if (strcmp(settings->workload,"none") == 0) {
    /* there is no workload file, use the internal defaults */
    return(allocate_ftp_request_ring_nofile(test));
  }

  if (debug) {
    fprintf(where,
	    "Allocating a request ring with requests from %s\n",
	    settings->workload);
    fflush(where);
  }


  if (stat(settings->workload,&statbuf) == -1) {
    perror("could not stat input file");
    exit(-1);
  }

  input = open(settings->workload,O_RDONLY);
  if (input == -1) {
    perror("could not open input file for reading");
    exit(-1);
  }
  
  settings->mmap_base_addr = mmap(NULL,
				  statbuf.st_size,
				  PROT_READ,
				  MAP_SHARED,
				  input,
				  0);
  
  if (settings->mmap_base_addr == (void *)-1) {
    perror("unable to mmap input file");
    exit(-1);
  }
  
  settings->mmap_max_addr = ((char *)(settings->mmap_base_addr) +
			     statbuf.st_size);

  close(input);

  srandom((unsigned)test->thread_num  + (unsigned)getpid());
  count = random() % 1024; /* perhaps one day I should cycle through
			      and count the number of commands in the
			      workload file... */

  if (debug) {
    printf("Thread %d is cycling through %d requests for first_request\n",
	   test->thread_num,
	   count);
  }

  temp_link = (ftp_request_t *)(settings->mmap_base_addr);
  for (i = 0; i < count; i++) {
    temp_link = (ftp_request_t *)((char *)temp_link +
				  temp_link->next_offset);
    if (temp_link >= (ftp_request_t *)(settings->mmap_max_addr)) {
      temp_link = (ftp_request_t *)(settings->mmap_base_addr);
    }
  }

  /* now, we want to go forward until we hit a login, or things might
     not work right... raj 3/99 */
  while (temp_link->op_type != OPEN) {
    temp_link = (ftp_request_t *)((char *)temp_link +
				  temp_link->next_offset);
    if (temp_link >= (ftp_request_t *)(settings->mmap_max_addr)) {
      temp_link = (ftp_request_t *)(settings->mmap_base_addr);
    }
  }

  return(temp_link);
}

static void
free_request_ring(test_t *test)
{
  int i;
  ftp_test_settings *settings;

  settings = (ftp_test_settings *)&(test->test_specific_settings);

  i = munmap(settings->mmap_base_addr,
	     settings->mmap_max_addr - settings->mmap_base_addr);

#ifdef DEBUG
  if (debug) {
    fprintf(where,"munmap returned %d\n",i);
    fflush(where);
  }
#endif /* DEBUG */

}
void
retrieve_ftp_settings(int temp_socket, test_t *test)
{
  ftp_test_results *results;
  ftp_test_settings *final_settings;

  int sock_opt_len;

  results = (ftp_test_results *)&(test->test_specific_results);
  final_settings = (ftp_test_settings *)&(results->final_settings);

  if (debug) {
    fprintf(where,
	    "retrieve_ftp_settings called socket %d test %p\n",
	    temp_socket,test);
    fflush(where);
  }

#ifdef TCP_MAXSEG  
  sock_opt_len = sizeof(int);
  if (getsockopt(temp_socket,
		 /* hmm, is this really such a good idea? */
		 getprotobyname("tcp")->p_proto,
		 TCP_MAXSEG,
		 (char *)&(results->segment_size),
		 &sock_opt_len) != 0) {
    fprintf(where,
	    "Thread %d could not retrieve the TCP_MSS - errno %d\n",
	    test->thread_num,
	    errno);
    fflush(where);
    results->segment_size = -1;
  }
#else /* TCP_MAXSEG */
  /* The TCP_MAXSEG option is not available */
  results->segment_size = -2;
#endif /* TCP_MAXSEG */

#ifdef SO_SNDBUF
  sock_opt_len = sizeof(int);
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_SNDBUF,
		 (char *)&(results->final_settings.client_socket_send),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "Thread %d could not retrieve final SO_SNDBUF - errno %d\n",
	    test->thread_num,
	    errno);
    fflush(where);
    results->final_settings.client_socket_send = -1;
  }
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_RCVBUF,
		 (char *)&(results->final_settings.client_socket_recv),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "Thread %d could not retreive final SO_RCVBUF - errno %d\n",
	    test->thread_num,
	    errno);
    fflush(where);
    results->final_settings.client_socket_recv;
  }
  
#else /* SO_SNDBUF */
  
  results->final_settings.local_socket_send = -1;
  results->final_settings.local_socket_recv = -1;
  
#endif /* SO_SNDBUF */

}

void
dump_ftp_settings_i(ftp_test_settings *settings)
{
  if (settings == NULL) {
    fprintf(where,"Cannot dump settings from a null pointer\n");
    exit(-1);
  }

  fprintf(where,"\ttest_dest |%s|\n",settings->test_dest);
  fprintf(where,"\ttest_source |%s|\n",settings->test_source);
  fprintf(where,"\tclient_socket_send %d\n",settings->client_socket_send);
  fprintf(where,"\tclient_socket_recv %d\n",settings->client_socket_recv);
  fprintf(where,"\tclient_nodelay %d\n",settings->client_nodelay);
  fprintf(where,"\tsend_size %d\n",settings->send_size);
  fprintf(where,"\trecv_size %d\n",settings->recv_size);
  fprintf(where,"\tclient_watermark %d\n",settings->client_watermark);
  fprintf(where,"\tuser %s\n",settings->user);
  fprintf(where,"\tworkload %s\n",settings->workload);
}

/* this probably needs to be made thread safe at some point. raj 3/99 */
char *
code_to_str(int command_code)
{
  switch (command_code) {
  case CLOS:
    return "CLOS";

  case CWD:
    return "CWD";
      
  case LIST:
    return "LIST";

  case MDTM:
    return "MDTM";

  case NLST:
    return "NLST";

  case NOOP:
    return "NOOP";

  case OPEN:
    return "OPEN";

  case PASS:
    return "PASS";

  case PASV:
    return "PASV";

  case PORT:
    return "PORT";

  case PWD:
    return "PWD";

  case QUIT:
    return "QUIT";

  case REST:
    return "REST";

  case RETR:
    return "RETR";

  case SIZE:
    return "SIZE";

  case STAT:
    return "STAT";

  case SYST:
    return "SYST";

  case TYPE:
    return "TYPE";

  case USER:
    return "USER";

  case WAIT:
    return "WAIT";

  default:
    return "UNKN";
  }
}

void
dump_ftp_settings(test_t *test)
{
  /* at some point, a better destination should be found */
  fprintf(where,
	  "Dumping the contents for ftp_test_settings for thread %d\n",
	  test->thread_num);
  dump_ftp_settings_i((ftp_test_settings *)&(test->test_specific_settings));
}

static void
dump_ftp_results_i(ftp_test_results *results)
{

  int i;

  if (results == NULL) {
    fprintf(where,"Cannot dump results from a null pointer\n");
    exit(-1);
  }
  fprintf(where,
	  "Final Settings\n");
  dump_ftp_settings_i((ftp_test_settings *)&(results->final_settings));

  fprintf(where,
	  "Final Results\n");
  fprintf(where,"\tbytes_received %d\n",results->bytes_received);
  fprintf(where,"\tsegment_size %d\n",results->segment_size);

  fprintf(where,"\tOP\tGood\tBad\tBytes\t\tSecs\tUsecs\n");
  for (i = 0; i < MAXOPS; i++) {
    fprintf(where,
	    "%8s %7d %7d %15llu %7d %7d\n",
	    code_to_str(i),
	    results->op_res[i].num_ok,
	    results->op_res[i].num_bad,
	    results->op_res[i].bytes_received,
	    results->op_res[i].elapsed.tv_sec,
	    results->op_res[i].elapsed.tv_usec);
  }
}

void
dump_ftp_results(test_t *test)
{
  /* at some point, a better destination should be found */
  fprintf(where,
	  "Dumping the contents of ftp_test_results for thread %d\n",
	  test->thread_num);
  dump_ftp_results_i((ftp_test_results *)test->test_specific_results);

  fflush(where);
}

 /* This routine will create a data (listen) socket with the apropriate */
 /* options set and return it to the caller. this replaces all the */
 /* duplicate code in each of the test routines and should help make */
 /* things a little easier to understand. since this routine can be */
 /* called by either the netperf or netserver programs, all output */
 /* should be directed towards "where." family is generally AF_INET, */
 /* and type will be either SOCK_STREAM or SOCK_DGRAM */
static int
create_data_socket(int family, int type, test_t *test)
{

  int temp_socket;
  int one;
  int sock_opt_len;
  uint32_t addr;
  struct sockaddr_in myaddr;
  struct hostent *hp;
  int32_t do_bind = 0;

  /* from this we will get what the values should be */
  ftp_test_settings *settings;

  /* and here we may put what the values became */
  ftp_test_results  *results;

  /* we need some VASSERTS here I think */
  settings = (ftp_test_settings *)&(test->test_specific_settings);
  results  = (ftp_test_results *)&(test->test_specific_results);

  /* set up the data socket */
  temp_socket = socket(family, type,  0);
  
#ifdef WIN32
  if (temp_socket == INVALID_SOCKET){
#else
  if (temp_socket < 0){
#endif /* WIN32 */
    fprintf(where,
	    "netperf: create_data_socket: socket: %d\n",
	    errno);
    fflush(where);
    exit(-1);
  }
  
  if (debug) {
    fprintf(where,"create_data_socket: socket %d obtained...\n",temp_socket);
    fflush(where);
  }
  
  /* Modify the local socket size. The reason we alter the send buffer */
  /* size here rather than when the connection is made is to take care */
  /* of decreases in buffer size. Decreasing the window size after */
  /* connection establishment is a TCP no-no. Also, by setting the */
  /* buffer (window) size before the connection is established, we can */
  /* control the TCP MSS (segment size). The MSS is never more that 1/2 */
  /* the minimum receive buffer size at each half of the connection. */
  /* This is why we are altering the receive buffer size on the sending */
  /* size of a unidirectional transfer. If the user has not requested */
  /* that the socket buffers be altered, we will try to find-out what */
  /* their values are. If we cannot touch the socket buffer in any way, */
  /* we will set the values to -1 to indicate that.  */
  
#ifdef SO_SNDBUF
  if (settings->client_socket_send > 0) {
    if(setsockopt(temp_socket, SOL_SOCKET, SO_SNDBUF,
		  (char *)&(settings->client_socket_send), 
		  sizeof(int)) < 0) {
      fprintf(where,
	      "netperf: create_data_socket: SO_SNDBUF option: errno %d\n",
	      errno);
      fflush(where);
      exit(1);
    }
    if (debug > 1) {
      fprintf(where,
	      "netperf: create_data_socket: SO_SNDBUF of %d requested.\n",
	      settings->client_socket_send);
      fflush(where);
    }
  }
  if (settings->client_socket_recv > 0) {
    if(setsockopt(temp_socket, SOL_SOCKET, SO_RCVBUF,
		  (char *)&(settings->client_socket_recv),
		  sizeof(int)) < 0) {
      fprintf(where,
	      "netperf: create_data_socket: SO_RCVBUF option: errno %d\n",
	      errno);
      fflush(where);
      exit(1);
    }
    if (debug > 1) {
      fprintf(where,
	      "netperf: create_data_socket: SO_RCVBUF of %d requested.\n",
	      settings->client_socket_recv);
      fflush(where);
    }
  }
  
  
  /* Now, we will find-out what the size actually became, and report */
  /* that back to the user. If the call fails, we will just report a -1 */
  /* back to the initiator for the recv buffer size. */
  
  sock_opt_len = sizeof(int);
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_SNDBUF,
		 (char *)&(results->final_settings.client_socket_send),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "netperf: create_data_socket: getsockopt SO_SNDBUF: errno %d\n",
	    errno);
    fflush(where);
    results->final_settings.client_socket_send = -1;
  }
  if (getsockopt(temp_socket,
		 SOL_SOCKET,	
		 SO_RCVBUF,
		 (char *)&(results->final_settings.client_socket_recv),
		 &sock_opt_len) < 0) {
    fprintf(where,
	    "netperf: create_data_socket: getsockopt SO_SNDBUF: errno %d\n",
	    errno);
    fflush(where);
    results->final_settings.client_socket_recv;
  }
  
  if (debug) {
    fprintf(where,
	    "netperf: create_data_socket: socket sizes determined...\n");
    fprintf(where,
	    "                       send: %d recv: %d\n",
	    results->final_settings.client_socket_send,
	    results->final_settings.client_socket_recv);
    fflush(where);
  }
  
#else /* SO_SNDBUF */
  
  results->final_settings.client_socket_send = -1;
  results->final_settings.client_socket_recv = -1;
  
#endif /* SO_SNDBUF */

  
  /* Now, we will see about setting the TCP_NODELAY flag on the local */
  /* socket. We will only do this for those systems that actually */
  /* support the option. If it fails, note the fact, but keep going. */
  /* If the user tries to enable TCP_NODELAY on a UDP socket, this */
  /* will cause an error to be displayed */
  
#ifdef TCP_NODELAY
  if (settings->client_nodelay) {
    one = 1;
    if(setsockopt(temp_socket,
		  getprotobyname("tcp")->p_proto,
		  TCP_NODELAY,
		  (char *)&one,
		  sizeof(one)) < 0) {
      fprintf(where,
	      "netperf: create_data_socket: nodelay: errno %d\n",
	      errno);
      fflush(where);
    }
    
    if (debug > 1) {
      fprintf(where,
	      "netperf: create_data_socket: TCP_NODELAY requested...\n");
      fflush(where);
    }
  }
#else /* TCP_NODELAY */
  
  settings->local_nodelay = 0;
  
#endif /* TCP_NODELAY */

  bzero(&myaddr,sizeof(myaddr));
  myaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  myaddr.sin_port = htons(0);
  myaddr.sin_family = AF_INET;
  
  /* if the user specified a different source IP address or host name,
     look that up.  */  

  if (strcmp(settings->test_source,"INADDR_ANY") != 0) {
    /* looks as though someone wanted to specify a local address */
    if (debug) {
      fprintf(where,"Asked to set the local address to %s\n",
	      settings->test_source);
    }

    if ((addr = inet_addr(settings->test_source)) == -1) {
      do_bind++;
      /* it was not an IP address, try it as a name */
      if ((hp = gethostbyname(settings->test_source)) == NULL) {
	/* we have no idea what address they wwere asking for */
	fprintf(where,
		"Could not translate %s for binding to a socket\n",
		settings->test_source);
	exit(-1);
      }
      else {
	/* it was a valid remote_host */
	bcopy(hp->h_addr,
	      (char *)&myaddr.sin_addr,
	      hp->h_length);
	myaddr.sin_family = hp->h_addrtype;
      }
    }
    else {
      /* it was a valid IP address */
      myaddr.sin_addr.s_addr = addr;
      myaddr.sin_family = AF_INET;
    }
  }

  return(temp_socket);

}

void
init_ftp_settings(ftp_test_settings *settings)
  {
    /* this will get fleshed-out more later */
    strcpy(settings->test_source,"INADDR_ANY");
    strcpy(settings->test_dest,"sut");
    strcpy(settings->workload,"none");
    strcpy(settings->user,"ftp");
    strcpy(settings->password,"netperf@");
  }

void
scan_ftp_args(test_t *test)
{

  /* it is important to remember that getopt is *NOT* thread-safe, so
     we want to make very certain that this routine is _only_ called
     from the main thread. */

  extern char	*optarg;	  /* pointer to option string	*/
  
  int		c;

  ftp_test_settings *settings;

  char	
    arg1[BUFSIZ],  /* argument holders		*/
    arg2[BUFSIZ];
  
  /* make sure we were not given a NULL pointer to our test structure */
  assert((test != NULL));

  /* make sure we have enough space for our test-specific settings. I
     would have prefered this to be compile-time check, but one cannot
     put a sizeof() into a #if. bummer. raj 2/98 */
  assert((sizeof(ftp_test_settings) < NETPERF_MAX_SETTINGS));

  settings = (ftp_test_settings *)test->test_specific_settings;

  /* while we are here, we might as well check the results structure
     too */

  init_ftp_settings(settings);

  assert((sizeof(ftp_test_results) < NETPERF_MAX_RESULTS));


  /* Go through all the command line arguments and break them */
  /* out. For those options that take two parms, specifying only */
  /* the first will set both to that value. Specifying only the */
  /* second will leave the first untouched. To change only the */
  /* first, use the form "first," (see the routine break_args.. */
  
  while ((c= getopt(test->argc, test->argv, FTP_ARGS)) != EOF) {
    switch (c) {
    case '?':	
    case 'h':
      print_sockets_usage();
      exit(1);
    case 'D':
      /* set the TCP nodelay flag */
      settings->client_nodelay = 1;
      break;
    case 'H':
      /* the user wishes to specify a remote hostname or IP address
	 which differs from the one used to establish the control
	 connection. of course if this path can stand the test
	 connection, it should be able to stand the control connection
	 too */
      strncpy(settings->test_dest,optarg,NETPERF_HOST_MAX);
      break;
    case 's':
      /* set ftp client socket sizes */
      break_args(optarg,arg1,arg2);
      if (arg1[0])
	settings->client_socket_send = convert(arg1);
      if (arg2[0])
	settings->client_socket_recv = convert(arg2);
      break;
    case 'm':
      /* set the send size */
      settings->send_size = convert(optarg);
      break;
    case 'M':
      /* set the recv size */
      settings->recv_size = convert(optarg);
      break;
    case 'p':
      /* override the default FTP login password */
      strncpy(settings->password,optarg,NETPERF_PWRD_MAX);
      break;
    case 'u':
      /* override the default user name */
      strncpy(settings->user,optarg,NETPERF_USER_MAX);
      break;
    case 'w':
      settings->client_watermark = (uint16_t) atoi(optarg);
      break;
    case 'W':
      /* state that we have a workload file */
      strncpy(settings->workload,optarg,NETPERF_FILE_MAX);
      break;
    };
  }
}

/* though slightly strange in name, this is the netperf side of the
   FTP download test. it is expected that the netperf side will be
   running either on the SUT, or a central coordinating machine. this
   is not entirely unlike what happens in a DNS_RR test. if one wants
   CPU utilization of the SUT, the netperf side should be started on
   the SUT, otherwise a different system is fine.
   
   this routine will send the test parameters to the netserver side,
   which will run the ftp client code (yes, "client" code running on
   the net_server_ :) raj 2/99 */
 
void
send_ftp_download(test_t *test)
{

  int data_socket;
  char *test_dest;
  int test_num;
  char test_message[MAX_CONTROL_MSG_LEN];
  int32_t bytes_to_send;
  int32_t bytes_sent;

  struct sockaddr_in server;
  struct hostent *hp;
  int32_t addr;

  double service_demand;
  int method;
  int numcpus;

  ring_elt_t *send_ring = NULL;

  ftp_test_settings *settings;
  ftp_test_settings *final_settings;
  ftp_test_results  *results;

  settings = (ftp_test_settings *)&(test->test_specific_settings);

  results = (ftp_test_results *)&(test->test_specific_results);

  final_settings = (ftp_test_settings *)&(results->final_settings);

  if (debug) {
    fprintf(where,
	    "Sending a FTP_DOWNLOAD test settings %p results %p final %p\n",
	    settings,
	    results,
	    final_settings);
    fflush(where);
  }

  if (debug > 1) {
    dump_ftp_settings(test);
  }

  /* if the user requested CPU utilization measurements, we want to
     get the CPU measurement infrastructure started and warmed-up. */

  calibrate_local_cpu(test);


  /* now that we are ready on our side, lets tell the remote what we
     want to do. admittedly, this is a triffle ugly - i guess that if
     I wanted to get fancy, I could define a parser for the format
     string that would automagically pick the proper variables, but
     that is a luxury. REMEMBER, we are sending the *remote* the
     settings it needs to know. raj 2/98 */

  sprintf(test_message,
	  FTP_DOWNLOAD_REQ,
	  /* first, the NETPERF_BUF_SETTINGS */
	  test->remote_cpu_rate,
	  test->remote_send_align,
	  test->remote_recv_align,
	  test->remote_send_offset,
	  test->remote_recv_offset,
	  test->send_width,
	  test->recv_width,
	  /* now, the FTP_DOWNLOAD_REQ_COMM settings */
	  test->test_length,
	  settings->client_socket_send,
	  settings->client_socket_recv,
	  settings->recv_size,
	  settings->send_size,
	  settings->client_nodelay,
	  settings->client_watermark,
	  settings->test_dest,
	  settings->workload,
	  settings->user,
	  settings->password);

  send_control_message(test->control_sock,
		       FTP_DOWNLOAD,
		       strlen(test_message),
		       test_message);

  
  /* once the remote has chewed on our request, he will send us a
     response */

  test_num = recv_control_message(test->control_sock,
				  0, /* we want the select */
				  sizeof(test_message),
				  test_message);

  if (test_num != FTP_DOWNLOAD) {
    /* something went wrong */
    fprintf(where,
	    "Remote error: %s\n",test_message);
    fflush(where);
    exit(-1);
  }
	
  /* so, at this point, we know that everything  is hunky-dory, so
     lets grab the information from the remote */

  sscanf(test_message,
	 FTP_DOWNLOAD_RSP,
	 &(test->test_length),
	 &(final_settings->client_socket_send),
	 &(final_settings->client_socket_recv),
	 &(final_settings->recv_size),
	 &(final_settings->send_size),
	 &(final_settings->client_nodelay),
	 &(final_settings->client_watermark),
	 /* remember, test_dest is a string ! */
	 final_settings->test_dest,
	 /* remember that workload file, username and password are
	    strings! */ 
	 final_settings->workload,
	 final_settings->user,
	 final_settings->password);

  /* we now wait for everyone else to say that they are ready */
  barrier_wait(netperf_barrier);

  /* now that we are all synced-up, we can tell the remote to go ahead
     and start actually running tests. we will do that with just a
     simple send across the control connection to pull the remote out
     of a receive. raj 2/99 */

  send_control_message(test->control_sock,
		       FTP_GO_MSG,
		       strlen("GO!"),
		       "GO!");

  /* ok, now set-up the conditions to tell us when it is time to stop
     sending data. in prior versions of netperf, this could be either
     time, or byte-count based, however, for netperf3, this will just
     be time-based until someone squawks. looks like I might be able
     to combine a couple calls here, but I'll wait until no-one
     squawks about the dropping of byte-length tests. raj 2/98 */
  
  /* this needs some non-trivial cleanup - do we really need to do the
     cpu_start/stop thing and sleep, or can we just sit in
     recv_control_message indefinitely? */
  cpu_start(test);

  /* time to receive the results from the remote */
  test_num = recv_control_message(test->control_sock,
				  1, /* we do not want the select,
					which means if something goes
					truely fubar, we might be here
					for a while. Perhaps we should
					enable SO_KEEPALIVE on the
					control socket? */ 
				  sizeof(test_message),
				  test_message);

  if (test_num != FTP_DOWNLOAD_RESULT) {
    /* something went wrong */
    fprintf(where,
	    "Remote error: %s\n",test_message);
    fflush(where);
    exit(-1);
  }

  cpu_stop(test);

  results->elapsed_time = test->elapsed_time;

  /* this is slightly screwey - we save-off the wall-clock elapsed
     time in the results structure so we can stuff the data transfer
     elapsed time in the test structure so the throughput can be
     calculated correctly, and we can still report the overall elapsed
     time. at some point I should resolve that I suppose... raj 4/99
     */ 
	
  /* so, at this point, we know that everything  is hunky-dory, so
     lets grab the information from the remote */

  if (debug) {
    printf("Received result message |%s|\n",test_message);
  }

  sscanf(test_message,
	 FTP_DOWNLOAD_RES,
	 &(results->bytes_received),
	 &(results->num_receives),
	 &(test->elapsed_time),
	 &(service_demand),
	 &(method),
	 &(numcpus));


  /* this test iteration is over, now sync-up with everyone else -
     either for output, or for the next iteration. */
  barrier_wait(netperf_stop_barrier);
}

void
recv_ftp_download(test_t *test)
{
  ftp_test_settings *settings;
  ftp_test_settings *final_settings;
  ftp_test_results *results;

  ftp_request_t *request_ring;

  struct timeval t1;
  struct timeval t2;

  struct timeval *time_now=&t1;
  struct timeval *time_prev=&t2;
  struct timeval *time_temp=NULL;

  FTP_t ftp;

  int len;
  int retcode;
  time_t start_time;

  sigset_t orig_mask;
  sigset_t new_mask;

  char reply_message[MAX_CONTROL_MSG_LEN];
  char temp_message[MAX_CONTROL_MSG_LEN];

  settings = (ftp_test_settings *)&(test->test_specific_settings);

  results = (ftp_test_results *)&(test->test_specific_results);

  final_settings = (ftp_test_settings *)&(results->final_settings);

  if (debug) {
    printf("Receiving a FTP_DOWNLOAD test\nMy initial message was |%s|\n",
	   test->setup_message);
  }

  /* pull the information from the test message */
  sscanf(test->setup_message,
	 FTP_DOWNLOAD_REQ,
	 /* first, the NETPERF_BUF_SETTINGS */
	 &(test->local_cpu_rate),
	 &(test->local_send_align),
	 &(test->local_recv_align),
	 &(test->local_send_offset),
	 &(test->local_recv_offset),
	 &(test->send_width),
	 &(test->recv_width),
	  /* now, the FTP_DOWNLOAD_REQ_COMM settings */
	 &(test->test_length),
	 &(settings->client_socket_send),
	 &(settings->client_socket_recv),
	 &(settings->recv_size),
	 &(settings->send_size),
	 &(settings->client_nodelay),
	 &(settings->client_watermark),
	 /* test source is a string!!! */
	 &(settings->test_dest[0]), 
	 &(settings->workload[0]),
	 &(settings->user[0]),
	 &(settings->password[0])); 

  request_ring = allocate_ftp_request_ring(test);

  if (request_ring == NULL) {
    fprintf(where,"request_ring allocation failled");
    fflush(where);
    send_control_message(test->control_sock,
			 999,
			 strlen("request_ring alloc failled"),
			 "request_ring alloc failled");
    exit(-1);
  }

  if (debug) {
    dump_ftp_settings(test);

    printf("about to reply to the ftp_download request\n");
  }

  sprintf(reply_message,
	  FTP_DOWNLOAD_RSP,
	  final_settings->client_socket_send,
	  final_settings->client_socket_recv,
	  final_settings->recv_size,
	  final_settings->send_size,
	  final_settings->client_nodelay,
	  final_settings->client_watermark,
	  settings->test_dest,
	  settings->workload,
	  settings->user,
	  settings->password);
  

  send_control_message(test->control_sock,
		       FTP_DOWNLOAD,
		       strlen(reply_message),
		       reply_message);

  /* we need another control message receive before this to tell us to
     get going */

  recv_control_message(test->control_sock,
		       1,
		       MAX_CONTROL_MSG_LEN,
		       temp_message);

  start_time = time(NULL);
  len = 0; /* otherwise, we think we received a boatload of bytes in
	      the open operation :) */

  cpu_start(test);

  /* initialize our timers for the per-operation timing. try not to
     call gettimeofday() more than once through each loop, and at some
     point, we need to use that call to trigger our end-of-test
     condition rather than making an extra call to time(). raj 4/99 */
  
  gettimeofday(time_prev,NULL);
  /* maybe one day we'll worry about skew */
  while ((time(NULL) - start_time) < test->test_length) {

    /* perform th eoperation indicated by the op_type - we try to have
       these in some semblence of order of frequency. raj 3/99 */

    switch (request_ring->op_type) {
    case CWD:
      len = ftpCWD(ftp,request_ring->parm);
      break;

    case TYPE:
      len = ftpTYPE(ftp,request_ring->parm);
#ifdef DEBUG
      if (debug) {
	printf("ftpTYPE got %d\n",len);
      }
#endif /* DEBUG */
      break;

    case PASV:
      len = ftpPASV(ftp);
      break;

    case OPEN:
      ftp = ftpOpen(request_ring->parm, /* where */
		    0,                  /* take the default port */
		    debug,                  /* do not be verbose */
		    &retcode);
#ifdef DEBUG 
      if (debug) {
	printf("login got ftp %p retcode %d\n",ftp,retcode);
      }
#endif /* DEBUG */
      break;

    case USER:
      len = ftpUSER(ftp,request_ring->parm);
      break;

    case PASS:
      len = ftpPASS(ftp,request_ring->parm);
      break;

    case LIST:
      if (request_ring->parm[0]) {
	len = ftpLIST(ftp,request_ring->parm);
      }
      else {
	len = ftpLIST(ftp,NULL);
      }

      len = ftpRecvData(ftp,
			0.0,
			1460);

      break;

    case RETR:
      len = ftpRETR(ftp,request_ring->parm);
#ifdef DEBUG
      if (debug) {
	printf("ftpRETR returned %d\n",len);
      }
#endif /* DEBUG */
      len = ftpRecvData(ftp,
			0.0,
			1460);
#ifdef DEBUG
      if (debug) {
	printf("got %d bytes on the RETR\n",len);      
      }
#endif /* DEBUG */
      results->bytes_received += len;
      break;

    case STAT:
      len = ftpSTAT(ftp,request_ring->parm);
      break;

    case SIZE:
      len = ftpSIZE(ftp,request_ring->parm);
      break;

    case PORT:
      len = ftpPORT(ftp);
      break;

    case MDTM:
      len = ftpMDTM(ftp,request_ring->parm);
      break;

    case SYST:
      len = ftpSYST(ftp);
      break;

    case QUIT:
      len = ftpQUIT(ftp);
      break;

    case CLOS:
      ftpCLOS(ftp);

      /* I'm not entirely sure if this should be burried in ftpClose()
	 but for the time being, i'll place it here. raj 3/99 */
      free(ftp);
      ftp == NULL;

#ifdef DEBUG
      if (debug) {
	printf("quit\n");
      }
#endif /* DEBUG */
      break;

    case REST:
      len = ftpREST(ftp,atoi(request_ring->parm));
      break;

    case PWD:
      len = ftpPWD(ftp);
      break;

    case NLST:
      if (request_ring->parm[0]) {
	len = ftpNLST(ftp,request_ring->parm);
      }
      else {
	len = ftpNLST(ftp,NULL);
      }
      break;

    case WAIT:
      /* should I use something other than sleep? */
      sleep(atoi(request_ring->parm));
      break;

    case NOOP:
      len = ftpNOOP(ftp);
      break;

    default:
      fprintf(where,"Skipping unknown op type!\n");
      fflush(where);
    }

    /* for now, we assume the op was a success - that it matched the
       expected return code */
    results->op_res[request_ring->op_type].num_ok++;

    /* we don't case off the op type for this and just ignore it for
       the non-data transfer ops. */
    results->op_res[request_ring->op_type].bytes_received += len;

    /* ok, take the end timestamp */
    gettimeofday(time_now,NULL);
    
    /* now, take the delta of the two timestamps, hopefully, in a
       manner that does not mess-up the structure pointed to by
       time_now so we can use it for time_prev next time through the
       loop. yes, we are assuming here that the time spent in getting
       to the next operation is trivial when compared with the time
       spent in that operation itself. */
    sub_add_timeval(*time_prev,
		    *time_now,
		    &(results->op_res[request_ring->op_type].elapsed));

    /* "flip" time_prev and time_now pointers */
    time_temp = time_prev;
    time_prev = time_now;
    time_now = time_temp;

    results->num_receives += 1;
    request_ring = get_next_request(test,request_ring);
  }

  cpu_stop(test);

  /* set-up the global numbers, first the bytes_received. since we are
   including the time for the PORT and PASV commands, we should also
   include the bytes from the LIST and NLST commands. we include the
   PORT and PASV commands bcause they are part of the time the user
   sees when asking for the download. this is especially true when a
   web browser is being used. i am not sure if the typical ftp client
   includes it, or iv the web browsers actually include that in their
   bitrate calculations, so that should be checked. until then, be
   conservative in what we send - include the time for the PORT and
   PASV commands. raj 4/99 */

  results->bytes_received = 0;
  results->bytes_received += results->op_res[RETR].bytes_received;
  results->bytes_received += results->op_res[LIST].bytes_received;
  results->bytes_received += results->op_res[NLST].bytes_received;

  /* reuse time_now to accumulate the times */
  time_now->tv_sec = time_now->tv_usec = 0;
  time_now->tv_sec += results->op_res[RETR].elapsed.tv_sec;
  time_now->tv_sec += results->op_res[LIST].elapsed.tv_sec;
  time_now->tv_sec += results->op_res[NLST].elapsed.tv_sec;
  time_now->tv_sec += results->op_res[PORT].elapsed.tv_sec;
  time_now->tv_sec += results->op_res[PASV].elapsed.tv_sec;

  time_now->tv_usec += results->op_res[RETR].elapsed.tv_usec;
  time_now->tv_usec += results->op_res[LIST].elapsed.tv_usec;
  if (time_now->tv_usec > 1000000) {
    time_now->tv_sec += 1;
    time_now->tv_usec -= 1000000;
  }
  time_now->tv_usec += results->op_res[NLST].elapsed.tv_usec;
  if (time_now->tv_usec > 1000000) {
    time_now->tv_sec += 1;
    time_now->tv_usec -= 1000000;
  }
  time_now->tv_usec += results->op_res[PORT].elapsed.tv_usec;
  if (time_now->tv_usec > 1000000) {
    time_now->tv_sec += 1;
    time_now->tv_usec -= 1000000;
  }
  time_now->tv_usec += results->op_res[PASV].elapsed.tv_usec;
  if (time_now->tv_usec > 1000000) {
    time_now->tv_sec += 1;
    time_now->tv_usec -= 1000000;
  }


  /* now we need to communicate these things back to the netperf side.
   at some point, we should really increase the things being reported
   back - for example, the per-operation stats, and perhaps let the
   totals be calculated by the netperf side. */
  sprintf(reply_message,
	  FTP_DOWNLOAD_RES,
	  results->bytes_received,
	  results->num_receives,
	  ((float)time_now->tv_sec + ((float)time_now->tv_usec /
				      (float)1000000)),
	  -1.0,
	  99,
	  -1);

  send_control_message(test->control_sock,
		       FTP_DOWNLOAD_RESULT,
		       strlen(reply_message),
		       reply_message);

  if (debug) {
    dump_ftp_results(test);
  }

  free_request_ring(test);
  /* did we leave an ftp behind? */
  if (ftp) {
    ftpCLOS(ftp);
    free(ftp);
  }
}

void
print_ftp_download(test_t *test)
{

  double bytes_sent;
  double throughput;
  ftp_test_results *results;
  ftp_test_settings *final_settings;

  results = (ftp_test_results *)&(test->test_specific_results);
  final_settings = (ftp_test_settings *)&(results->final_settings);


  if (debug > 1) {
    dump_ftp_results(test);
  }

  bytes_sent = 
    (double)results->bytes_received;

  throughput = calc_thruput(test,bytes_sent);

  printf("Thread %d elapsed time %f format %c tput %f\n",
	 test->thread_num,
	 results->elapsed_time,
	 test->format_units,
	 throughput);

}

void
nettest3_ftp_init()
{
  /* stick our function pointers into the apropriate places in the
     netperf_switch */  

  if (debug) {
    fprintf(where,"Initializing the FTP Tests with test %d\n",send_ftp_download);
    fflush(where);
  }

  netperf_switch[FTP_DOWNLOAD][SCAN] = scan_ftp_args;
  netperf_switch[FTP_DOWNLOAD][SEND] = send_ftp_download;
  netperf_switch[FTP_DOWNLOAD][RECV] = recv_ftp_download;
  netperf_switch[FTP_DOWNLOAD][PRINT] = print_ftp_download;

}




