#include "CipePanel.h"
#include "CipePanel.rh"
#include "hexdump.c"
#include "string"

using namespace std;

//=====================================================================================
//                                   Class Members
//=====================================================================================
char *CipeControlPanel::m_CipherList[] = {"NONE", "BLOWFISH", "IDEA"};

BOOL CipeControlPanel::m_RestartWhenDone = FALSE;

int CipeControlPanel::m_EditFieldID[] =
   {
    CIPEPANEL_LOCAL_IP,
    CIPEPANEL_LOCAL_PORT,
    CIPEPANEL_REMOTE_IP,
    CIPEPANEL_REMOTE_PORT,
    CIPEPANEL_REMOTE_PTP,
    CIPEPANEL_STATIC_KEY,
    CIPEPANEL_CIPHER,
    CIPEPANEL_KEY_TIMEOUT,
    CIPEPANEL_STARTUP_SCRIPT,
    CIPEPANEL_SHUTDOWN_SCRIPT
   };

int CipeControlPanel::m_IconID[] =
   {
    CIPEPANEL_ADAPTER_ICON,
    CIPEPANEL_PEER_ICON
   };

int CipeControlPanel::m_EditFieldSize =
   (
    sizeof (CipeControlPanel::m_EditFieldID) /
    sizeof (CipeControlPanel::m_EditFieldID [0])
   );

int CipeControlPanel::m_IconArraySize =
   (
    sizeof (CipeControlPanel::m_IconID) /
    sizeof (CipeControlPanel::m_IconID [0])
   );

HCURSOR CipeControlPanel::m_SystemCursor;
HCURSOR CipeControlPanel::m_Hourglass;

//=====================================================================================
//                                Factory Methods
//=====================================================================================
CipeControlPanel::CipeControlPanel() : CJBControlPanelApplet (CIPEPANEL_CIPEWIN32_ICON, CIPEPANEL_CIPEWIN32_TITLE, CIPEPANEL_CIPEWIN32_DESCRIPTION)
   {
    m_RestartWhenDone = (BOOL) (CipeServiceBase::GetSvcState ("CIPE_Daemon") == SERVICE_RUNNING);
    m_IsModified = FALSE;
   }

//=====================================================================================
//                                Event Handling
//=====================================================================================
BOOL CipeControlPanel::OnDoubleClick (HWND hWnd, LONG appletData)
   {
    DialogBoxParam (GetInstanceHandle(), "CipePanel", hWnd, (DLGPROC) OnDialogMessage, (unsigned long) this);
    return TRUE;
   }

BOOL CALLBACK CipeControlPanel::OnDialogMessage (HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
   {
    typedef enum
       {
        CIPE_DIALOG_EXIT,
        CIPE_DIALOG_QUIT,
        CIPE_DIALOG_SAVE,
        CIPE_DIALOG_ADD,
        CIPE_DIALOG_DELETE,
        CIPE_SELECT_TREE_ITEM,
        CIPE_REFRESH_TREE,
        CIPE_CREATE_TREE,
        CIPE_BEGIN_LABEL_EDIT,
        CIPE_END_LABEL_EDIT,
        CIPE_VERIFY,
        CIPE_DIALOG_NULL
       }
    CipeDialogAction;

    CipeControlPanel *l_ControlPanel = (CipeControlPanel *)
       (
        uMsg == WM_INITDIALOG ? lParam : GetWindowLong (hwndDlg, GWL_USERDATA)
       );

    CipeDialogAction l_Action = CIPE_DIALOG_NULL;
    BOOL l_Value = TRUE;

    if (l_ControlPanel) switch (uMsg)
       {
        case WM_COMMAND:
           {
            switch (LOWORD (wParam)) 
               {
                case CIPEPANEL_CANCEL_BUTTON: l_Action = CIPE_DIALOG_QUIT;   break;
                case CIPEPANEL_DONE_BUTTON:   l_Action = CIPE_DIALOG_EXIT;   break;
                case CIPEPANEL_ADD_BUTTON:    l_Action = CIPE_DIALOG_ADD;    break;
                case CIPEPANEL_UPDATE_BUTTON: l_Action = CIPE_DIALOG_SAVE;   break;
                case CIPEPANEL_DELETE_BUTTON: l_Action = CIPE_DIALOG_DELETE; break;
                case CIPEPANEL_CHECK_BOX:     l_Action = CIPE_VERIFY;        break;

                case CIPEPANEL_CIPHER:
                case CIPEPANEL_LOCAL_IP:
                case CIPEPANEL_REMOTE_IP:
                case CIPEPANEL_REMOTE_PTP:
                case CIPEPANEL_LOCAL_PORT:
                case CIPEPANEL_REMOTE_PORT:
                case CIPEPANEL_STATIC_KEY:
                case CIPEPANEL_KEY_TIMEOUT:
                   {
                    l_Action = (HIWORD (wParam) == EN_KILLFOCUS || HIWORD (wParam) == CBN_KILLFOCUS ? CIPE_VERIFY : CIPE_DIALOG_NULL);
                    break;
                   }
               }
            break;
           }

        case WM_INITDIALOG:
           {
            m_Hourglass = LoadCursor (NULL, IDC_WAIT);
            m_SystemCursor = GetCursor();
            l_ControlPanel->StatusMsg();
            l_Action = CIPE_CREATE_TREE;
            break;
           }

        case WM_CLOSE:
           {
            l_Action = CIPE_DIALOG_QUIT;
            break;
           }

        case WM_NOTIFY:
           {
            switch (((LPNMTREEVIEW) lParam)->hdr.code)
               {
                case TVN_SELCHANGING:    l_Action = CIPE_DIALOG_SAVE; l_Value = FALSE; break;
                case TVN_SELCHANGED:     l_Action = CIPE_SELECT_TREE_ITEM; break;
                case TVN_BEGINLABELEDIT: l_Action = CIPE_BEGIN_LABEL_EDIT; break;
                case TVN_ENDLABELEDIT:   l_Action = CIPE_END_LABEL_EDIT;   break;
               }

            break;
           }
       }

    switch (l_Action)
       {
        case CIPE_DIALOG_EXIT:
           if (l_ControlPanel->IsModified())
              {
               SetCursor (m_Hourglass);
               l_ControlPanel->StatusMsg ("Saving Adapter settings...");
               l_ControlPanel->SavePeerData();
               l_ControlPanel->m_PeerList.Commit();

               if (m_RestartWhenDone)
                  {
                   l_ControlPanel->StatusMsg ("Restarting CIPE service...");
                   CipeServiceMgr::StopCipeService();
                   CipeServiceMgr::StartCipeService();
                  }
               else if (! CipeServiceMgr::CipeServiceExists())
                  {
                   l_ControlPanel->StatusMsg ("Installing CIPE service...");
                   CipeServiceMgr::InstallCipeService();
                   CipeServiceMgr::StartCipeService();
                  }

               l_ControlPanel->StatusMsg();
               SetCursor (m_SystemCursor);
              }

           EndDialog (l_ControlPanel->m_DialogWindow, wParam);
           break;

        case CIPE_DIALOG_QUIT:
           EndDialog (l_ControlPanel->m_DialogWindow, wParam);
           break;

        case CIPE_DIALOG_SAVE:
           l_ControlPanel->SavePeerData();
           break;

        case CIPE_DIALOG_ADD:
           l_ControlPanel->AddNewPeer();
           break;

        case CIPE_DIALOG_DELETE:
           {
            CipePeer *l_Peer;
            TVITEM l_Item;

            l_Item.mask = TVIF_HANDLE | TVIF_CHILDREN | TVIF_PARAM;
            l_Item.hItem = TreeView_GetSelection (l_ControlPanel->m_TreeView);

            if (TreeView_GetItem (l_ControlPanel->m_TreeView, &l_Item))
            if (l_Peer = (CipePeer *) l_Item.lParam)
            if (l_Peer->m_UserData) // Only allow if it's a peer
               {
                TreeView_DeleteItem (l_ControlPanel->m_TreeView, l_Item.hItem);
                l_ControlPanel->m_IsModified = TRUE;
                l_Peer->MarkForDeletion();
               }

            break;
           }

        case CIPE_SELECT_TREE_ITEM:
           l_ControlPanel->LoadPeerData();
           break;

        case CIPE_VERIFY:
           if (l_ControlPanel->Verify (LOWORD (wParam)) == FALSE)
              {
               SetFocus (GetDlgItem (l_ControlPanel->m_DialogWindow, LOWORD (wParam)));
               l_Value = FALSE;
               Beep (192, 100);
              }

           break;

        case CIPE_BEGIN_LABEL_EDIT:
           {
            CipePeer *l_Peer = (CipePeer *) ((LPNMTVDISPINFO) lParam)->item.lParam;
            l_Value = ! (BOOL) (l_Peer->m_UserData); // Don't allow editing (TRUE) if it's an adapter entry
            break;
           }

        case CIPE_END_LABEL_EDIT:
           {
            CipePeer *l_Peer = (CipePeer *) ((LPNMTVDISPINFO) lParam)->item.lParam;
            char *l_Name = ((LPNMTVDISPINFO) lParam)->item.pszText;

            l_Value = FALSE;

            if (l_Peer && l_Name)
               {
                TVITEM l_Item;

                l_Item.hItem = ((LPNMTVDISPINFO) lParam)->item.hItem; //TreeView_GetSelection (l_ControlPanel->m_TreeView);
                l_Item.mask = TVIF_TEXT | TVIF_HANDLE;
                l_Item.pszText = l_Name;

                TreeView_SetItem (l_ControlPanel->m_TreeView, &l_Item);

                l_Peer->SetName (string (l_Name));
                l_Value = TRUE;
               }

            break;
           }

        case CIPE_CREATE_TREE:
           SetCursor (m_Hourglass);
           l_ControlPanel->StatusMsg ("Loading Adapter information");
           SetWindowLong (l_ControlPanel->m_DialogWindow = hwndDlg, GWL_USERDATA, (long) l_ControlPanel);
           l_ControlPanel->FormatCipherList();
           l_ControlPanel->CreateTreeView();
           l_ControlPanel->StatusMsg();
           SetCursor (m_SystemCursor);
           break;

        default:
           l_Value = FALSE;
           break;
       }

    return l_Value;
   }

//=====================================================================================
//                              TreeView Maintenance
//=====================================================================================
void CipeControlPanel::CreateTreeView()
   {
    HINSTANCE l_Instance = GetInstanceHandle();
    HICON l_Icon;

    if (m_ImageList = ImageList_Create (32, 32, TRUE, CipeControlPanel::m_IconArraySize, 0))
    if (m_TreeView = GetDlgItem (m_DialogWindow, CIPEPANEL_TREEVIEW))
       {
        for (int l_Index = 0; l_Index < CipeControlPanel::m_IconArraySize; ++l_Index)
           {
            if (l_Icon = LoadIcon (l_Instance, MAKEINTRESOURCE (CipeControlPanel::m_IconID [l_Index])))
               {
                ImageList_AddIcon (m_ImageList, l_Icon);
               }
           }

        TreeView_SetImageList (m_TreeView, m_ImageList, TVSIL_NORMAL);

        RefreshTreeView();
       }
   }

void CipeControlPanel::RefreshTreeView()
   {
    TreeView_DeleteAllItems (m_TreeView);

    for (int l_Index = 0; l_Index < m_PeerList.AdapterList().size(); ++l_Index)
       {
        HTREEITEM l_AdapterItem = InsertAdapterItem (m_PeerList.AdapterList() [l_Index]);

        for (int l_PeerIndex = 0; l_PeerIndex < m_PeerList.size(); ++l_PeerIndex)
           {
            if (&m_PeerList.AdapterList() [l_Index] == &m_PeerList [l_PeerIndex].Adapter())
               {
                HTREEITEM l_PeerItem = InsertPeerItem (m_PeerList [l_PeerIndex], l_AdapterItem);
               }
           }
       }
   }

//=====================================================================================
//                       TreeView Item Maintenance
//=====================================================================================
HTREEITEM CipeControlPanel::InsertAdapterItem (CipeAdapter &p_Adapter) throw (CipeControlPanelException)
   {
    TV_INSERTSTRUCT tvins;
    HTREEITEM l_Return;
    TV_ITEM l_Item; 

    p_Adapter.m_UserData = 0; // Adapter tree entries have no parent

    l_Item.mask = TVIF_TEXT | TVIF_IMAGE | TVIF_SELECTEDIMAGE | TVIF_PARAM; 
    l_Item.cchTextMax = p_Adapter.DisplayName().size();
    l_Item.pszText = (char *) p_Adapter.DisplayName().c_str();
    l_Item.iImage = l_Item.iSelectedImage = 0;
    l_Item.lParam = (LPARAM) &p_Adapter;

    tvins.hInsertAfter = (HTREEITEM) TVI_LAST; 
    tvins.hParent = TVI_ROOT; 
    tvins.item = l_Item;

    if (! (l_Return = TreeView_InsertItem (m_TreeView, &tvins)))
       {
        throw CipeControlPanelException();
       }

    return l_Return;
   }

HTREEITEM CipeControlPanel::InsertPeerItem (CipePeer &p_Peer, HTREEITEM p_AdapterItem) throw (CipeControlPanelException)
   {
    TV_INSERTSTRUCT l_InsertItem;
    HTREEITEM l_Return;
    TV_ITEM l_Item; 

    p_Peer.m_UserData = (unsigned long) p_AdapterItem; // Save parent item with Peer data

    l_Item.mask = TVIF_TEXT | TVIF_IMAGE | TVIF_SELECTEDIMAGE | TVIF_PARAM; 
    l_Item.cchTextMax = p_Peer.DisplayName().size();
    l_Item.pszText = (char *) p_Peer.DisplayName().c_str(); 
    l_Item.iImage = l_Item.iSelectedImage = 1;
    l_Item.lParam = (LPARAM) &p_Peer;

    l_InsertItem.hInsertAfter = (HTREEITEM) TVI_LAST; 
    l_InsertItem.hParent = p_AdapterItem; 
    l_InsertItem.item = l_Item;

    if (! (l_Return = TreeView_InsertItem (m_TreeView, &l_InsertItem)))
       {
        throw CipeControlPanelException();
       }

    return l_Return;
   }

void CipeControlPanel::SetEditTextState (BOOL p_Enable)
   {
    for (int l_Index = 0; l_Index < CipeControlPanel::m_EditFieldSize; ++l_Index)
       {
        EnableWindow (GetDlgItem (m_DialogWindow, CipeControlPanel::m_EditFieldID [l_Index]), p_Enable);
       }

    EnableWindow (GetDlgItem (m_DialogWindow, CIPEPANEL_CHECK_BOX), p_Enable);
   }

//=====================================================================================
//                                Peer Item Routines
//=====================================================================================
void CipeControlPanel::AddNewPeer()
   {
    HTREEITEM l_AdapterItem = 0, l_PeerItem = 0;
    CipeAdapter *l_Adapter = 0;
    CipePeer *l_Peer = 0;
    TVITEM l_Item;

    //=================================================================
    // Find the adapter parent of the currently selected item. If it IS
    // an adapter then use the adapter
    //=================================================================
    l_Item.mask = TVIF_HANDLE | TVIF_CHILDREN | TVIF_PARAM;
    l_Item.hItem = TreeView_GetSelection (m_TreeView);

    if (TreeView_GetItem (m_TreeView, &l_Item))
       {
        if ((l_Adapter = (CipeAdapter *) l_Item.lParam)->m_UserData == 0) // Adapter selected
           l_AdapterItem = l_Item.hItem;
        else if (l_Adapter = &((l_Peer = (CipePeer *) l_Adapter)->Adapter()))
           l_AdapterItem = (HTREEITEM) l_Peer->m_UserData;

        try
           {
            m_PeerList.push_back (l_Peer = new CipePeer (*l_Adapter));

            HTREEITEM l_PeerItem = InsertPeerItem (*l_Peer, l_AdapterItem);

            TreeView_SelectItem (m_TreeView, l_PeerItem);
            TreeView_EnsureVisible (m_TreeView, l_PeerItem);
            TreeView_EditLabel (m_TreeView, l_PeerItem);
           }
        catch (CipePeerException &e_Exception)
           {
           }
        catch (CipeControlPanelException &e_Exception)
           {
           }
       }
   }

//=====================================================================================
//                             Peer Registry Data
//=====================================================================================
void CipeControlPanel::LoadPeerData()
   {
    CipeAdapter *l_Adapter = 0;
    CipePeer *l_Peer = 0;
    TVITEM l_Item;

    l_Item.mask = TVIF_HANDLE | TVIF_CHILDREN | TVIF_PARAM;
    l_Item.hItem = TreeView_GetSelection (m_TreeView);

    if (! TreeView_GetItem (m_TreeView, &l_Item)) // Selected nothing
       {
       }
    else if ((l_Adapter = (CipeAdapter *) l_Item.lParam)->m_UserData == 0) // Selected an adapter entry
       {
        for (int l_Index = 0; l_Index < CipeControlPanel::m_EditFieldSize; ++l_Index)
           {
            SetDlgItemText (m_DialogWindow, CipeControlPanel::m_EditFieldID [l_Index], "");
           }

        SetDlgItemText (m_DialogWindow, CIPEPANEL_LOCAL_PTP, inet_str (l_Adapter->Address()).c_str());
        SetEditTextState (FALSE);
       }
    else if (l_Peer = (CipePeer *) l_Item.lParam) // Selected a peer entry
       {
        SetDlgItemText (m_DialogWindow, CIPEPANEL_LOCAL_PTP,       inet_str (l_Peer->LocalPTP()).c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_LOCAL_IP,        l_Peer->LocalIP().c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_LOCAL_PORT,      port_str (l_Peer->LocalPort()).c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_REMOTE_PTP,      l_Peer->PeerPTP().c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_REMOTE_IP,       l_Peer->PeerIP().c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_REMOTE_PORT,     port_str (l_Peer->PeerPort()).c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_STATIC_KEY,      l_Peer->TextKey().c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_CIPHER,          l_Peer->EncryptionType().c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_KEY_TIMEOUT,     num_str (l_Peer->KeyTimeout()).c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_STARTUP_SCRIPT,  l_Peer->StartupScript().c_str());
        SetDlgItemText (m_DialogWindow, CIPEPANEL_SHUTDOWN_SCRIPT, l_Peer->ShutdownScript().c_str());
        CheckDlgButton (m_DialogWindow, CIPEPANEL_CHECK_BOX,       (int) l_Peer->Enabled());
        SetEditTextState (TRUE);
       }
   }

void CipeControlPanel::SavePeerData()
   {
    HTREEITEM l_AdapterItem, l_PeerItem;
    static char l_Buffer [4096];
    CipePeer *l_Peer = 0;
    TVITEM l_Item;

    l_Item.mask = TVIF_HANDLE | TVIF_CHILDREN | TVIF_PARAM;
    l_Item.hItem = l_PeerItem = TreeView_GetSelection (m_TreeView);

    if (TreeView_GetItem (m_TreeView, &l_Item) && (l_Peer = (CipePeer *) l_Item.lParam)->m_UserData)
       {
        for (int l_Index = 0; l_Index < CipeControlPanel::m_EditFieldSize; ++l_Index) if (Verify (CipeControlPanel::m_EditFieldID [l_Index]))
           {
            GetDlgItemText (m_DialogWindow, CipeControlPanel::m_EditFieldID [l_Index], l_Buffer, sizeof (l_Buffer));
            string l_Value (l_Buffer);

            switch (CipeControlPanel::m_EditFieldID [l_Index])
               {
                case CIPEPANEL_LOCAL_IP:        l_Peer->SetLocalIP        (l_Value); break;
                case CIPEPANEL_LOCAL_PORT:      l_Peer->SetLocalPort      (l_Value); break;
                case CIPEPANEL_REMOTE_IP:       l_Peer->SetPeerIP         (l_Value); break;
                case CIPEPANEL_REMOTE_PORT:     l_Peer->SetPeerPort       (l_Value); break;
                case CIPEPANEL_REMOTE_PTP:      l_Peer->SetPeerPTP        (l_Value); break;
                case CIPEPANEL_STATIC_KEY:      l_Peer->SetTextKey        (l_Value); break;
                case CIPEPANEL_CIPHER:          l_Peer->SetEncryptionType (l_Value); break;
                case CIPEPANEL_KEY_TIMEOUT:     l_Peer->SetKeyTimeout     (l_Value); break;
                case CIPEPANEL_STARTUP_SCRIPT:  l_Peer->SetStartupScript  (l_Value); break;
                case CIPEPANEL_SHUTDOWN_SCRIPT: l_Peer->SetShutdownScript (l_Value); break;
               }
           }

        l_Peer->SetEnabled ((BOOL) IsDlgButtonChecked (m_DialogWindow, CIPEPANEL_CHECK_BOX));
       }
   }

//=====================================================================================
//                                Utility Routines
//=====================================================================================
BOOL CipeControlPanel::Verify (UINT p_Control)
   {
    unsigned long l_Value, l_Index = 0, l_Flag = 1;
    static char l_Buffer [1024];
    BOOL l_Return = TRUE;

    switch (p_Control)
       {
        case CIPEPANEL_CIPHER:
           {
            GetDlgItemText (m_DialogWindow, p_Control, l_Buffer, sizeof (l_Buffer));
            l_Return = FALSE;

            for (l_Index = 0; l_Index < (sizeof (m_CipherList) / sizeof (char *)) && m_CipherList [l_Index] && ! l_Return; ++l_Index)
               {
                l_Return = (BOOL) (! strcmp (m_CipherList [l_Index], l_Buffer));
               }

            break;
           }

        case CIPEPANEL_LOCAL_IP:
        case CIPEPANEL_REMOTE_IP:
        case CIPEPANEL_REMOTE_PTP:
           {
            GetDlgItemText (m_DialogWindow, p_Control, l_Buffer, sizeof (l_Buffer));

            if (inet_addr (l_Buffer) == INADDR_NONE) for (l_Index = 0; l_Return && l_Index < strlen (l_Buffer); ++l_Index) l_Return =
               (
                (l_Buffer [l_Index] >= '0' && l_Buffer [l_Index] <= '9') ||
                (l_Buffer [l_Index] >= 'a' && l_Buffer [l_Index] <= 'z') ||
                (l_Buffer [l_Index] >= 'A' && l_Buffer [l_Index] <= 'Z') ||
                (l_Buffer [l_Index] == '_') ||
                (l_Buffer [l_Index] == '-') ||
                (l_Buffer [l_Index] == '.')
               );

            break;
           }

        case CIPEPANEL_LOCAL_PORT:
        case CIPEPANEL_REMOTE_PORT:
           {
            l_Value = GetDlgItemInt (m_DialogWindow, p_Control, &l_Return, FALSE);
            l_Return = (l_Return && l_Value <= 32767);
            break;
           }

        case CIPEPANEL_STATIC_KEY:
           {
            GetDlgItemText (m_DialogWindow, p_Control, l_Buffer, sizeof (l_Buffer));

            for (l_Index = 0; l_Return && l_Index < strlen (l_Buffer); ++l_Index) l_Return =
               (
                (l_Buffer [l_Index] >= '0' && l_Buffer [l_Index] <= '9') ||
                (l_Buffer [l_Index] >= 'a' && l_Buffer [l_Index] <= 'f') ||
                (l_Buffer [l_Index] >= 'A' && l_Buffer [l_Index] <= 'F')
               );

            break;
           }

        case CIPEPANEL_KEY_TIMEOUT:
           {
            l_Value = GetDlgItemInt (m_DialogWindow, p_Control, &l_Return, FALSE);
            l_Return = (l_Return && l_Value <= 1800);
            break;
           }

        case CIPEPANEL_STARTUP_SCRIPT:
        case CIPEPANEL_SHUTDOWN_SCRIPT:
        default:
           {
            l_Return = TRUE;
            break;
           }
       }

    if (l_Return)
       {
        m_IsModified = TRUE;
       }

    return l_Return;
   }

void CipeControlPanel::StatusMsg (const char *p_Message)
   {
    SetDlgItemText (m_DialogWindow, CIPEPANEL_STATUS, p_Message ? p_Message : "Ready");
   }

void CipeControlPanel::FormatCipherList()
   {
    for (int l_Index = 0; l_Index < (sizeof (m_CipherList) / sizeof (char *)) && m_CipherList [l_Index]; ++l_Index)
       {
        ComboBox_AddString (GetDlgItem (m_DialogWindow, CIPEPANEL_CIPHER), m_CipherList [l_Index]);
       }
   }

//=====================================================================================
//                                        END
//=====================================================================================
CipeControlPanel CipeControlPanelInstance;
